import asyncio
import urllib.parse
import webbrowser
from typing import Dict, Optional, Tuple

from pixelarraythirdparty.client import AsyncClient


class GoogleLogin(AsyncClient):
    """
    Google OAuth2 登录客户端

    使用示例:
    ```
    google = GoogleLogin(api_key="your_api_key")
    user_info, success = await google.login()
    ```
    """

    def __init__(self, api_key: str):
        super().__init__(api_key)

    async def _get_auth_url(self) -> Tuple[Optional[Dict[str, str]], bool]:
        data, success = await self._request(
            "POST", "/api/unified-login/google/auth-url"
        )
        if not success:
            return None, False
        auth_url = data.get("auth_url")
        if not auth_url:
            return None, False
        return data, True

    async def login(self, timeout: int = 180) -> Tuple[Dict, bool]:
        """
        仿 Supabase CLI 的一键登录流程：打开浏览器完成授权，
        终端端轮询等待登录结果

        :param timeout: 等待用户完成授权的超时时间（秒）
        """
        auth_data, success = await self._get_auth_url()
        if not success or not auth_data:
            return {}, False

        auth_url = auth_data.get("auth_url")
        state = auth_data.get("state") or self._extract_state(auth_url)

        if not auth_url or not state:
            return {}, False

        webbrowser.open(auth_url, new=2)

        return await self._wait_for_google_login(state, timeout)

    def _extract_state(self, auth_url: Optional[str]) -> Optional[str]:
        if not auth_url:
            return None
        try:
            parsed = urllib.parse.urlparse(auth_url)
            query = urllib.parse.parse_qs(parsed.query)
            values = query.get("state")
            if values:
                return values[0]
        except Exception:
            return None
        return None

    async def _wait_for_google_login(
        self, state: str, timeout: int
    ) -> Tuple[Dict, bool]:
        interval = 2
        total_checks = max(1, timeout // interval) if timeout > 0 else 1

        for _ in range(total_checks):
            status, response = await self._request_raw(
                "POST",
                "/api/unified-login/google/wait-login",
                json={"state": state},
            )

            if status == 200 and response.get("success") is True:
                return response.get("data", {}), True

            if status in (400, 408):
                break

            await asyncio.sleep(interval)

        return {}, False


class WechatLogin(AsyncClient):
    """
    微信 OAuth2 登录客户端

    使用示例:
    ```
    wechat = WechatLogin(api_key="your_api_key")
    user_info, success = await wechat.login()
    ```
    """

    def __init__(self, api_key: str):
        super().__init__(api_key)

    async def _get_auth_url(self) -> Tuple[Optional[Dict[str, str]], bool]:
        data, success = await self._request(
            "POST", "/api/unified-login/wechat/auth-url"
        )
        if not success:
            return None, False
        auth_url = data.get("auth_url")
        if not auth_url:
            return None, False
        return data, True

    async def login(self, timeout: int = 180) -> Tuple[Dict, bool]:
        """
        仿 Supabase CLI 的一键登录流程：打开浏览器完成授权，
        终端端轮询等待登录结果

        :param timeout: 等待用户完成授权的超时时间（秒）
        """
        auth_data, success = await self._get_auth_url()
        if not success or not auth_data:
            return {}, False

        auth_url = auth_data.get("auth_url")
        state = auth_data.get("state") or self._extract_state(auth_url)

        if not auth_url or not state:
            return {}, False

        webbrowser.open(auth_url, new=2)

        return await self._wait_for_wechat_login(state, timeout)

    def _extract_state(self, auth_url: Optional[str]) -> Optional[str]:
        if not auth_url:
            return None
        try:
            parsed = urllib.parse.urlparse(auth_url)
            query = urllib.parse.parse_qs(parsed.query)
            values = query.get("state")
            if values:
                return values[0]
        except Exception:
            return None
        return None

    async def _wait_for_wechat_login(
        self, state: str, timeout: int
    ) -> Tuple[Dict, bool]:
        interval = 2
        total_checks = max(1, timeout // interval) if timeout > 0 else 1

        for _ in range(total_checks):
            status, response = await self._request_raw(
                "POST",
                "/api/unified-login/wechat/wait-login",
                json={"state": state},
            )

            if status == 200 and response.get("success") is True:
                return response.get("data", {}), True

            if status in (400, 408):
                break

            await asyncio.sleep(interval)

        return {}, False


class GitHubLogin(AsyncClient):
    """
    GitHub OAuth2 登录客户端

    使用示例:
    ```
    github = GitHubLogin(api_key="your_api_key")
    user_info, success = await github.login()
    ```
    """

    def __init__(self, api_key: str):
        super().__init__(api_key)

    async def _get_auth_url(self) -> Tuple[Optional[Dict[str, str]], bool]:
        data, success = await self._request(
            "POST", "/api/unified-login/github/auth-url"
        )
        if not success:
            return None, False
        auth_url = data.get("auth_url")
        if not auth_url:
            return None, False
        return data, True

    async def login(self, timeout: int = 180) -> Tuple[Dict, bool]:
        """
        仿 Supabase CLI 的一键登录流程：打开浏览器完成授权，
        终端端轮询等待登录结果

        :param timeout: 等待用户完成授权的超时时间（秒）
        """
        auth_data, success = await self._get_auth_url()
        if not success or not auth_data:
            return {}, False

        auth_url = auth_data.get("auth_url")
        state = auth_data.get("state") or self._extract_state(auth_url)

        if not auth_url or not state:
            return {}, False

        webbrowser.open(auth_url, new=2)

        return await self._wait_for_github_login(state, timeout)

    def _extract_state(self, auth_url: Optional[str]) -> Optional[str]:
        if not auth_url:
            return None
        try:
            parsed = urllib.parse.urlparse(auth_url)
            query = urllib.parse.parse_qs(parsed.query)
            values = query.get("state")
            if values:
                return values[0]
        except Exception:
            return None
        return None

    async def _wait_for_github_login(
        self, state: str, timeout: int
    ) -> Tuple[Dict, bool]:
        interval = 2
        total_checks = max(1, timeout // interval) if timeout > 0 else 1

        for _ in range(total_checks):
            status, response = await self._request_raw(
                "POST",
                "/api/unified-login/github/wait-login",
                json={"state": state},
            )

            if status == 200 and response.get("success") is True:
                return response.get("data", {}), True

            if status in (400, 408):
                break

            await asyncio.sleep(interval)

        return {}, False


class UnifiedLogin(AsyncClient):
    def __init__(self, api_key: str):
        super().__init__(api_key)

    async def login(self, provider: str, timeout: int = 180) -> Tuple[Dict, bool]:
        if provider == "google":
            return await GoogleLogin(self.api_key).login(timeout)
        elif provider == "wechat":
            return await WechatLogin(self.api_key).login(timeout)
        elif provider == "github":
            return await GitHubLogin(self.api_key).login(timeout)
        else:
            return {}, False
