import contextlib
import uuid as py_uuid
from collections.abc import Callable, Generator, Iterator
from typing import Protocol

import pytest
import sqlalchemy as sa
import sqlalchemy.orm as sa_orm
import textcase
from fastapi import FastAPI
from iker.common.utils.dbutils import ConnectionMaker, Dialects, Drivers
from iker.common.utils.dbutils import make_scheme
from iker.common.utils.jsonutils import JsonType
from iker.common.utils.jsonutils import json_difference, json_reformat
from sqlmodel import SQLModel

from plexus.common.utils.datautils import compute_vin_code_check_digit

__all__ = [
    "generate_dummy_uuid_str",
    "generate_dummy_uuid",
    "generate_dummy_vin_code",
    "case_insensitive_json_compare",
    "patched_value_factory",
    "patched_postgresql_session_maker",
]


def generate_dummy_uuid_str(*nums: int) -> str:
    if len(nums) > 8:
        raise ValueError("a maximum of 8 integers can be provided")
    if not all(0 <= num <= 0xFFFF for num in nums):
        raise ValueError("all integers must be in the range 0 to 65535 (0xFFFF)")
    i0, i1, i2, i3, i4, i5, i6, i7 = list(nums) + [0] * (8 - len(nums))
    return f"{i0:04x}{i1:04x}-{i2:04x}-{i3:04x}-{i4:04x}-{i5:04x}{i6:04x}{i7:04x}"


def generate_dummy_uuid(*nums: int) -> py_uuid.UUID:
    return py_uuid.UUID(generate_dummy_uuid_str(*nums))


def generate_dummy_vin_code(*nums: int) -> str:
    if len(nums) > 4:
        raise ValueError("a maximum of 4 integers can be provided")
    if not all(0 <= num <= 9999 for num in nums):
        raise ValueError("all integers must be in the range 0 to 9999")
    i0, i1, i2, i3 = list(nums) + [0] * (4 - len(nums))
    unchecked_vin_code = f"{i0:04d}{i1:04d}0{i2:04d}{i3:04d}"
    check_digit = compute_vin_code_check_digit(unchecked_vin_code)
    return unchecked_vin_code[:8] + check_digit + unchecked_vin_code[-8:]


def case_insensitive_json_compare(a: JsonType, b: JsonType, *, print_diff_messages: bool = True) -> bool:
    identical = True
    for node_path, diff_message in json_difference(json_reformat(a, key_formatter=textcase.camel),
                                                   json_reformat(b, key_formatter=textcase.camel),
                                                   []):
        if print_diff_messages:
            print(node_path, diff_message)
        identical = False
    return identical


class PatcherProtocol[T](Protocol):

    def value_factory(self) -> Generator[T, None, None]:
        ...

    @contextlib.contextmanager
    def use(self, value: T) -> Generator[T, None, None]:
        ...


def patched_value_factory[T](
    app: FastAPI,
    value_factory: Callable[[], Generator[T, None, None]],
) -> Callable[[], Generator[PatcherProtocol[T], None, None]]:
    """
    Create a pytest fixture that allows patching a FastAPI dependency value factory.

    :param app: FastAPI application instance.
    :param value_factory: Original value factory to be overridden.
    :return:
    """

    class Patcher(object):
        value: T | None = None

        def value_factory(self) -> Generator[T, None, None]:
            if self.value is not None:
                yield self.value
            else:
                yield from value_factory()

        @contextlib.contextmanager
        def use(self, value: T) -> Generator[T, None, None]:
            self.value = value
            try:
                yield self.value
            finally:
                self.value = None

    @pytest.fixture(scope="function")
    def fixture() -> Generator[PatcherProtocol[T], None, None]:
        patcher = Patcher()
        app.dependency_overrides[value_factory] = patcher.value_factory
        try:
            yield patcher
        finally:
            app.dependency_overrides[value_factory] = value_factory

    return fixture


def patched_postgresql_session_maker(
    postgresql_test_fixture_name: str,
    base_model: type[SQLModel],
    app: FastAPI,
    session_maker: Callable[[], Generator[sa_orm.Session]],
) -> Callable[[pytest.FixtureRequest], Generator[Callable[[], Iterator[sa_orm.Session]]]]:
    """
    Create a pytest fixture that provides a patched session maker for PostgreSQL tests.

    :param postgresql_test_fixture_name: Name of the PostgreSQL test fixture to use.
    :param base_model: The base SQLModel model containing metadata.
    :param app: FastAPI application instance.
    :param session_maker: Original session maker to be overridden.
    :return: A pytest fixture function.
    """

    @pytest.fixture(scope="function")
    def fixture(request: pytest.FixtureRequest):
        postgresql_test_fixture = request.getfixturevalue(postgresql_test_fixture_name)

        cm = ConnectionMaker.create(
            scheme=make_scheme(Dialects.postgresql, Drivers.psycopg),
            host=postgresql_test_fixture.info.host,
            port=postgresql_test_fixture.info.port,
            username=postgresql_test_fixture.info.user,
            password="",
            database=postgresql_test_fixture.info.dbname,
        )

        def make_session_maker() -> Iterator[sa_orm.Session]:
            with cm.make_session(autocommit=False, autoflush=False) as db:
                try:
                    # language=postgresql
                    db.execute(sa.sql.text("SET TIMEZONE TO 'UTC'"))
                    yield db
                finally:
                    db.close()

        app.dependency_overrides[session_maker] = make_session_maker

        with cm.make_session() as session:
            for table in base_model.metadata.tables.values():
                session.execute(sa.schema.CreateSchema(table.schema, if_not_exists=True))
            session.commit()
        base_model.metadata.create_all(cm.engine)

        try:
            yield make_session_maker
        finally:
            base_model.metadata.drop_all(cm.engine)
            with cm.make_session() as session:
                for table in base_model.metadata.tables.values():
                    session.execute(sa.schema.DropSchema(table.schema, if_exists=True))
                session.commit()

            app.dependency_overrides[session_maker] = session_maker

    return fixture
