import importlib.metadata
import os
from collections.abc import Iterable

try:
    __version__ = importlib.metadata.version("plexus-python-protobuf-setup")
except importlib.metadata.PackageNotFoundError:
    __version__ = "unknown"


def compile_protos(
    out_dir: str | os.PathLike[str],
    proto_dirs: Iterable[str | os.PathLike[str]],
    include_dirs: Iterable[str | os.PathLike[str]],
    *,
    package_root_dir: str | os.PathLike[str] | None = None,
    descriptor_path: str | os.PathLike[str] | None = None,
    with_grpc: bool = False,
) -> None:
    """
    Compiles .proto files in the specified directory

    :param out_dir: Directory where the compiled files will be placed.
    :param proto_dirs: Directories containing ``.proto`` files which are used for generating target source codes.
    :param include_dirs: Directories containing ``.proto`` files which are used only in compiling.
    :param package_root_dir: Root directory to create ``__init__.py`` files for packages. If ``None``, uses ``out_dir``.
    :param descriptor_path: Optional path to output the descriptor set file.
    :param with_grpc: If ``True``, also generates gRPC Python code.
    :raises FileNotFoundError: If any of the proto directories and include directories does not exist.
    """
    import glob
    import os
    import subprocess
    import sys

    out_dir = os.path.abspath(out_dir)

    proto_dirs = [os.path.abspath(d) for d in proto_dirs]
    for proto_dir in proto_dirs:
        if not os.path.exists(proto_dir):
            raise FileNotFoundError(f"proto directory '{proto_dir}' does not exist")

    include_dirs = [os.path.abspath(d) for d in include_dirs]
    for include_dir in include_dirs:
        if not os.path.exists(include_dir):
            raise FileNotFoundError(f"include directory '{include_dir}' does not exist")

    if package_root_dir is not None:
        package_root_dir = os.path.abspath(package_root_dir)
    else:
        package_root_dir = out_dir
    if os.path.commonpath([out_dir, package_root_dir]) != out_dir:
        raise ValueError("'package_root_dir' must be inside 'out_dir' or equal to it")

    if descriptor_path is not None:
        descriptor_path = os.path.abspath(descriptor_path)

    proto_targets: list[tuple[str, str]] = []
    for proto_dir in proto_dirs:
        for proto_file in glob.glob(os.path.join(proto_dir, "**", "*.proto"), recursive=True):
            proto_targets.append((proto_dir, proto_file))

    if not proto_targets:
        print("No .proto files found")
        return

    cmd = [sys.executable, "-m", "grpc_tools.protoc"]

    for include_dir in include_dirs:
        cmd.extend(("--proto_path", include_dir))

    cmd.extend(("--python_out", out_dir))
    cmd.extend(("--pyi_out", out_dir))
    if with_grpc:
        cmd.extend(("--grpc_python_out", out_dir))

    if descriptor_path:
        cmd.extend(("--descriptor_set_out",
                    descriptor_path,
                    "--include_imports",
                    "--include_source_info",
                    "--retain_options",
                    ))
    cmd.extend(proto_file for _, proto_file in proto_targets)

    if not os.path.exists(out_dir):
        os.makedirs(out_dir)

    subprocess.check_call(cmd)

    # Create `__init__.py` files to make packages, starting from `package_root_dir`
    for proto_dir, proto_file in proto_targets:
        relative_path = os.path.relpath(proto_file, start=proto_dir)
        dir_path = os.path.dirname(os.path.join(out_dir, relative_path))

        while True:
            if os.path.commonpath([dir_path, package_root_dir]) != package_root_dir:
                break

            init_py_file = os.path.join(dir_path, "__init__.py")
            if not os.path.exists(init_py_file):
                with open(init_py_file, "w", encoding="utf-8") as fh:
                    fh.write("# Generated by plexus.protobuf.setup.compile_protos")

            parent_dir = os.path.dirname(dir_path)
            if dir_path == parent_dir:
                break
            dir_path = parent_dir
