

import dessia_common as dc
from matplotlib.colors import hsv_to_rgb

class Color(dc.DessiaObject):
    def __init__(self, red:float, green:float, blue:float):
        self.red = red
        self.green = green
        self.blue = blue

        self.rgb = (red, green, blue)

    @classmethod
    def from_hex(cls, hex_code):
        hex_code = hex_code.replace('#', '')
        r, g, b = (int(hex_code[i:i + 2], 16)/255. for i in (0, 2, 4))
        return cls(r, g, b)
    @classmethod
    def from_hsv(cls, h:float, s:float, v:float):
        return cls(hsv_to_rgb(h, s ,v))

    def __str__(self):
        return 'rgb({},{},{})'.format(round(self.red*255),
                                      round(self.green*255),
                                      round(self.blue*255))

    def to_dict(self):
        return str(self)

    @classmethod
    def dict_to_object(cls, d):
        if not d.startswith('rgb('):
            raise ValueError('Color should be string starting with rgb(')
        return cls(*(int(v)/255. for v in d[4:-1].split(',')))

RED = Color.dict_to_object('rgb(247,0,0)')
LIGHTRED = Color.dict_to_object('rgb(237,128,128)')
BLUE = Color.dict_to_object('rgb(0,19,254)')
LIGHTBLUE = Color.dict_to_object('rgb(195,230,252)')
GREEN = Color.dict_to_object('rgb(0,193,18)')
LIGHTGREEN = Color.dict_to_object('rgb(137,232,146)')
YELLOW = Color.dict_to_object('rgb(244,255,0)')
LIGHTYELLOW = Color.dict_to_object('rgb(249,255,123)')
ORANGE = Color.dict_to_object('rgb(255,135,0)')
LIGHTORANGE = Color.dict_to_object('rgb(255,175,96)')
CYAN = Color.dict_to_object('rgb(19,240,240)')
LIGHTCYAN = Color.dict_to_object('rgb(144,247,247)')
ROSE = Color.dict_to_object('rgb(255,105,180)')
LIGHTROSE = Color.dict_to_object('rgb(255,192,203)')
VIOLET = Color.dict_to_object('rgb(238,130,238)')
LIGHTVIOLET = Color.dict_to_object('rgb(234,165,246)')
WHITE = Color.dict_to_object('rgb(255,255,255)')
BLACK = Color.dict_to_object('rgb(0,0,0)')
BROWN = Color.dict_to_object('rgb(205,143,64)')
LIGHTBROWN = Color.dict_to_object('rgb(222,184,135)')
GREY = Color.dict_to_object('rgb(169,169,169)')
LIGHTGREY = Color.dict_to_object('rgb(211,211,211)')
LIGHTSKYBLUE = Color.dict_to_object('rgb(135,206,250)')

# Pantone
EGRET = Color.from_hex('#f3ece0')
SNOW_WHITE = Color.from_hex('#f2f0eb')
BRIGHT_WHITE = Color.from_hex('#f4f5f0')
CLOUD_DANCER = Color.from_hex('#f0eee9')
GARDENIA = Color.from_hex('#f1e8df')
MARSHMALLOW = Color.from_hex('#f0eee4')
BLANC_DE_BLANC = Color.from_hex('#e7e9e7')
PRISTINE = Color.from_hex('#f2e8da')
WHISPER_WHITE = Color.from_hex('#ede6db')
WHITE_ASPARAGUS = Color.from_hex('#e1dbc8')
BIRCH = Color.from_hex('#ddd5c7')
TURTLEDOVE = Color.from_hex('#ded7c8')
BONE_WHITE = Color.from_hex('#d7d0c0')
SILVER_BIRCH = Color.from_hex('#d2cfc4')
VANILLA_ICE = Color.from_hex('#f0eada')
PAPYRUS = Color.from_hex('#f5edd6')
ANTIQUE_WHITE = Color.from_hex('#ede3d2')
WINTER_WHITE = Color.from_hex('#f5ecd2')
CLOUD_CREAM = Color.from_hex('#e6ddc5')
ANGORA = Color.from_hex('#dfd1bb')
SEEDPEARL = Color.from_hex('#e6dac4')
VANILLA_CUSTARD = Color.from_hex('#f3e0be')
ALMOND_OIL = Color.from_hex('#f4efc1')
ALABASTER_GLEAM = Color.from_hex('#f0debd')
VANILLA = Color.from_hex('#f4e1c1')
RUTABAGA = Color.from_hex('#ecddbe')
BANANA_CREPE = Color.from_hex('#e7d3ad')
ITALIAN_STRAW = Color.from_hex('#e7d1a1')
WHITECAP_GRAY = Color.from_hex('#e0d5c6')
FOG = Color.from_hex('#d0c5b1')
WHITE_SWAN = Color.from_hex('#e4d7c5')
SANDSHELL = Color.from_hex('#d8ccbb')
TAPIOCA = Color.from_hex('#dccdbc')
CREME_BRULEE = Color.from_hex('#dbccb5')
PARCHMENT = Color.from_hex('#dfd1be')
SHEER_PINK = Color.from_hex('#f6e5db')
DEW = Color.from_hex('#eeded1')
POWDER_PUFF = Color.from_hex('#f3e0d6')
PEARLED_IVORY = Color.from_hex('#f0dfcc')
WHITE_SMOKE = Color.from_hex('#eddcc9')
ECRU = Color.from_hex('#f3dfca')
NAVAJO = Color.from_hex('#efdcc3')
ALMOST_MAUVE = Color.from_hex('#e7dcd9')
DELICACY = Color.from_hex('#f5e3e2')
PETAL_PINK = Color.from_hex('#f2e2e0')
BRIDAL_BLUSH = Color.from_hex('#eee2dd')
CREAM_PINK = Color.from_hex('#f6e4d9')
ANGEL_WING = Color.from_hex('#f3dfd7')
PASTEL_PARCHMENT = Color.from_hex('#e5d9d3')
STAR_WHITE = Color.from_hex('#efefe8')
LILY_WHITE = Color.from_hex('#e2e2da')
VAPOROUS_GRAY = Color.from_hex('#dfddd7')
SUMMER_SHOWER = Color.from_hex('#e5ebe3')
ICE = Color.from_hex('#e0e4d9')
FROST = Color.from_hex('#dde2d6')
ICICLE = Color.from_hex('#dadcd0')
BIT_OF_BLUE = Color.from_hex('#e2eaeb')
MYSTIC_BLUE = Color.from_hex('#e1e3de')
BLUEWASH = Color.from_hex('#e2e6e0')
SPA_BLUE = Color.from_hex('#d3dedf')
LIGHTEST_SKY = Color.from_hex('#e4eadf')
HINT_OF_MINT = Color.from_hex('#d8e8e6')
MURMUR = Color.from_hex('#d2d8d2')
BARELY_BLUE = Color.from_hex('#dde0df')
BLUE_BLUSH = Color.from_hex('#d6dbd9')
ZEPHYR_BLUE = Color.from_hex('#d3d9d1')
BLUE_FLOWER = Color.from_hex('#d0d9d4')
SPROUT_GREEN = Color.from_hex('#cbd7d2')
BILLOWING_SAIL = Color.from_hex('#d8e7e7')
HUSHED_GREEN = Color.from_hex('#d8e9e5')
LAMBS_WOOL = Color.from_hex('#e5d0b1')
WINTER_WHEAT = Color.from_hex('#dfc09f')
SUMMER_MELON = Color.from_hex('#ead3ae')
CHAMOMILE = Color.from_hex('#e8d0a7')
CORNHUSK = Color.from_hex('#f2d6ae')
APRICOT_GELATO = Color.from_hex('#f5d7af')
BISCOTTI = Color.from_hex('#dac7ab')
ASPARAGUS_GREEN = Color.from_hex('#d2cdb4')
OYSTER_WHITE = Color.from_hex('#d2caaf')
PUTTY = Color.from_hex('#d4cab0')
MOTH = Color.from_hex('#d2cbaf')
WOOD_ASH = Color.from_hex('#d7cab0')
GRAVEL = Color.from_hex('#cbbfa2')
PALE_KHAKI = Color.from_hex('#bfaf92')
LIGHT_GRAY = Color.from_hex('#dad8c9')
SILVER_GREEN = Color.from_hex('#d7d7c7')
PELICAN = Color.from_hex('#c1bcac')
OVERCAST = Color.from_hex('#c3bdab')
TIDAL_FOAM = Color.from_hex('#bfb9a3')
AGATE_GRAY = Color.from_hex('#b1b09f')
ALFALFA = Color.from_hex('#b7b59f')
CASTLE_WALL = Color.from_hex('#c8c1ab')
OYSTER_GRAY = Color.from_hex('#cbc1ae')
CEMENT = Color.from_hex('#c4b6a6')
SPRAY_GREEN = Color.from_hex('#aea692')
EUCALYPTUS = Color.from_hex('#b1a992')
TWILL = Color.from_hex('#a79b82')
OLIVE_GRAY = Color.from_hex('#a6997a')
CHINCHILLA = Color.from_hex('#9c8e7b')
SENECA_ROCK = Color.from_hex('#9a927f')
LAUREL_OAK = Color.from_hex('#918c7e')
CORIANDER = Color.from_hex('#938772')
DUNE = Color.from_hex('#998978')
LEAD_GRAY = Color.from_hex('#8a7963')
COVERT_GREEN = Color.from_hex('#80765f')
OXFORD_TAN = Color.from_hex('#b8a99a')
PLAZA_TAUPE = Color.from_hex('#aea393')
TUFFET = Color.from_hex('#a59788')
SILVER_MINK = Color.from_hex('#9f8d7c')
TIMBER_WOLF = Color.from_hex('#8d8070')
TAUPE_GRAY = Color.from_hex('#8e7c71')
PINE_BARK = Color.from_hex('#827064')
PUMICE_STONE = Color.from_hex('#cac2b9')
SIMPLY_TAUPE = Color.from_hex('#ad9f93')
ALUMINUM = Color.from_hex('#9f9586')
COBBLESTONE = Color.from_hex('#a89a8e')
BRINDLE = Color.from_hex('#82776b')
WALNUT = Color.from_hex('#776a5f')
BUNGEE_CORD = Color.from_hex('#696156')
OATMEAL = Color.from_hex('#cbc3b4')
MOONBEAM = Color.from_hex('#cdc6bd')
RAINY_DAY = Color.from_hex('#cfc8bd')
GRAY_MORN = Color.from_hex('#cabeb5')
PEYOTE = Color.from_hex('#c5bbae')
FEATHER_GRAY = Color.from_hex('#b8ad9e')
GOAT = Color.from_hex('#a89a91')
WHITE_SAND = Color.from_hex('#dbd5d1')
SILVER_GRAY = Color.from_hex('#c1b7b0')
CHATEAU_GRAY = Color.from_hex('#bbb1a8')
STRING = Color.from_hex('#aa9f96')
ATMOSPHERE = Color.from_hex('#a89c94')
MOON_ROCK = Color.from_hex('#958b84')
FUNGI = Color.from_hex('#8f8177')
SILVER_LINING = Color.from_hex('#bdb6ab')
MOONSTRUCK = Color.from_hex('#c2beb6')
PUSSYWILLOW_GRAY = Color.from_hex('#aeaca1')
LONDON_FOG = Color.from_hex('#a29e92')
ROCK_RIDGE = Color.from_hex('#918c86')
MOON_MIST = Color.from_hex('#80817d')
CASTOR_GRAY = Color.from_hex('#646762')
GLACIER_GRAY = Color.from_hex('#c5c6c7')
LUNAR_ROCK = Color.from_hex('#c5c5c5')
DAWN_BLUE = Color.from_hex('#cacccb')
GRAY_VIOLET = Color.from_hex('#bbbcbc')
VAPOR_BLUE = Color.from_hex('#bebdbd')
HIGH_RISE = Color.from_hex('#aeb2b5')
LIMESTONE = Color.from_hex('#989a98')
SILVER_CLOUD = Color.from_hex('#beb7b0')
DOVE = Color.from_hex('#b3ada7')
FLINT_GRAY = Color.from_hex('#a09c98')
DRIZZLE = Color.from_hex('#a09f9c')
ELEPHANT_SKIN = Color.from_hex('#8f8982')
CINDER = Color.from_hex('#8a7e78')
STEEPLE_GRAY = Color.from_hex('#827e7c')
METAL = Color.from_hex('#babfbc')
BLUE_FOX = Color.from_hex('#b9bcb6')
STORM_GRAY = Color.from_hex('#b5bab6')
PIGEON = Color.from_hex('#a9afaa')
MIRAGE_GRAY = Color.from_hex('#abafae')
PURITAN_GRAY = Color.from_hex('#a8b0ae')
WROUGHT_IRON = Color.from_hex('#999e98')
OPAL_GRAY = Color.from_hex('#a49e9e')
WILD_DOVE = Color.from_hex('#8b8c89')
NEUTRAL_GRAY = Color.from_hex('#8e918f')
GARGOYLE = Color.from_hex('#686767')
SMOKED_PEARL = Color.from_hex('#656466')
SEDONA_SAGE = Color.from_hex('#686d6c')
GUNMETAL = Color.from_hex('#5c5d5b')
WIND_CHIME = Color.from_hex('#cac5c2')
PALOMA = Color.from_hex('#9f9c99')
CHARCOAL_GRAY = Color.from_hex('#6c6868')
STEEL_GRAY = Color.from_hex('#726f70')
PEWTER = Color.from_hex('#666564')
CASTLEROCK = Color.from_hex('#5f5e62')
NINE_IRON = Color.from_hex('#46434a')
ASH = Color.from_hex('#a09998')
CLOUDBURST = Color.from_hex('#837f7f')
FROST_GRAY = Color.from_hex('#848283')
EXCALIBUR = Color.from_hex('#676168')
DARK_GULL_GRAY = Color.from_hex('#625d5d')
RABBIT = Color.from_hex('#5f575c')
SHALE = Color.from_hex('#4a3f41')
FOSSIL = Color.from_hex('#806f63')
MAJOR_BROWN = Color.from_hex('#5b5149')
CHOCOLATE_CHIP = Color.from_hex('#685a4e')
CANTEEN = Color.from_hex('#5e5347')
STONE_GRAY = Color.from_hex('#685e4f')
CAPERS = Color.from_hex('#695e4b')
BEECH = Color.from_hex('#5b4f3b')
TARMAC = Color.from_hex('#5a5348')
WREN = Color.from_hex('#4a4139')
BLACK_OLIVE = Color.from_hex('#48413b')
BELUGA = Color.from_hex('#4a4843')
BLACK_INK = Color.from_hex('#44413c')
PEAT = Color.from_hex('#3b3a36')
JET_SET = Color.from_hex('#262c2a')
IRON = Color.from_hex('#736460')
PLUM_KITTEN = Color.from_hex('#625b5c')
TURKISH_COFFEE = Color.from_hex('#483f39')
BLACK_COFFEE = Color.from_hex('#3b302f')
AFTER_DARK = Color.from_hex('#3c3535')
LICORICE = Color.from_hex('#3a3536')
RAVEN = Color.from_hex('#413e3d')
JET_BLACK = Color.from_hex('#2d2c2f')
PHANTOM = Color.from_hex('#39373b')
STRETCH_LIMO = Color.from_hex('#2b2c30')
MOONLESS_NIGHT = Color.from_hex('#2f2d30')
CAVIAR = Color.from_hex('#292a2d')
PIRATE_BLACK = Color.from_hex('#363838')
ANTHRACITE = Color.from_hex('#28282d')
VANILLA_CREAM = Color.from_hex('#f4d8c6')
DAWN = Color.from_hex('#ebd2b7')
GRAY_SAND = Color.from_hex('#e5ccaf')
AUTUMN_BLONDE = Color.from_hex('#eed0ae')
APRICOT_ILLUSION = Color.from_hex('#e2c4a6')
MELLOW_BUFF = Color.from_hex('#d8b998')
SHEEPSKIN = Color.from_hex('#dab58f')
ALMOND_BUFF = Color.from_hex('#ccb390')
BEIGE = Color.from_hex('#d5ba98')
SAND = Color.from_hex('#cca67f')
LATTE = Color.from_hex('#c5a582')
TAN = Color.from_hex('#b69574')
DOE = Color.from_hex('#b98e68')
INDIAN_TAN = Color.from_hex('#ad8567')
SAFARI = Color.from_hex('#baaa91')
CANDIED_GINGER = Color.from_hex('#bfa387')
WARM_SAND = Color.from_hex('#c5ae91')
CUBAN_SAND = Color.from_hex('#c1a68d')
NOUGAT = Color.from_hex('#b69885')
NATURAL = Color.from_hex('#aa907d')
NOMAD = Color.from_hex('#b49f89')
FROZEN_DEW = Color.from_hex('#d8cfb2')
BLEACHED_SAND = Color.from_hex('#daccb4')
PEBBLE = Color.from_hex('#cab698')
CROISSANT = Color.from_hex('#c4ab86')
INCENSE = Color.from_hex('#af9a7e')
CORNSTALK = Color.from_hex('#a9947a')
TANNIN = Color.from_hex('#a68a6d')
GREEN_HAZE = Color.from_hex('#cac4a4')
MOJAVE_DESERT = Color.from_hex('#c7b595')
TAOS_TAUPE = Color.from_hex('#bfa77f')
LARK = Color.from_hex('#b89b72')
KELP = Color.from_hex('#988467')
ANTIQUE_BRONZE = Color.from_hex('#907954')
DULL_GOLD = Color.from_hex('#8a6f48')
BROWN_SUGAR = Color.from_hex('#a17249')
CHIPMUNK = Color.from_hex('#976f4c')
TOBACCO_BROWN = Color.from_hex('#9a7352')
BISON = Color.from_hex('#6e4f3a')
MONKS_ROBE = Color.from_hex('#704822')
DACHSHUND = Color.from_hex('#704f37')
TOFFEE = Color.from_hex('#755139')
AZTEC = Color.from_hex('#7a5747')
COCOA_BROWN = Color.from_hex('#6c5043')
PARTRIDGE = Color.from_hex('#725440')
FRIAR_BROWN = Color.from_hex('#6e493a')
MUSTANG = Color.from_hex('#684b40')
PINECONE = Color.from_hex('#61473b')
POTTING_SOIL = Color.from_hex('#54392d')
ERMINE = Color.from_hex('#836b4f')
OTTER = Color.from_hex('#7f674f')
KANGAROO = Color.from_hex('#725e43')
SEPIA = Color.from_hex('#6b543e')
COFFEE_LIQUEUR = Color.from_hex('#6a513b')
DESERT_PALM = Color.from_hex('#5a4632')
TEAK = Color.from_hex('#655341')
SHITAKE = Color.from_hex('#736253')
CUB = Color.from_hex('#6e5c4b')
CARAFE = Color.from_hex('#5d473a')
DARK_EARTH = Color.from_hex('#5c4939')
SLATE_BLACK = Color.from_hex('#4b3d33')
CHOCOLATE_BROWN = Color.from_hex('#4e403b')
DEMITASSE = Color.from_hex('#40342b')
DEEP_TAUPE = Color.from_hex('#7b6660')
SHOPPING_BAG = Color.from_hex('#5a4743')
CHESTNUT = Color.from_hex('#584039')
BRACKEN = Color.from_hex('#4f3f3b')
SEAL_BROWN = Color.from_hex('#493b39')
JAVA = Color.from_hex('#433331')
COFFEE_BEAN = Color.from_hex('#40312f')
MOTHER_OF_PEARL = Color.from_hex('#e9d4c3')
PASTEL_ROSE_TAN = Color.from_hex('#e9d1bf')
NOVELLE_PEACH = Color.from_hex('#e7cfbd')
SUN_KISS = Color.from_hex('#ebd1bb')
IVORY_CREAM = Color.from_hex('#dac0a7')
SHIFTING_SAND = Color.from_hex('#d8c0ad')
APPLEBLOSSOM = Color.from_hex('#ddbca0')
EGGNOG = Color.from_hex('#ece1d3')
CREAM_TAN = Color.from_hex('#e4c7b8')
SAND_DOLLAR = Color.from_hex('#decdbe')
SMOKE_GRAY = Color.from_hex('#cebaa8')
DOESKIN = Color.from_hex('#bdab9b')
SESAME = Color.from_hex('#baa38b')
LIGHT_TAUPE = Color.from_hex('#b19d8d')
WARM_TAUPE = Color.from_hex('#af9483')
STUCCO = Color.from_hex('#a58d7f')
ALMONDINE = Color.from_hex('#a78c8b')
CHANTERELLE = Color.from_hex('#a28776')
GINGER_SNAP = Color.from_hex('#977d70')
WOODSMOKE = Color.from_hex('#947764')
AMPHORA = Color.from_hex('#9f8672')
MOONLIGHT = Color.from_hex('#c5b1a0')
FRAPPE = Color.from_hex('#d1b7a0')
RUGBY_TAN = Color.from_hex('#c2a594')
ROEBUCK = Color.from_hex('#b09080')
PRALINE = Color.from_hex('#ad8b75')
BURRO = Color.from_hex('#947764')
BEAVER_FUR = Color.from_hex('#997867')
TOASTED_ALMOND = Color.from_hex('#d2b49c')
TAWNY_BIRCH = Color.from_hex('#ae856c')
MACAROON = Color.from_hex('#b38b71')
TAWNY_BROWN = Color.from_hex('#ab856f')
CAMEL = Color.from_hex('#b0846a')
TOAST = Color.from_hex('#ca9978')
TOASTED_NUT = Color.from_hex('#c08768')
NUDE = Color.from_hex('#f2d3bc')
TENDER_PEACH = Color.from_hex('#f8d5b8')
ALESAN = Color.from_hex('#f1ceb3')
PALE_PEACH = Color.from_hex('#fed1bd')
PEACH_PUREE = Color.from_hex('#efcfba')
BELLINI = Color.from_hex('#f4c9b1')
AMBERLIGHT = Color.from_hex('#e2bea2')
PEACH_DUST = Color.from_hex('#f0d8cc')
LINEN = Color.from_hex('#edd2c0')
SCALLOP_SHELL = Color.from_hex('#fbd8c9')
SOFT_PINK = Color.from_hex('#f2d8cd')
PALE_DOGWOOD = Color.from_hex('#edcdc2')
SILVER_PEONY = Color.from_hex('#e7cfc7')
ROSE_DUST = Color.from_hex('#cdb2a5')
SHELL = Color.from_hex('#e1cfc6')
WHISPER_PINK = Color.from_hex('#dacbbe')
PINK_TINT = Color.from_hex('#dbcbbd')
EVENING_SAND = Color.from_hex('#ddb6ab')
SIROCCO = Color.from_hex('#c39d88')
BRUSH = Color.from_hex('#b99984')
CAFE_AU_LAIT = Color.from_hex('#ae8774')
CAMEO_ROSE = Color.from_hex('#d7b8ab')
PALE_BLUSH = Color.from_hex('#e4bfb3')
ROSE_CLOUD = Color.from_hex('#dbb0a2')
SPANISH_VILLA = Color.from_hex('#dfbaa9')
MAPLE_SUGAR = Color.from_hex('#c9a38d')
TUSCANY = Color.from_hex('#be9785')
CORK = Color.from_hex('#ba8671')
BISQUE = Color.from_hex('#edcab5')
ALMOST_APRICOT = Color.from_hex('#e5b39b')
PINK_SAND = Color.from_hex('#dfb19b')
PEACH_NOUGAT = Color.from_hex('#e6af91')
PEACH_BLOOM = Color.from_hex('#d99b7c')
DUSTY_CORAL = Color.from_hex('#d29b83')
CAFE_CREME = Color.from_hex('#c79685')
SANDSTORM = Color.from_hex('#bd8b69')
BUTTERUM = Color.from_hex('#c68f65')
BISCUIT = Color.from_hex('#b4835b')
CASHEW = Color.from_hex('#a47149')
ALMOND = Color.from_hex('#a7754d')
LION = Color.from_hex('#a0714f')
THRUSH = Color.from_hex('#936b4f')
MOCHA_MOUSSE = Color.from_hex('#a47864')
PECAN_BROWN = Color.from_hex('#a36e51')
HAZEL = Color.from_hex('#ae7250')
BRAN = Color.from_hex('#a66e4a')
ADOBE = Color.from_hex('#a3623b')
LEATHER_BROWN = Color.from_hex('#97572b')
GLAZED_GINGER = Color.from_hex('#91552b')
SANDSTONE = Color.from_hex('#c48a69')
CARAMEL = Color.from_hex('#c37c54')
AMBER_BROWN = Color.from_hex('#a66646')
SIERRA = Color.from_hex('#985c41')
GINGER_BREAD = Color.from_hex('#8c4a2f')
MOCHA_BISQUE = Color.from_hex('#8c543a')
TORTOISE_SHELL = Color.from_hex('#754734')
PHEASANT = Color.from_hex('#c68463')
SUNBURN = Color.from_hex('#b37256')
RAW_SIENNA = Color.from_hex('#b9714f')
AUTUMN_LEAF = Color.from_hex('#b56a4c')
MECCA_ORANGE = Color.from_hex('#bd5745')
RUST = Color.from_hex('#b55a30')
BOMBAY_BROWN = Color.from_hex('#9f5130')
FROSTED_ALMOND = Color.from_hex('#d2c2ac')
GILDED_BEIGE = Color.from_hex('#b39f8d')
PALE_GOLD = Color.from_hex('#bd9865')
RICH_GOLD = Color.from_hex('#c8b273')
COPPER = Color.from_hex('#c47e5a')
COPPER_COIN = Color.from_hex('#ba6b57')
SILVER = Color.from_hex('#a2a2a1')
RAW_UMBER = Color.from_hex('#92705f')
BROWNIE = Color.from_hex('#8f7265')
ACORN = Color.from_hex('#7e5e52')
CLOVE = Color.from_hex('#876155')
CAROB_BROWN = Color.from_hex('#855c4c')
RUSSET = Color.from_hex('#8f5f50')
RAWHIDE = Color.from_hex('#865e49')
CHUTNEY = Color.from_hex('#98594b')
BAKED_CLAY = Color.from_hex('#9c5642')
COPPER_BROWN = Color.from_hex('#9a6051')
BROWN_PATINA = Color.from_hex('#834f3d')
RUSTIC_BROWN = Color.from_hex('#855141')
COCONUT_SHELL = Color.from_hex('#874e3c')
SEQUOIA = Color.from_hex('#804839')
ROOT_BEER = Color.from_hex('#714a41')
BRUNETTE = Color.from_hex('#664238')
SABLE = Color.from_hex('#6e403c')
CINNAMON = Color.from_hex('#6b4139')
FUDGESICKLE = Color.from_hex('#63403a')
MINK = Color.from_hex('#734b42')
CAPPUCCINO = Color.from_hex('#633f33')
COGNAC = Color.from_hex('#8b645a')
NUTMEG = Color.from_hex('#7e5c54')
FRENCH_ROAST = Color.from_hex('#58423f')
DEEP_MAHOGANY = Color.from_hex('#553b39')
RUM_RAISIN = Color.from_hex('#583432')
BROWN_STONE = Color.from_hex('#593c39')
BITTER_CHOCOLATE = Color.from_hex('#503130')
MAHOGANY = Color.from_hex('#824d46')
HENNA = Color.from_hex('#7c423c')
ARABIAN_SPICE = Color.from_hex('#884332')
HOT_CHOCOLATE = Color.from_hex('#683b39')
RUSSET_BROWN = Color.from_hex('#743332')
MADDER_BROWN = Color.from_hex('#6a3331')
ANDORRA = Color.from_hex('#603535')
AFTERGLOW = Color.from_hex('#f3e6c9')
TRANSPARENT_YELLOW = Color.from_hex('#f4ecc2')
DOUBLE_CREAM = Color.from_hex('#f3e0ac')
SUNLIGHT = Color.from_hex('#edd59e')
STRAW = Color.from_hex('#e0c992')
JOJOBA = Color.from_hex('#dabe81')
RATTAN = Color.from_hex('#d1b272')
BOULDER = Color.from_hex('#d1be9b')
SEA_MIST = Color.from_hex('#d8c9a3')
REED_YELLOW = Color.from_hex('#dcc99e')
CHINO_GREEN = Color.from_hex('#d9caa5')
PARSNIP = Color.from_hex('#d6c69a')
DUSTY_YELLOW = Color.from_hex('#d4cc9a')
SILVER_FERN = Color.from_hex('#bbaa7e')
LEMON_GRASS = Color.from_hex('#dcd494')
RAFFIA = Color.from_hex('#dac483')
GOLDEN_MIST = Color.from_hex('#d5cd94')
PAMPAS = Color.from_hex('#cfbb7b')
BAMBOO = Color.from_hex('#d2b04c')
CRESS_GREEN = Color.from_hex('#bca949')
OLIVE_OIL = Color.from_hex('#a98b2d')
DRIED_MOSS = Color.from_hex('#ccb97e')
CELERY = Color.from_hex('#cec153')
ACACIA = Color.from_hex('#dacd65')
SULPHUR = Color.from_hex('#ddb614')
OIL_YELLOW = Color.from_hex('#c4a647')
GREEN_SULPHUR = Color.from_hex('#ae8e2c')
GOLDEN_PALM = Color.from_hex('#aa8805')
COCOON = Color.from_hex('#c9b27c')
HEMP = Color.from_hex('#c0ad7c')
SOUTHERN_MOSS = Color.from_hex('#bca66a')
OLIVENITE = Color.from_hex('#c1a65c')
GOLDEN_GREEN = Color.from_hex('#bdb369')
ANTIQUE_GOLD = Color.from_hex('#b59e5f')
BURNISHED_GOLD = Color.from_hex('#aa9855')
FRENCH_VANILLA = Color.from_hex('#efe1a7')
PASTEL_YELLOW = Color.from_hex('#f2e6b1')
TENDER_YELLOW = Color.from_hex('#ededb7')
WAX_YELLOW = Color.from_hex('#ede9ad')
LEMONADE = Color.from_hex('#f0e79d')
ELFIN_YELLOW = Color.from_hex('#eeea97')
LIMELIGHT = Color.from_hex('#f0e87d')
DUSKY_CITRON = Color.from_hex('#e3cc81')
MUTED_LIME = Color.from_hex('#d1c87c')
ENDIVE = Color.from_hex('#d2cc81')
CUSTARD = Color.from_hex('#e5d68e')
CANARY_YELLOW = Color.from_hex('#dfd87e')
YELLOW_CREAM = Color.from_hex('#efdc75')
CREAM_GOLD = Color.from_hex('#dec05f')
AURORA = Color.from_hex('#eddd59')
GREEN_SHEEN = Color.from_hex('#d9ce52')
MAIZE = Color.from_hex('#eec843')
BLAZING_YELLOW = Color.from_hex('#fee715')
BUTTERCUP = Color.from_hex('#fae03c')
EMPIRE_YELLOW = Color.from_hex('#f7d000')
LEMON = Color.from_hex('#f3bf08')
MIMOSA = Color.from_hex('#f0c05a')
ASPEN_GOLD = Color.from_hex('#ffd662')
DANDELION = Color.from_hex('#ffd02e')
VIBRANT_YELLOW = Color.from_hex('#ffda29')
CYBER_YELLOW = Color.from_hex('#ffd400')
FREESIA = Color.from_hex('#f3c12c')
LEMON_CHROME = Color.from_hex('#ffc300')
MELLOW_YELLOW = Color.from_hex('#f0dd9d')
PALE_BANANA = Color.from_hex('#fae199')
POPCORN = Color.from_hex('#f8de8d')
SUNSHINE = Color.from_hex('#fade85')
LEMON_DROP = Color.from_hex('#fdd878')
PRIMROSE_YELLOW = Color.from_hex('#f6d155')
SUPER_LEMON = Color.from_hex('#e4bf45')
MISTED_YELLOW = Color.from_hex('#dab965')
SAUTERNE = Color.from_hex('#c5a253')
HONEY = Color.from_hex('#ba9238')
ARROWWOOD = Color.from_hex('#bc8d1f')
TAWNY_OLIVE = Color.from_hex('#c4962c')
CEYLON_YELLOW = Color.from_hex('#d4ae40')
LEMON_CURRY = Color.from_hex('#cda323')
FALL_LEAF = Color.from_hex('#c9a86a')
ANTELOPE = Color.from_hex('#b19664')
MUSTARD_GOLD = Color.from_hex('#b08e51')
HARVEST_GOLD = Color.from_hex('#b68a3a')
NUGGET_GOLD = Color.from_hex('#c89720')
GOLDEN_SPICE = Color.from_hex('#c6973f')
GOLDEN_YELLOW = Color.from_hex('#cb8e16')
OCHRE = Color.from_hex('#d6af66')
TINSEL = Color.from_hex('#c3964d')
BRIGHT_GOLD = Color.from_hex('#cf9f52')
HONEY_GOLD = Color.from_hex('#d1a054')
AMBER_GOLD = Color.from_hex('#c19552')
MINERAL_YELLOW = Color.from_hex('#d39c43')
NARCISSUS = Color.from_hex('#c39449')
MARZIPAN = Color.from_hex('#d8c09d')
CURRY = Color.from_hex('#be9e6f')
PRAIRIE_SAND = Color.from_hex('#b59a6a')
HONEY_MUSTARD = Color.from_hex('#b68f52')
WOOD_THRUSH = Color.from_hex('#a47d43')
GOLDEN_BROWN = Color.from_hex('#91672f')
BRONZE_BROWN = Color.from_hex('#825e2f')
APPLE_CINNAMON = Color.from_hex('#b0885a')
BONE_BROWN = Color.from_hex('#9d7446')
DIJON = Color.from_hex('#97754c')
BISTRE = Color.from_hex('#98754a')
MEDAL_BRONZE = Color.from_hex('#977547')
CUMIN = Color.from_hex('#927240')
BREEN = Color.from_hex('#795d34')
SNAPDRAGON = Color.from_hex('#fed777')
BANANA_CREAM = Color.from_hex('#ffcf73')
DAFFODIL = Color.from_hex('#fdc04e')
YOLK_YELLOW = Color.from_hex('#e2b051')
GOLDEN_ROD = Color.from_hex('#e2a829')
OLD_GOLD = Color.from_hex('#eca825')
SPECTRA_YELLOW = Color.from_hex('#f7b718')
GOLDEN_HAZE = Color.from_hex('#fbd897')
SAHARA_SUN = Color.from_hex('#dfc08a')
NEW_WHEAT = Color.from_hex('#d7b57f')
CORNSILK = Color.from_hex('#edc373')
BUFF_YELLOW = Color.from_hex('#f1bf70')
SUNSET_GOLD = Color.from_hex('#f7c46c')
GOLDEN_CREAM = Color.from_hex('#f7b768')
IMPALA = Color.from_hex('#f8ce97')
FLAX = Color.from_hex('#ffc87d')
PALE_MARIGOLD = Color.from_hex('#ffc66e')
AMBER_YELLOW = Color.from_hex('#fab75a')
AMBER = Color.from_hex('#efad55')
GOLDEN_APRICOT = Color.from_hex('#dda758')
BEESWAX = Color.from_hex('#eba851')
BANANA = Color.from_hex('#fcb953')
CITRUS = Color.from_hex('#f9ac2f')
GOLDEN_GLOW = Color.from_hex('#d99938')
ARTISANS_GOLD = Color.from_hex('#f2ab46')
SUNFLOWER = Color.from_hex('#d39237')
BUCKTHORN_BROWN = Color.from_hex('#a76f1f')
CATHAY_SPICE = Color.from_hex('#99642c')
TAFFY = Color.from_hex('#c39b6a')
OAK_BUFF = Color.from_hex('#cf9c63')
HONEY_YELLOW = Color.from_hex('#ca9456')
SPRUCE_YELLOW = Color.from_hex('#be8a4a')
INCA_GOLD = Color.from_hex('#bb7a2c')
SUDAN_BROWN = Color.from_hex('#ac6b29')
RUBBER = Color.from_hex('#815b37')
WHEAT = Color.from_hex('#dec5a5')
HONEY_PEACH = Color.from_hex('#dcbd9e')
DESERT_DUST = Color.from_hex('#e3bc8e')
GOLDEN_STRAW = Color.from_hex('#e6bd8f')
BUFF = Color.from_hex('#ebc396')
DESERT_MIST = Color.from_hex('#e0b589')
CLAY = Color.from_hex('#d2a172')
GOLDEN_FLEECE = Color.from_hex('#f2d1a0')
APRICOT_SHERBET = Color.from_hex('#facd9e')
SUNBURST = Color.from_hex('#f6c289')
APRICOT_CREAM = Color.from_hex('#f1bd89')
BUFF_ORANGE = Color.from_hex('#ffbb7c')
CHAMOIS = Color.from_hex('#f7b26a')
WARM_APRICOT = Color.from_hex('#ffb865')
MARIGOLD = Color.from_hex('#fadc53')
GOLDEN_NUGGET = Color.from_hex('#db9b59')
BUTTERSCOTCH = Color.from_hex('#e19640')
NUGGET = Color.from_hex('#cf8848')
BUCKSKIN = Color.from_hex('#d18e54')
YAM = Color.from_hex('#d0893f')
GOLDEN_OAK = Color.from_hex('#be752d')
GOLD_FUSION = Color.from_hex('#ffb000')
SAFFRON = Color.from_hex('#ffa500')
CADMIUM_YELLOW = Color.from_hex('#ee9626')
ZINNIA = Color.from_hex('#ffa010')
RADIANT_YELLOW = Color.from_hex('#fc9e21')
APRICOT = Color.from_hex('#f19035')
DARK_CHEDDAR = Color.from_hex('#e08119')
APRICOT_ICE = Color.from_hex('#fbbe99')
APRICOT_NECTAR = Color.from_hex('#ecaa79')
GOLD_EARTH = Color.from_hex('#dd9c6b')
APRICOT_TAN = Color.from_hex('#dd9760')
TOPAZ = Color.from_hex('#d08344')
GOLDEN_OCHRE = Color.from_hex('#c77943')
APRICOT_BUFF = Color.from_hex('#cd7e4d')
PEACH_COBBLER = Color.from_hex('#ffb181')
SALMON_BUFF = Color.from_hex('#feaa7b')
PUMPKIN = Color.from_hex('#f5a26f')
MOCK_ORANGE = Color.from_hex('#ffa368')
MUSKMELON = Color.from_hex('#ec935e')
COPPER_TAN = Color.from_hex('#de8e65')
CORAL_GOLD = Color.from_hex('#d27d56')
RUSSET_ORANGE = Color.from_hex('#e47127')
ORANGE_OCHRE = Color.from_hex('#dc793a')
AMBERGLOW = Color.from_hex('#dc793e')
JAFFA_ORANGE = Color.from_hex('#d86d39')
APRICOT_ORANGE = Color.from_hex('#c86b3c')
BURNT_ORANGE = Color.from_hex('#c86733')
HARVEST_PUMPKIN = Color.from_hex('#d56231')
BLAZING_ORANGE = Color.from_hex('#ffa64f')
FLAME_ORANGE = Color.from_hex('#fb8b23')
BRIGHT_MARIGOLD = Color.from_hex('#ff8d00')
AUTUMN_GLORY = Color.from_hex('#ff8812')
SUN_ORANGE = Color.from_hex('#f48037')
PERSIMMON_ORANGE = Color.from_hex('#f47327')
ORANGE_POPSICLE = Color.from_hex('#ff7913')
AUTUMN_SUNSET = Color.from_hex('#f38554')
TANGERINE = Color.from_hex('#f88f58')
BIRD_OF_PARADISE = Color.from_hex('#ff8c55')
ORANGE_PEEL = Color.from_hex('#fa7a35')
MANDARIN_ORANGE = Color.from_hex('#ec6a37')
GOLDEN_POPPY = Color.from_hex('#f56733')
VIBRANT_ORANGE = Color.from_hex('#ff7420')
NECTARINE = Color.from_hex('#ff8656')
CORAL_ROSE = Color.from_hex('#f3774d')
CARROT = Color.from_hex('#fd6f3b')
FIRECRACKER = Color.from_hex('#f36944')
RED_ORANGE = Color.from_hex('#f05627')
VERMILLION_ORANGE = Color.from_hex('#f9633b')
FLAME = Color.from_hex('#f2552c')
CREAMPUFF = Color.from_hex('#ffcda8')
BLEACHED_APRICOT = Color.from_hex('#fccaac')
ALMOND_CREAM = Color.from_hex('#f4c29f')
BEACH_SAND = Color.from_hex('#fbb995')
CREAM_BLUSH = Color.from_hex('#f8c19a')
CARAMEL_CREAM = Color.from_hex('#f4ba94')
PEACH_FUZZ = Color.from_hex('#ffbe98')
PRAIRIE_SUNSET = Color.from_hex('#ffbb9e')
CORAL_SANDS = Color.from_hex('#edaa86')
APRICOT_WASH = Color.from_hex('#fbac82')
CANYON_SUNSET = Color.from_hex('#e1927a')
BRANDIED_MELON = Color.from_hex('#ce7b5b')
CARNELIAN = Color.from_hex('#ce785d')
MANGO = Color.from_hex('#b75e41')
PEACH = Color.from_hex('#f2a987')
CANTALOUPE = Color.from_hex('#ffa177')
CORAL_REEF = Color.from_hex('#faa181')
SHELL_CORAL = Color.from_hex('#ea9575')
CADMIUM_ORANGE = Color.from_hex('#f99471')
MELON = Color.from_hex('#fe8863')
DUSTY_ORANGE = Color.from_hex('#e27a53')
ARABESQUE = Color.from_hex('#d16f52')
LANGOUSTINO = Color.from_hex('#ca6c56')
GINGER = Color.from_hex('#c96551')
FLAMINGO = Color.from_hex('#df7253')
ORANGE_RUST = Color.from_hex('#c25a3c')
BURNT_OCHRE = Color.from_hex('#bb4f35')
CHILI = Color.from_hex('#be5141')
GINGER_SPICE = Color.from_hex('#b65d48')
AUTUMN_GLAZE = Color.from_hex('#b3573f')
AUBURN = Color.from_hex('#a15843')
PICANTE = Color.from_hex('#8d3f2d')
TANDORI_SPICE = Color.from_hex('#9f4440')
CINNABAR = Color.from_hex('#9c453b')
BOSSA_NOVA = Color.from_hex('#973a36')
TROPICAL_PEACH = Color.from_hex('#ffc4b2')
PEACH_PARFAIT = Color.from_hex('#f8bfa8')
CORAL_PINK = Color.from_hex('#e8a798')
DUSTY_PINK = Color.from_hex('#deaa9b')
MUTED_CLAY = Color.from_hex('#d29380')
SHRIMP = Color.from_hex('#e29a86')
TAWNY_ORANGE = Color.from_hex('#d37f6f')
CORAL_HAZE = Color.from_hex('#e38e84')
CANYON_CLAY = Color.from_hex('#ce8477')
TERRA_COTTA = Color.from_hex('#d38377')
DESERT_SAND = Color.from_hex('#bd7b74')
LIGHT_MAHOGANY = Color.from_hex('#ad6d68')
CEDAR_WOOD = Color.from_hex('#a1655b')
WITHERED_ROSE = Color.from_hex('#a26666')
ROSE_DAWN = Color.from_hex('#c2877b')
ASH_ROSE = Color.from_hex('#b5817d')
OLD_ROSE = Color.from_hex('#b47b77')
BRICK_DUST = Color.from_hex('#b07069')
CANYON_ROSE = Color.from_hex('#af6c67')
DUSTY_CEDAR = Color.from_hex('#ad5d5d')
MARSALA = Color.from_hex('#964f4c')
APRICOT_BRANDY = Color.from_hex('#c26a5a')
ARAGON = Color.from_hex('#b06455')
HOT_SAUCE = Color.from_hex('#ab4f41')
BRUSCHETTA = Color.from_hex('#a75949')
ETRUSCAN_RED = Color.from_hex('#a2574b')
REDWOOD = Color.from_hex('#a6594c')
BURNT_BRICK = Color.from_hex('#a14d3a')
FADED_ROSE = Color.from_hex('#bf6464')
BAKED_APPLE = Color.from_hex('#b34646')
POMPEIAN_RED = Color.from_hex('#a4292e')
KETCHUP = Color.from_hex('#9a382d')
RED_OCHRE = Color.from_hex('#913832')
BARN_RED = Color.from_hex('#8f423b')
BURNT_HENNA = Color.from_hex('#7e392f')
PEACH_PEARL = Color.from_hex('#ffb2a5')
PEACH_MELBA = Color.from_hex('#fbbdaf')
APRICOT_BLUSH = Color.from_hex('#feaea5')
PEACH_BUD = Color.from_hex('#fdb2ab')
CORAL_ALMOND = Color.from_hex('#e29d94')
LOBSTER_BISQUE = Color.from_hex('#dd9289')
LANTANA = Color.from_hex('#da7e7a')
PEACH_NECTAR = Color.from_hex('#ffb59b')
SALMON = Color.from_hex('#faaa94')
PEACH_AMBER = Color.from_hex('#fb9f93')
DESERT_FLOWER = Color.from_hex('#ff9687')
PEACH_PINK = Color.from_hex('#fa9a85')
BURNT_CORAL = Color.from_hex('#e9897e')
CRABAPPLE = Color.from_hex('#d77e70')
PAPAYA_PUNCH = Color.from_hex('#fca289')
FUSION_CORAL = Color.from_hex('#ff8576')
FRESH_SALMON = Color.from_hex('#ff7f6a')
PERSIMMON = Color.from_hex('#f67866')
CORAL = Color.from_hex('#ed7464')
LIVING_CORAL = Color.from_hex('#ff6f61')
HOT_CORAL = Color.from_hex('#f35b53')
SHELL_PINK = Color.from_hex('#f88180')
GEORGIA_PEACH = Color.from_hex('#f97272')
SUGAR_CORAL = Color.from_hex('#f56c73')
DUBARRY = Color.from_hex('#f25f66')
PORCELAIN_ROSE = Color.from_hex('#ea6b6a')
SPICED_CORAL = Color.from_hex('#d75c5d')
DEEP_SEA_CORAL = Color.from_hex('#d9615b')
ROSE_OF_SHARON = Color.from_hex('#dc5b62')
CAYENNE = Color.from_hex('#e04951')
HIBISCUS = Color.from_hex('#dd3848')
POINSETTIA = Color.from_hex('#cb3441')
CHRYSANTHEMUM = Color.from_hex('#be454f')
CRANBERRY = Color.from_hex('#bb4a4d')
CARDINAL = Color.from_hex('#ad3e48')
TIGERLILY = Color.from_hex('#e2583e')
GRENADINE = Color.from_hex('#df3f32')
MANDARIN_RED = Color.from_hex('#e74a33')
FIESTA = Color.from_hex('#dd4132')
CHERRY_TOMATO = Color.from_hex('#eb3c27')
ORANGE_COM = Color.from_hex('#da321c')
SPICY_ORANGE = Color.from_hex('#d73c26')
CAMELLIA = Color.from_hex('#f6745f')
NASTURTIUM = Color.from_hex('#fe6347')
EMBERGLOW = Color.from_hex('#ea6759')
BURNT_SIENNA = Color.from_hex('#c65d52')
PAPRIKA = Color.from_hex('#ce4d42')
RED_CLAY = Color.from_hex('#c2452d')
MOLTEN_LAVA = Color.from_hex('#b5332e')
BITTERSWEET = Color.from_hex('#d93744')
POPPY_RED = Color.from_hex('#dc343b')
TOMATO = Color.from_hex('#ce2939')
FIERY_RED = Color.from_hex('#d01c1f')
FLAME_SCARLET = Color.from_hex('#cd212a')
HIGH_RISK_RED = Color.from_hex('#c71f2d')
AURORA_RED = Color.from_hex('#b93a32')
ROCOCCO_RED = Color.from_hex('#bb363f')
TOMATO_PUREE = Color.from_hex('#c53346')
LOLLIPOP = Color.from_hex('#cc1c3b')
SKI_PATROL = Color.from_hex('#bb1237')
SCARLET = Color.from_hex('#bc2b3d')
LIPSTICK_RED = Color.from_hex('#b31a38')
CRIMSON = Color.from_hex('#ae0e36')
RACING_RED = Color.from_hex('#bd162c')
MARS_RED = Color.from_hex('#bc2731')
TANGO_RED = Color.from_hex('#ac0e2e')
CHINESE_RED = Color.from_hex('#be132d')
RIBBON_RED = Color.from_hex('#b92636')
TRUE_RED = Color.from_hex('#bf1932')
CHILI_PEPPER = Color.from_hex('#9b1b30')
QUARTZ_PINK = Color.from_hex('#efa6aa')
PINK_ICING = Color.from_hex('#eea0a6')
BLOSSOM = Color.from_hex('#f2b2ae')
PEACHES_N_CREAM = Color.from_hex('#f4a6a3')
CANDLELIGHT_PEACH = Color.from_hex('#f8a39d')
STRAWBERRY_ICE = Color.from_hex('#e78b90')
PEACH_BLOSSOM = Color.from_hex('#de8286')
FLAMINGO_PINK = Color.from_hex('#f7969e')
CONFETTI = Color.from_hex('#e6798e')
BUBBLEGUM = Color.from_hex('#ea738d')
PINK_LEMONADE = Color.from_hex('#ee6d8a')
CAMELLIA_ROSE = Color.from_hex('#eb6081')
RAPTURE_ROSE = Color.from_hex('#d16277')
DESERT_ROSE = Color.from_hex('#cf6977')
GERANIUM_PINK = Color.from_hex('#f6909d')
CONCH_SHELL = Color.from_hex('#fc8f9b')
SALMON_ROSE = Color.from_hex('#ff8d94')
STRAWBERRY_PINK = Color.from_hex('#f57f8e')
SUNKIST_CORAL = Color.from_hex('#ea6676')
CALYPSO_CORAL = Color.from_hex('#ee5c6c')
TEA_ROSE = Color.from_hex('#dc7178')
GERANIUM = Color.from_hex('#da3d58')
PARADISE_PINK = Color.from_hex('#e4445e')
TEABERRY = Color.from_hex('#dc3855')
ROUGE_RED = Color.from_hex('#e24666')
RASPBERRY = Color.from_hex('#d32e5e')
AZALEA = Color.from_hex('#d42e5b')
VIRTUAL_PINK = Color.from_hex('#c6174e')
CLARET_RED = Color.from_hex('#c84c61')
RASPBERRY_WINE = Color.from_hex('#b63753')
ROSE_RED = Color.from_hex('#c92351')
BARBERRY = Color.from_hex('#bf1945')
BRIGHT_ROSE = Color.from_hex('#c51959')
PERSIAN_RED = Color.from_hex('#a21441')
CERISE = Color.from_hex('#a41247')
PINK_LADY = Color.from_hex('#efc1d6')
LILAC_SACHET = Color.from_hex('#e9adca')
PRISM_PINK = Color.from_hex('#f0a1bf')
BEGONIA_PINK = Color.from_hex('#ec9abe')
FUCHSIA_PINK = Color.from_hex('#df88b7')
ROSEBLOOM = Color.from_hex('#e290b2')
IBIS_ROSE = Color.from_hex('#ca628f')
SACHET_PINK = Color.from_hex('#f18aad')
WILD_ORCHID = Color.from_hex('#d979a2')
AURORA_PINK = Color.from_hex('#e881a6')
CHATEAU_ROSE = Color.from_hex('#d2738f')
MORNING_GLORY = Color.from_hex('#ee819f')
AZALEA_PINK = Color.from_hex('#e96a97')
SHOCKING_PINK = Color.from_hex('#de5b8c')
HOT_PINK = Color.from_hex('#e55982')
FANDANGO_PINK = Color.from_hex('#e04f80')
HONEYSUCKLE = Color.from_hex('#d94f70')
RASPBERRY_SORBET = Color.from_hex('#d2386c')
CARMINE = Color.from_hex('#bc4869')
FUCHSIA_ROSE = Color.from_hex('#c74375')
BEETROOT_PURPLE = Color.from_hex('#cf2d71')
PINK_CARNATION = Color.from_hex('#ed7a9e')
CARMINE_ROSE = Color.from_hex('#e35b8f')
MAGENTA = Color.from_hex('#d23c77')
PINK_FLAMBE = Color.from_hex('#d3507a')
FUCHSIA_PURPLE = Color.from_hex('#d33479')
LILAC_ROSE = Color.from_hex('#bd4275')
VERY_BERRY = Color.from_hex('#b73275')
SUPER_PINK = Color.from_hex('#ce6ba4')
PHLOX_PINK = Color.from_hex('#ce5e9a')
RASPBERRY_ROSE = Color.from_hex('#cc4385')
ROSE_VIOLET = Color.from_hex('#c0428a')
FUCHSIA_RED = Color.from_hex('#ab3475')
CACTUS_FLOWER = Color.from_hex('#a83e6c')
MAGENTA_HAZE = Color.from_hex('#9d446e')
SHRINKING_VIOLET = Color.from_hex('#f4e1e6')
PRIMROSE_PINK = Color.from_hex('#eed4d9')
SILVER_PINK = Color.from_hex('#dcb1af')
POWDER_PINK = Color.from_hex('#ecb2b3')
MAUVEGLOW = Color.from_hex('#d18489')
BRANDIED_APRICOT = Color.from_hex('#ca848a')
DUSTY_ROSE = Color.from_hex('#ba797d')
MAUVE_MORN = Color.from_hex('#ecd6d6')
MAUVE_CHALK = Color.from_hex('#e5d0cf')
PEARL = Color.from_hex('#f9dbd8')
BRIDAL_ROSE = Color.from_hex('#d69fa2')
BLUSH = Color.from_hex('#d1969a')
BAROQUE_ROSE = Color.from_hex('#b35a66')
SLATE_ROSE = Color.from_hex('#b45865')
MINERAL_RED = Color.from_hex('#b35457')
GARNET_ROSE = Color.from_hex('#ac4b55')
HOLLY_BERRY = Color.from_hex('#b44e5d')
AMERICAN_BEAUTY = Color.from_hex('#a73340')
JESTER_RED = Color.from_hex('#9e1030')
RIO_RED = Color.from_hex('#8a2232')
RUMBA_RED = Color.from_hex('#7c2439')
EARTH_RED = Color.from_hex('#95424e')
DEEP_CLARET = Color.from_hex('#973443')
GARNET = Color.from_hex('#953640')
BRICK_RED = Color.from_hex('#8c373e')
ROSEWOOD = Color.from_hex('#813639')
TIBETAN_RED = Color.from_hex('#782a39')
BIKING_RED = Color.from_hex('#77212e')
APPLE_BUTTER = Color.from_hex('#844b4d')
OXBLOOD_RED = Color.from_hex('#70393f')
COWHIDE = Color.from_hex('#884344')
BURNT_RUSSET = Color.from_hex('#7e3940')
RUBY_WINE = Color.from_hex('#77333b')
CORDOVAN = Color.from_hex('#702f3b')
TAWNY_PORT = Color.from_hex('#5c2c35')
CREOLE_PINK = Color.from_hex('#f7d5cc')
PEACH_BLUSH = Color.from_hex('#e4ccc6')
CLOUD_PINK = Color.from_hex('#f5d1c8')
VEILED_ROSE = Color.from_hex('#f8cdc9')
PEARL_BLUSH = Color.from_hex('#f4cec5')
ENGLISH_ROSE = Color.from_hex('#f4c6c3')
LOTUS = Color.from_hex('#e2c1c0')
ROSEWATER = Color.from_hex('#f6dbd8')
PEACH_WHIP = Color.from_hex('#dbbeb7')
ROSE_SMOKE = Color.from_hex('#d3b4ad')
CORAL_CLOUD = Color.from_hex('#e2a9a1')
MISTY_ROSE = Color.from_hex('#caa39a')
PEACH_BEIGE = Color.from_hex('#d3a297')
CAMEO_BROWN = Color.from_hex('#c08a80')
SEASHELL_PINK = Color.from_hex('#f7c8c2')
CHINTZ_ROSE = Color.from_hex('#eec4be')
IMPATIENS_PINK = Color.from_hex('#ffc4bc')
PEACHSKIN = Color.from_hex('#dfb8b6')
MELLOW_ROSE = Color.from_hex('#d9a6a1')
ROSE_TAN = Color.from_hex('#d19c97')
ROSETTE = Color.from_hex('#ce8e8b')
MAUVEWOOD = Color.from_hex('#a75d67')
ROSE_WINE = Color.from_hex('#a4596d')
MALAGA = Color.from_hex('#9f5069')
DRY_ROSE = Color.from_hex('#8c4759')
HAWTHORN_ROSE = Color.from_hex('#884c5e')
MAROON = Color.from_hex('#834655')
WILD_GINGER = Color.from_hex('#7c4c53')
SANGRIA = Color.from_hex('#982551')
RED_BUD = Color.from_hex('#962d49')
BEAUJOLAIS = Color.from_hex('#80304c')
ANEMONE = Color.from_hex('#842c48')
BEET_RED = Color.from_hex('#7a1f3d')
RED_PLUM = Color.from_hex('#7c2946')
RHODODENDRON = Color.from_hex('#722b3f')
BARELY_PINK = Color.from_hex('#f8d7dd')
BLUSHING_BRIDE = Color.from_hex('#fbd3d9')
CRADLE_PINK = Color.from_hex('#edd0dd')
PALE_LILAC = Color.from_hex('#e1c6cc')
CHALK_PINK = Color.from_hex('#e6c5ca')
LIGHT_LILAC = Color.from_hex('#dec6d3')
PINK_NECTAR = Color.from_hex('#d8aab7')
HEAVENLY_PINK = Color.from_hex('#f4dede')
POTPOURRI = Color.from_hex('#e7c9ca')
CRYSTAL_PINK = Color.from_hex('#edd0ce')
PINK_DOGWOOD = Color.from_hex('#f7d1d1')
CRYSTAL_ROSE = Color.from_hex('#fdc3c6')
STRAWBERRY_CREAM = Color.from_hex('#f4c3c4')
GOSSAMER_PINK = Color.from_hex('#fac8c3')
ROSE_SHADOW = Color.from_hex('#f9c2cd')
ORCHID_PINK = Color.from_hex('#f3bbca')
ALMOND_BLOSSOM = Color.from_hex('#f5bec7')
CORAL_BLUSH = Color.from_hex('#e6b2b8')
CANDY_PINK = Color.from_hex('#f5b0bd')
PEONY = Color.from_hex('#ed9ca8')
SEA_PINK = Color.from_hex('#de98ab')
CASHMERE_ROSE = Color.from_hex('#ce879f')
WILD_ROSE = Color.from_hex('#ce8498')
ORCHID_SMOKE = Color.from_hex('#d294aa')
POLIGNAC = Color.from_hex('#c28799')
LILAS = Color.from_hex('#b88995')
MAUVE_ORCHID = Color.from_hex('#b58299')
ORCHID_HAZE = Color.from_hex('#b0879b')
PARFAIT_PINK = Color.from_hex('#e9c3cf')
PINK_MIST = Color.from_hex('#e6bccd')
CAMEO_PINK = Color.from_hex('#dba9b8')
SWEET_LILAC = Color.from_hex('#e8b5ce')
PINK_LAVENDER = Color.from_hex('#d9afca')
PASTEL_LAVENDER = Color.from_hex('#d8a1c4')
ORCHID = Color.from_hex('#d198c5')
LILAC_CHIFFON = Color.from_hex('#de9bc4')
MOONLITE_MAUVE = Color.from_hex('#d28fb0')
CYCLAMEN = Color.from_hex('#d687ba')
OPERA_MAUVE = Color.from_hex('#ca80b1')
CROCUS = Color.from_hex('#c67fae')
MULBERRY = Color.from_hex('#a76c97')
STRIKING_PURPLE = Color.from_hex('#944e87')
VIOLET = Color.from_hex('#c17fb5')
IRIS_ORCHID = Color.from_hex('#a767a2')
RADIANT_ORCHID = Color.from_hex('#ad5e99')
SPRING_CROCUS = Color.from_hex('#ba69a1')
MEADOW_MAUVE = Color.from_hex('#a9568c')
AMETHYST = Color.from_hex('#864d75')
MAGENTA_PURPLE = Color.from_hex('#6b264b')
ROSEBUD = Color.from_hex('#b65f9a')
PURPLE_ORCHID = Color.from_hex('#ad4d8c')
FESTIVAL_FUCHSIA = Color.from_hex('#9e2c6a')
BATON_ROUGE = Color.from_hex('#973c6c')
BOYSENBERRY = Color.from_hex('#85325c')
RASPBERRY_RADIANCE = Color.from_hex('#802a50')
PURPLE_POTION = Color.from_hex('#692746')
DAHLIA_MAUVE = Color.from_hex('#a64f82')
VIVID_VIOLA = Color.from_hex('#993c7c')
WILD_ASTER = Color.from_hex('#92316f')
DEEP_ORCHID = Color.from_hex('#903f75')
CLOVER = Color.from_hex('#8a3371')
PURPLE_WINE = Color.from_hex('#8c3573')
HOLLYHOCK = Color.from_hex('#823270')
HYACINTH_VIOLET = Color.from_hex('#8d4687')
DAHLIA = Color.from_hex('#843e83')
SPARKLING_GRAPE = Color.from_hex('#773376')
BYZANTIUM = Color.from_hex('#853b7b')
PHLOX = Color.from_hex('#692d5d')
GRAPE_JUICE = Color.from_hex('#682961')
GLOXINIA = Color.from_hex('#622e5a')
CRYSTAL_GRAY = Color.from_hex('#d7cbc4')
MUSHROOM = Color.from_hex('#bdaca3')
SHADOW_GRAY = Color.from_hex('#bba5a0')
SPHINX = Color.from_hex('#ab9895')
BARK = Color.from_hex('#a99592')
FAWN = Color.from_hex('#ae9490')
ADOBE_ROSE = Color.from_hex('#ba9f99')
PALE_MAUVE = Color.from_hex('#c6a4a4')
WOODROSE = Color.from_hex('#ae8c8e')
DEAUVILLE_MAUVE = Color.from_hex('#af9294')
TWILIGHT_MAUVE = Color.from_hex('#8b6f70')
ROSE_TAUPE = Color.from_hex('#806062')
ROSE_BROWN = Color.from_hex('#80565b')
ROAN_ROUGE = Color.from_hex('#885157')
ANTLER = Color.from_hex('#957a76')
PEPPERCORN = Color.from_hex('#6c5656')
RAISIN = Color.from_hex('#524144')
HUCKLEBERRY = Color.from_hex('#5b4349')
CATAWBA_GRAPE = Color.from_hex('#5d3c43')
PUCE = Color.from_hex('#503938')
FUDGE = Color.from_hex('#493338')
MAHOGANY_ROSE = Color.from_hex('#c5a193')
BURLWOOD = Color.from_hex('#9b716b')
MARRON = Color.from_hex('#6e4c4b')
DECADENT_CHOCOLATE = Color.from_hex('#513235')
RED_MAHOGANY = Color.from_hex('#60373d')
VINEYARD_WINE = Color.from_hex('#58363d')
WINETASTING = Color.from_hex('#492a34')
PORT = Color.from_hex('#663336')
CHOCOLATE_TRUFFLE = Color.from_hex('#612e35')
BURGUNDY = Color.from_hex('#64313e')
ZINFANDEL = Color.from_hex('#5c2935')
WINDSOR_WINE = Color.from_hex('#582b36')
PORT_ROYALE = Color.from_hex('#502b33')
FIG = Color.from_hex('#532d3b')
VIOLET_ICE = Color.from_hex('#c2acb1')
BURNISHED_LILAC = Color.from_hex('#c5aeb1')
KEEPSAKE_LILAC = Color.from_hex('#c0a5ae')
MAUVE_SHADOWS = Color.from_hex('#b598a3')
DAWN_PINK = Color.from_hex('#bfa3af')
FRAGRANT_LILAC = Color.from_hex('#ceadbe')
MAUVE_MIST = Color.from_hex('#c49bd4')
HEATHER_ROSE = Color.from_hex('#ad6d7f')
RED_VIOLET = Color.from_hex('#a35776')
MELLOW_MAUVE = Color.from_hex('#996378')
BORDEAUX = Color.from_hex('#96637b')
VIOLET_QUARTZ = Color.from_hex('#8b4963')
DAMSON = Color.from_hex('#854c65')
AMARANTH = Color.from_hex('#6f3c56')
ZEPHYR = Color.from_hex('#c89fa5')
DUSKY_ORCHID = Color.from_hex('#9a7182')
GRAPE_SHAKE = Color.from_hex('#886971')
WISTFUL_MAUVE = Color.from_hex('#946c74')
TULIPWOOD = Color.from_hex('#805466')
GRAPE_NECTAR = Color.from_hex('#8d5c74')
ARGYLE_PURPLE = Color.from_hex('#895c79')
NOSTALGIA_ROSE = Color.from_hex('#a4777e')
DECO_ROSE = Color.from_hex('#985f68')
RENAISSANCE_ROSE = Color.from_hex('#865560')
NOCTURNE = Color.from_hex('#7a4b56')
CRUSHED_BERRY = Color.from_hex('#804f5a')
CRUSHED_VIOLETS = Color.from_hex('#643a4c')
MAUVE_WINE = Color.from_hex('#5b3644')
PLUM_WINE = Color.from_hex('#674550')
EGGPLANT = Color.from_hex('#613f4c')
PRUNE = Color.from_hex('#603749')
PRUNE_PURPLE = Color.from_hex('#5c3a4d')
GRAPE_WINE = Color.from_hex('#5a2f43')
ITALIAN_PLUM = Color.from_hex('#533146')
POTENT_PURPLE = Color.from_hex('#462639')
LAVENDER_HERB = Color.from_hex('#b18eaa')
LAVENDER_MIST = Color.from_hex('#ae90a7')
VALERIAN = Color.from_hex('#9f7a93')
VERY_GRAPE = Color.from_hex('#927288')
GRAPEADE = Color.from_hex('#85677b')
PURPLE_GUMDROP = Color.from_hex('#7a596f')
BERRY_CONSERVE = Color.from_hex('#765269')
CHINESE_VIOLET = Color.from_hex('#835e81')
CRUSHED_GRAPE = Color.from_hex('#7a547f')
CONCORD_GRAPE = Color.from_hex('#7c5379')
SUNSET_PURPLE = Color.from_hex('#6f456e')
WOOD_VIOLET = Color.from_hex('#75406a')
PURPLE_PASSION = Color.from_hex('#683d62')
DARK_PURPLE = Color.from_hex('#582147')
GRAPE_JAM = Color.from_hex('#725671')
DEEP_PURPLE = Color.from_hex('#50314c')
WINEBERRY = Color.from_hex('#5a395b')
GRAPE_ROYALE = Color.from_hex('#4f2d54')
PLUM_PURPLE = Color.from_hex('#51304e')
HORTENSIA = Color.from_hex('#553b50')
BLACKBERRY_WINE = Color.from_hex('#4d3246')
NAVY_COSMOS = Color.from_hex('#503b53')
INDIGO = Color.from_hex('#4c3957')
PURPLE_PENNANT = Color.from_hex('#432c47')
PLUM_PERFECT = Color.from_hex('#473442')
SWEET_GRAPE = Color.from_hex('#4b3b4f')
SHADOW_PURPLE = Color.from_hex('#4e334e')
BLACKBERRY_CORDIAL = Color.from_hex('#3f2a47')
PURPLE_REIGN = Color.from_hex('#56456b')
MULBERRY_PURPLE = Color.from_hex('#493c62')
GOTHIC_GRAPE = Color.from_hex('#473951')
GRAPE = Color.from_hex('#433455')
MYSTERIOSO = Color.from_hex('#46394b')
PURPLE_VELVET = Color.from_hex('#41354d')
NIGHTSHADE = Color.from_hex('#433748')
ORCHID_TINT = Color.from_hex('#dbd2db')
LILAC_ASH = Color.from_hex('#d7cdcd')
GRAY_LILAC = Color.from_hex('#d4cacd')
HUSHED_VIOLET = Color.from_hex('#d1c0bf')
CLOUD_GRAY = Color.from_hex('#b7a9ac')
QUAIL = Color.from_hex('#98868c')
NIRVANA = Color.from_hex('#a2919b')
ORCHID_HUSH = Color.from_hex('#cec3d2')
IRIS = Color.from_hex('#baafbc')
SEA_FOG = Color.from_hex('#a5929d')
ELDERBERRY = Color.from_hex('#9d848e')
BLACK_PLUM = Color.from_hex('#6c5765')
FLINT = Color.from_hex('#705861')
SASSAFRAS = Color.from_hex('#54353b')
EVENING_HAZE = Color.from_hex('#bdb8c7')
THISTLE = Color.from_hex('#b9b3c5')
LAVENDER_GRAY = Color.from_hex('#9890a2')
MINIMAL_GRAY = Color.from_hex('#948d99')
PURPLE_ASH = Color.from_hex('#8f8395')
GRAY_RIDGE = Color.from_hex('#847986')
PURPLE_SAGE = Color.from_hex('#75697e')
HEIRLOOM_LILAC = Color.from_hex('#9d96b2')
WISTERIA = Color.from_hex('#a198af')
DUSK = Color.from_hex('#897f98')
DAYBREAK = Color.from_hex('#8981a0')
CADET = Color.from_hex('#6a6378')
MULLED_GRAPE = Color.from_hex('#675a74')
PURPLE_PLUMERIA = Color.from_hex('#473854')
LILAC_MARBLE = Color.from_hex('#c3babf')
ASHES_OF_ROSES = Color.from_hex('#b5acab')
GULL_GRAY = Color.from_hex('#a49ca0')
ZINC = Color.from_hex('#92898a')
GULL = Color.from_hex('#918c8f')
SHARK = Color.from_hex('#6d636b')
SPARROW = Color.from_hex('#69595c')
ORCHID_ICE = Color.from_hex('#e0d0db')
LILAC_SNOW = Color.from_hex('#e0c7d7')
WINSOME_ORCHID = Color.from_hex('#d4b9cb')
FAIR_ORCHID = Color.from_hex('#c0aac0')
LAVENDER_FROST = Color.from_hex('#bdabbe')
ORCHID_PETAL = Color.from_hex('#bfb4cb')
PASTEL_LILAC = Color.from_hex('#bdb0d0')
ORCHID_BLOOM = Color.from_hex('#c5aecf')
ORCHID_BOUQUET = Color.from_hex('#d1acce')
LUPINE = Color.from_hex('#be9cc1')
VIOLET_TULLE = Color.from_hex('#c193c0')
SHEER_LILAC = Color.from_hex('#b793c0')
AFRICAN_VIOLET = Color.from_hex('#b085b7')
DUSTY_LAVENDER = Color.from_hex('#a1759c')
PAISLEY_PURPLE = Color.from_hex('#8b79b1')
HYACINTH = Color.from_hex('#936ca7')
AMETHYST_ORCHID = Color.from_hex('#926aa6')
DEWBERRY = Color.from_hex('#8b5987')
PURPLE_HEART = Color.from_hex('#745587')
MEADOW_VIOLET = Color.from_hex('#764f82')
ROYAL_PURPLE = Color.from_hex('#603f83')
DEEP_LAVENDER = Color.from_hex('#775496')
ROYAL_LILAC = Color.from_hex('#774d8e')
PANSY = Color.from_hex('#653d7c')
BRIGHT_VIOLET = Color.from_hex('#784384')
AMARANTH_PURPLE = Color.from_hex('#6a397b')
PURPLE_MAGIC = Color.from_hex('#663271')
PLUM = Color.from_hex('#5a315d')
IMPERIAL_PALACE = Color.from_hex('#604e7a')
PATRICIAN_PURPLE = Color.from_hex('#6c4e79')
LOGANBERRY = Color.from_hex('#5a4769')
MAJESTY = Color.from_hex('#593761')
IMPERIAL_PURPLE = Color.from_hex('#542c5d')
CROWN_JEWEL = Color.from_hex('#482d54')
PARACHUTE_PURPLE = Color.from_hex('#392852')
LAVENDER_FOG = Color.from_hex('#d2c4d6')
LAVENDULA = Color.from_hex('#bca4cb')
LAVENDER = Color.from_hex('#afa4ce')
BOUGAINVILLEA = Color.from_hex('#9884b9')
VIOLET_TULIP = Color.from_hex('#9e91c3')
CHALK_VIOLET = Color.from_hex('#8f7da5')
PURPLE_HAZE = Color.from_hex('#807396')
SMOKY_GRAPE = Color.from_hex('#b88aac')
REGAL_ORCHID = Color.from_hex('#a98baf')
VIOLA = Color.from_hex('#a692ba')
ORCHID_MIST = Color.from_hex('#917798')
GRAPE_COMPOTE = Color.from_hex('#6b5876')
MONTANA_GRAPE = Color.from_hex('#6c5971')
VINTAGE_VIOLET = Color.from_hex('#634f62')
ASTER_PURPLE = Color.from_hex('#7d74a8')
DAHLIA_PURPLE = Color.from_hex('#7e6eac')
PASSION_FLOWER = Color.from_hex('#6d5698')
ULTRA_VIOLET = Color.from_hex('#5f4b8b')
PRISM_VIOLET = Color.from_hex('#53357d')
HELIOTROPE = Color.from_hex('#4f3872')
PETUNIA = Color.from_hex('#4f3466')
CORSICAN_BLUE = Color.from_hex('#646093')
VERONICA = Color.from_hex('#6d6695')
BLUE_IRIS = Color.from_hex('#5a5b9f')
PURPLE_OPULENCE = Color.from_hex('#60569a')
GENTIAN_VIOLET = Color.from_hex('#544275')
LIBERTY = Color.from_hex('#4d448a')
DEEP_BLUE = Color.from_hex('#44377d')
BLEACHED_DENIM = Color.from_hex('#646f9b')
HERON = Color.from_hex('#62617e')
SKIPPER_BLUE = Color.from_hex('#484a72')
NAVY_BLUE = Color.from_hex('#403f6f')
DEEP_WISTERIA = Color.from_hex('#443f6f')
BLUE_RIBBON = Color.from_hex('#3a395f')
ASTRAL_AURA = Color.from_hex('#363151')
LILAC_HINT = Color.from_hex('#d0d0da')
MISTY_LILAC = Color.from_hex('#bcb4c4')
LAVENDER_BLUE = Color.from_hex('#c5c0d0')
PURPLE_HEATHER = Color.from_hex('#bab8d3')
COSMIC_SKY = Color.from_hex('#aaaac4')
LANGUID_LAVENDER = Color.from_hex('#a2a1ba')
DAPPLE_GRAY = Color.from_hex('#9c9ba7')
SWEET_LAVENDER = Color.from_hex('#9a9bc1')
EASTER_EGG = Color.from_hex('#919bc9')
JACARANDA = Color.from_hex('#848dc5')
DEEP_PERIWINKLE = Color.from_hex('#7c83bc')
DUSTED_PERI = Color.from_hex('#696ba0')
VIOLET_STORM = Color.from_hex('#5c619d')
BAJA_BLUE = Color.from_hex('#5f6db0')
THISTLE_DOWN = Color.from_hex('#9499bb')
PERSIAN_VIOLET = Color.from_hex('#8c8eb2')
TWILIGHT_PURPLE = Color.from_hex('#66648b')
ORIENT_BLUE = Color.from_hex('#47457a')
CLEMATIS_BLUE = Color.from_hex('#363b7c')
ROYAL_BLUE = Color.from_hex('#3d428b')
SPECTRUM_BLUE = Color.from_hex('#3d3c7c')
LAVENDER_VIOLET = Color.from_hex('#767ba5')
BLUE_ICE = Color.from_hex('#70789b')
VELVET_MORNING = Color.from_hex('#60688d')
MARLIN = Color.from_hex('#515b87')
BLUEPRINT = Color.from_hex('#2d3359')
BLUE_DEPTHS = Color.from_hex('#263056')
MEDIEVAL_BLUE = Color.from_hex('#29304e')
LAVENDER_AURA = Color.from_hex('#9f99aa')
STONEWASH = Color.from_hex('#74809a')
NIGHTSHADOW_BLUE = Color.from_hex('#4e5368')
BLUE_INDIGO = Color.from_hex('#49516d')
GRAYSTONE = Color.from_hex('#4d495b')
CROWN_BLUE = Color.from_hex('#464b65')
DEEP_COBALT = Color.from_hex('#404466')
ARCTIC_ICE = Color.from_hex('#bfc7d6')
GRAY_DAWN = Color.from_hex('#bbc1cc')
HEATHER = Color.from_hex('#b7c0d6')
EVENTIDE = Color.from_hex('#959eb7')
SILVER_LAKE_BLUE = Color.from_hex('#618bb9')
BLUE_BONNET = Color.from_hex('#6384b8')
BLUE_YONDER = Color.from_hex('#5a77a8')
LAVENDER_LUSTRE = Color.from_hex('#8c9cc1')
PURPLE_IMPRESSION = Color.from_hex('#858fb1')
GRAPEMIST = Color.from_hex('#8398ca')
VISTA_BLUE = Color.from_hex('#81a0d4')
CORNFLOWER_BLUE = Color.from_hex('#7391c8')
PERSIAN_JEWEL = Color.from_hex('#6e81be')
WEDGEWOOD = Color.from_hex('#6479b3')
SKYWAY = Color.from_hex('#adbed3')
CASHMERE_BLUE = Color.from_hex('#a5b8d0')
BLUE_BELL = Color.from_hex('#93b4d7')
PLACID_BLUE = Color.from_hex('#8cadd3')
DELLA_ROBBIA_BLUE = Color.from_hex('#7a9dcb')
PROVENCE = Color.from_hex('#658dc6')
ULTRAMARINE = Color.from_hex('#5b7ebd')
ALLURE = Color.from_hex('#7291b4')
COLONY_BLUE = Color.from_hex('#65769a')
MOONLIGHT_BLUE = Color.from_hex('#506886')
DUTCH_BLUE = Color.from_hex('#4a638d')
DELFT = Color.from_hex('#3d5e8c')
LIMOGES = Color.from_hex('#243f6c')
ESTATE_BLUE = Color.from_hex('#233658')
INFINITY = Color.from_hex('#6e7e99')
BIJOU_BLUE = Color.from_hex('#4e5e7f')
COASTAL_FJORD = Color.from_hex('#505d7e')
TRUE_NAVY = Color.from_hex('#3f5277')
ENSIGN_BLUE = Color.from_hex('#384c67')
DARK_DENIM = Color.from_hex('#35465e')
INSIGNIA_BLUE = Color.from_hex('#2f3e55')
AIR_BLUE = Color.from_hex('#77acc7')
HERITAGE_BLUE = Color.from_hex('#5d96bc')
ETHEREAL_BLUE = Color.from_hex('#5ca6ce')
BONNIE_BLUE = Color.from_hex('#539ccc')
CENDRE_BLUE = Color.from_hex('#3e7fa5')
PARISIAN_BLUE = Color.from_hex('#4f7ca4')
FAIENCE = Color.from_hex('#2a6a8b')
ALASKAN_BLUE = Color.from_hex('#6da9d2')
LITTLE_BOY_BLUE = Color.from_hex('#6ea2d5')
AZURE_BLUE = Color.from_hex('#4d91c6')
RIVIERA = Color.from_hex('#5879a2')
FEDERAL_BLUE = Color.from_hex('#43628b')
STAR_SAPPHIRE = Color.from_hex('#386192')
BRIGHT_COBALT = Color.from_hex('#385d8d')
DUSK_BLUE = Color.from_hex('#7ba0c0')
REGATTA = Color.from_hex('#487ab7')
PALACE_BLUE = Color.from_hex('#346cb0')
STRONG_BLUE = Color.from_hex('#1f5da0')
TURKISH_SEA = Color.from_hex('#195190')
OLYMPIAN_BLUE = Color.from_hex('#1a4c8b')
CLASSIC_BLUE = Color.from_hex('#0f4c81')
MARINA = Color.from_hex('#4f84c4')
CAMPANULA = Color.from_hex('#3272af')
DAPHNE = Color.from_hex('#0f5f9a')
VICTORIA_BLUE = Color.from_hex('#08589d')
SNORKEL_BLUE = Color.from_hex('#034f84')
NAUTICAL_BLUE = Color.from_hex('#1a5091')
PRINCESS_BLUE = Color.from_hex('#00539c')
DAZZLING_BLUE = Color.from_hex('#3850a0')
AMPARO_BLUE = Color.from_hex('#4960a8')
DEEP_ULTRAMARINE = Color.from_hex('#384883')
SURF_THE_WEB = Color.from_hex('#203c7f')
MAZARINE_BLUE = Color.from_hex('#273c76')
TRUE_BLUE = Color.from_hex('#1e4477')
TWILIGHT_BLUE = Color.from_hex('#313d64')
KENTUCKY_BLUE = Color.from_hex('#a5b3cc')
CERULEAN = Color.from_hex('#9bb7d4')
POWDER_BLUE = Color.from_hex('#96b3d2')
FOREVER_BLUE = Color.from_hex('#899bb8')
TEMPEST = Color.from_hex('#79839b')
COUNTRY_BLUE = Color.from_hex('#717f9b')
ENGLISH_MANOR = Color.from_hex('#7181a4')
ILLUSION_BLUE = Color.from_hex('#c9d3dc')
BALLAD_BLUE = Color.from_hex('#c0ceda')
BABY_BLUE = Color.from_hex('#b5c7d3')
CELESTIAL_BLUE = Color.from_hex('#a3b4c4')
BLUE_FOG = Color.from_hex('#9babbb')
FLINT_STONE = Color.from_hex('#677283')
FOLKSTONE_GRAY = Color.from_hex('#626879')
PEARL_BLUE = Color.from_hex('#b0b7be')
MONUMENT = Color.from_hex('#84898c')
DARK_SLATE = Color.from_hex('#46515a')
MIDNIGHT_NAVY = Color.from_hex('#34414e')
TOTAL_ECLIPSE = Color.from_hex('#2c313d')
BLUE_GRAPHITE = Color.from_hex('#323137')
DARK_NAVY = Color.from_hex('#232f36')
ICE_FLOW = Color.from_hex('#c6d2d2')
QUARRY = Color.from_hex('#98a0a5')
GRIFFIN = Color.from_hex('#8d8f8f')
DARK_SHADOW = Color.from_hex('#4a4b4d')
OMBRE_BLUE = Color.from_hex('#434854')
INDIA_INK = Color.from_hex('#3c3f4a')
EBONY = Color.from_hex('#41424a')
PATRIOT_BLUE = Color.from_hex('#363756')
ECLIPSE = Color.from_hex('#343148')
MOOD_INDIGO = Color.from_hex('#353a4c')
PEACOAT = Color.from_hex('#2b2e43')
BLACK_IRIS = Color.from_hex('#2b3042')
DRESS_BLUES = Color.from_hex('#2a3244')
BLUE_NIGHTS = Color.from_hex('#363b48')
ANGEL_FALLS = Color.from_hex('#a3bdd3')
DREAM_BLUE = Color.from_hex('#a0bcd0')
ASHLEY_BLUE = Color.from_hex('#8699ab')
DUSTY_BLUE = Color.from_hex('#8c9dad')
INDIAN_TEAL = Color.from_hex('#3c586b')
STARGAZER = Color.from_hex('#39505c')
ORION_BLUE = Color.from_hex('#3e4f5c')
FORGET_ME_NOT = Color.from_hex('#8fadbd')
FADED_DENIM = Color.from_hex('#798ea4')
BLUE_SHADOW = Color.from_hex('#66829a')
CORONET_BLUE = Color.from_hex('#59728e')
CAPTAINS_BLUE = Color.from_hex('#557088')
COPEN_BLUE = Color.from_hex('#516b84')
CHINA_BLUE = Color.from_hex('#546477')
ADRIATIC_BLUE = Color.from_hex('#5c899b')
PROVINCIAL_BLUE = Color.from_hex('#5c798e')
NIAGARA = Color.from_hex('#5487a4')
BLUE_HEAVEN = Color.from_hex('#5b7e98')
STELLAR = Color.from_hex('#46647e')
REAL_TEAL = Color.from_hex('#405d73')
MAJOLICA_BLUE = Color.from_hex('#274357')
STARLIGHT_BLUE = Color.from_hex('#b5ced4')
WINTER_SKY = Color.from_hex('#a9c0cb')
STRATOSPHERE = Color.from_hex('#9ec1cc')
STERLING_BLUE = Color.from_hex('#a2b9c2')
ARONA = Color.from_hex('#879ba3')
CITADEL = Color.from_hex('#748995')
BLUE_MIRAGE = Color.from_hex('#5c6d7c')
CLOUD_BLUE = Color.from_hex('#a2b6b9')
ETHER = Color.from_hex('#9eb6b8')
CAMEO_BLUE = Color.from_hex('#769da6')
STONE_BLUE = Color.from_hex('#829ca5')
TOURMALINE = Color.from_hex('#86a1a9')
SMOKE_BLUE = Color.from_hex('#6d8994')
BLUESTONE = Color.from_hex('#577284')
AQUAMARINE = Color.from_hex('#9dc3d4')
SKY_BLUE = Color.from_hex('#8abad3')
MILKY_BLUE = Color.from_hex('#72a8ba')
BLUE_GROTTO = Color.from_hex('#5cacce')
NORSE_BLUE = Color.from_hex('#4ca5c7')
AQUARIUS = Color.from_hex('#3cadd4')
MAUI_BLUE = Color.from_hex('#52a2b4')
BLUE_MIST = Color.from_hex('#5bacc3')
RIVER_BLUE = Color.from_hex('#38afcd')
CYAN_BLUE = Color.from_hex('#14a3c7')
HORIZON_BLUE = Color.from_hex('#289dbe')
BLUE_MOON = Color.from_hex('#3686a0')
BLUEJAY = Color.from_hex('#157ea0')
MEDITERRANEAN_BLUE = Color.from_hex('#1478a7')
BACHELOR_BUTTON = Color.from_hex('#4abbd5')
BLUE_ATOLL = Color.from_hex('#00b1d2')
VIVID_BLUE = Color.from_hex('#0088b0')
HAWAIIAN_OCEAN = Color.from_hex('#008db9')
BLUE_DANUBE = Color.from_hex('#0087b6')
BLUE_JEWEL = Color.from_hex('#007baa')
METHYL_BLUE = Color.from_hex('#0074a8')
MALIBU_BLUE = Color.from_hex('#008cc1')
BLITHE = Color.from_hex('#0084bd')
SWEDISH_BLUE = Color.from_hex('#007eb1')
DRESDEN_BLUE = Color.from_hex('#0086bb')
DIVA_BLUE = Color.from_hex('#007bb2')
BLUE_ASTER = Color.from_hex('#0077b3')
CLOISONNE = Color.from_hex('#0075af')
FRENCH_BLUE = Color.from_hex('#0072b5')
BRILLIANT_BLUE = Color.from_hex('#0075b3')
DIRECTOIRE_BLUE = Color.from_hex('#0061a3')
SKYDIVER = Color.from_hex('#00589b')
IMPERIAL_BLUE = Color.from_hex('#005a92')
DEEP_WATER = Color.from_hex('#266691')
DARK_BLUE = Color.from_hex('#305679')
PASTEL_BLUE = Color.from_hex('#bcd3d5')
CLEARWATER = Color.from_hex('#aad5db')
BLUE_GLOW = Color.from_hex('#b2d4dd')
PLUME = Color.from_hex('#a5cfd5')
PORCELAIN_BLUE = Color.from_hex('#95c0cb')
CRYSTAL_BLUE = Color.from_hex('#a1c8db')
PETIT_FOUR = Color.from_hex('#87c2d4')
WAN_BLUE = Color.from_hex('#cbdcdf')
WHISPERING_BLUE = Color.from_hex('#c9dcdc')
SKYLIGHT = Color.from_hex('#c8e0e0')
AQUATIC = Color.from_hex('#99c1cc')
MARINE_BLUE = Color.from_hex('#76afb6')
REEF_WATERS = Color.from_hex('#6f9fa9')
ARCTIC = Color.from_hex('#648589')
CHALK_BLUE = Color.from_hex('#ccdad7')
PALE_BLUE = Color.from_hex('#c4d6d3')
MISTY_BLUE = Color.from_hex('#bfcdcc')
SKY_GRAY = Color.from_hex('#bcc8c6')
SURF_SPRAY = Color.from_hex('#b4c8c2')
GRAY_MIST = Color.from_hex('#99aeae')
AQUIFER = Color.from_hex('#89acac')
BLUE_GLASS = Color.from_hex('#c6e3e1')
ICY_MORN = Color.from_hex('#b0d3d1')
CANAL_BLUE = Color.from_hex('#9cc2c5')
PASTEL_TURQUOISE = Color.from_hex('#99c5c4')
AQUA_HAZE = Color.from_hex('#87b9bc')
AQUA_SEA = Color.from_hex('#6baaae')
MEADOWBROOK = Color.from_hex('#60a0a3')
GLACIER = Color.from_hex('#c3dbd4')
FAIR_AQUA = Color.from_hex('#b8e2dc')
SOOTHING_SEA = Color.from_hex('#c3e9e4')
BLEACHED_AQUA = Color.from_hex('#bce3df')
BLUE_LIGHT = Color.from_hex('#acdfdd')
BLUE_TINT = Color.from_hex('#9fd9d7')
AQUA_SKY = Color.from_hex('#7bc4c4')
MORNING_MIST = Color.from_hex('#cfdfdb')
HARBOR_GRAY = Color.from_hex('#a8c0bb')
EGGSHELL_BLUE = Color.from_hex('#a3ccc9')
DUSTY_TURQUOISE = Color.from_hex('#649b9e')
PORCELAIN = Color.from_hex('#5d9ca4')
BRITTANY_BLUE = Color.from_hex('#4c7e86')
HYDRO = Color.from_hex('#426972')
BLUE_HAZE = Color.from_hex('#a5bcbb')
NILE_BLUE = Color.from_hex('#76a7ab')
MINERAL_BLUE = Color.from_hex('#6d9192')
BRISTOL_BLUE = Color.from_hex('#558f91')
TEAL = Color.from_hex('#478589')
BLUE_SPRUCE = Color.from_hex('#486b67')
SAGEBRUSH_GREEN = Color.from_hex('#567572')
GREEN_MILIEU = Color.from_hex('#8a9992')
JADEITE = Color.from_hex('#95a69f')
BLUE_SURF = Color.from_hex('#90a8a4')
OIL_BLUE = Color.from_hex('#658c88')
TRELLIS = Color.from_hex('#6a8988')
NORTH_ATLANTIC = Color.from_hex('#536d70')
SEA_PINE = Color.from_hex('#4c6969')
SLATE = Color.from_hex('#8c9fa1')
SILVER_BLUE = Color.from_hex('#8a9a9a')
ABYSS = Color.from_hex('#8f9e9d')
LEAD = Color.from_hex('#7a898f')
STORMY_SEA = Color.from_hex('#6e8082')
TROOPER = Color.from_hex('#697a7e')
GOBLIN_BLUE = Color.from_hex('#5f7278')
SLATE_GRAY = Color.from_hex('#8a9691')
CHINOIS_GREEN = Color.from_hex('#7c8c87')
DARK_FOREST = Color.from_hex('#556962')
BALSAM_GREEN = Color.from_hex('#576664')
BEETLE = Color.from_hex('#55584c')
URBAN_CHIC = Color.from_hex('#464e4d')
DARKEST_SPRUCE = Color.from_hex('#303d3c')
MALLARD_BLUE = Color.from_hex('#3a5c6e')
CELESTIAL = Color.from_hex('#006380')
SAXONY_BLUE = Color.from_hex('#1f6680')
LYONS_BLUE = Color.from_hex('#005871')
INK_BLUE = Color.from_hex('#0b5369')
CORSAIR = Color.from_hex('#18576c')
LEGION_BLUE = Color.from_hex('#1f495b')
AEGEAN_BLUE = Color.from_hex('#4e6e81')
BLUESTEEL = Color.from_hex('#35637c')
BLUE_ASHES = Color.from_hex('#3b5f78')
MIDNIGHT = Color.from_hex('#325b74')
BLUE_SAPPHIRE = Color.from_hex('#09577b')
SEAPORT = Color.from_hex('#005e7d')
MOROCCAN_BLUE = Color.from_hex('#0f4e67')
OCEAN_DEPTHS = Color.from_hex('#006175')
BLUE_CORAL = Color.from_hex('#1b5366')
DRAGONFLY = Color.from_hex('#2a5c6a')
PACIFIC = Color.from_hex('#1f595c')
BALSAM = Color.from_hex('#33565e')
MEDITERRANEA = Color.from_hex('#32575d')
ATLANTIC_DEEP = Color.from_hex('#274e55')
AQUA = Color.from_hex('#64a1ad')
STILLWATER = Color.from_hex('#70a4b0')
DELPHINIUM_BLUE = Color.from_hex('#6198ae')
LARKSPUR = Color.from_hex('#3c7d90')
STORM_BLUE = Color.from_hex('#47788a')
TAPESTRY = Color.from_hex('#436573')
COLONIAL_BLUE = Color.from_hex('#2d6471')
PEACOCK_BLUE = Color.from_hex('#00a0b0')
CAPRI_BREEZE = Color.from_hex('#008799')
ALGIERS_BLUE = Color.from_hex('#00859c')
CANEEL_BAY = Color.from_hex('#00849f')
CARIBBEAN_SEA = Color.from_hex('#00819d')
MOSAIC_BLUE = Color.from_hex('#00758f')
TURKISH_TILE = Color.from_hex('#00698b')
ANGEL_BLUE = Color.from_hex('#83c5cd')
BLUE_RADIANCE = Color.from_hex('#58c9d4')
CAPRI = Color.from_hex('#44bbca')
BLUE_CURACAO = Color.from_hex('#32becc')
SCUBA_BLUE = Color.from_hex('#00abc0')
BLUEBIRD = Color.from_hex('#009dae')
ENAMEL_BLUE = Color.from_hex('#007a8e')
POOL_BLUE = Color.from_hex('#67bcb3')
BLUE_TURQUOISE = Color.from_hex('#53b0ae')
BALTIC = Color.from_hex('#279d9f')
LAKE_BLUE = Color.from_hex('#008c96')
TILE_BLUE = Color.from_hex('#008491')
PAGODA_BLUE = Color.from_hex('#1a7f8e')
BISCAY_BAY = Color.from_hex('#097988')
ARUBA_BLUE = Color.from_hex('#81d7d3')
CERAMIC = Color.from_hex('#00aaa9')
VIRIDIAN_GREEN = Color.from_hex('#009499')
TROPICAL_GREEN = Color.from_hex('#008786')
NAVIGATE = Color.from_hex('#008583')
DEEP_PEACOCK_BLUE = Color.from_hex('#008381')
LAPIS = Color.from_hex('#008684')
TURQUOISE = Color.from_hex('#45b5aa')
WATERFALL = Color.from_hex('#3ab0a2')
LAGOON = Color.from_hex('#4d9e9a')
BRIGHT_AQUA = Color.from_hex('#30a299')
PORCELAIN_GREEN = Color.from_hex('#108780')
BLUE_GRASS = Color.from_hex('#007c7a')
FANFARE = Color.from_hex('#006d70')
ATLANTIS = Color.from_hex('#00af9f')
POOL_GREEN = Color.from_hex('#00af9d')
DYNASTY_GREEN = Color.from_hex('#008e80')
SPECTRA_GREEN = Color.from_hex('#009b8c')
COLUMBIA = Color.from_hex('#009288')
TEAL_BLUE = Color.from_hex('#007f7c')
PARASAILING = Color.from_hex('#00736c')
WASABI = Color.from_hex('#73a89e')
BERYL_GREEN = Color.from_hex('#619187')
DEEP_SEA = Color.from_hex('#4f7c74')
BOTTLE_GREEN = Color.from_hex('#427d6d')
GALAPAGOS_GREEN = Color.from_hex('#29685f')
ANTIQUE_GREEN = Color.from_hex('#29675c')
STORM = Color.from_hex('#035453')
MARINE_GREEN = Color.from_hex('#40a48e')
SEA_GREEN = Color.from_hex('#149c88')
GREENLAKE = Color.from_hex('#007d69')
TIDEPOOL = Color.from_hex('#0a6f69')
IVY = Color.from_hex('#226c63')
CADMIUM_GREEN = Color.from_hex('#00675b')
ALPINE_GREEN = Color.from_hex('#005f56')
CANTON = Color.from_hex('#6da29e')
AGATE_GREEN = Color.from_hex('#599f99')
SEA_BLUE = Color.from_hex('#549f98')
LATIGO_BAY = Color.from_hex('#379190')
GREEN_BLUE_SLATE = Color.from_hex('#358082')
BAYOU = Color.from_hex('#20706f')
NORTH_SEA = Color.from_hex('#316c6b')
DEEP_JUNGLE = Color.from_hex('#36716f')
EVERGLADE = Color.from_hex('#005b5d')
TEAL_GREEN = Color.from_hex('#006361')
HARBOR_BLUE = Color.from_hex('#00656e')
DEEP_LAKE = Color.from_hex('#00656b')
SHADED_SPRUCE = Color.from_hex('#00585e')
DEEP_TEAL = Color.from_hex('#18454b')
SILVER_PINE = Color.from_hex('#4e6866')
MALLARD_GREEN = Color.from_hex('#405e5c')
BISTRO_GREEN = Color.from_hex('#395551')
JASPER = Color.from_hex('#335959')
BAYBERRY = Color.from_hex('#255958')
JUNE_BUG = Color.from_hex('#264a48')
PONDEROSA_PINE = Color.from_hex('#203b3d')
AQUA_GLASS = Color.from_hex('#d2e8e0')
OPAL_BLUE = Color.from_hex('#c3ddd6')
DUSTY_AQUA = Color.from_hex('#c0dccd')
OCEAN_WAVE = Color.from_hex('#8ec5b6')
HOLIDAY = Color.from_hex('#81c3b4')
CASCADE = Color.from_hex('#76c1b2')
DUSTY_JADE_GREEN = Color.from_hex('#7bb5a3')
HONEYDEW = Color.from_hex('#bae1d3')
BROOK_GREEN = Color.from_hex('#afddcc')
CABBAGE = Color.from_hex('#87d7be')
BEVELED_GLASS = Color.from_hex('#7accb8')
OPAL = Color.from_hex('#77cfb7')
BISCAY_GREEN = Color.from_hex('#55c6a9')
SPEARMINT = Color.from_hex('#64bfa4')
MOONLIGHT_JADE = Color.from_hex('#c7e5df')
BAY = Color.from_hex('#bae5d6')
YUCCA = Color.from_hex('#a1d7c9')
BEACH_GLASS = Color.from_hex('#96dfce')
ICE_GREEN = Color.from_hex('#87d8c3')
COCKATOO = Color.from_hex('#58c8b6')
FLORIDA_KEYS = Color.from_hex('#56beab')
BERMUDA = Color.from_hex('#60c9b3')
ELECTRIC_GREEN = Color.from_hex('#4bc3a8')
AQUA_GREEN = Color.from_hex('#00b89f')
BILLIARD = Color.from_hex('#00aa92')
ARCADIA = Color.from_hex('#00a28a')
ALHAMBRA = Color.from_hex('#008778')
DEEP_GREEN = Color.from_hex('#009276')
MINT_LEAF = Color.from_hex('#00b694')
PEACOCK_GREEN = Color.from_hex('#00a78b')
VIVID_GREEN = Color.from_hex('#009e82')
EMERALD = Color.from_hex('#009473')
VIRIDIS = Color.from_hex('#00846b')
SHADY_GLADE = Color.from_hex('#006e5b')
ULTRAMARINE_GREEN = Color.from_hex('#006b54')
SILT_GREEN = Color.from_hex('#a9bdb1')
FROSTY_GREEN = Color.from_hex('#a3b5a6')
ICEBERG_GREEN = Color.from_hex('#8c9c92')
GRANITE_GREEN = Color.from_hex('#86a293')
GREEN_BAY = Color.from_hex('#7e9285')
LILY_PAD = Color.from_hex('#818f84')
LAUREL_WREATH = Color.from_hex('#616f65')
GREEN_SPRUCE = Color.from_hex('#589f7e')
COMFREY = Color.from_hex('#5b7961')
DARK_IVY = Color.from_hex('#5b7763')
FOLIAGE_GREEN = Color.from_hex('#3e6f58')
MYRTLE = Color.from_hex('#4f6b58')
POSY_GREEN = Color.from_hex('#325b51')
PINENEEDLE = Color.from_hex('#334d41')
SEA_SPRAY = Color.from_hex('#717e6f')
DUCK_GREEN = Color.from_hex('#53665c')
FROSTY_SPRUCE = Color.from_hex('#578270')
FIR = Color.from_hex('#3a725f')
EVERGREEN = Color.from_hex('#11574a')
HUNTER_GREEN = Color.from_hex('#335749')
DARK_GREEN = Color.from_hex('#314f40')
FELDSPAR = Color.from_hex('#729b8b')
SMOKE_PINE = Color.from_hex('#3e6257')
TREKKING_GREEN = Color.from_hex('#355048')
GARDEN_TOPIARY = Color.from_hex('#3e524b')
JUNGLE_GREEN = Color.from_hex('#3c4e47')
SYCAMORE = Color.from_hex('#35463d')
GREEN_GABLES = Color.from_hex('#324241')
VETIVER = Color.from_hex('#807d6f')
DEEP_LICHEN_GREEN = Color.from_hex('#6e6e5c')
THYME = Color.from_hex('#50574c')
KOMBU_GREEN = Color.from_hex('#3a4032')
DEEP_FOREST = Color.from_hex('#37413a')
FOREST_NIGHT = Color.from_hex('#434237')
ROSIN = Color.from_hex('#36362d')
CELADON = Color.from_hex('#b8ccba')
PALE_AQUA = Color.from_hex('#c1ccc2')
SMOKE = Color.from_hex('#bfc8c3')
FOGGY_DEW = Color.from_hex('#d1d5d0')
MERCURY = Color.from_hex('#bac2ba')
MINERAL_GRAY = Color.from_hex('#b2b6ac')
AQUA_GRAY = Color.from_hex('#a5b2aa')
FAIREST_JADE = Color.from_hex('#d8e3d7')
WATER_LILY = Color.from_hex('#dde3d5')
CANARY_GREEN = Color.from_hex('#d6dec9')
ALMOST_AQUA = Color.from_hex('#cad3c1')
GREEN_TINT = Color.from_hex('#c5ccc0')
SEA_FOAM = Color.from_hex('#b7c2b2')
DESERT_SAGE = Color.from_hex('#a7ae9e')
WHISPER_GREEN = Color.from_hex('#e0e6d7')
CELADON_TINT = Color.from_hex('#cbcebe')
DEWKIST = Color.from_hex('#c4d1c2')
GREEN_LILY = Color.from_hex('#c1cec1')
CAMEO_GREEN = Color.from_hex('#aac0ad')
SEAGRASS = Color.from_hex('#959889')
SHADOW = Color.from_hex('#888d82')
CLEARLY_AQUA = Color.from_hex('#cee1d4')
MISTY_JADE = Color.from_hex('#bcd9c8')
SUBTLE_GREEN = Color.from_hex('#b5cbbb')
AQUA_FOAM = Color.from_hex('#adc3b4')
GOSSAMER_GREEN = Color.from_hex('#b2cfbe')
LICHEN = Color.from_hex('#9bc2b1')
GRAYED_JADE = Color.from_hex('#9bbea9')
MILKY_GREEN = Color.from_hex('#cfdbd1')
PHANTOM_GREEN = Color.from_hex('#dce4d7')
MIST_GREEN = Color.from_hex('#aacebc')
BIRDS_EGG_GREEN = Color.from_hex('#aaccb9')
BOK_CHOY = Color.from_hex('#bccab3')
SMOKE_GREEN = Color.from_hex('#a8bba2')
MALACHITE_GREEN = Color.from_hex('#709a89')
MISTLETOE = Color.from_hex('#8aa282')
BASIL = Color.from_hex('#879f84')
MINERAL_GREEN = Color.from_hex('#7a9b78')
GREEN_EYES = Color.from_hex('#7d956d')
TURF_GREEN = Color.from_hex('#6f8c69')
WATERCRESS = Color.from_hex('#748c69')
ELM_GREEN = Color.from_hex('#547053')
HEDGE_GREEN = Color.from_hex('#768a75')
LODEN_FROST = Color.from_hex('#788f74')
SHALE_GREEN = Color.from_hex('#739072')
KASHMIR = Color.from_hex('#6f8d6a')
STONE_GREEN = Color.from_hex('#658e67')
ENGLISH_IVY = Color.from_hex('#61845b')
DEEP_GRASS_GREEN = Color.from_hex('#558367')
PIQUANT_GREEN = Color.from_hex('#769358')
FOREST_GREEN = Color.from_hex('#6b8d53')
FLUORITE_GREEN = Color.from_hex('#699158')
CACTUS = Color.from_hex('#53713d')
GARDEN_GREEN = Color.from_hex('#495e35')
ARTICHOKE_GREEN = Color.from_hex('#4b6d41')
WILLOW_BOUGH = Color.from_hex('#59754d')
ASPEN_GREEN = Color.from_hex('#7e9b76')
MEDIUM_GREEN = Color.from_hex('#3c824e')
JUNIPER = Color.from_hex('#3d7245')
FAIRWAY = Color.from_hex('#477050')
VINEYARD_GREEN = Color.from_hex('#5f7355')
DILL = Color.from_hex('#6f7755')
GREENER_PASTURES = Color.from_hex('#37503d')
FOUR_LEAF_CLOVER = Color.from_hex('#616652')
BRONZE_GREEN = Color.from_hex('#525f48')
CHIVE = Color.from_hex('#4a5335')
CYPRESS = Color.from_hex('#545a3e')
BLACK_FOREST = Color.from_hex('#414f3c')
RIFLE_GREEN = Color.from_hex('#414832')
DUFFEL_BAG = Color.from_hex('#394034')
AMBROSIA = Color.from_hex('#d2e7ca')
SPRAY = Color.from_hex('#bed3bb')
PASTEL_GREEN = Color.from_hex('#b4d3b2')
HEMLOCK = Color.from_hex('#97c1a1')
SPRUCESTONE = Color.from_hex('#9fc09c')
MEADOW = Color.from_hex('#8bba94')
JADESHEEN = Color.from_hex('#77a276')
GREEN_ASH = Color.from_hex('#a0daa9')
GREENGAGE = Color.from_hex('#8bc28c')
MING = Color.from_hex('#7cb08a')
ZEPHYR_GREEN = Color.from_hex('#7cb083')
PEAPOD = Color.from_hex('#82b185')
LIGHT_GRASS_GREEN = Color.from_hex('#7cb68e')
ABSINTHE_GREEN = Color.from_hex('#76b583')
NEPTUNE_GREEN = Color.from_hex('#7fbb9e')
CREME_DE_MENTHE = Color.from_hex('#70a38d')
WINTER_GREEN = Color.from_hex('#4f9e81')
GUMDROP_GREEN = Color.from_hex('#2ea785')
HOLLY_GREEN = Color.from_hex('#0f9d76')
PARAKEET = Color.from_hex('#008c69')
GOLF_GREEN = Color.from_hex('#008763')
SPRING_BUD = Color.from_hex('#6bcd9c')
KATYDID = Color.from_hex('#66bc91')
JADE_CREAM = Color.from_hex('#60b892')
MING_GREEN = Color.from_hex('#3aa278')
GREENBRIAR = Color.from_hex('#4b9b69')
LEPRECHAUN = Color.from_hex('#378661')
PINE_GREEN = Color.from_hex('#3a795e')
BLARNEY = Color.from_hex('#00a776')
MINT = Color.from_hex('#00a170')
DEEP_MINT = Color.from_hex('#009e6d')
SIMPLY_GREEN = Color.from_hex('#009b75')
PEPPER_GREEN = Color.from_hex('#007d60')
BOSPHORUS = Color.from_hex('#007558')
VERDANT_GREEN = Color.from_hex('#12674a')
SEACREST = Color.from_hex('#bfd1b3')
GLEAM = Color.from_hex('#bfd1ad')
NILE_GREEN = Color.from_hex('#a7c796')
QUIET_GREEN = Color.from_hex('#9ebc97')
FAIR_GREEN = Color.from_hex('#92af88')
FOREST_SHADE = Color.from_hex('#91ac80')
JADE_GREEN = Color.from_hex('#759465')
PATINA_GREEN = Color.from_hex('#b9eab3')
PISTACHIO_GREEN = Color.from_hex('#a9d39e')
ARCADIAN_GREEN = Color.from_hex('#a3c893')
GRASS_GREEN = Color.from_hex('#7bb369')
BUD_GREEN = Color.from_hex('#79b465')
GREEN_TEA = Color.from_hex('#86a96f')
TENDRIL = Color.from_hex('#89a06b')
PARADISE_GREEN = Color.from_hex('#b2e79f')
LIME_GREEN = Color.from_hex('#9fc131')
JASMINE_GREEN = Color.from_hex('#7ec845')
GREEN_FLASH = Color.from_hex('#79c753')
CLASSIC_GREEN = Color.from_hex('#39a845')
ONLINE_LIME = Color.from_hex('#44883c')
TREETOP = Color.from_hex('#476a30')
SUMMER_GREEN = Color.from_hex('#7ed37f')
SPRING_BOUQUET = Color.from_hex('#6dce87')
ISLAND_GREEN = Color.from_hex('#2bae66')
IRISH_GREEN = Color.from_hex('#45be76')
SHAMROCK = Color.from_hex('#6fa26b')
PEPPERMINT = Color.from_hex('#699e6d')
MINT_GREEN = Color.from_hex('#487d49')
POISON_GREEN = Color.from_hex('#4db560')
VIBRANT_GREEN = Color.from_hex('#55a860')
KELLY_GREEN = Color.from_hex('#339c5e')
BRIGHT_GREEN = Color.from_hex('#009b5c')
FERN_GREEN = Color.from_hex('#008c45')
JELLY_BEAN = Color.from_hex('#008658')
AMAZON = Color.from_hex('#1f7349')
GREEN_GLOW = Color.from_hex('#b0c965')
BRIGHT_LIME_GREEN = Color.from_hex('#97bc62')
GREENERY = Color.from_hex('#88b04b')
FOLIAGE = Color.from_hex('#75a14f')
PERIDOT = Color.from_hex('#819548')
MEADOW_GREEN = Color.from_hex('#739957')
WOODBINE = Color.from_hex('#7b7f32')
JADE_LIME = Color.from_hex('#a1ca7b')
HERBAL_GARDEN = Color.from_hex('#9cad60')
LEAF_GREEN = Color.from_hex('#9faf6c')
PARROT_GREEN = Color.from_hex('#8db051')
DARK_CITRON = Color.from_hex('#a0ac4f')
MACAW_GREEN = Color.from_hex('#9bb53e')
KIWI = Color.from_hex('#7aab55')
SHARP_GREEN = Color.from_hex('#c6ec7a')
DAIQUIRI_GREEN = Color.from_hex('#c9d77e')
WILD_LIME = Color.from_hex('#c3d363')
LINDEN_GREEN = Color.from_hex('#c4bf71')
BRIGHT_CHARTREUSE = Color.from_hex('#b5bf50')
TENDER_SHOOTS = Color.from_hex('#b5cc39')
LIME_PUNCH = Color.from_hex('#c0d725')
SUNNY_LIME = Color.from_hex('#dfef87')
LIMEADE = Color.from_hex('#d3d95f')
SULPHUR_SPRING = Color.from_hex('#d5d717')
CITRONELLE = Color.from_hex('#b8af23')
APPLE_GREEN = Color.from_hex('#b5b644')
WARM_OLIVE = Color.from_hex('#c7b63c')
ANTIQUE_MOSS = Color.from_hex('#b9a023')
LIME_CREAM = Color.from_hex('#d7e8bc')
SHADOW_LIME = Color.from_hex('#cfe09d')
LIME_SHERBET = Color.from_hex('#cdd78a')
LETTUCE_GREEN = Color.from_hex('#bed38e')
SAP_GREEN = Color.from_hex('#afcb80')
OPALINE_GREEN = Color.from_hex('#a3c57d')
WINTER_PEAR = Color.from_hex('#b0b487')
SYLVAN_GREEN = Color.from_hex('#e7eacb')
GLASS_GREEN = Color.from_hex('#ecead0')
GREEN_ESSENCE = Color.from_hex('#e9eac8')
ETHEREAL_GREEN = Color.from_hex('#f1ecca')
GARDEN_GLADE = Color.from_hex('#dcd8a8')
HAY = Color.from_hex('#d3cca3')
PALE_GREEN = Color.from_hex('#cbce91')
YOUNG_WHEAT = Color.from_hex('#e1e3a9')
CITRON = Color.from_hex('#dfde9b')
LUMINARY_GREEN = Color.from_hex('#e3eaa5')
PALE_LIME_YELLOW = Color.from_hex('#dfe69f')
CHARDONNAY = Color.from_hex('#e7df99')
LIMA_BEAN = Color.from_hex('#e1d590')
CHARLOCK = Color.from_hex('#e5e790')
MELLOW_GREEN = Color.from_hex('#d5d593')
SHADOW_GREEN = Color.from_hex('#cfc486')
CELERY_GREEN = Color.from_hex('#c5cc7b')
GREEN_BANANA = Color.from_hex('#babc72')
GREEN_OASIS = Color.from_hex('#b0b454')
LEEK_GREEN = Color.from_hex('#b7b17a')
WEEPING_WILLOW = Color.from_hex('#b3b17b')
PALM = Color.from_hex('#afaf5e')
GOLDEN_OLIVE = Color.from_hex('#af9841')
OASIS = Color.from_hex('#a3a04e')
MOSS = Color.from_hex('#a09d59')
AMBER_GREEN = Color.from_hex('#9a803a')
ECRU_OLIVE = Color.from_hex('#927b3c')
GREEN_MOSS = Color.from_hex('#857946')
KHAKI = Color.from_hex('#a39264')
FENNEL_SEED = Color.from_hex('#998456')
WILLOW = Color.from_hex('#9a8b4f')
BRONZE_MIST = Color.from_hex('#9c7e41')
DRIED_TOBACCO = Color.from_hex('#997b38')
TAPENADE = Color.from_hex('#805d24')
PLANTATION = Color.from_hex('#7a6332')
FOG_GREEN = Color.from_hex('#c2cbb4')
TENDER_GREENS = Color.from_hex('#c5cfb6')
ALOE_WASH = Color.from_hex('#d0d3b7')
CELADON_GREEN = Color.from_hex('#b5c1a5')
LAUREL_GREEN = Color.from_hex('#adbba1')
SWAMP = Color.from_hex('#a8b197')
RESEDA = Color.from_hex('#a1ad92')
MEADOW_MIST = Color.from_hex('#d3dec4')
BUTTERFLY = Color.from_hex('#cadea5')
WHITE_JADE = Color.from_hex('#d4dbb2')
SEAFOAM_GREEN = Color.from_hex('#cbd5b1')
REED = Color.from_hex('#c3d3a8')
SEEDLING = Color.from_hex('#c0cba1')
FOAM_GREEN = Color.from_hex('#b4c79c')
LILY_GREEN = Color.from_hex('#c5cf98')
BEECHNUT = Color.from_hex('#c2c18d')
NILE = Color.from_hex('#b4bb85')
SWEET_PEA = Color.from_hex('#a3a969')
SPINACH_GREEN = Color.from_hex('#909b4c')
FERN = Color.from_hex('#9aa067')
GREEN_OLIVE = Color.from_hex('#8d8b55')
EPSOM = Color.from_hex('#849161')
GRASSHOPPER = Color.from_hex('#77824a')
TURTLE_GREEN = Color.from_hex('#81894e')
CALLISTE_GREEN = Color.from_hex('#757a4e')
CALLA_GREEN = Color.from_hex('#6a6f34')
CEDAR_GREEN = Color.from_hex('#5e6737')
PESTO = Color.from_hex('#595f34')
TARRAGON = Color.from_hex('#a4ae77')
SAGE = Color.from_hex('#91946e')
IGUANA = Color.from_hex('#818455')
OIL_GREEN = Color.from_hex('#80856d')
LODEN_GREEN = Color.from_hex('#6e7153')
CAPULET_OLIVE = Color.from_hex('#656344')
OLIVINE = Color.from_hex('#666b54')
LINT = Color.from_hex('#b6ba99')
PALE_OLIVE_GREEN = Color.from_hex('#b5ad88')
SAGE_GREEN = Color.from_hex('#b2ac88')
GRAY_GREEN = Color.from_hex('#a49a79')
SPONGE = Color.from_hex('#a49775')
MERMAID = Color.from_hex('#817a65')
DUSKY_GREEN = Color.from_hex('#746c57')
TEA = Color.from_hex('#999b85')
SILVER_SAGE = Color.from_hex('#938b78')
SLATE_GREEN = Color.from_hex('#a0987c')
ELM = Color.from_hex('#a39f86')
MOSSTONE = Color.from_hex('#858961')
ALOE = Color.from_hex('#817a60')
OLIVE_DRAB = Color.from_hex('#756d47')
CEDAR = Color.from_hex('#928e64')
BOA = Color.from_hex('#8e855f')
DRIED_HERB = Color.from_hex('#847a59')
OLIVE_BRANCH = Color.from_hex('#646a45')
LIZARD = Color.from_hex('#71643e')
AVOCADO = Color.from_hex('#676232')
FIR_GREEN = Color.from_hex('#67592a')
BOG = Color.from_hex('#bab696')
ELMWOOD = Color.from_hex('#8c7c61')
GOTHIC_OLIVE = Color.from_hex('#7c6e4f')
BUTTERNUT = Color.from_hex('#7a643f')
NUTRIA = Color.from_hex('#75663e')
MILITARY_OLIVE = Color.from_hex('#63563b')
DARK_OLIVE = Color.from_hex('#574d35')
MOSS_GRAY = Color.from_hex('#afab97')
ABBEY_STONE = Color.from_hex('#aba798')
BURNT_OLIVE = Color.from_hex('#646049')
DUSTY_OLIVE = Color.from_hex('#646356')
IVY_GREEN = Color.from_hex('#585442')
OLIVE_NIGHT = Color.from_hex('#535040')
GRAPE_LEAF = Color.from_hex('#545144')
PORPOISE = Color.from_hex('#a7a19e')
SATELLITE = Color.from_hex('#9f8d89')
DRIFTWOOD = Color.from_hex('#847a75')
FALCON = Color.from_hex('#6d625b')
MOREL = Color.from_hex('#685c53')
FALLEN_ROCK = Color.from_hex('#807669')
VINTAGE_KHAKI = Color.from_hex('#9a9186')
CROCKERY = Color.from_hex('#a49887')
GREIGE = Color.from_hex('#928475')
DESERT_TAUPE = Color.from_hex('#8d7e71')
WHITE_PEPPER = Color.from_hex('#b6a893')
HUMUS = Color.from_hex('#b7a793')
PORTABELLA = Color.from_hex('#937b6a')
CARIBOU = Color.from_hex('#816d5e')
TRAVERTINE = Color.from_hex('#ae997d')
STARFISH = Color.from_hex('#b09a77')
SEMOLINA = Color.from_hex('#ceb899')
CURDS_AND_WHEY = Color.from_hex('#bca483')
TIGERS_EYE = Color.from_hex('#977c61')
TOASTED_COCONUT = Color.from_hex('#8b6a4f')
RAIN_DRUM = Color.from_hex('#5f4c40')
PEAR_SORBET = Color.from_hex('#f3eac3')
PINEAPPLE_SLICE = Color.from_hex('#e7d391')
YARROW = Color.from_hex('#face6d')
ANISE_FLOWER = Color.from_hex('#f4e3b5')
FLAN = Color.from_hex('#f6e3b4')
SUNDRESS = Color.from_hex('#ebcf89')
MACADAMIA = Color.from_hex('#e4cfb6')
LEMON_MERINGUE = Color.from_hex('#f6e199')
YELLOW_IRIS = Color.from_hex('#eee78e')
GOLDFINCH = Color.from_hex('#f8dc6c')
LEMON_ZEST = Color.from_hex('#f9d857')
SOLAR_POWER = Color.from_hex('#f4bf3a')
SAMOAN_SUN = Color.from_hex('#fbc85f')
DESERT_SUN = Color.from_hex('#c87629')
PUMPKIN_SPICE = Color.from_hex('#a05c17')
ORANGE_PEPPER = Color.from_hex('#df7500')
MARMALADE = Color.from_hex('#c16512')
HAWAIIAN_SUNSET = Color.from_hex('#bb5c14')
AUTUMNAL = Color.from_hex('#a15325')
UMBER = Color.from_hex('#944a1f')
EXUBERANCE = Color.from_hex('#e86800')
PUFFINS_BILL = Color.from_hex('#e95c20')
CARAMEL_CAFE = Color.from_hex('#864c24')
GOLD_FLAME = Color.from_hex('#b45422')
CINNAMON_STICK = Color.from_hex('#9b4722')
POTTERS_CLAY = Color.from_hex('#9e4624')
ROOIBOS_TEA = Color.from_hex('#a23c26')
CELOSIA_ORANGE = Color.from_hex('#e8703a')
ORANGEADE = Color.from_hex('#e2552c')
PUREED_PUMPKIN = Color.from_hex('#c34121')
TANGERINE_TANGO = Color.from_hex('#dd4124')
POINCIANA = Color.from_hex('#ca3422')
KOI = Color.from_hex('#d15837')
SAMBA = Color.from_hex('#a2242f')
BARBADOS_CHERRY = Color.from_hex('#aa0a27')
HAUTE_RED = Color.from_hex('#a11729')
SALSA = Color.from_hex('#aa182b')
SCARLET_SAGE = Color.from_hex('#9d202f')
SCOOTER = Color.from_hex('#941e32')
RED_DAHLIA = Color.from_hex('#7d2027')
SUN_DRIED_TOMATO = Color.from_hex('#752329')
FIRED_BRICK = Color.from_hex('#6a2e2a')
RHUBARB = Color.from_hex('#77202f')
SYRAH = Color.from_hex('#6a282c')
POMEGRANATE = Color.from_hex('#6c2831')
CABERNET = Color.from_hex('#64242e')
BALLERINA = Color.from_hex('#f2cfdc')
FAIRY_TALE = Color.from_hex('#f2c1d1')
ETHEREA = Color.from_hex('#a5958f')
FOXGLOVE = Color.from_hex('#b98391')
MESA_ROSE = Color.from_hex('#a66e7a')
JAZZY = Color.from_hex('#b61c50')
GRANITA = Color.from_hex('#a52350')
CHERRIES_JUBILEE = Color.from_hex('#a22452')
CABARET = Color.from_hex('#cb3373')
VIVACIOUS = Color.from_hex('#a32857')
BELLFLOWER = Color.from_hex('#9469a2')
ENGLISH_LAVENDAR = Color.from_hex('#9d7bb0')
RHAPSODY = Color.from_hex('#9f86aa')
ACAI = Color.from_hex('#46295a')
TILLANDSIA_PURPLE = Color.from_hex('#563474')
PICASSO_LILY = Color.from_hex('#634878')
MYSTICAL = Color.from_hex('#5f4e72')
ICELANDIC_BLUE = Color.from_hex('#a9adc2')
ALEUTIAN = Color.from_hex('#9a9eb3')
SILVER_BULLET = Color.from_hex('#81839a')
BLUE_GRANITE = Color.from_hex('#717388')
EVENING_BLUE = Color.from_hex('#2a293e')
DEEP_WELL = Color.from_hex('#2c2a33')
NIGHT_SKY = Color.from_hex('#2a2a35')
BLUE_HERON = Color.from_hex('#96a3c7')
HYDRANGEA = Color.from_hex('#849bcc')
XENON_BLUE = Color.from_hex('#b7c0d7')
BRUNNERA_BLUE = Color.from_hex('#9ba9ca')
SKY_CAPTAIN = Color.from_hex('#262934')
NAVY_BLAZER = Color.from_hex('#282d3c')
DARK_SAPPHIRE = Color.from_hex('#262b37')
PLEIN_AIR = Color.from_hex('#bfcad6')
HALOGEN_BLUE = Color.from_hex('#bdc6dc')
CHAMBRAY_BLUE = Color.from_hex('#9eb4d3')
BEL_AIR_BLUE = Color.from_hex('#819ac1')
VINTAGE_INDIGO = Color.from_hex('#4a556b')
SODALITE_BLUE = Color.from_hex('#253668')
PARISIAN_NIGHT = Color.from_hex('#323441')
MONACO_BLUE = Color.from_hex('#274374')
VALLARTA_BLUE = Color.from_hex('#30658e')
SALUTE = Color.from_hex('#282b34')
OUTER_SPACE = Color.from_hex('#2f3441')
BLUEBERRY = Color.from_hex('#2c333e')
CARBON = Color.from_hex('#272f38')
VULCAN = Color.from_hex('#2d3036')
OMPHALODES = Color.from_hex('#b5cedf')
COOL_BLUE = Color.from_hex('#a5c5d9')
BERING_SEA = Color.from_hex('#4b5b6e')
BLUE_WING_TEAL = Color.from_hex('#2c4053')
POSEIDON = Color.from_hex('#123955')
MYKONOS_BLUE = Color.from_hex('#005780')
REFLECTING_POND = Color.from_hex('#203e4a')
CORYDALIS_BLUE = Color.from_hex('#a9cada')
BLUE_TOPAZ = Color.from_hex('#78bdd4')
GULF_STREAM = Color.from_hex('#88c3d0')
AQUARELLE = Color.from_hex('#61aab1')
AQUA_SPLASH = Color.from_hex('#85ced1')
BOTANICAL_GARDEN = Color.from_hex('#12403c')
SCARAB = Color.from_hex('#23312d')
NIMBUS_CLOUD = Color.from_hex('#d5d5d8')
MICRO_CHIP = Color.from_hex('#babcc0')
WET_WEATHER = Color.from_hex('#929090')
TITANIUM = Color.from_hex('#807d7f')
DECEMBER_SKY = Color.from_hex('#767275')
PAVEMENT = Color.from_hex('#524d50')
MAGNET = Color.from_hex('#4d4b4f')
SILVER_SCONCE = Color.from_hex('#a19fa5')
SILVER_FILIGREE = Color.from_hex('#7f7c81')
QUICKSILVER = Color.from_hex('#7e7d88')
STORM_FRONT = Color.from_hex('#787376')
TORNADO = Color.from_hex('#5e5b60')
EIFFEL_TOWER = Color.from_hex('#5c5658')
GRAPHITE = Color.from_hex('#3b3b48')
ALLOY = Color.from_hex('#98979a')
SLEET = Color.from_hex('#92949b')
TRADEWINDS = Color.from_hex('#7f8793')
GRISAILLE = Color.from_hex('#585e6f')
PERISCOPE = Color.from_hex('#46444c')
QUIET_SHADE = Color.from_hex('#66676d')
TURBULENCE = Color.from_hex('#4e545b')
STORMY_WEATHER = Color.from_hex('#58646d')
IRON_GATE = Color.from_hex('#4e5055')
FORGED_IRON = Color.from_hex('#48464a')
ASPHALT = Color.from_hex('#434447')
GHOST_GRAY = Color.from_hex('#9c9b98')
BRUSHED_NICKEL = Color.from_hex('#73706f')
MOURNING_DOVE = Color.from_hex('#94908b')
BELGIAN_BLOCK = Color.from_hex('#a3a9a6')
AGAVE_GREEN = Color.from_hex('#6b7169')
CILANTRO = Color.from_hex('#43544b')
PINE_GROVE = Color.from_hex('#213631')
EDEN = Color.from_hex('#264e36')
JOLLY_GREEN = Color.from_hex('#007844')
MOUNTAIN_VIEW = Color.from_hex('#2e3d30')
MARGARITA = Color.from_hex('#b5c38e')
WINTER_MOSS = Color.from_hex('#5b5a41')
CLIMBING_IVY = Color.from_hex('#444940')
DELICIOSO = Color.from_hex('#3f352f')
MULCH = Color.from_hex('#433937')
MOLE = Color.from_hex('#392d2b')
CHOCOLATE_TORTE = Color.from_hex('#382e2d')
GANACHE = Color.from_hex('#34292a')
BLACK_BEAN = Color.from_hex('#2e272a')
ESPRESSO = Color.from_hex('#363031')
METEORITE = Color.from_hex('#2b2929')
TAP_SHOE = Color.from_hex('#2a2b2d')
WHITE_ALYSSUM = Color.from_hex('#efebe7')
JET_STREAM = Color.from_hex('#ede6de')
SWEET_CREAM = Color.from_hex('#f0ead6')
BUTTERCREAM = Color.from_hex('#efe0cd')
LEMON_ICING = Color.from_hex('#f6ebc8')
SUGAR_SWIZZLE = Color.from_hex('#f3eee7')
COCONUT_MILK = Color.from_hex('#f0ede5')
YELLOW_PEAR = Color.from_hex('#ece99b')
SEA_SALT = Color.from_hex('#f1e6de')
BRILLIANT_WHITE = Color.from_hex('#edf1fe')
CANNOLI_CREAM = Color.from_hex('#f0efe2')
TOFU = Color.from_hex('#e8e3d9')
PISTACHIO_SHELL = Color.from_hex('#d7cfbb')
CELANDINE = Color.from_hex('#ebdf67')
LEMON_VERBENA = Color.from_hex('#f3e779')
CREME_DE_PECHE = Color.from_hex('#f5d6c6')
MARYS_ROSE = Color.from_hex('#f7d1d4')
MORGANITE = Color.from_hex('#dfcdc6')
ROSE_WATER = Color.from_hex('#f8e0e7')
ALMOND_MILK = Color.from_hex('#d6cebe')
LIME_POPSICLE = Color.from_hex('#c0db3a')
GOLDEN_KIWI = Color.from_hex('#f3dd3e')
MEADOWLARK = Color.from_hex('#ead94e')
EVENING_PRIMROSE = Color.from_hex('#ccdb1e')
HABANERO_GOLD = Color.from_hex('#fed450')
MINION_YELLOW = Color.from_hex('#fed55d')
SOYBEAN = Color.from_hex('#d2c29d')
JURASSIC_GOLD = Color.from_hex('#e7aa56')
BROWN_RICE = Color.from_hex('#c7bba4')
PEACH_QUARTZ = Color.from_hex('#f5b895')
PEACHY_KEEN = Color.from_hex('#e2bdb3')
BRAZILIAN_SAND = Color.from_hex('#dacab7')
PINK_SALT = Color.from_hex('#f7cdc7')
ROSE_QUARTZ = Color.from_hex('#f7cac9')
BALLET_SLIPPER = Color.from_hex('#ebced5')
CHERRY_BLOSSOM = Color.from_hex('#f7cee0')
ANTARCTICA = Color.from_hex('#c6c5c6')
OYSTER_MUSHROOM = Color.from_hex('#c3c6c8')
TANAGER_TURQUOISE = Color.from_hex('#91dce8')
LIMPET_SHELL = Color.from_hex('#98ddde')
ICED_AQUA = Color.from_hex('#abd3db')
ACID_LIME = Color.from_hex('#badf30')
SPICY_MUSTARD = Color.from_hex('#d8ae47')
KUMQUAT = Color.from_hex('#fbaa4c')
IRISH_CREAM = Color.from_hex('#c0ac92')
ORANGE_CHIFFON = Color.from_hex('#f9aa7d')
HAZELNUT = Color.from_hex('#cfb095')
SEPIA_ROSE = Color.from_hex('#d4bab6')
RAINDROPS = Color.from_hex('#b1aab3')
ZEN_BLUE = Color.from_hex('#9fa9be')
QUIET_GRAY = Color.from_hex('#b9babd')
AIRY_BLUE = Color.from_hex('#92b6d5')
HARBOR_MIST = Color.from_hex('#afb1b4')
SEA_ANGEL = Color.from_hex('#98bfca')
BALTIC_SEA = Color.from_hex('#79b5db')
ANTIQUA_SAND = Color.from_hex('#83c2cd')
ISLAND_PARADISE = Color.from_hex('#95dee3')
TIBETAN_STONE = Color.from_hex('#82c2c7')
MANGO_MOJITO = Color.from_hex('#d69c2f')
GINGER_ROOT = Color.from_hex('#bfa58a')
ICED_COFFEE = Color.from_hex('#b18f6a')
AUTUMN_BLAZE = Color.from_hex('#d9922e')
GOLDEN_ORANGE = Color.from_hex('#d7942d')
PORCINI = Color.from_hex('#cca580')
ICELAND_POPPY = Color.from_hex('#f4963a')
PAPAYA = Color.from_hex('#fea166')
CARROT_CURL = Color.from_hex('#fe8c18')
TURMERIC = Color.from_hex('#fe840e')
TANGELO = Color.from_hex('#fe7e03')
FENUGREEK = Color.from_hex('#c0916c')
DUSTED_CLAY = Color.from_hex('#cc7357')
PASTRY_SHELL = Color.from_hex('#bd8c66')
BLOOMING_DAHLIA = Color.from_hex('#eb9687')
CROCUS_PETAL = Color.from_hex('#b99bc5')
PURPLE_ROSE = Color.from_hex('#b09fca')
LILAC_BREEZE = Color.from_hex('#b3a0c9')
SERENITY = Color.from_hex('#91a8d0')
CRYSTAL_SEAS = Color.from_hex('#5dafce')
GOLDEN_LIME = Color.from_hex('#9a9738')
SPLIT_PEA = Color.from_hex('#9c9a40')
LENTIL_SPROUT = Color.from_hex('#aba44d')
PURE_CASHMERE = Color.from_hex('#ada396')
SUN_BAKED = Color.from_hex('#d27f63')
PEACH_CARAMEL = Color.from_hex('#c5733d')
TOMATO_CREAM = Color.from_hex('#c57644')
ORANGE_TIGER = Color.from_hex('#f96714')
MEERKAT = Color.from_hex('#a46f44')
EXOTIC_ORANGE = Color.from_hex('#f96531')
DRAGON_FIRE = Color.from_hex('#fc642d')
CORAL_QUARTZ = Color.from_hex('#f77464')
PEACH_ECHO = Color.from_hex('#f7786b')
PURPLE_DOVE = Color.from_hex('#98878c')
SAND_VERBENA = Color.from_hex('#9f90c1')
LILAC_GRAY = Color.from_hex('#9896a4')
GRANADA_SKY = Color.from_hex('#5d81bb')
TREE_HOUSE = Color.from_hex('#988c75')
CHAI_TEA = Color.from_hex('#b1832f')
ROASTED_PECAN = Color.from_hex('#93592b')
ROASTED_CASHEW = Color.from_hex('#918579')
WINTER_TWIG = Color.from_hex('#948a7a')
PETRIFIED_OAK = Color.from_hex('#8d7960')
ARGAN_OIL = Color.from_hex('#8b593e')
AUTUMN_MAPLE = Color.from_hex('#c46215')
SEPIA_TINT = Color.from_hex('#897560')
SPICE_ROUTE = Color.from_hex('#b95b3f')
SCARLET_IBIS = Color.from_hex('#f45520')
SUMMER_FIG = Color.from_hex('#be4b3b')
MOONSCAPE = Color.from_hex('#725f69')
FRUIT_DOVE = Color.from_hex('#ce5b78')
PINK_YARROW = Color.from_hex('#ce3175')
TOADSTOOL = Color.from_hex('#988088')
BODACIOUS = Color.from_hex('#b76ba3')
DIFFUSED_ORCHID = Color.from_hex('#9879a2')
FAIRY_WREN = Color.from_hex('#9479af')
SUNLIT_ALLIUM = Color.from_hex('#9787bb')
SHARKSKIN = Color.from_hex('#838487')
PALE_IRIS = Color.from_hex('#8895c5')
IOLITE = Color.from_hex('#707bb4')
GRAY_FLANNEL = Color.from_hex('#848182')
RIVERSIDE = Color.from_hex('#4c6a92')
QUIET_HARBOR = Color.from_hex('#5a789a')
LICHEN_BLUE = Color.from_hex('#5d89b3')
PACIFIC_COAST = Color.from_hex('#5480ac')
IBIZA_BLUE = Color.from_hex('#007cb7')
NAVAGIO_BAY = Color.from_hex('#3183a0')
BARRIER_REEF = Color.from_hex('#0084a1')
GUACAMOLE = Color.from_hex('#797b3a')
KALE = Color.from_hex('#5a7247')
MAYFLY = Color.from_hex('#65663f')
TWIST_OF_LIME = Color.from_hex('#4e632c')
MARTINI_OLIVE = Color.from_hex('#716a4d')
EMPERADOR = Color.from_hex('#684832')
THAI_CURRY = Color.from_hex('#ab6819')
HONEY_GINGER = Color.from_hex('#a86217')
SUGAR_ALMOND = Color.from_hex('#935529')
SPICED_APPLE = Color.from_hex('#783937')
CHILI_OIL = Color.from_hex('#8e3c36')
PLUM_TRUFFLE = Color.from_hex('#675657')
BRANDY_BROWN = Color.from_hex('#73362a')
VALIANT_POPPY = Color.from_hex('#bc322c')
AURA_ORANGE = Color.from_hex('#b4262a')
TOREADOR = Color.from_hex('#b61032')
LYCHEE = Color.from_hex('#ba0b32')
GOJI_BERRY = Color.from_hex('#b91228')
ARCTIC_DUSK = Color.from_hex('#735b6a')
EPHEMERA = Color.from_hex('#6f5965')
JALAPENO_RED = Color.from_hex('#b2103c')
LOVE_POTION = Color.from_hex('#c01352')
PINK_PEACOCK = Color.from_hex('#c62168')
GRAPE_KISS = Color.from_hex('#7b4368')
WILLOWHERB = Color.from_hex('#8e4483')
CHARISMA = Color.from_hex('#632a60')
PLUM_JAM = Color.from_hex('#624076')
LAVENDER_CRYSTAL = Color.from_hex('#936a98')
PURPLE_SAPPHIRE = Color.from_hex('#6f4685')
CHIVE_BLOSSOM = Color.from_hex('#7d5d99')
PURPLE_CORALLITE = Color.from_hex('#5a4e8f')
VOLCANIC_GLASS = Color.from_hex('#615c60')
GRAY_BLUE = Color.from_hex('#4d587a')
BLUE_HORIZON = Color.from_hex('#4e6482')
IRIS_BLOOM = Color.from_hex('#5b609e')
NEBULAS_BLUE = Color.from_hex('#2d62a3')
INDIGO_BUNTING = Color.from_hex('#006ca9')
FJORD_BLUE = Color.from_hex('#007290')
HAWAIIAN_SURF = Color.from_hex('#0078a7')
TAHITIAN_TIDE = Color.from_hex('#006b7e')
QUETZAL_GREEN = Color.from_hex('#006865')
GRANITE_GRAY = Color.from_hex('#615e5f')
LUSH_MEADOW = Color.from_hex('#006e51')
GRAY_PINSTRIPE = Color.from_hex('#49494d')
SEA_TURTLE = Color.from_hex('#5e5749')
DEEP_DEPTHS = Color.from_hex('#46483c')
KALAMATA = Color.from_hex('#5f5b4c')
CROCODILE = Color.from_hex('#5d5348')
CHOCOLATE_PLUM = Color.from_hex('#3c2d2e')
CHOCOLATE_LAB = Color.from_hex('#5c3e35')
SHAVED_CHOCOLATE = Color.from_hex('#543b35')
FONDUE_FUDGE = Color.from_hex('#5d4236')
TIRAMISU = Color.from_hex('#634235')
ROCKY_ROAD = Color.from_hex('#5a3e36')
CHICORY_COFFEE = Color.from_hex('#4a342e')
SMOKED_PAPRIKA = Color.from_hex('#6e362c')
CHOCOLATE_FONDANT = Color.from_hex('#56352d')
CHERRY_MAHOGANY = Color.from_hex('#66352b')
MERLOT = Color.from_hex('#72262c')
RED_PEAR = Color.from_hex('#7b3539')
PICKLED_BEET = Color.from_hex('#4d233d')
PLUM_CASPIA = Color.from_hex('#61224a')
WINTER_BLOOM = Color.from_hex('#47243b')
SPICED_PLUM = Color.from_hex('#6d4773')
VIOLET_INDIGO = Color.from_hex('#3e285c')
MARITIME_BLUE = Color.from_hex('#27293d')
OBSIDIAN = Color.from_hex('#3a363b')
BLACK_BEAUTY = Color.from_hex('#26262a')
BLACKENED_PEARL = Color.from_hex('#4d4b50')
ODYSSEY_GRAY = Color.from_hex('#434452')
BLACK_ONYX = Color.from_hex('#2b272b')
NAVY_PEONY = Color.from_hex('#223a5e')
SARGASSO_SEA = Color.from_hex('#35435a')
SAILOR_BLUE = Color.from_hex('#0e3a53')
GIBRALTAR_SEA = Color.from_hex('#123850')
LAPIS_BLUE = Color.from_hex('#004b8d')
BALEINE_BLUE = Color.from_hex('#155187')
GALAXY_BLUE = Color.from_hex('#2a4b7c')
BLUE_OPAL = Color.from_hex('#0f3b57')
MOONLIT_OCEAN = Color.from_hex('#293b4d')
DEEP_DIVE = Color.from_hex('#29495c')
CRYSTAL_TEAL = Color.from_hex('#00637c')
DEEP_LAGOON = Color.from_hex('#005265')
SEA_MOSS = Color.from_hex('#254445')
FOREST_BIOME = Color.from_hex('#184a45')
RAIN_FOREST = Color.from_hex('#15463e')
