import warnings

from numpy.ma.core import append

warnings.simplefilter("ignore", UserWarning)

import time
import datetime

import pandas as pd
import numpy as np
import polars as pl

import math
import json
import ast
import os

from dash import dcc, html    ## pip install dash
import dash_bootstrap_components as dbc
import plotly.express as px
import plotly.graph_objs as go
from plotly.subplots import make_subplots

import plotguy
#import dash_dangerously_set_inner_html   # for all Plot Class



class Components:
    chart_bg = '#1f2c56'

    sort_method_dropdown = html.Div()

    filter_dropdown = html.Div(id='filter_dropdown',
                               children=dbc.Select(id='filter_name',
                                                   placeholder="Select Filter",
                                                   options=[
                                                       {'label': 'Exclude Stock', 'value': 'exclude'},
                                                       {'label': 'Return on Capital >', 'value': 'equity_return_on_capital>'},
                                                       {'label': 'Return on Capital <', 'value': 'equity_return_on_capital<'},
                                                       {'label': 'Annualized Return >', 'value': 'equity_annualized_return>'},
                                                       {'label': 'Annualized Return <', 'value': 'equity_annualized_return<'},
                                                       {'label': 'Return-BaH % Diff. >', 'value': 'return_on_capital_diff>'},
                                                       {'label': 'Return-BaH % Diff. <', 'value': 'return_on_capital_diff<'},
                                                       {'label': 'Sharpe Ratio >', 'value': 'equity_annualized_sr>'},
                                                       {'label': 'Sharpe Ratio <', 'value': 'equity_annualized_sr<'},
                                                       {'label': 'MDD Percentage >', 'value': 'equity_mdd_pct>'},
                                                       {'label': 'MDD Percentage <', 'value': 'equity_mdd_pct<'},
                                                       {'label': 'Trade Count >', 'value': 'num_of_trade>'},
                                                       {'label': 'Trade Count <', 'value': 'num_of_trade<'},
                                                       {'label': 'COV (Count) >', 'value': 'cov_count>'},
                                                       {'label': 'COV (Count) <', 'value': 'cov_count<'},
                                                       {'label': 'COV (Return) >', 'value': 'cov_return>'},
                                                       {'label': 'COV (Return) <', 'value': 'cov_return<'},
                                                       {'label': 'Win Rate >', 'value': 'win_rate>'},
                                                       {'label': 'Win Rate <', 'value': 'win_rate<'},
                                                   ],
                                                   style={'border-radius': '5px', 'font-size': '12px'}),
                               style={'padding-left': '15px', 'width': '175px'})

    filter_dropdown_disabled = html.Div(id='filter_dropdown',
                                        children=dbc.Select(id='filter_name', disabled=True,
                                                            placeholder="Select Filter",
                                                            style={'border-radius': '5px', 'font-size': '12px',
                                                                   'backgroundColor': 'Gray'}),
                                        style={'padding-left': '15px', 'width': '180px'})

    filter_input = dbc.Input(id='filter_input', value=None, size="md", type='text',
                             style={'width': '50px', 'margin-right': '5px', 'border-radius': '3px',
                                      'padding': '6px 5px', 'font-size': '12px', })

    filter_input_disabled = dbc.Input(id='filter_input', value=None, size="md", disabled=True,
                                      style={'width': '50px', 'margin-right': '5px', 'border-radius': '3px',
                                               'padding': '6px 5px', 'font-size': '12px', 'backgroundColor': 'Gray'})

    add_filter_btn_style = {'margin-left': '50px', 'width': '150px', 'backgroundColor': 'blue',
                        'border-radius': '5px', 'text-align': 'center', 'cursor': 'pointer',
                        'font-size': '13px', 'height':'30px'}

    add_filter_btn_style_disabled = {'margin-left': '50px', 'width': '150px', 'color': 'Silver', 'backgroundColor': 'Gray',
                                 'border-radius': '5px', 'text-align': 'center',
                                 'font-size': '13px', 'height':'30px'}

    def __init__(self,number_of_curves):
        self.sort_method_dropdown = html.Div(
            dbc.Select(id='sort_method',
                       placeholder="Select Sorting Method",
                       value=f'Top Sharpe Ratio',
                       options=[{'label': f'Top {number_of_curves} Sharpe Ratio', 'value': 'Top Sharpe Ratio'},
                                {'label': f'Top {number_of_curves} Net Profit', 'value': 'Top Net Profit'}
                                # {'label': f'Top {number_of_curves} Return-BaH % Difference', 'value': 'Top Return to BaH Ratio'}
                                , ],
                       style={'border-radius': '5px', 'font-size': '12px', }),
            style={'padding-left': '15px', 'width': '235px'})
        self.df_daily_dict = {}

    def empty_line_chart(self):
        chart_bg = self.chart_bg
        fig_line = px.line()
        fig_line.update_layout(plot_bgcolor=chart_bg, paper_bgcolor=chart_bg, margin=dict(l=85, r=60, t=30, b=40),
                               height=500, font={"color": chart_bg})
        fig_line.update_xaxes(showline=True, zeroline=False, linecolor='#979A9A', gridcolor=chart_bg)
        fig_line.update_yaxes(showline=True, zeroline=False, linecolor='#979A9A', gridcolor=chart_bg)

        return fig_line


    def update_checkbox_div(self, para_keys, backtest_result_df):
        checkbox_values = {}

        for key in para_keys:
            unique_values = backtest_result_df[key].unique().tolist()
            unique_values.sort()
            checkbox_values[key] = unique_values

        checkbox_div = []
        for i, para_name in enumerate (para_keys):
            _options = []
            if para_name == 'code':
                for option in checkbox_values[para_name]:
                    if isinstance(option, int):
                        _options.append(str(option).zfill(5))
                    else:
                        _options.append(option)

                code_list = _options
            else:
                _options = checkbox_values[para_name]


            if _options == [False, True]:
                _checklist = dcc.Checklist([str(tf) for tf in _options], [str(tf) for tf in _options], inline=True,
                                           id={'type': 'para-checklist', 'index': i},
                                           labelStyle={},
                                           inputStyle={'margin-left': '10px', 'margin-right': '3px'})
            else:
                if para_name == 'code':
                    _checklist = dbc.Row([
                        dcc.Checklist(
                            id="all-or-none",
                            options=[{"label": "Select All", "value": "All"}],
                            labelStyle={'font-size': '12px'},
                            inputStyle={'margin-left': '10px', 'margin-right': '3px',
                                        'background-color': 'red',
                                        'vertical-align': 'middle', 'position': 'relative',
                                        'bottom': '.15em'}
                        ),

                        html.Div(style={'height': '5px'}),

                        dcc.Checklist(_options, _options, inline=True,
                                      id={'type': 'para-checklist', 'index': i},
                                      labelStyle={'font-size': '12px'},
                                      inputStyle={'margin-left': '10px', 'margin-right': '3px',
                                                  'background-color': 'red',
                                                  'vertical-align': 'middle', 'position': 'relative',
                                                  'bottom': '.15em'})
                    ])

                else:
                    _checklist = dcc.Checklist(_options, _options, inline=True,
                                               id={'type': 'para-checklist', 'index': i},
                                               labelStyle={'font-size': '12px'},
                                               inputStyle={'margin-left': '10px', 'margin-right': '3px',
                                                           'background-color': 'red',
                                                           'vertical-align': 'middle', 'position': 'relative',
                                                           'bottom': '.15em'})
            row = html.Div(
                dbc.Row([
                    html.Div(para_name),
                    html.Div(style={'height': '5px'}),
                    html.Div(_checklist),
                    html.Div(style={'height': '5px'}),
                ]), style={'padding': '0px 20px', 'font-size': '12px'})

            checkbox_div.append(row)

        return checkbox_div, code_list

    def update_stat_div(self, period, pct_mean, rise_mean, fall_mean):
        div = html.Div([

            dbc.Row([
                 dbc.Col(width=2),
                 dbc.Col(str(period[0]) + ' Days', width=2),
                 dbc.Col(str(period[1]) + ' Days', width=2),
                 dbc.Col(str(period[2]) + ' Days', width=2),
                 dbc.Col(str(period[3]) + ' Days', width=2),
                 dbc.Col(str(period[4]) + ' Days', width=2),
             ]),

            dbc.Row([
                 dbc.Col('pct_change', width=2),
                 dbc.Col(f'{pct_mean[0]}%', width=2),
                 dbc.Col(f'{pct_mean[1]}%', width=2),
                 dbc.Col(f'{pct_mean[2]}%', width=2),
                 dbc.Col(f'{pct_mean[3]}%', width=2),
                 dbc.Col(f'{pct_mean[4]}%', width=2),
             ]),

            dbc.Row([
                 dbc.Col('max_rise', width=2),
                 dbc.Col(f'{rise_mean[0]}%', width=2),
                 dbc.Col(f'{rise_mean[1]}%', width=2),
                 dbc.Col(f'{rise_mean[2]}%', width=2),
                 dbc.Col(f'{rise_mean[3]}%', width=2),
                 dbc.Col(f'{rise_mean[4]}%', width=2),
             ]),

            dbc.Row([
                 dbc.Col('max_fall', width=2),
                 dbc.Col(f'{fall_mean[0]}%', width=2),
                 dbc.Col(f'{fall_mean[1]}%', width=2),
                 dbc.Col(f'{fall_mean[2]}%', width=2),
                 dbc.Col(f'{fall_mean[3]}%', width=2),
                 dbc.Col(f'{fall_mean[4]}%', width=2),
             ]),

        ])

        return div


    def generate_radioitems(self, para_dict):
        radioitems_div = []

        for i, key in enumerate(para_dict):
            options = para_dict[key]

            radioitems_div.append(html.Div(key,style={'color': 'Yellow','font-size': '14px'}))

            radioitems_div.append(dcc.RadioItems(
                id={'type': 'para_radioitems', 'index': i},
                options=[{'label': k,
                          'value': k} for k in options],
                labelStyle={'font-size': '13px'},
                inputStyle={'margin-left': '10px', 'margin-right': '5px'}

            ), )

            radioitems_div.append(html.Div(html.Img(),style={'height':'10px'}))

        return radioitems_div

    def selection_title(self, para_dict, values):
        title = []
        for i, key in enumerate(para_dict):
            title.append(dbc.Row(
                [dbc.Col(key, width=6),
                 dbc.Col(f'{values[i]}', style={'text-align': 'center'}, width=6)]))

        return title


    def update_performance_matrix(self, init_chart, para_keys_list, result_df, line_selected = 0):

        if init_chart:
            ref_code = '-----'
            result_df = result_df.reset_index()
        else:
            ref_code = result_df.loc[line_selected].ref_code

        yearly_stats_string = result_df.loc[line_selected]['yearly_stats_string']

        per_col1 =[]
        per_col2 = []
        per_col_1_1 = []
        per_col_1_2 = []
        per_col_2_1 = [html.Div(html.Img())]
        per_col_2_2 = [html.Div('Selected',style={'text-align': 'center', 'font-weight': 'bold', 'color':'yellow'})]
        per_col_2_3 = [html.Div('BaH',style={'text-align': 'center', 'font-weight': 'bold', 'color':'yellow'})]

        keys = ['num_of_trade',
                'equity_net_profit',
                'equity_net_profit_to_mdd',
                'return_on_capital_diff',
                'win_rate',
                'cov_count',
                'cov_return',
                'total_commission',
                'risk_free_rate',

                'equity_return_on_capital',
                'equity_annualized_return',
                'equity_annualized_std',
                'equity_annualized_sr',
                'equity_mdd_dollar',
                'equity_mdd_pct',

                'price_return_on_capital',
                'price_annualized_return',
                'price_annualized_std',
                'price_annualized_sr',
                'price_mdd_dollar',
                'price_mdd_pct',
                ]

        ### {x:,.2f} 2 decail points ###
        ### {int(x):,} integer ###

        if init_chart:
            result_df = pd.DataFrame({key: ['-----'] for key in keys + para_keys_list}, index=[0])
        else:
            result_df['equity_net_profit'] = result_df['equity_net_profit'].apply(lambda x: f"{int(x):,}")
            result_df['equity_net_profit_to_mdd'] = result_df['equity_net_profit_to_mdd'].apply(
                lambda x: 'inf' if x == np.inf else ('-inf' if x == -np.inf else f"{x:,.2f}"))

            result_df['total_commission'] = result_df['total_commission'].apply(lambda x: f"{int(x):,}")

            result_df['cov_count'] = result_df['cov_count'].apply(lambda x: f"{x:,.1f}")
            result_df['cov_return'] = result_df['cov_return'].apply(lambda x: f"{x:,.1f}")
            result_df['win_rate'] = result_df['win_rate'].apply(lambda x: f"{x:.1f} %")

            result_df['equity_mdd_dollar'] = result_df['equity_mdd_dollar'].apply(lambda x: f"{int(x):,}")
            result_df['equity_mdd_pct'] = result_df['equity_mdd_pct'].apply(lambda x: f"{x:,.2f} %")

            result_df['equity_return_on_capital'] = result_df['equity_return_on_capital'].apply(lambda x: f"{x:,.0f} %")
            result_df['equity_annualized_return'] = result_df['equity_annualized_return'].apply(lambda x: f"{x:,.0f} %")
            result_df['equity_annualized_std']    = result_df['equity_annualized_std'].apply(lambda x: f"{x:,.2f} %")
            result_df['equity_annualized_sr']     = result_df['equity_annualized_sr'].apply(lambda x: f"{x:,.2f}")

            result_df['return_on_capital_diff']     = result_df['return_on_capital_diff'].apply(lambda x: f"{x:,.0f}")

            result_df['price_mdd_dollar'] = result_df['price_mdd_dollar'].apply(lambda x: f"{int(x):,}")
            result_df['price_mdd_pct']    = result_df['price_mdd_pct'].apply(lambda x: f"{x:,.2f} %")

            result_df['price_return_on_capital'] = result_df['price_return_on_capital'].apply(lambda x: f"{x:.0f} %")
            result_df['price_annualized_return'] = result_df['price_annualized_return'].apply(lambda x: f"{x:.0f} %")
            result_df['price_annualized_std']    = result_df['price_annualized_std'].apply(lambda x: f"{x:.2f} %")
            result_df['price_annualized_sr']     = result_df['price_annualized_sr'].apply(lambda x: f"{x:.2f}")

            result_df['risk_free_rate']     = result_df['risk_free_rate'].apply(lambda x: f"{x:.2f} %")

        per_col_1_1.append(html.Div('Number of Trade', style={'background-color': 'darkblue' }))
        per_col_1_1.append(html.Div('Net Profit'))
        per_col_1_1.append(html.Div('Net Profit/MDD', style={'background-color': 'darkblue' }))
        per_col_1_1.append(html.Div('Return-BaH % Diff.'))
        per_col_1_1.append(html.Div('Win Rate', style={'background-color': 'darkblue' }))
        per_col_1_1.append(html.Div('COV (Count)'))
        per_col_1_1.append(html.Div('COV (Return)', style={'background-color': 'darkblue' }))
        per_col_1_1.append(html.Div('Total Commission'))
        per_col_1_1.append(html.Div('Risk Free Rate', style={'background-color': 'darkblue' }))

        per_col_2_1.append(html.Div('Return on Capital'))
        per_col_2_1.append(html.Div('Ann. Return', style={'background-color': 'darkblue' }))
        per_col_2_1.append(html.Div('Ann. Std'))
        per_col_2_1.append(html.Div('Ann. Sharpe Ratio', style={'background-color': 'darkblue' }))
        per_col_2_1.append(html.Div('MDD Dollar'))
        per_col_2_1.append(html.Div('MDD Percentage', style={'background-color': 'darkblue' }))

        for i in range(9):
            content = result_df.iloc[0][keys[i]]
            if i%2 == 1:
                per_col_1_2.append(html.Div(content, style={'text-align': 'center', }))
            else:
                per_col_1_2.append(html.Div(content, style={'text-align': 'center','background-color': 'darkblue' }))
        for i in range(9, 15):
            content = result_df.iloc[0][keys[i]]
            if i%2 == 1:
                per_col_2_2.append(html.Div(content, style={'text-align': 'center'}),)
            else:
                per_col_2_2.append(html.Div(content, style={'text-align': 'center', 'background-color': 'darkblue'}),)
        for i in range(15, 21):
            content = result_df.iloc[0][keys[i]]
            if i%2 == 1:
                per_col_2_3.append(html.Div(content, style={'text-align': 'center'}))
            else:
                per_col_2_3.append(html.Div(content, style={'text-align': 'center', 'background-color': 'darkblue'}))

        per_col1.append(dbc.Row([dbc.Col(html.Div(per_col_1_1), width=6),
                                 dbc.Col(per_col_1_2, style={'padding': '0'}, width=6)]))
        per_col2.append(dbc.Row([dbc.Col(html.Div(per_col_2_1), width=6),
                                 dbc.Col(per_col_2_2, style={'padding': '0'}, width=3),
                                 dbc.Col(per_col_2_3, style={'padding': '0'}, width=3)
                                 ]))

        year_col1 = [dbc.Row([dbc.Col(style={'margin-left':'15px'}, width=2),
                              dbc.Col('Count', style={'text-align': 'center', 'font-weight': 'bold', 'color':'yellow'},width=3),
                              dbc.Col('WinRate', style={'text-align': 'center', 'font-weight': 'bold', 'color':'yellow'},width=3),
                              dbc.Col('Return', style={'text-align': 'center', 'font-weight': 'bold', 'color': 'yellow'}, width=3)
                              ], style={'font-size': '11px'})]

        yearly_stats_dict = {}
        years_data = yearly_stats_string.split("|")

        for year_data in years_data:
            parts = year_data.split(",")
            year = parts[0]
            yearly_stats_dict[year] = {}

            for item in parts[1:]:
                key, value = item.split(":")
                yearly_stats_dict[year][key] = float(value)

        for idx, (year, value) in enumerate(yearly_stats_dict.items()):

            if init_chart:
                year_trade_count = '-----'
                year_win_rate    = '-----'
                year_return      = '-----'
            else:
                format_content = value['year_win_rate']
                year_win_rate = "{:d} %".format(int(format_content))

                format_content = value['year_return']
                year_return = "{:d} %".format(int(format_content))

                format_content = value['year_trade_count']
                year_trade_count = str(int(format_content))

            if idx%2 == 1:
                year_col1.append(
                    dbc.Row(
                        [
                            dbc.Col(year, style={'margin-left': '15px'}, width=2),
                            dbc.Col(year_trade_count, style={'text-align': 'center'}, width=3),
                            dbc.Col(year_win_rate, style={'text-align': 'center'}, width=3),
                            dbc.Col(year_return, style={'text-align': 'center'}, width=3),
                        ],
                    ))
            else:
                year_col1.append(
                    dbc.Row(
                        [
                            dbc.Col(year, style={'margin-left': '15px', 'background-color': 'darkblue'}, width=2),
                            dbc.Col(year_trade_count, style={'text-align': 'center', 'background-color': 'darkblue'}, width=3),
                            dbc.Col(year_win_rate, style={'text-align': 'center', 'background-color': 'darkblue'}, width=3),
                            dbc.Col(year_return, style={'text-align': 'center', 'background-color': 'darkblue'}, width=3),
                        ],
                    ))


        # Selected Equity Curve
        title = []
        for idx , key in enumerate(para_keys_list):
            if idx%2 == 0:
                title.append(dbc.Row(
                    [dbc.Col(key,width=6),
                     dbc.Col(f'{result_df.iloc[0][key]}',style={ 'text-align': 'center'},width=6)]))
            else:
                title.append(dbc.Row(
                    [dbc.Col(key,width=6, style={  'background-color': 'darkblue'}),
                     dbc.Col(f'{result_df.iloc[0][key]}',style={ 'text-align': 'center', 'background-color': 'darkblue'},width=6)]))

        matrix_div = html.Div([
            html.Div(style={'height': '8px', }),

            html.Div([
                html.Div('Selected Curve', style={'color': 'Cyan',
                                                  'display': 'inline',
                                                  'font-size': '15px'}),
                html.Div('', id='save_status',
                         style={'color': 'Cyan',
                                'display': 'inline',
                                'padding-left': '5px',
                                'font-size': '15px'}),
            ]),

            dbc.Row(html.Div(title), style={'font-size': '11px','padding-left':'1px'}),

            html.Div(style={'height': '15px', }),

            html.Div('Performance', style={'color': 'Cyan', 'font-size': '15px'}),
            html.Div(children=per_col1, style={'font-size': '11px','padding-left':'1px'}),
            html.Div(style={'height': '15px', }),

            html.Div('Comparison', style={'color': 'Cyan', 'font-size': '15px'}),
            html.Div(children=per_col2, style={'font-size': '11px','padding-right': '10px','padding-left':'1px'}),
            html.Div(style={'height': '15px', }),

            html.Div('Performance by Year', style={'color': 'Cyan', 'font-size': '15px'}),
            html.Div(style={'height': '5px', }),
            html.Div(children=year_col1, style={'font-size': '11px'}),

            html.Div(style={'height': '10px'}),

            html.Div(f'Ref: {ref_code}', style={'font-size': '11px'}),

            html.Div(style={'height': '20px'}),

            html.Div([
                html.Div(style={'height': '5px'}),
                html.Div('', id='save_string'),
                html.Div(style={'height': '5px'}),
            ], id='save_btn'),

        ],style={'padding': '0px',})


        return matrix_div


    filter_options = {
        'num_of_trade':'Number of Trade',
        'equity_return_on_capital': 'Return on Capital',
        'annualized_return': 'Annualized Return',
        'equity_annualized_return': 'Sharpe Ratio',
        'equity_mdd_pct':'MDD Percentage',
        'cov_count':'COV (Count)',
        'cov_return': 'COV (Return)',
        'win_rate':'Win Rate',
        'return_on_capital_diff': 'Return/BnH % Diff.',
        #'exclude': 'Exclude',
        }

    def update_filter_div(self, filter_list):
        filter_btn = []
        for i, element in enumerate(filter_list):
            element = filter_list[i]
            filter_full = []
            filter_full.append(html.Div(self. filter_options[element[0]], style={'margin-right': '15px', 'display': 'inline'}))
            filter_full.append(html.Div(element[1], style={'margin-right': '15px', 'display': 'inline'}))
            filter_full.append(html.Div(element[2], style={'margin-right': '15px', 'display': 'inline'}))
            filter_btn.append(dbc.Row([
                dbc.Col(html.Div(filter_full,
                                 style={'font-size': '12px', 'padding': '0px', 'margin': '0px'}), width=10),
                dbc.Col(html.Div(children=html.Div('✗', style={'padding': '0px', 'margin': '0px'}),
                                 id='btn_' + str(i), n_clicks=i,
                                 style={'font-size': '12px', 'backgroundColor': 'rgba(0, 0, 0, 0)',
                                        'border': '0px black solid', 'padding': '0px', 'padding-bottom': '10px',
                                        'margin': '0px', 'width': '5px', 'cursor': 'pointer'}), width=2)
            ]))

        for i in range(len(filter_list), 10):
            filter_btn.append(html.Div(id='btn_' + str(i), n_clicks=i))

        return filter_btn


    def generate_sorted_df(self, sort_method, df_checked, number_of_curves):

        if sort_method == 'Top Sharpe Ratio':
            df_checked.sort_values(by='equity_annualized_sr', ascending=False, inplace=True)
            df_checked = df_checked.head(number_of_curves)
        elif sort_method == 'Top Net Profit':
            df_checked = df_checked.sort_values(by='equity_net_profit', ascending=False, inplace=True)
            df_checked = df_checked.head(number_of_curves)
        else:
            df_checked = df_checked.head(number_of_curves)
            df_checked = df_checked.copy()

        df_checked = df_checked.reset_index(drop=True)

        line_colour = []
        for c in range(len(df_checked)):
            profile = c % 6
            degree = (c // 6) / math.ceil(len(df_checked) / 6)
            line_colour.append(self.assign_colour(profile, degree))
        df_checked['line_colour'] = line_colour

        return df_checked


    def assign_colour(self, profile, degree):
        if profile == 0:    rgb = (0, int(252 - 252 * degree), 252)
        elif profile == 1:  rgb = (int(252 - 252 * degree), 252, 0)
        elif profile == 2:  rgb = (252, 0, int(252 - 252 * degree))
        elif profile == 3:  rgb = (0, 252, int(252 * degree))
        elif profile == 4:  rgb = (252, int(252 * degree), 0)
        elif profile == 5:  rgb = (int(252 * degree), 0, 252)
        return 'rgb' + str(rgb)

    def generate_chart_3(self, para_combination, chart3_start, chart3_end, freq):

        data_folder = para_combination['data_folder']
        code = para_combination['code']
        start_date = para_combination['start_date']
        end_date = para_combination['end_date']
        data_path = os.path.join(data_folder, f'{code}_{freq}.parquet')

        df = pd.read_parquet(data_path)  # Daraframe that may not be daily

        df = df.reset_index(drop=False)

        df['date'] = pd.to_datetime(df['date'], format='%Y-%m-%d')
        df['datetime'] = pd.to_datetime(df['datetime'], format='%Y-%m-%d %H:%M:%S')
        df = df.loc[(df['date'] >= chart3_start) & (df['date'] <= chart3_end)]

        # grab first and last observations from df.date and make a continuous date range from that
        dt_all = pd.date_range(start=df['datetime'].iloc[0], end=df['datetime'].iloc[-1], freq=freq)
        # check which dates from your source that also accur in the continuous date range
        dt_obs = [d.strftime("%Y-%m-%d %H:%M:%S") for d in df['datetime']]
        # isolate missing timestamps
        dt_breaks = [d for d in dt_all.strftime("%Y-%m-%d %H:%M:%S").tolist() if not d in dt_obs]

        ###### chagne ######
        save_path = plotguy.generate_save_path(para_combination=para_combination)
        df_signal_list = plotguy.read_daily_df(save_path, start_date, end_date, freq)
        ####################

        df_signal_list = df_signal_list.loc[
            (df_signal_list['date'] >= chart3_start) & (df_signal_list['date'] <= chart3_end)]

        df_open = df_signal_list.loc[df_signal_list['action'] == 'open']
        df_profit = df_signal_list.loc[df_signal_list['action'] == 'profit_target']
        df_stop_loss = df_signal_list.loc[df_signal_list['action'] == 'stop_loss']
        df_close = df_signal_list.loc[df_signal_list['action'] == 'close_logic']
        df_close = df_signal_list.loc[df_signal_list['action'] == 'close_logic']

        fig = go.Figure(data=[go.Candlestick(x=df['datetime'],
                                             open=df['open'],
                                             high=df['high'],
                                             low=df['low'],
                                             close=df['close'],
                                             increasing_line_color='white')
                              ])

        fig.update_layout(title = {'text': 'Candlestick Chart' , 'font': {'size': 12 }  })

        fig.update_xaxes(rangebreaks=[dict(dvalue=15 * 60 * 1000, values=dt_breaks)])

        fig.add_trace(go.Scatter(mode='markers',
                                 #x=df_open['datetime'], y=df_open['close'],  # visible='legendonly',
                                 x=df_open.index, y=df_open['close'],  # visible='legendonly',
                                 marker=dict(color='rgba(0, 0, 0, 0)', size=18,
                                             line=dict(color='yellow', width=2.5)), name='Open'))
        fig.add_trace(go.Scatter(mode='markers',
                                 #x=df_profit['datetime'], y=df_profit['close'],  # visible='legendonly',
                                 x=df_profit.index, y=df_profit['close'],  # visible='legendonly',
                                 marker=dict(color='rgba(0, 0, 0, 0)', size=18,
                                             line=dict(color='Cyan', width=2.5)), name='Profit Target'))
        fig.add_trace(go.Scatter(mode='markers',
                                 #x=df_stop_loss['datetime'], y=df_stop_loss['close'],  # visible='legendonly',
                                 x=df_stop_loss.index, y=df_stop_loss['close'],  # visible='legendonly',
                                 marker=dict(color='rgba(0, 0, 0, 0)', size=18,
                                             line=dict(color='Cyan', width=2.5)), name='Stop Loss'))
        fig.add_trace(go.Scatter(mode='markers',
                                 #x=df_close['datetime'], y=df_close['close'],  # visible='legendonly',
                                 x=df_close.index, y=df_close['close'],  # visible='legendonly',
                                 marker=dict(color='rgba(0, 0, 0, 0)', size=18,
                                             line=dict(color='Cyan', width=2.5)), name='Close Logic'))

        fig.update_layout(xaxis_rangeslider_visible=True)

        chart_bg = '#1f2c56'
        fig.update_layout(plot_bgcolor=chart_bg, paper_bgcolor=chart_bg, height=500,
                          margin=dict(l=85, r=25, t=60, b=0),
                          showlegend=True,
                          font={"color": "white", 'size': 10.5}, yaxis={'title': 'Equity'},
                          xaxis={'title': ''}
                          )
        fig.update_xaxes(showline=True, zeroline=False, linecolor='#979A9A', gridcolor=chart_bg)
        fig.update_yaxes(showline=True, zeroline=False, linecolor='#979A9A', gridcolor=chart_bg)




        return fig


    def generate_main_chart(self, graph_df, backtest_name, equity_curve_folder, para_keys_list):


        fig_line = px.line()
        fig_line.update_layout(title = {'text': backtest_name})
        fig_line.update_xaxes(showline=True, zeroline=False, linecolor='white', gridcolor='rgba(0, 0, 0, 0)')
        fig_line.update_yaxes(showline=True, zeroline=False, linecolor='white', gridcolor='rgba(0, 0, 0, 0)')
        fig_line.update_layout(plot_bgcolor=self.chart_bg, paper_bgcolor=self.chart_bg, height=500,
                               margin=dict(l=85, r=25, t=60, b=0),
                               showlegend=False,
                               font={"color": "white", 'size': 10.5}, yaxis={'title': 'Equity'},
                               xaxis={'title': ''}
                               )

        for i in graph_df.index:

            hovertemplate = "%{x}<br>"
            hovertemplate = hovertemplate + "Equity : %{y:,.0f}"
            hovertemplate = hovertemplate + "<br>"
            hovertemplate = hovertemplate + "====================="
            hovertemplate = hovertemplate + "<br>"

            hovertemplate = hovertemplate + "Annual Return : " + "{:.0%}".format(graph_df.iloc[i]['equity_annualized_return'] / 100) + "<br>"
            hovertemplate = hovertemplate + "Sharpe Ratio : " + str(graph_df.iloc[i]['equity_annualized_sr']) + "<br>"
            hovertemplate = hovertemplate + "MDD Percentage : " + "{:.0%}".format(graph_df.iloc[i]['equity_mdd_pct'] / 100) + "<br>"
            hovertemplate = hovertemplate + "Return on Capital : " + "{:.0%}".format(graph_df.iloc[i]['equity_return_on_capital']/100) + "<br>"
            hovertemplate = hovertemplate + "====================="
            hovertemplate = hovertemplate + "<br>"

            for key in para_keys_list:
                value = str(graph_df.iloc[i][key])
                hovertemplate = hovertemplate + key + " : " + value + "<br>"

            hovertemplate = hovertemplate + "====================="
            hovertemplate = hovertemplate + "<br>"

            hovertemplate = hovertemplate + "Trade Count : " + str(graph_df.iloc[i]['num_of_trade']) + "<br>"
            hovertemplate = hovertemplate + "Net Profit : " + "{:,}".format(int(round(graph_df.iloc[i]['equity_net_profit'], 0))) + "<br>"
            hovertemplate = hovertemplate + "Net Profit to MDD: " + str(round(graph_df.iloc[i]['equity_net_profit_to_mdd'], 2)) + "<br>"
            hovertemplate = hovertemplate + "Return-BaH % Diff. : " + "{:.0%}".format(float(graph_df.iloc[i]['return_on_capital_diff']/100)) + "<br>"
            try:
                hovertemplate = hovertemplate + "Win Rate : " + "{:.0%}".format(float(graph_df.iloc[i]['win_rate']) / 100) + "<br>"
            except:
                hovertemplate = hovertemplate + "Win Rate : --" + "<br>"
            hovertemplate = hovertemplate + "COV (Count) : " + str(round(graph_df.iloc[i]['cov_count'], 2)) + "<br>"
            hovertemplate = hovertemplate + "COV (Return) : " + str(round(graph_df.iloc[i]['cov_return'], 2)) + "<br>"
            hovertemplate = hovertemplate + "Total Commissionn : " + "{:,}".format(int(round(graph_df.iloc[i]['total_commission'], 2))) + "<br>"
            hovertemplate = hovertemplate + "====================="

            hovertemplate = hovertemplate + "<br>"
            hovertemplate = hovertemplate + "ref_code : " + str(graph_df.iloc[i]['ref_code']) + "<br>"

            # prepare to read equity curve
            ref_code    = str(graph_df.iloc[i].ref_code)
            line_colour = graph_df.loc[i].line_colour

            if ref_code in self.df_daily_dict:
                df_daily = self.df_daily_dict[ref_code]
            else:
                file_path = os.path.join(equity_curve_folder, ref_code + '.parquet')
                df_daily = pd.read_parquet(file_path)
                self.df_daily_dict[ref_code] = df_daily

            fig_line.add_trace(go.Scatter(mode='lines', hovertemplate=hovertemplate,
                                          x=df_daily.index.date, y=df_daily['equity_value'],
                                          line=dict(color=line_colour, width=1.5), name=''), )


        return fig_line



    def generate_stra_alys_chart(self, intraday, graph_df, line_selected,
                                 para_keys_list, backtest_result_dict):

        line_colour = graph_df.iloc[line_selected].line_colour
        ref_code    = graph_df.iloc[line_selected].ref_code
        code        = graph_df.iloc[line_selected].code

        df_chart = self.df_daily_dict[str(ref_code)].copy()

        initial_equity_value = df_chart.iloc[0].equity_value

        df_chart['date'] = df_chart.index.date
        df_chart['adj_price'] = df_chart['price'] * (initial_equity_value / df_chart.iloc[0].price)

        if not intraday:

            df_chart['open_date'] = None
            df_chart['open_price'] = np.nan
            df_chart['close_date'] = None
            df_chart['close_price'] = np.nan
            df_chart['close_reason'] = None
            df_chart['pct_change'] = np.nan

            df_chart['close_logic_price'] = np.nan
            df_chart['profit_target_price'] = np.nan
            df_chart['stop_loss_price'] = np.nan

            df_chart['close_logic_price'] = df_chart['adj_price'].where(df_chart['action'] == 'close_logic')
            df_chart['profit_target_price'] = df_chart['adj_price'].where(df_chart['action'] == 'profit_target')
            df_chart['stop_loss_price'] = df_chart['adj_price'].where(df_chart['action'] == 'stop_loss')

            df_chart['filled_open_date'] = df_chart['date'].where(df_chart['action'] == 'open').ffill()
            df_chart['filled_open_price'] = df_chart['adj_price'].where(df_chart['action'] == 'open').ffill()

            close_mask = df_chart['action'].isin(['close_logic', 'profit_target', 'stop_loss'])

            df_chart['open_date'] = df_chart['filled_open_date'].where(close_mask)
            df_chart['open_price'] = df_chart['filled_open_price'].where(close_mask)

            df_chart['close_date'] = df_chart['date'].where(close_mask)
            df_chart['close_price'] = df_chart['adj_price'].where(close_mask)
            df_chart['close_reason'] = df_chart['action'].where(close_mask)

            df_chart['pct_change'] = (100 * ((df_chart['close_price'] / df_chart['open_price']) - 1)).where(close_mask)

            df_chart.drop(columns=['filled_open_date', 'filled_open_price'], inplace=True)

            df_chart_hover = df_chart[['price', 'open_date','open_price','close_date','close_price','close_reason','pct_change']].copy()
            df_chart_hover['open_price'] = df_chart_hover['open_price'] *  ( df_chart.iloc[0].price/ initial_equity_value)
            df_chart_hover['close_price'] = df_chart_hover['close_price'] *  ( df_chart.iloc[0].price/ initial_equity_value)
            df_chart       = df_chart[['date', 'equity_value', 'adj_price','open_price',
                                       'close_logic_price','profit_target_price','stop_loss_price', 'price']]

            df_chart_hover['open_price'] = df_chart_hover['open_price'].round(2)
            df_chart_hover['close_price'] = df_chart_hover['close_price'].round(2)
            df_chart_hover['pct_change'] = df_chart_hover['pct_change'].round(2)

            ### customdata index is the order of df column ###
            hover_template = "<br>".join([
                "Close Reason: %{customdata[5]}",
                "Open Date:    %{customdata[1]}",
                "Close Date:   %{customdata[3]}",
                "Open Price:   %{customdata[2]}",
                "Close Price:  %{customdata[4]}",
                "Pct Change:   %{customdata[6]} %",
            ])

        title = ''
        for key in para_keys_list:
            if len(title) > 100:
                title = title + '<br>'
            title = title + f'{key}:{backtest_result_dict[ref_code][key]} '

        fig_line = make_subplots(rows=1, cols=1, row_heights=[1],shared_xaxes=True)

        hover_template_equity_cruve = "<br>".join([
            "Equity Value: %{customdata[1]}",
        ])

        fig_line.update_layout(title = {'text': title , 'font': {'size': 12 }, 'y':0.97,  })
        fig_line.update_xaxes(showline=True, zeroline=False, linecolor='white', gridcolor='rgba(0, 0, 0, 0)')
        fig_line.update_yaxes(showline=True, zeroline=False, linecolor='white', gridcolor='rgba(0, 0, 0, 0)')
        fig_line.add_trace(go.Scatter(mode='lines', customdata=df_chart, hovertemplate=hover_template_equity_cruve,
                                      x=df_chart['date'], y=df_chart['equity_value'],
                                      line=dict(color=line_colour, width=1), name='Strategy Equity'), row=1, col=1)

        if intraday:
            hover_template_stock_price = "<br>".join([
                "Stock Price: %{customdata[0]}",
            ])
        elif not intraday:
            hover_template_stock_price = "<br>".join([
                "Stock Price: %{customdata[7]}",
            ])

        fig_line.add_trace(go.Scatter(mode='lines', customdata=df_chart, hovertemplate=hover_template_stock_price,
                                      # hoverinfo='skip',
                                      x=df_chart['date'], y=df_chart['adj_price'],
                                      line=dict(color='Grey', width=1), name=code ),row=1, col=1 )

        fig_line.update_layout(plot_bgcolor=self.chart_bg, paper_bgcolor=self.chart_bg, height=500,
                               margin=dict(l=85, r=25, t=35, b=20),
                               font={"color": "white", 'size': 9}, yaxis={'title': 'Equity'},
                               xaxis={'title': ''}
                               )

        if not intraday:
            fig_line.add_trace(go.Scatter(mode='markers', customdata=df_chart_hover, hovertemplate=hover_template,
                                          x=df_chart['date'], y=df_chart['open_price'], # visible='legendonly',
                                          marker=dict(color='rgba(0, 0, 0, 0)', size=9,
                                                      line=dict(color='yellow', width=2.5)), name='open'), row=1, col=1)

            fig_line.add_trace(go.Scatter(mode='markers', customdata=df_chart_hover, hovertemplate=hover_template,
                                          x=df_chart['date'], y=df_chart['close_logic_price'], visible='legendonly',
                                          marker=dict(color='rgba(0, 0, 0, 0)', size=9,
                                                      line=dict(color='green', width=2.5)), name='close_logic'), row=1, col=1)

            fig_line.add_trace(go.Scatter(mode='markers', customdata=df_chart_hover, hovertemplate=hover_template,
                                          x=df_chart['date'], y=df_chart['profit_target_price'], visible='legendonly',
                                          marker=dict(color='rgba(0, 0, 0, 0)', size=9,
                                                      line=dict(color='red', width=2.5)), name='profit_target'), row=1, col=1)

            fig_line.add_trace(go.Scatter(mode='markers', customdata=df_chart_hover, hovertemplate=hover_template,
                                          x=df_chart['date'], y=df_chart['stop_loss_price'], visible='legendonly',
                                          marker=dict(color='rgba(0, 0, 0, 0)', size=9,
                                                      line=dict(color='Cyan', width=2.5)), name='stop_loss'), row=1, col=1)

        return fig_line


    ### for class signal ###
    def generate_df_stat(self, df, period):
        col_pct = 'pct_change_' + str(period)
        col_rise = 'max_rise_' + str(period)
        col_fall = 'max_fall_' + str(period)
        df_his = df.copy()
        df_his[col_pct] = df_his['close'].pct_change(period)
        df_his[col_pct] = df_his[col_pct].shift(-1 * period)
        df_his[col_pct] = df_his[col_pct] * 100
        df_his[col_pct] = df_his[col_pct].map(lambda x: round(x, 2))
        df_his[col_rise] = (df_his['high'].rolling(period).max().shift(-1 * (period)) / df_his['close']) - 1
        df_his[col_fall] = (df_his['low'].rolling(period).min().shift(-1 * (period)) / df_his['close']) - 1

        df_his = df_his[df_his['logic'] == 'trade_logic']

        df_his[col_rise] = df_his[col_rise] * 100
        df_his[col_fall] = df_his[col_fall] * 100

        df_his[col_rise] = df_his[col_rise].map(lambda x: round(x, 2))
        df_his[col_fall] = df_his[col_fall].map(lambda x: round(x, 2))

        return round(df_his[col_pct].mean(), 2), round(df_his[col_rise].mean(), 2), round(df_his[col_fall].mean(), 2)


    def aggregate_df(self):
        all_str_dir = []
        for folder_name in os.listdir():
            if os.path.isfile(os.path.join(folder_name, 'saved_strategies.parquet')):
                all_str_dir.append(folder_name)
        all_str_dir.sort()
        equity_curves_dict_list = []

        for str_dir in all_str_dir:
            saved_strategies_path = os.path.join(str_dir, 'saved_strategies.parquet')
            saved_strategies_df   = pd.read_parquet(saved_strategies_path)

            for i, row in saved_strategies_df.iterrows():
                equity_path = os.path.join(str_dir, row["equity_curve_folder"], str(row['ref_code']) + '.parquet')
                #df_equity = pd.read_parquet(equity_path)

                yearly_stats_string  = row['yearly_stats_string']
                para_keys_str  = row['para_keys_str']
                para_keys_list = para_keys_str.split('|')

                parameters = {}
                for para_keys in para_keys_list:
                    parameters[para_keys] = str(row[para_keys])

                equity_curve_dict = {}

                equity_curve_dict['folder']        = str_dir
                equity_curve_dict['backtest_name'] = row['backtest_name']
                equity_curve_dict['equity_path']   =  equity_path
                equity_curve_dict['parameters']    = parameters

                equity_curve_dict['performance'] = {
                    'initial_capital':   row['initial_capital'],
                    'sharpe_ratio':      row['equity_annualized_sr'],
                    'annualized_return': row['equity_annualized_return'],
                    'net_profit_to_mdd': row['equity_net_profit_to_mdd'],
                    'net_profit':        row['equity_net_profit'],
                    'mdd_dollar':        row['equity_mdd_dollar'],
                    'mdd_pct':           row['equity_mdd_pct'],
                    'num_of_trade':      row['num_of_trade'],
                    'num_of_win':        row['num_of_win'],
                    'return_on_capital': row['equity_return_on_capital'],
                    'total_commission':  row['total_commission'],
                }

                ref_code = str(row['ref_code'])
                equity_curve_dict['ref_code'] = ref_code

                yearly_stats_dict = {}
                years_data = yearly_stats_string.split("|")

                for year_data in years_data:
                    parts = year_data.split(",")
                    year = parts[0]
                    yearly_stats_dict[year] = {}

                    for item in parts[1:]:
                        key, value = item.split(":")
                        yearly_stats_dict[year][key] = float(value)

                for idx, (year, value) in enumerate(yearly_stats_dict.items()):
                    year_trade_count = value['year_trade_count']
                    year_win_rate = value['year_win_rate']
                    year_return = value['year_return']

                    year_win_count =  year_trade_count * year_win_rate * 0.01

                    ### dont remove ###
                    # year_trade_count = "{:d}".format(int(year_trade_count))
                    # year_win_count = "{:d}".format(int(year_win_count))
                    # year_win_rate = "{:d}%".format(int(year_win_rate))
                    # year_return = "{:.2f} %".format(int(year_return))

                    equity_curve_dict['performance'][f'{year} Count']    = year_trade_count
                    equity_curve_dict['performance'][f'{year} Win Rate'] = year_win_rate
                    equity_curve_dict['performance'][f'{year} Win Count'] = year_win_count
                    equity_curve_dict['performance'][f'{year} Return']   = year_return

                equity_curves_dict_list.append(equity_curve_dict)

        line_colour = []
        for c in range(len(equity_curves_dict_list)):
            profile = c % 6
            degree = (c // 6) / math.ceil(len(equity_curves_dict_list) / 6)
            line_colour.append(self.assign_colour(profile, degree))

        df_show_agg = pd.DataFrame.from_dict(equity_curves_dict_list, orient='columns', dtype=None, columns=None)
        df_show_agg['line_colour'] = line_colour
        df_show_agg.index += 1

        choices = []

        ###
        for i in list(df_show_agg.index):
            initial_capital_text = []
            parameters_text      = []
            performance_text     = []
            performance_dict     = df_show_agg.iloc[i - 1].performance

            line_height = '120%'

            initial_capital = performance_dict['initial_capital']
            initial_capital = "{:,}".format(int(initial_capital))
            initial_capital_text.append(html.Div(f'{initial_capital}', style={'line-height': line_height}))

            for key, value in df_show_agg.iloc[i - 1].parameters.items():
                parameters_text.append(html.Div(f'{key} : {value}',style={'line-height': line_height}))


            for key, value in performance_dict.items():

                if key == 'initial_capital':
                    item = 'Initial Capital'
                    _value = "{:,}".format(int(value))
                    performance_text.append(html.Div(f'{item} : {_value}', style={'line-height': line_height}))

                if key == 'annualized_return':
                    item = 'Ann. Return'
                    _value = "{:.0%}".format(value/100)
                    performance_text.append(html.Div(f'{item} : {_value}', style={'line-height': line_height}))

                if key == 'sharpe_ratio':
                    item = 'Sharpe Ratio'
                    _value = "{:,.2f}".format(value)
                    performance_text.append(html.Div(f'{item} : {_value}', style={'line-height': line_height}))

                if key == 'mdd_pct':
                    item = 'MDD Percentage'
                    _value = "{:.0%}".format(value/100)
                    performance_text.append(html.Div(f'{item} : {_value}', style={'line-height': line_height}))

                if key == 'return_on_capital':
                    item = 'ReturnOnCapital'
                    _value = "{:.0%}".format(value/100)
                    performance_text.append(html.Div(f'{item} : {_value}', style={'line-height': line_height}))


            choices.append({
                "label": html.Div([

                    html.Div('-----------------------------------------------------------------'),

                    dbc.Row([
                        dbc.Col([

                            dbc.Row([
                                dbc.Col([html.Div('Curve', style={'line-height': line_height})],
                                        width=3),
                                dbc.Col([html.Div(str(i).zfill(3), style={'line-height': line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '5px'}),

                            dbc.Row([
                                dbc.Col([html.Div('folder',style={'line-height':line_height})],
                                        width=3),
                                dbc.Col([html.Div(df_show_agg.iloc[i - 1].folder,style={'line-height':line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '5px'}),

                            dbc.Row([
                                dbc.Col([html.Div('name',style={'line-height':line_height})],
                                        width=3),
                                dbc.Col([html.Div(df_show_agg.iloc[i - 1].backtest_name,style={'line-height':line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '10px'}),

                            dbc.Row([
                                dbc.Col([html.Div('initial_capital', style={'line-height': line_height})],
                                        width=3),
                                dbc.Col([html.Div(initial_capital_text, style={'line-height': line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '10px'}),

                            dbc.Row([
                                dbc.Col([html.Div('parameters',style={'line-height':line_height})],
                                        width=3),
                                dbc.Col([html.Div(parameters_text,style={'line-height':line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '10px'}),

                            dbc.Row([
                                dbc.Col([html.Div('performance', style={'line-height': line_height})],
                                        width=3),
                                dbc.Col([html.Div(performance_text, style={'line-height': line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '5px'}),

                            dbc.Row([
                                dbc.Col([html.Div('ref_code', style={'line-height': line_height})],
                                        width=3),
                                dbc.Col([html.Div(df_show_agg.iloc[i - 1].ref_code, style={'line-height': line_height})],
                                        width=6),
                            ]),

                            html.Div(style={'height': '5px'}),


                        ],width=12),

                    ]),

                ],style={"display": "inline-block", 'width':'320px'}

                ),

                "value": i,
            })

        _checklist = html.Div([
            dcc.Checklist(
                options=choices,
                id='curve_checklist',
                value=list(df_show_agg.index),
                inputStyle={'margin-left': '3px', 'margin-right': '10px',
                               'background-color': 'red',
                               'vertical-align': 'top', 'position': 'relative',
                               'top': '1.6em'}
            )],

        ),

        row = html.Div(
            dbc.Row([
                html.Div(_checklist),
                html.Div(style={'height': '5px'}),
                html.Div('--------------------------------------------------------------------------------',
                         style={'padding-left': '23px'}),
                html.Div(style={'height': '10px', }),
            ]), style={'padding': '0px', 'font-size': '12px'})

        return df_show_agg, row


    def aggregate_performance(self, total_dict, year_list, risk_free_rate):
        num_of_trade = "{:,}".format(int(total_dict['num_of_trade']))
        net_profit = "{:,}".format(int(round(total_dict['net_profit'],0)) )
        net_profit_to_mdd = round(total_dict['net_profit_to_mdd'],2)
        try:
            win_rate = "{:.0%}".format( float(total_dict['win_rate']) )
        except:
            win_rate = '--'

        # Sharpe Ratio
        annualized_std = total_dict['annualized_std']
        annualized_return = total_dict['annualized_return']

        if annualized_std > 0:
            annualized_sr = (annualized_return - float(risk_free_rate) / 100) / annualized_std
        else:
            annualized_sr = 0

        total_commission = "{:,}".format(int(total_dict['total_commission']))
        return_on_capital = "{:.0%}".format(total_dict['return_on_capital'])
        annualized_return = "{:.0%}".format(total_dict['annualized_return'])
        annualized_std = "{:.0%}".format(total_dict['annualized_std'])
        annualized_sr = round(annualized_sr,2)
        mdd_dollar = "{:,}".format(int(total_dict['mdd_dollar']))
        mdd_pct = "{:.0%}".format(total_dict['mdd_pct'])

        year_col = []
        year_count = []
        year_count_col = []
        year_win_rate_col = []
        year_return_col = []

        year_return_list = []
        for year in year_list:

            year_col.append(html.Div(year))

            year_count.append(int(total_dict[f'{year} Count']))
            year_count_col.append(html.Div(int(total_dict[f'{year} Count'])))

            rate = total_dict[f'{year} Win Count']/total_dict[f'{year} Count']
            year_win_rate_col.append(html.Div( "{:.0%}".format(rate) ))

            year_return = total_dict[f'{year}_return']
            year_return_col.append(html.Div( "{:.0%}".format(year_return) ))

            try:
                year_return_list.append(float(year_return))
            except:
                year_return_list.append(0)


        cov_count = round(np.std(year_count) / np.mean(year_count), 2)

        if np.mean(year_return_list) == 0:
            cov_return = 0
        else:
            cov_return = round(np.std(year_return_list) / np.mean(year_return_list), 2)

        div = html.Div([
            html.Div(style={'height': '10px'}),

            html.Div('Performance', style={'color': 'Cyan', 'font-size': '15px'}),

            html.Div(style={'height': '5px'}),

            dbc.Row([
                dbc.Col([html.Div('Number of Trade'),
                         html.Div('Net Profit'),
                         html.Div('Net Profit/MDD'),
                         html.Div('Win Rate'),
                         html.Div('COV (Count)'),
                         html.Div('COV (Return)'),
                         html.Div('Total Commission'),
                         html.Div('Risk Free Rate'),
                         html.Div(style={'height': '10px'}),
                         html.Div('Return on Capital'),
                         html.Div('Ann. Return',
                                  style={ 'color': '#FFD700','backgroundColor': '#333333'}),
                         html.Div('Ann. Std'),
                         html.Div('Ann. Sharpe Ratio',
                                    style={ 'color': '#FFD700', 'backgroundColor': '#333333'}),
                         html.Div('MDD Dollar'),
                         html.Div('MDD Percentage',
                                  style={ 'color': '#FFD700', 'backgroundColor': '#333333'}),
                         ],
                        style={'padding-left':'13px'},
                        width=7),
                dbc.Col([html.Div(num_of_trade),
                         html.Div(net_profit),
                         html.Div(net_profit_to_mdd),
                         html.Div(win_rate),
                         html.Div(cov_count),
                         html.Div(cov_return),
                         html.Div(total_commission),
                         html.Div("{:.2%}".format(risk_free_rate / 100)),
                         html.Div(style={'height': '10px'}),
                         html.Div(return_on_capital),
                         html.Div(annualized_return, style={ 'color': '#FFD700', 'backgroundColor': '#333333'}),
                         html.Div(annualized_std),
                         html.Div(annualized_sr, style={ 'color': '#FFD700', 'backgroundColor': '#333333'}),
                         html.Div(mdd_dollar),
                         html.Div(mdd_pct, style={ 'color': '#FFD700', 'backgroundColor': '#333333'}),
                         ],
                        style={'text-align': 'center',},width=5),
            ]),

            html.Div(style={'height': '30px'}),

            html.Div('Performance by Year', style={'color': 'Cyan', 'font-size': '15px'}),

            html.Div(style={'height': '5px'}),

            dbc.Row([
                dbc.Col(width=3),
                dbc.Col('Count', style={'color': 'Yellow', 'font-size': '15px'},width=3),
                dbc.Col('WinRate', style={'color': 'Yellow', 'font-size': '15px'},width=3),
                dbc.Col('Return', style={'color': 'Yellow', 'font-size': '15px'}, width=3),
            ], style={'text-align': 'center'}),

            dbc.Row([
                dbc.Col(year_col, width=3),
                dbc.Col(year_count_col, width=3),
                dbc.Col(year_win_rate_col, width=3),
                dbc.Col(year_return_col, width=3),
            ], style={'text-align': 'center'})



        ])

        return div