import logging
import math

from .....utility.list_tools import bisect_left_cmp, bisect_right_cmp
from ..datacube_mappers import DatacubeMapper


class OctahedralGridMapper(DatacubeMapper):
    def __init__(self, base_axis, mapped_axes, resolution, md5_hash=None, local_area=[], axis_reversed=None):
        # TODO: if local area is not empty list, raise NotImplemented
        self._mapped_axes = mapped_axes
        self._base_axis = base_axis
        self._resolution = resolution
        self._first_axis_vals = self.first_axis_vals()
        self._first_idx_map = self.create_first_idx_map()
        self._second_axis_spacing = {}
        self._axis_reversed = {mapped_axes[0]: True, mapped_axes[1]: False}
        if self._axis_reversed[mapped_axes[1]]:
            raise NotImplementedError("Octahedral grid with second axis in decreasing order is not supported")
        if not self._axis_reversed[mapped_axes[0]]:
            raise NotImplementedError("Octahedral grid with first axis in increasing order is not supported")
        self.compressed_grid_axes = [self._mapped_axes[1]]
        if md5_hash is not None:
            self.md5_hash = md5_hash
        else:
            self.md5_hash = _md5_hash.get(resolution, None)

    def gauss_first_guess(self):
        i = 0
        gvals = [
            2.4048255577e0,
            5.5200781103e0,
            8.6537279129e0,
            11.7915344391e0,
            14.9309177086e0,
            18.0710639679e0,
            21.2116366299e0,
            24.3524715308e0,
            27.4934791320e0,
            30.6346064684e0,
            33.7758202136e0,
            36.9170983537e0,
            40.0584257646e0,
            43.1997917132e0,
            46.3411883717e0,
            49.4826098974e0,
            52.6240518411e0,
            55.7655107550e0,
            58.9069839261e0,
            62.0484691902e0,
            65.1899648002e0,
            68.3314693299e0,
            71.4729816036e0,
            74.6145006437e0,
            77.7560256304e0,
            80.8975558711e0,
            84.0390907769e0,
            87.1806298436e0,
            90.3221726372e0,
            93.4637187819e0,
            96.6052679510e0,
            99.7468198587e0,
            102.8883742542e0,
            106.0299309165e0,
            109.1714896498e0,
            112.3130502805e0,
            115.4546126537e0,
            118.5961766309e0,
            121.7377420880e0,
            124.8793089132e0,
            128.0208770059e0,
            131.1624462752e0,
            134.3040166383e0,
            137.4455880203e0,
            140.5871603528e0,
            143.7287335737e0,
            146.8703076258e0,
            150.0118824570e0,
            153.1534580192e0,
            156.2950342685e0,
        ]

        numVals = len(gvals)
        vals = []
        for i in range(self._resolution):
            if i < numVals:
                vals.append(gvals[i])
            else:
                vals.append(vals[i - 1] + math.pi)
        return vals

    def get_precomputed_values_N2560(self):
        lats = [
            89.97309123040117,
            89.93823314562859,
            89.90316920502394,
            89.86805875280535,
            89.83293065763206,
            89.79779402499162,
            89.76265262112551,
            89.72750828294004,
            89.69236201232401,
            89.65721440193403,
            89.62206582467452,
            89.5869165268675,
            89.55176667774543,
            89.51661639740729,
            89.48146577344028,
            89.44631487122197,
            89.41116374053657,
            89.37601241996336,
            89.34086093986036,
            89.30570932444132,
            89.27055759325799,
            89.23540576226809,
            89.20025384462546,
            89.16510185127088,
            89.12994979137731,
            89.09479767269328,
            89.05964550180977,
            89.02449328436776,
            88.98934102522412,
            88.95418872858265,
            88.91903639810036,
            88.8838840369745,
            88.84873164801114,
            88.81357923368387,
            88.77842679618206,
            88.7432743374502,
            88.70812185922085,
            88.67296936304294,
            88.63781685030615,
            88.60266432226004,
            88.56751178003105,
            88.5323592246387,
            88.49720665700596,
            88.46205407797261,
            88.42690148830275,
            88.39174888869346,
            88.35659627978269,
            88.32144366215479,
            88.28629103634599,
            88.25113840284968,
            88.21598576212041,
            88.18083311457711,
            88.14568046060737,
            88.11052780056978,
            88.0753751347971,
            88.04022246359764,
            88.00506978725831,
            87.96991710604617,
            87.9347644202099,
            87.89961172998179,
            87.86445903557853,
            87.8293063372031,
            87.79415363504548,
            87.75900092928332,
            87.7238482200839,
            87.68869550760395,
            87.65354279199096,
            87.61839007338347,
            87.58323735191227,
            87.54808462770063,
            87.5129319008645,
            87.47777917151384,
            87.44262643975223,
            87.40747370567779,
            87.37232096938324,
            87.33716823095666,
            87.30201549048122,
            87.266862748036,
            87.23171000369605,
            87.19655725753272,
            87.16140450961363,
            87.1262517600033,
            87.09109900876308,
            87.05594625595114,
            87.0207935016234,
            86.98564074583264,
            86.9504879886297,
            86.91533523006262,
            86.88018247017756,
            86.84502970901876,
            86.80987694662807,
            86.77472418304592,
            86.73957141831082,
            86.70441865245964,
            86.66926588552778,
            86.634113117549,
            86.59896034855585,
            86.56380757857941,
            86.52865480764962,
            86.49350203579505,
            86.45834926304326,
            86.42319648942066,
            86.38804371495281,
            86.35289093966418,
            86.31773816357806,
            86.28258538671719,
            86.24743260910334,
            86.21227983075751,
            86.17712705169994,
            86.14197427194992,
            86.10682149152628,
            86.07166871044704,
            86.03651592872987,
            86.00136314639137,
            85.96621036344783,
            85.931057579915,
            85.89590479580797,
            85.86075201114136,
            85.82559922592937,
            85.7904464401856,
            85.75529365392322,
            85.72014086715508,
            85.68498807989361,
            85.64983529215064,
            85.61468250393777,
            85.57952971526623,
            85.54437692614691,
            85.50922413659035,
            85.47407134660659,
            85.43891855620564,
            85.40376576539705,
            85.36861297419011,
            85.33346018259377,
            85.2983073906168,
            85.26315459826759,
            85.22800180555454,
            85.19284901248552,
            85.15769621906828,
            85.12254342531035,
            85.08739063121912,
            85.05223783680162,
            85.01708504206468,
            84.98193224701521,
            84.94677945165958,
            84.91162665600422,
            84.87647386005523,
            84.84132106381864,
            84.80616826730024,
            84.77101547050583,
            84.73586267344089,
            84.70070987611072,
            84.66555707852066,
            84.63040428067582,
            84.59525148258108,
            84.56009868424141,
            84.52494588566144,
            84.48979308684572,
            84.45464028779887,
            84.41948748852516,
            84.38433468902892,
            84.34918188931425,
            84.3140290893852,
            84.2788762892457,
            84.24372348889973,
            84.20857068835086,
            84.17341788760294,
            84.13826508665947,
            84.10311228552392,
            84.06795948419972,
            84.03280668269028,
            83.99765388099878,
            83.96250107912836,
            83.92734827708223,
            83.89219547486339,
            83.85704267247478,
            83.82188986991927,
            83.78673706719977,
            83.75158426431902,
            83.7164314612797,
            83.68127865808447,
            83.64612585473593,
            83.61097305123657,
            83.57582024758892,
            83.54066744379534,
            83.50551463985829,
            83.47036183578001,
            83.43520903156282,
            83.40005622720888,
            83.36490342272042,
            83.32975061809955,
            83.2945978133483,
            83.25944500846877,
            83.224292203463,
            83.18913939833283,
            83.15398659308029,
            83.11883378770712,
            83.08368098221528,
            83.04852817660655,
            83.01337537088263,
            82.9782225650453,
            82.9430697590963,
            82.90791695303719,
            82.87276414686964,
            82.83761134059527,
            82.80245853421565,
            82.76730572773225,
            82.73215292114665,
            82.69700011446027,
            82.66184730767465,
            82.62669450079105,
            82.59154169381104,
            82.55638888673587,
            82.5212360795669,
            82.48608327230549,
            82.45093046495289,
            82.41577765751036,
            82.38062484997923,
            82.34547204236057,
            82.31031923465576,
            82.27516642686582,
            82.24001361899201,
            82.2048608110354,
            82.16970800299714,
            82.1345551948783,
            82.09940238667998,
            82.06424957840325,
            82.02909677004911,
            81.9939439616186,
            81.95879115311274,
            81.92363834453245,
            81.88848553587873,
            81.85333272715256,
            81.81817991835484,
            81.78302710948653,
            81.74787430054847,
            81.71272149154161,
            81.67756868246677,
            81.64241587332481,
            81.60726306411662,
            81.572110254843,
            81.53695744550481,
            81.50180463610273,
            81.46665182663769,
            81.43149901711035,
            81.39634620752157,
            81.36119339787204,
            81.32604058816251,
            81.29088777839375,
            81.25573496856643,
            81.22058215868122,
            81.1854293487389,
            81.15027653874006,
            81.11512372868543,
            81.07997091857564,
            81.04481810841138,
            81.00966529819324,
            80.9745124879219,
            80.93935967759795,
            80.904206867222,
            80.86905405679461,
            80.83390124631644,
            80.79874843578806,
            80.76359562521,
            80.72844281458288,
            80.69329000390721,
            80.65813719318359,
            80.62298438241248,
            80.58783157159452,
            80.55267876073015,
            80.51752594981987,
            80.48237313886426,
            80.44722032786379,
            80.41206751681895,
            80.37691470573023,
            80.34176189459811,
            80.30660908342306,
            80.27145627220554,
            80.23630346094598,
            80.20115064964492,
            80.1659978383027,
            80.13084502691984,
            80.09569221549675,
            80.06053940403386,
            80.02538659253156,
            79.99023378099028,
            79.95508096941045,
            79.91992815779244,
            79.88477534613668,
            79.84962253444357,
            79.81446972271347,
            79.77931691094673,
            79.74416409914382,
            79.70901128730503,
            79.67385847543072,
            79.6387056635213,
            79.60355285157715,
            79.56840003959856,
            79.53324722758587,
            79.4980944155395,
            79.46294160345968,
            79.42778879134681,
            79.39263597920124,
            79.35748316702326,
            79.32233035481318,
            79.28717754257134,
            79.25202473029802,
            79.21687191799359,
            79.18171910565829,
            79.14656629329242,
            79.11141348089633,
            79.07626066847028,
            79.04110785601456,
            79.00595504352944,
            78.9708022310152,
            78.93564941847217,
            78.90049660590059,
            78.86534379330068,
            78.83019098067278,
            78.79503816801711,
            78.75988535533398,
            78.7247325426236,
            78.68957972988625,
            78.65442691712212,
            78.61927410433155,
            78.58412129151476,
            78.54896847867194,
            78.51381566580338,
            78.47866285290928,
            78.4435100399899,
            78.40835722704547,
            78.3732044140762,
            78.33805160108233,
            78.30289878806408,
            78.26774597502167,
            78.23259316195528,
            78.19744034886517,
            78.16228753575152,
            78.1271347226146,
            78.09198190945453,
            78.05682909627157,
            78.02167628306591,
            77.98652346983775,
            77.95137065658729,
            77.91621784331471,
            77.88106503002021,
            77.84591221670398,
            77.81075940336622,
            77.77560659000709,
            77.7404537766268,
            77.7053009632255,
            77.67014814980341,
            77.63499533636069,
            77.5998425228975,
            77.56468970941404,
            77.52953689591047,
            77.49438408238693,
            77.45923126884364,
            77.42407845528075,
            77.38892564169842,
            77.35377282809678,
            77.31862001447601,
            77.2834672008363,
            77.2483143871778,
            77.21316157350064,
            77.17800875980494,
            77.14285594609095,
            77.10770313235871,
            77.07255031860848,
            77.0373975048403,
            77.0022446910544,
            76.96709187725088,
            76.93193906342987,
            76.89678624959154,
            76.86163343573602,
            76.82648062186347,
            76.79132780797399,
            76.75617499406773,
            76.7210221801448,
            76.6858693662054,
            76.65071655224956,
            76.6155637382775,
            76.5804109242893,
            76.54525811028509,
            76.51010529626504,
            76.4749524822292,
            76.43979966817773,
            76.40464685411077,
            76.36949404002841,
            76.33434122593079,
            76.299188411818,
            76.26403559769017,
            76.22888278354743,
            76.1937299693899,
            76.15857715521766,
            76.12342434103081,
            76.08827152682954,
            76.05311871261388,
            76.01796589838393,
            75.98281308413988,
            75.9476602698818,
            75.91250745560974,
            75.87735464132389,
            75.84220182702428,
            75.80704901271106,
            75.7718961983843,
            75.73674338404413,
            75.70159056969062,
            75.66643775532391,
            75.63128494094406,
            75.59613212655114,
            75.56097931214532,
            75.52582649772664,
            75.4906736832952,
            75.4555208688511,
            75.42036805439444,
            75.38521523992533,
            75.35006242544378,
            75.31490961094997,
            75.27975679644392,
            75.24460398192576,
            75.20945116739557,
            75.17429835285344,
            75.13914553829942,
            75.10399272373363,
            75.06883990915613,
            75.03368709456701,
            74.99853427996639,
            74.96338146535429,
            74.92822865073084,
            74.89307583609606,
            74.85792302145009,
            74.82277020679298,
            74.78761739212479,
            74.75246457744564,
            74.71731176275556,
            74.68215894805466,
            74.647006133343,
            74.61185331862066,
            74.57670050388771,
            74.54154768914422,
            74.50639487439025,
            74.4712420596259,
            74.43608924485122,
            74.4009364300663,
            74.36578361527117,
            74.33063080046593,
            74.29547798565063,
            74.26032517082535,
            74.22517235599017,
            74.19001954114512,
            74.15486672629031,
            74.11971391142576,
            74.08456109655155,
            74.04940828166777,
            74.01425546677446,
            73.97910265187167,
            73.9439498369595,
            73.90879702203799,
            73.87364420710718,
            73.83849139216716,
            73.80333857721797,
            73.76818576225969,
            73.73303294729237,
            73.69788013231607,
            73.66272731733085,
            73.62757450233674,
            73.59242168733385,
            73.5572688723222,
            73.52211605730184,
            73.48696324227285,
            73.45181042723527,
            73.41665761218917,
            73.38150479713458,
            73.34635198207157,
            73.3111991670002,
            73.2760463519205,
            73.24089353683253,
            73.20574072173635,
            73.17058790663201,
            73.13543509151957,
            73.10028227639904,
            73.06512946127053,
            73.02997664613406,
            72.99482383098967,
            72.95967101583742,
            72.92451820067735,
            72.88936538550952,
            72.85421257033397,
            72.81905975515078,
            72.7839069399599,
            72.7487541247615,
            72.71360130955557,
            72.67844849434215,
            72.64329567912128,
            72.60814286389304,
            72.57299004865742,
            72.5378372334145,
            72.50268441816432,
            72.46753160290694,
            72.43237878764238,
            72.39722597237069,
            72.36207315709191,
            72.3269203418061,
            72.29176752651328,
            72.25661471121349,
            72.22146189590678,
            72.18630908059319,
            72.15115626527279,
            72.11600344994555,
            72.08085063461158,
            72.0456978192709,
            72.0105450039235,
            71.97539218856949,
            71.94023937320887,
            71.9050865578417,
            71.86993374246799,
            71.8347809270878,
            71.79962811170117,
            71.76447529630813,
            71.72932248090869,
            71.69416966550294,
            71.6590168500909,
            71.62386403467256,
            71.58871121924801,
            71.55355840381726,
            71.51840558838035,
            71.48325277293733,
            71.44809995748821,
            71.41294714203303,
            71.37779432657186,
            71.34264151110469,
            71.30748869563158,
            71.27233588015254,
            71.2371830646676,
            71.20203024917683,
            71.16687743368026,
            71.13172461817786,
            71.09657180266971,
            71.06141898715586,
            71.02626617163634,
            70.99111335611113,
            70.95596054058028,
            70.92080772504387,
            70.88565490950188,
            70.85050209395436,
            70.81534927840131,
            70.78019646284282,
            70.74504364727889,
            70.70989083170953,
            70.6747380161348,
            70.6395852005547,
            70.60443238496927,
            70.56927956937857,
            70.53412675378257,
            70.49897393818136,
            70.46382112257493,
            70.42866830696332,
            70.39351549134655,
            70.35836267572463,
            70.32320986009763,
            70.28805704446557,
            70.25290422882847,
            70.21775141318632,
            70.1825985975392,
            70.14744578188711,
            70.11229296623009,
            70.07714015056814,
            70.04198733490132,
            70.00683451922961,
            69.97168170355312,
            69.93652888787177,
            69.90137607218566,
            69.86622325649478,
            69.83107044079917,
            69.79591762509887,
            69.76076480939385,
            69.7256119936842,
            69.69045917796988,
            69.65530636225097,
            69.62015354652748,
            69.58500073079942,
            69.54984791506679,
            69.51469509932967,
            69.47954228358806,
            69.44438946784197,
            69.40923665209144,
            69.37408383633647,
            69.3389310205771,
            69.30377820481336,
            69.26862538904524,
            69.23347257327282,
            69.19831975749605,
            69.16316694171502,
            69.1280141259297,
            69.09286131014014,
            69.05770849434634,
            69.02255567854833,
            68.98740286274617,
            68.95225004693981,
            68.91709723112932,
            68.8819444153147,
            68.846791599496,
            68.81163878367319,
            68.77648596784634,
            68.74133315201544,
            68.70618033618052,
            68.6710275203416,
            68.63587470449872,
            68.60072188865185,
            68.56556907280105,
            68.53041625694631,
            68.49526344108769,
            68.46011062522518,
            68.4249578093588,
            68.3898049934886,
            68.35465217761453,
            68.31949936173666,
            68.28434654585503,
            68.24919372996959,
            68.21404091408043,
            68.17888809818751,
            68.14373528229089,
            68.10858246639056,
            68.07342965048656,
            68.0382768345789,
            68.00312401866758,
            67.96797120275262,
            67.93281838683407,
            67.89766557091191,
            67.86251275498618,
            67.8273599390569,
            67.79220712312406,
            67.7570543071877,
            67.72190149124783,
            67.6867486753045,
            67.65159585935764,
            67.61644304340737,
            67.58129022745362,
            67.54613741149646,
            67.51098459553589,
            67.4758317795719,
            67.44067896360457,
            67.40552614763385,
            67.37037333165979,
            67.33522051568241,
            67.30006769970169,
            67.26491488371768,
            67.22976206773039,
            67.19460925173982,
            67.159456435746,
            67.12430361974893,
            67.08915080374865,
            67.05399798774513,
            67.01884517173846,
            66.98369235572856,
            66.94853953971553,
            66.91338672369935,
            66.87823390768001,
            66.84308109165757,
            66.807928275632,
            66.77277545960334,
            66.7376226435716,
            66.7024698275368,
            66.66731701149894,
            66.63216419545803,
            66.5970113794141,
            66.56185856336717,
            66.52670574731725,
            66.49155293126431,
            66.45640011520842,
            66.42124729914957,
            66.38609448308779,
            66.35094166702306,
            66.31578885095543,
            66.28063603488488,
            66.24548321881144,
            66.21033040273511,
            66.17517758665592,
            66.14002477057389,
            66.10487195448903,
            66.0697191384013,
            66.03456632231078,
            65.99941350621744,
            65.96426069012132,
            65.92910787402242,
            65.89395505792075,
            65.85880224181632,
            65.82364942570915,
            65.78849660959925,
            65.75334379348664,
            65.71819097737131,
            65.68303816125328,
            65.64788534513258,
            65.61273252900921,
            65.57757971288316,
            65.54242689675448,
            65.50727408062315,
            65.4721212644892,
            65.43696844835264,
            65.40181563221347,
            65.36666281607171,
            65.33150999992735,
            65.29635718378044,
            65.26120436763098,
            65.22605155147896,
            65.1908987353244,
            65.1557459191673,
            65.1205931030077,
            65.0854402868456,
            65.05028747068098,
            65.0151346545139,
            64.97998183834434,
            64.94482902217233,
            64.90967620599784,
            64.87452338982094,
            64.83937057364157,
            64.80421775745981,
            64.76906494127562,
            64.73391212508905,
            64.69875930890007,
            64.6636064927087,
            64.62845367651497,
            64.59330086031886,
            64.55814804412043,
            64.52299522791964,
            64.48784241171651,
            64.45268959551109,
            64.41753677930333,
            64.38238396309325,
            64.34723114688092,
            64.31207833066627,
            64.27692551444935,
            64.24177269823016,
            64.20661988200874,
            64.17146706578505,
            64.13631424955913,
            64.10116143333097,
            64.06600861710061,
            64.03085580086803,
            63.99570298463323,
            63.96055016839627,
            63.92539735215709,
            63.89024453591576,
            63.85509171967227,
            63.81993890342659,
            63.784786087178794,
            63.74963327092884,
            63.71448045467677,
            63.679327638422556,
            63.644174822166235,
            63.6090220059078,
            63.57386918964728,
            63.53871637338468,
            63.503563557119975,
            63.46841074085321,
            63.433257924584396,
            63.39810510831351,
            63.362952292040575,
            63.3277994757656,
            63.2926466594886,
            63.257493843209566,
            63.222341026928525,
            63.187188210645466,
            63.15203539436042,
            63.11688257807337,
            63.081729761784345,
            63.04657694549334,
            63.011424129200385,
            62.97627131290544,
            62.94111849660856,
            62.905965680309734,
            62.87081286400896,
            62.83566004770627,
            62.800507231401646,
            62.76535441509511,
            62.73020159878666,
            62.69504878247631,
            62.65989596616408,
            62.62474314984995,
            62.58959033353395,
            62.55443751721608,
            62.51928470089635,
            62.484131884574744,
            62.44897906825131,
            62.413826251926025,
            62.378673435598905,
            62.34352061926995,
            62.308367802939195,
            62.273214986606604,
            62.23806217027221,
            62.202909353936015,
            62.16775653759803,
            62.132603721258256,
            62.09745090491671,
            62.06229808857337,
            62.027145272228275,
            61.99199245588144,
            61.956839639532824,
            61.92168682318246,
            61.88653400683038,
            61.851381190476545,
            61.816228374120996,
            61.78107555776372,
            61.74592274140473,
            61.71076992504404,
            61.67561710868164,
            61.64046429231755,
            61.60531147595177,
            61.57015865958431,
            61.53500584321517,
            61.49985302684436,
            61.4647002104719,
            61.42954739409777,
            61.39439457772199,
            61.35924176134456,
            61.32408894496549,
            61.288936128584794,
            61.25378331220247,
            61.21863049581852,
            61.18347767943296,
            61.14832486304581,
            61.113172046657034,
            61.07801923026666,
            61.042866413874705,
            61.00771359748115,
            60.97256078108603,
            60.937407964689335,
            60.902255148291076,
            60.86710233189123,
            60.83194951548984,
            60.796796699086926,
            60.76164388268243,
            60.7264910662764,
            60.691338249868856,
            60.65618543345977,
            60.62103261704917,
            60.58587980063704,
            60.55072698422339,
            60.515574167808246,
            60.4804213513916,
            60.44526853497346,
            60.41011571855382,
            60.37496290213271,
            60.339810085710106,
            60.30465726928603,
            60.269504452860495,
            60.234351636433495,
            60.19919882000503,
            60.1640460035751,
            60.12889318714374,
            60.09374037071092,
            60.05858755427666,
            60.02343473784099,
            59.988281921403875,
            59.95312910496534,
            59.917976288525395,
            59.88282347208403,
            59.84767065564125,
            59.81251783919707,
            59.777365022751496,
            59.742212206304536,
            59.707059389856184,
            59.671906573406446,
            59.636753756955315,
            59.60160094050283,
            59.56644812404897,
            59.53129530759374,
            59.49614249113716,
            59.46098967467922,
            59.425836858219924,
            59.390684041759286,
            59.35553122529733,
            59.32037840883401,
            59.285225592369365,
            59.2500727759034,
            59.2149199594361,
            59.1797671429675,
            59.144614326497575,
            59.109461510026335,
            59.0743086935538,
            59.03915587707997,
            59.004003060604845,
            58.968850244128426,
            58.93369742765071,
            58.89854461117173,
            58.863391794691445,
            58.82823897820992,
            58.7930861617271,
            58.757933345243046,
            58.722780528757696,
            58.687627712271116,
            58.652474895783264,
            58.61732207929418,
            58.58216926280386,
            58.54701644631229,
            58.51186362981948,
            58.47671081332545,
            58.44155799683018,
            58.4064051803337,
            58.371252363835985,
            58.33609954733708,
            58.30094673083695,
            58.26579391433561,
            58.23064109783308,
            58.195488281329354,
            58.16033546482443,
            58.125182648318315,
            58.090029831811,
            58.054877015302516,
            58.01972419879286,
            57.98457138228201,
            57.94941856577002,
            57.914265749256835,
            57.87911293274251,
            57.84396011622701,
            57.80880729971035,
            57.773654483192566,
            57.738501666673606,
            57.70334885015353,
            57.6681960336323,
            57.63304321710993,
            57.59789040058643,
            57.56273758406181,
            57.52758476753605,
            57.49243195100917,
            57.45727913448118,
            57.422126317952085,
            57.38697350142187,
            57.35182068489056,
            57.31666786835813,
            57.28151505182461,
            57.24636223529,
            57.211209418754294,
            57.17605660221748,
            57.1409037856796,
            57.10575096914063,
            57.070598152600596,
            57.03544533605948,
            57.00029251951729,
            56.965139702974035,
            56.92998688642972,
            56.89483406988434,
            56.85968125333789,
            56.82452843679041,
            56.789375620241856,
            56.75422280369227,
            56.719069987141644,
            56.68391717058997,
            56.64876435403726,
            56.613611537483514,
            56.578458720928744,
            56.543305904372936,
            56.50815308781611,
            56.473000271258265,
            56.4378474546994,
            56.40269463813953,
            56.367541821578634,
            56.332389005016736,
            56.29723618845384,
            56.262083371889936,
            56.22693055532504,
            56.191777738759136,
            56.15662492219225,
            56.12147210562437,
            56.086319289055524,
            56.051166472485676,
            56.01601365591485,
            55.980860839343045,
            55.94570802277027,
            55.910555206196534,
            55.875402389621826,
            55.840249573046144,
            55.8050967564695,
            55.76994393989192,
            55.734791123313364,
            55.69963830673386,
            55.66448549015341,
            55.629332673572016,
            55.59417985698967,
            55.559027040406384,
            55.52387422382217,
            55.488721407237016,
            55.45356859065093,
            55.41841577406392,
            55.38326295747598,
            55.3481101408871,
            55.31295732429732,
            55.27780450770662,
            55.242651691115015,
            55.207498874522486,
            55.17234605792905,
            55.1371932413347,
            55.102040424739464,
            55.066887608143304,
            55.03173479154627,
            54.99658197494833,
            54.96142915834949,
            54.92627634174978,
            54.89112352514916,
            54.855970708547666,
            54.82081789194529,
            54.78566507534204,
            54.75051225873791,
            54.71535944213291,
            54.680206625527035,
            54.64505380892029,
            54.60990099231268,
            54.57474817570421,
            54.53959535909487,
            54.50444254248469,
            54.469289725873644,
            54.43413690926174,
            54.39898409264899,
            54.3638312760354,
            54.328678459420956,
            54.293525642805676,
            54.25837282618955,
            54.22322000957259,
            54.18806719295479,
            54.15291437633617,
            54.117761559716705,
            54.08260874309642,
            54.047455926475315,
            54.01230310985337,
            53.97715029323062,
            53.94199747660706,
            53.90684465998267,
            53.87169184335747,
            53.836539026731465,
            53.80138621010464,
            53.76623339347702,
            53.73108057684859,
            53.69592776021936,
            53.660774943589345,
            53.62562212695852,
            53.5904693103269,
            53.5553164936945,
            53.520163677061305,
            53.48501086042732,
            53.449858043792545,
            53.41470522715701,
            53.37955241052068,
            53.344399593883566,
            53.30924677724568,
            53.27409396060703,
            53.23894114396761,
            53.20378832732741,
            53.16863551068646,
            53.13348269404472,
            53.09832987740223,
            53.063177060758996,
            53.02802424411499,
            52.99287142747023,
            52.95771861082471,
            52.92256579417845,
            52.88741297753144,
            52.852260160883674,
            52.81710734423518,
            52.78195452758594,
            52.746801710935955,
            52.71164889428523,
            52.676496077633765,
            52.64134326098158,
            52.60619044432865,
            52.571037627675,
            52.53588481102062,
            52.50073199436551,
            52.465579177709685,
            52.43042636105314,
            52.39527354439586,
            52.36012072773788,
            52.324967911079185,
            52.28981509441977,
            52.254662277759635,
            52.2195094610988,
            52.184356644437266,
            52.14920382777502,
            52.11405101111207,
            52.07889819444841,
            52.04374537778406,
            52.00859256111902,
            51.97343974445329,
            51.938286927786855,
            51.903134111119726,
            51.867981294451916,
            51.832828477783416,
            51.797675661114226,
            51.76252284444436,
            51.72737002777381,
            51.69221721110258,
            51.65706439443068,
            51.6219115777581,
            51.58675876108484,
            51.55160594441091,
            51.51645312773632,
            51.48130031106105,
            51.44614749438512,
            51.41099467770852,
            51.37584186103126,
            51.34068904435335,
            51.30553622767477,
            51.270383410995535,
            51.235230594315645,
            51.2000777776351,
            51.16492496095391,
            51.12977214427206,
            51.09461932758957,
            51.059466510906425,
            51.02431369422265,
            50.98916087753823,
            50.95400806085317,
            50.91885524416746,
            50.88370242748112,
            50.848549610794144,
            50.81339679410654,
            50.77824397741829,
            50.74309116072942,
            50.70793834403992,
            50.672785527349795,
            50.63763271065904,
            50.60247989396767,
            50.567327077275664,
            50.53217426058305,
            50.497021443889814,
            50.46186862719597,
            50.426715810501506,
            50.391562993806424,
            50.356410177110725,
            50.32125736041443,
            50.28610454371753,
            50.250951727020016,
            50.215798910321894,
            50.180646093623174,
            50.145493276923844,
            50.110340460223924,
            50.0751876435234,
            50.040034826822286,
            50.004882010120575,
            49.96972919341826,
            49.93457637671536,
            49.89942356001188,
            49.8642707433078,
            49.82911792660313,
            49.79396510989788,
            49.75881229319205,
            49.723659476485636,
            49.68850665977864,
            49.65335384307106,
            49.61820102636292,
            49.58304820965418,
            49.54789539294488,
            49.512742576235006,
            49.477589759524555,
            49.442436942813536,
            49.40728412610195,
            49.37213130938979,
            49.336978492677076,
            49.3018256759638,
            49.26667285924995,
            49.231520042535536,
            49.19636722582058,
            49.16121440910506,
            49.12606159238898,
            49.09090877567235,
            49.05575595895515,
            49.02060314223742,
            48.985450325519125,
            48.95029750880029,
            48.9151446920809,
            48.87999187536097,
            48.8448390586405,
            48.809686241919465,
            48.774533425197916,
            48.73938060847581,
            48.70422779175317,
            48.66907497502999,
            48.63392215830627,
            48.59876934158202,
            48.563616524857245,
            48.52846370813193,
            48.49331089140608,
            48.45815807467971,
            48.4230052579528,
            48.38785244122537,
            48.35269962449742,
            48.31754680776895,
            48.282393991039946,
            48.247241174310425,
            48.21208835758039,
            48.176935540849826,
            48.141782724118755,
            48.10662990738716,
            48.07147709065506,
            48.036324273922446,
            48.00117145718931,
            47.966018640455665,
            47.930865823721525,
            47.89571300698687,
            47.86056019025171,
            47.825407373516036,
            47.790254556779864,
            47.7551017400432,
            47.71994892330603,
            47.68479610656835,
            47.64964328983018,
            47.614490473091514,
            47.57933765635235,
            47.5441848396127,
            47.509032022872546,
            47.4738792061319,
            47.43872638939076,
            47.40357357264914,
            47.36842075590702,
            47.33326793916443,
            47.298115122421336,
            47.26296230567777,
            47.22780948893371,
            47.19265667218917,
            47.15750385544417,
            47.12235103869867,
            47.0871982219527,
            47.05204540520624,
            47.016892588459314,
            46.98173977171191,
            46.94658695496403,
            46.9114341382157,
            46.87628132146687,
            46.84112850471759,
            46.80597568796783,
            46.770822871217604,
            46.73567005446692,
            46.70051723771576,
            46.66536442096413,
            46.63021160421206,
            46.595058787459514,
            46.55990597070651,
            46.52475315395304,
            46.489600337199114,
            46.45444752044474,
            46.4192947036899,
            46.384141886934614,
            46.348989070178874,
            46.31383625342268,
            46.278683436666036,
            46.24353061990893,
            46.208377803151386,
            46.17322498639339,
            46.13807216963495,
            46.10291935287606,
            46.06776653611673,
            46.03261371935696,
            45.99746090259675,
            45.96230808583608,
            45.92715526907498,
            45.89200245231345,
            45.85684963555147,
            45.82169681878906,
            45.78654400202622,
            45.75139118526293,
            45.716238368499205,
            45.68108555173506,
            45.64593273497048,
            45.61077991820547,
            45.575627101440034,
            45.54047428467417,
            45.50532146790787,
            45.47016865114115,
            45.435015834374,
            45.399863017606435,
            45.36471020083844,
            45.32955738407002,
            45.294404567301186,
            45.25925175053193,
            45.22409893376226,
            45.18894611699216,
            45.15379330022165,
            45.118640483450726,
            45.08348766667938,
            45.04833484990764,
            45.01318203313547,
            44.97802921636289,
            44.9428763995899,
            44.9077235828165,
            44.87257076604269,
            44.837417949268485,
            44.802265132493865,
            44.76711231571883,
            44.731959498943404,
            44.69680668216757,
            44.66165386539134,
            44.62650104861469,
            44.59134823183765,
            44.556195415060216,
            44.52104259828237,
            44.48588978150414,
            44.45073696472551,
            44.41558414794647,
            44.38043133116706,
            44.34527851438723,
            44.31012569760702,
            44.27497288082641,
            44.23982006404542,
            44.20466724726404,
            44.16951443048226,
            44.13436161370011,
            44.09920879691755,
            44.06405598013462,
            44.0289031633513,
            43.993750346567595,
            43.958597529783496,
            43.923444712999036,
            43.88829189621418,
            43.85313907942894,
            43.81798626264333,
            43.782833445857335,
            43.747680629070956,
            43.712527812284215,
            43.67737499549709,
            43.64222217870959,
            43.60706936192171,
            43.57191654513346,
            43.536763728344845,
            43.50161091155585,
            43.46645809476648,
            43.43130527797675,
            43.396152461186645,
            43.36099964439617,
            43.32584682760533,
            43.29069401081413,
            43.255541194022555,
            43.22038837723062,
            43.18523556043832,
            43.15008274364566,
            43.11492992685264,
            43.079777110059254,
            43.0446242932655,
            43.009471476471404,
            42.97431865967693,
            42.93916584288211,
            42.90401302608693,
            42.86886020929139,
            42.8337073924955,
            42.79855457569926,
            42.76340175890265,
            42.7282489421057,
            42.693096125308394,
            42.65794330851074,
            42.622790491712735,
            42.587637674914376,
            42.55248485811567,
            42.51733204131662,
            42.48217922451722,
            42.447026407717466,
            42.41187359091739,
            42.37672077411695,
            42.341567957316165,
            42.30641514051504,
            42.27126232371358,
            42.236109506911774,
            42.20095669010963,
            42.16580387330714,
            42.13065105650431,
            42.09549823970115,
            42.060345422897655,
            42.02519260609381,
            41.99003978928963,
            41.954886972485134,
            41.91973415568028,
            41.88458133887511,
            41.8494285220696,
            41.81427570526376,
            41.779122888457586,
            41.743970071651084,
            41.70881725484425,
            41.67366443803709,
            41.6385116212296,
            41.60335880442178,
            41.568205987613645,
            41.53305317080517,
            41.49790035399638,
            41.462747537187255,
            41.42759472037782,
            41.39244190356805,
            41.35728908675797,
            41.32213626994756,
            41.286983453136834,
            41.251830636325785,
            41.21667781951441,
            41.18152500270273,
            41.14637218589072,
            41.111219369078405,
            41.076066552265765,
            41.04091373545281,
            41.005760918639545,
            40.97060810182596,
            40.935455285012075,
            40.900302468197864,
            40.86514965138334,
            40.82999683456852,
            40.79484401775338,
            40.75969120093792,
            40.72453838412217,
            40.68938556730611,
            40.65423275048973,
            40.61907993367305,
            40.58392711685607,
            40.54877430003877,
            40.51362148322117,
            40.47846866640327,
            40.44331584958507,
            40.40816303276656,
            40.37301021594776,
            40.337857399128644,
            40.30270458230923,
            40.26755176548952,
            40.23239894866951,
            40.1972461318492,
            40.1620933150286,
            40.126940498207695,
            40.091787681386485,
            40.056634864565,
            40.021482047743206,
            39.98632923092113,
            39.95117641409874,
            39.91602359727607,
            39.8808707804531,
            39.84571796362985,
            39.8105651468063,
            39.77541232998246,
            39.74025951315834,
            39.705106696333914,
            39.669953879509215,
            39.63480106268422,
            39.59964824585894,
            39.56449542903337,
            39.52934261220752,
            39.494189795381374,
            39.459036978554956,
            39.42388416172825,
            39.38873134490127,
            39.35357852807398,
            39.31842571124643,
            39.2832728944186,
            39.24812007759048,
            39.21296726076208,
            39.17781444393341,
            39.142661627104445,
            39.10750881027522,
            39.072355993445704,
            39.03720317661592,
            39.002050359785855,
            38.966897542955515,
            38.93174472612489,
            38.89659190929401,
            38.861439092462845,
            38.82628627563141,
            38.791133458799685,
            38.75598064196771,
            38.72082782513546,
            38.68567500830294,
            38.65052219147014,
            38.615369374637076,
            38.58021655780374,
            38.545063740970136,
            38.509910924136264,
            38.47475810730213,
            38.43960529046773,
            38.40445247363306,
            38.36929965679812,
            38.33414683996292,
            38.29899402312746,
            38.26384120629173,
            38.22868838945575,
            38.19353557261949,
            38.158382755782974,
            38.123229938946196,
            38.08807712210916,
            38.05292430527186,
            38.017771488434306,
            37.982618671596484,
            37.94746585475841,
            37.91231303792008,
            37.87716022108148,
            37.842007404242636,
            37.80685458740354,
            37.77170177056418,
            37.73654895372457,
            37.7013961368847,
            37.66624332004458,
            37.631090503204206,
            37.59593768636357,
            37.56078486952269,
            37.525632052681566,
            37.490479235840176,
            37.455326418998546,
            37.42017360215666,
            37.38502078531453,
            37.34986796847214,
            37.314715151629514,
            37.27956233478663,
            37.24440951794352,
            37.209256701100145,
            37.17410388425652,
            37.138951067412656,
            37.10379825056855,
            37.06864543372419,
            37.03349261687959,
            36.99833980003475,
            36.96318698318967,
            36.92803416634435,
            36.89288134949877,
            36.85772853265296,
            36.82257571580691,
            36.78742289896063,
            36.7522700821141,
            36.717117265267326,
            36.68196444842032,
            36.64681163157307,
            36.611658814725594,
            36.57650599787787,
            36.54135318102991,
            36.506200364181716,
            36.471047547333285,
            36.43589473048463,
            36.40074191363572,
            36.365589096786586,
            36.33043627993722,
            36.29528346308762,
            36.26013064623778,
            36.224977829387726,
            36.18982501253743,
            36.1546721956869,
            36.119519378836145,
            36.084366561985156,
            36.04921374513395,
            36.0140609282825,
            35.978908111430826,
            35.94375529457892,
            35.908602477726795,
            35.87344966087443,
            35.838296844021855,
            35.80314402716905,
            35.767991210316005,
            35.73283839346276,
            35.69768557660927,
            35.662532759755564,
            35.62737994290163,
            35.59222712604747,
            35.5570743091931,
            35.52192149233849,
            35.48676867548368,
            35.45161585862863,
            35.41646304177337,
            35.38131022491789,
            35.34615740806218,
            35.31100459120626,
            35.27585177435012,
            35.24069895749376,
            35.20554614063719,
            35.170393323780395,
            35.13524050692339,
            35.10008769006616,
            35.06493487320871,
            35.02978205635106,
            34.99462923949318,
            34.95947642263509,
            34.92432360577679,
            34.88917078891827,
            34.85401797205955,
            34.818865155200605,
            34.783712338341445,
            34.74855952148208,
            34.71340670462251,
            34.67825388776272,
            34.643101070902716,
            34.60794825404251,
            34.572795437182094,
            34.53764262032147,
            34.502489803460634,
            34.46733698659959,
            34.43218416973834,
            34.39703135287689,
            34.361878536015226,
            34.326725719153345,
            34.29157290229127,
            34.25642008542899,
            34.2212672685665,
            34.18611445170381,
            34.15096163484091,
            34.115808817977815,
            34.08065600111451,
            34.045503184251004,
            34.010350367387296,
            33.97519755052339,
            33.94004473365928,
            33.90489191679496,
            33.86973909993044,
            33.83458628306573,
            33.79943346620082,
            33.76428064933571,
            33.7291278324704,
            33.693975015604885,
            33.65882219873917,
            33.62366938187327,
            33.58851656500717,
            33.55336374814086,
            33.518210931274375,
            33.483058114407676,
            33.44790529754079,
            33.4127524806737,
            33.37759966380643,
            33.34244684693896,
            33.3072940300713,
            33.272141213203426,
            33.23698839633538,
            33.20183557946714,
            33.1666827625987,
            33.131529945730065,
            33.09637712886125,
            33.061224311992234,
            33.026071495123034,
            32.99091867825364,
            32.95576586138406,
            32.92061304451429,
            32.88546022764433,
            32.85030741077418,
            32.81515459390384,
            32.78000177703332,
            32.74484896016261,
            32.709696143291715,
            32.674543326420626,
            32.63939050954935,
            32.6042376926779,
            32.56908487580625,
            32.53393205893442,
            32.49877924206241,
            32.46362642519021,
            32.42847360831784,
            32.39332079144527,
            32.358167974572524,
            32.3230151576996,
            32.28786234082648,
            32.25270952395319,
            32.21755670707972,
            32.18240389020606,
            32.14725107333222,
            32.112098256458204,
            32.076945439584016,
            32.041792622709636,
            32.00663980583508,
            31.97148698896035,
            31.93633417208543,
            31.901181355210348,
            31.866028538335073,
            31.83087572145963,
            31.795722904584004,
            31.76057008770821,
            31.725417270832235,
            31.690264453956086,
            31.655111637079763,
            31.61995882020326,
            31.584806003326584,
            31.54965318644973,
            31.514500369572712,
            31.479347552695515,
            31.44419473581814,
            31.409041918940602,
            31.373889102062883,
            31.338736285185,
            31.303583468306947,
            31.268430651428705,
            31.233277834550314,
            31.198125017671735,
            31.162972200793,
            31.127819383914087,
            31.092666567035003,
            31.057513750155756,
            31.022360933276335,
            30.987208116396754,
            30.952055299517,
            30.916902482637067,
            30.881749665756985,
            30.84659684887672,
            30.8114440319963,
            30.776291215115712,
            30.741138398234956,
            30.70598558135403,
            30.670832764472948,
            30.635679947591697,
            30.60052713071028,
            30.5653743138287,
            30.530221496946947,
            30.495068680065046,
            30.459915863182975,
            30.42476304630074,
            30.38961022941834,
            30.35445741253579,
            30.319304595653072,
            30.284151778770187,
            30.248998961887146,
            30.213846145003945,
            30.178693328120577,
            30.14354051123706,
            30.108387694353375,
            30.07323487746953,
            30.038082060585527,
            30.002929243701374,
            29.96777642681705,
            29.932623609932573,
            29.89747079304794,
            29.862317976163148,
            29.8271651592782,
            29.792012342393093,
            29.75685952550783,
            29.721706708622417,
            29.686553891736843,
            29.651401074851115,
            29.616248257965232,
            29.581095441079192,
            29.545942624193,
            29.51078980730665,
            29.475636990420156,
            29.4404841735335,
            29.4053313566467,
            29.37017853975974,
            29.335025722872626,
            29.299872905985364,
            29.264720089097946,
            29.22956727221038,
            29.19441445532266,
            29.159261638434792,
            29.124108821546777,
            29.088956004658602,
            29.053803187770285,
            29.018650370881815,
            28.983497553993203,
            28.948344737104435,
            28.913191920215514,
            28.878039103326454,
            28.842886286437242,
            28.807733469547884,
            28.77258065265838,
            28.737427835768727,
            28.70227501887892,
            28.667122201988978,
            28.631969385098884,
            28.596816568208652,
            28.561663751318267,
            28.526510934427737,
            28.491358117537064,
            28.456205300646246,
            28.421052483755282,
            28.38589966686418,
            28.350746849972932,
            28.31559403308154,
            28.280441216190003,
            28.24528839929832,
            28.210135582406505,
            28.174982765514542,
            28.139829948622435,
            28.104677131730188,
            28.0695243148378,
            28.034371497945276,
            27.999218681052607,
            27.964065864159796,
            27.92891304726685,
            27.893760230373758,
            27.85860741348053,
            27.823454596587165,
            27.78830177969366,
            27.753148962800015,
            27.717996145906234,
            27.682843329012314,
            27.64769051211826,
            27.612537695224063,
            27.57738487832973,
            27.542232061435264,
            27.50707924454066,
            27.47192642764592,
            27.43677361075104,
            27.401620793856026,
            27.36646797696088,
            27.331315160065596,
            27.29616234317018,
            27.261009526274627,
            27.22585670937894,
            27.190703892483125,
            27.15555107558717,
            27.120398258691083,
            27.08524544179486,
            27.05009262489851,
            27.014939808002026,
            26.979786991105414,
            26.944634174208666,
            26.909481357311783,
            26.874328540414773,
            26.83917572351763,
            26.804022906620357,
            26.768870089722956,
            26.73371727282542,
            26.69856445592776,
            26.663411639029963,
            26.628258822132043,
            26.59310600523399,
            26.557953188335812,
            26.522800371437505,
            26.48764755453907,
            26.452494737640507,
            26.417341920741812,
            26.38218910384299,
            26.347036286944046,
            26.311883470044968,
            26.27673065314577,
            26.241577836246446,
            26.206425019346995,
            26.171272202447415,
            26.136119385547712,
            26.100966568647884,
            26.06581375174793,
            26.03066093484785,
            25.995508117947647,
            25.960355301047322,
            25.925202484146872,
            25.8900496672463,
            25.8548968503456,
            25.819744033444778,
            25.784591216543838,
            25.74943839964277,
            25.71428558274158,
            25.679132765840272,
            25.643979948938842,
            25.608827132037288,
            25.573674315135612,
            25.538521498233816,
            25.503368681331903,
            25.468215864429865,
            25.433063047527707,
            25.39791023062543,
            25.362757413723035,
            25.32760459682052,
            25.292451779917887,
            25.25729896301513,
            25.22214614611226,
            25.18699332920927,
            25.15184051230616,
            25.11668769540293,
            25.08153487849959,
            25.046382061596127,
            25.01122924469255,
            24.976076427788854,
            24.94092361088504,
            24.905770793981116,
            24.87061797707707,
            24.83546516017291,
            24.800312343268633,
            24.76515952636424,
            24.730006709459733,
            24.694853892555113,
            24.659701075650382,
            24.62454825874553,
            24.589395441840566,
            24.55424262493549,
            24.519089808030298,
            24.483936991124992,
            24.448784174219576,
            24.413631357314046,
            24.378478540408402,
            24.343325723502645,
            24.308172906596777,
            24.2730200896908,
            24.237867272784708,
            24.202714455878503,
            24.16756163897219,
            24.13240882206577,
            24.097256005159235,
            24.062103188252593,
            24.02695037134583,
            23.991797554438968,
            23.956644737531995,
            23.92149192062491,
            23.886339103717713,
            23.851186286810414,
            23.816033469903,
            23.78088065299548,
            23.745727836087855,
            23.71057501918012,
            23.675422202272276,
            23.640269385364327,
            23.605116568456264,
            23.5699637515481,
            23.53481093463983,
            23.49965811773145,
            23.464505300822967,
            23.429352483914375,
            23.39419966700568,
            23.359046850096878,
            23.32389403318797,
            23.288741216278957,
            23.253588399369836,
            23.218435582460614,
            23.18328276555129,
            23.14812994864186,
            23.11297713173232,
            23.077824314822685,
            23.04267149791294,
            23.007518681003095,
            22.972365864093145,
            22.937213047183096,
            22.90206023027294,
            22.866907413362682,
            22.831754596452324,
            22.796601779541867,
            22.761448962631302,
            22.726296145720635,
            22.691143328809872,
            22.655990511899006,
            22.62083769498804,
            22.58568487807697,
            22.5505320611658,
            22.515379244254536,
            22.480226427343165,
            22.4450736104317,
            22.409920793520133,
            22.374767976608464,
            22.339615159696702,
            22.304462342784838,
            22.269309525872877,
            22.234156708960814,
            22.199003892048655,
            22.163851075136396,
            22.12869825822404,
            22.093545441311594,
            22.05839262439904,
            22.023239807486394,
            21.988086990573652,
            21.952934173660815,
            21.917781356747874,
            21.882628539834847,
            21.84747572292172,
            21.812322906008493,
            21.777170089095172,
            21.74201727218176,
            21.70686445526825,
            21.671711638354648,
            21.636558821440946,
            21.601406004527153,
            21.566253187613263,
            21.531100370699285,
            21.495947553785207,
            21.46079473687104,
            21.425641919956778,
            21.390489103042423,
            21.355336286127972,
            21.320183469213433,
            21.285030652298797,
            21.249877835384076,
            21.21472501846926,
            21.179572201554347,
            21.14441938463935,
            21.109266567724255,
            21.07411375080907,
            21.038960933893797,
            21.00380811697843,
            20.96865530006297,
            20.93350248314743,
            20.898349666231788,
            20.86319684931606,
            20.828044032400246,
            20.79289121548434,
            20.757738398568343,
            20.722585581652258,
            20.687432764736084,
            20.65227994781982,
            20.617127130903473,
            20.581974313987033,
            20.546821497070507,
            20.511668680153893,
            20.47651586323719,
            20.441363046320397,
            20.40621022940352,
            20.371057412486554,
            20.3359045955695,
            20.300751778652366,
            20.265598961735144,
            20.23044614481783,
            20.19529332790043,
            20.16014051098295,
            20.124987694065382,
            20.089834877147727,
            20.054682060229986,
            20.019529243312164,
            19.984376426394252,
            19.94922360947626,
            19.914070792558178,
            19.878917975640018,
            19.84376515872177,
            19.80861234180344,
            19.773459524885027,
            19.738306707966526,
            19.703153891047947,
            19.668001074129283,
            19.632848257210533,
            19.597695440291705,
            19.562542623372792,
            19.527389806453797,
            19.49223698953472,
            19.45708417261556,
            19.42193135569632,
            19.386778538776998,
            19.351625721857594,
            19.31647290493811,
            19.281320088018546,
            19.246167271098898,
            19.21101445417917,
            19.175861637259366,
            19.14070882033948,
            19.10555600341951,
            19.070403186499462,
            19.03525036957934,
            19.00009755265913,
            18.964944735738847,
            18.929791918818484,
            18.89463910189804,
            18.859486284977514,
            18.824333468056917,
            18.78918065113624,
            18.75402783421548,
            18.718875017294653,
            18.683722200373737,
            18.64856938345275,
            18.61341656653168,
            18.57826374961054,
            18.543110932689316,
            18.50795811576802,
            18.472805298846644,
            18.4376524819252,
            18.402499665003667,
            18.367346848082065,
            18.332194031160387,
            18.297041214238636,
            18.261888397316806,
            18.2267355803949,
            18.191582763472923,
            18.156429946550862,
            18.121277129628737,
            18.08612431270653,
            18.050971495784253,
            18.0158186788619,
            17.980665861939475,
            17.945513045016977,
            17.9103602280944,
            17.875207411171754,
            17.840054594249036,
            17.804901777326243,
            17.769748960403373,
            17.734596143480434,
            17.699443326557425,
            17.66429050963434,
            17.629137692711186,
            17.59398487578796,
            17.55883205886466,
            17.52367924194129,
            17.48852642501785,
            17.45337360809433,
            17.41822079117075,
            17.38306797424709,
            17.347915157323367,
            17.31276234039957,
            17.277609523475704,
            17.242456706551767,
            17.20730388962776,
            17.17215107270368,
            17.13699825577953,
            17.101845438855317,
            17.06669262193103,
            17.031539805006677,
            16.996386988082254,
            16.961234171157763,
            16.9260813542332,
            16.89092853730857,
            16.855775720383875,
            16.820622903459107,
            16.78547008653428,
            16.75031726960938,
            16.71516445268441,
            16.68001163575937,
            16.64485881883427,
            16.609706001909103,
            16.574553184983866,
            16.539400368058562,
            16.50424755113319,
            16.469094734207754,
            16.433941917282255,
            16.398789100356687,
            16.363636283431052,
            16.328483466505354,
            16.293330649579588,
            16.258177832653757,
            16.223025015727863,
            16.1878721988019,
            16.15271938187588,
            16.11756656494979,
            16.082413748023637,
            16.04726093109742,
            16.012108114171134,
            15.976955297244794,
            15.941802480318382,
            15.906649663391912,
            15.871496846465373,
            15.836344029538775,
            15.801191212612112,
            15.766038395685385,
            15.730885578758597,
            15.69573276183175,
            15.660579944904834,
            15.625427127977863,
            15.590274311050823,
            15.555121494123721,
            15.51996867719656,
            15.484815860269338,
            15.449663043342058,
            15.414510226414713,
            15.379357409487309,
            15.34420459255984,
            15.309051775632312,
            15.273898958704725,
            15.238746141777076,
            15.20359332484937,
            15.168440507921602,
            15.133287690993775,
            15.098134874065886,
            15.062982057137939,
            15.027829240209932,
            14.992676423281866,
            14.957523606353742,
            14.922370789425557,
            14.887217972497318,
            14.852065155569013,
            14.816912338640655,
            14.781759521712239,
            14.746606704783762,
            14.711453887855232,
            14.676301070926637,
            14.64114825399799,
            14.605995437069284,
            14.570842620140517,
            14.535689803211701,
            14.500536986282821,
            14.465384169353886,
            14.430231352424897,
            14.395078535495852,
            14.359925718566746,
            14.324772901637589,
            14.289620084708373,
            14.254467267779102,
            14.219314450849774,
            14.184161633920393,
            14.149008816990957,
            14.113856000061462,
            14.078703183131916,
            14.043550366202313,
            14.008397549272656,
            13.973244732342945,
            13.938091915413178,
            13.90293909848336,
            13.867786281553485,
            13.832633464623557,
            13.797480647693574,
            13.76232783076354,
            13.72717501383345,
            13.692022196903311,
            13.656869379973115,
            13.621716563042867,
            13.586563746112567,
            13.551410929182213,
            13.516258112251808,
            13.48110529532135,
            13.445952478390842,
            13.41079966146028,
            13.375646844529665,
            13.340494027599,
            13.305341210668281,
            13.270188393737513,
            13.235035576806695,
            13.199882759875823,
            13.164729942944902,
            13.12957712601393,
            13.094424309082909,
            13.059271492151833,
            13.024118675220711,
            12.988965858289538,
            12.953813041358314,
            12.91866022442704,
            12.883507407495717,
            12.848354590564345,
            12.813201773632924,
            12.778048956701452,
            12.742896139769933,
            12.707743322838365,
            12.672590505906745,
            12.637437688975078,
            12.602284872043365,
            12.567132055111601,
            12.531979238179792,
            12.496826421247933,
            12.461673604316024,
            12.426520787384069,
            12.391367970452068,
            12.356215153520019,
            12.321062336587921,
            12.285909519655778,
            12.250756702723589,
            12.215603885791351,
            12.180451068859067,
            12.145298251926736,
            12.11014543499436,
            12.074992618061938,
            12.03983980112947,
            12.004686984196953,
            11.969534167264394,
            11.934381350331789,
            11.899228533399137,
            11.864075716466438,
            11.828922899533698,
            11.793770082600911,
            11.758617265668079,
            11.723464448735204,
            11.688311631802282,
            11.653158814869316,
            11.618005997936308,
            11.582853181003255,
            11.547700364070158,
            11.512547547137018,
            11.477394730203832,
            11.442241913270603,
            11.407089096337334,
            11.371936279404018,
            11.33678346247066,
            11.30163064553726,
            11.266477828603817,
            11.23132501167033,
            11.196172194736803,
            11.161019377803232,
            11.12586656086962,
            11.090713743935966,
            11.055560927002269,
            11.02040811006853,
            10.985255293134749,
            10.950102476200929,
            10.914949659267066,
            10.87979684233316,
            10.844644025399216,
            10.809491208465232,
            10.774338391531206,
            10.739185574597139,
            10.704032757663033,
            10.668879940728884,
            10.633727123794698,
            10.598574306860469,
            10.563421489926201,
            10.528268672991894,
            10.493115856057548,
            10.457963039123163,
            10.422810222188739,
            10.387657405254272,
            10.35250458831977,
            10.317351771385228,
            10.282198954450648,
            10.247046137516026,
            10.211893320581368,
            10.176740503646672,
            10.14158768671194,
            10.106434869777168,
            10.071282052842356,
            10.03612923590751,
            10.000976418972625,
            9.965823602037702,
            9.930670785102743,
            9.895517968167747,
            9.860365151232713,
            9.82521233429764,
            9.790059517362534,
            9.75490670042739,
            9.71975388349221,
            9.684601066556993,
            9.649448249621742,
            9.614295432686454,
            9.579142615751127,
            9.543989798815767,
            9.508836981880373,
            9.473684164944942,
            9.438531348009473,
            9.403378531073974,
            9.368225714138436,
            9.333072897202864,
            9.29792008026726,
            9.262767263331618,
            9.227614446395943,
            9.192461629460235,
            9.15730881252449,
            9.122155995588713,
            9.087003178652902,
            9.051850361717056,
            9.016697544781179,
            8.981544727845264,
            8.946391910909322,
            8.911239093973341,
            8.87608627703733,
            8.840933460101287,
            8.805780643165209,
            8.770627826229099,
            8.735475009292957,
            8.700322192356781,
            8.665169375420575,
            8.630016558484336,
            8.594863741548064,
            8.559710924611762,
            8.524558107675428,
            8.489405290739063,
            8.454252473802663,
            8.419099656866235,
            8.383946839929775,
            8.348794022993285,
            8.313641206056763,
            8.278488389120213,
            8.24333557218363,
            8.208182755247016,
            8.173029938310373,
            8.1378771213737,
            8.102724304436997,
            8.067571487500262,
            8.0324186705635,
            7.997265853626706,
            7.962113036689885,
            7.926960219753031,
            7.891807402816152,
            7.856654585879242,
            7.821501768942302,
            7.786348952005335,
            7.75119613506834,
            7.716043318131316,
            7.680890501194263,
            7.645737684257181,
            7.610584867320071,
            7.575432050382934,
            7.5402792334457684,
            7.505126416508576,
            7.469973599571356,
            7.434820782634108,
            7.399667965696834,
            7.364515148759531,
            7.329362331822201,
            7.294209514884845,
            7.2590566979474636,
            7.223903881010052,
            7.188751064072616,
            7.153598247135154,
            7.118445430197665,
            7.083292613260151,
            7.04813979632261,
            7.012986979385044,
            6.9778341624474525,
            6.942681345509834,
            6.9075285285721915,
            6.872375711634522,
            6.837222894696829,
            6.802070077759111,
            6.766917260821368,
            6.731764443883599,
            6.696611626945806,
            6.661458810007989,
            6.626305993070147,
            6.5911531761322815,
            6.556000359194392,
            6.520847542256478,
            6.48569472531854,
            6.450541908380579,
            6.415389091442594,
            6.380236274504585,
            6.345083457566554,
            6.3099306406285,
            6.274777823690423,
            6.239625006752321,
            6.204472189814198,
            6.1693193728760525,
            6.134166555937884,
            6.099013738999694,
            6.06386092206148,
            6.028708105123244,
            5.993555288184988,
            5.958402471246708,
            5.923249654308408,
            5.888096837370086,
            5.8529440204317424,
            5.817791203493377,
            5.782638386554992,
            5.747485569616583,
            5.712332752678156,
            5.677179935739707,
            5.642027118801237,
            5.606874301862748,
            5.571721484924238,
            5.536568667985707,
            5.501415851047156,
            5.466263034108586,
            5.431110217169994,
            5.395957400231384,
            5.360804583292754,
            5.325651766354105,
            5.290498949415436,
            5.255346132476749,
            5.220193315538042,
            5.185040498599316,
            5.149887681660571,
            5.114734864721807,
            5.079582047783026,
            5.044429230844225,
            5.009276413905406,
            4.974123596966568,
            4.938970780027715,
            4.903817963088841,
            4.868665146149951,
            4.833512329211042,
            4.798359512272117,
            4.7632066953331735,
            4.728053878394213,
            4.692901061455236,
            4.657748244516241,
            4.62259542757723,
            4.587442610638202,
            4.552289793699156,
            4.517136976760096,
            4.481984159821018,
            4.446831342881925,
            4.411678525942815,
            4.3765257090036895,
            4.341372892064547,
            4.30622007512539,
            4.271067258186218,
            4.235914441247029,
            4.200761624307825,
            4.165608807368606,
            4.130455990429373,
            4.095303173490125,
            4.060150356550859,
            4.024997539611582,
            3.9898447226722893,
            3.9546919057329823,
            3.91953908879366,
            3.8843862718543245,
            3.8492334549149745,
            3.814080637975611,
            3.778927821036234,
            3.7437750040968427,
            3.7086221871574376,
            3.67346937021802,
            3.6383165532785893,
            3.6031637363391447,
            3.568010919399687,
            3.5328581024602173,
            3.497705285520734,
            3.4625524685812388,
            3.42739965164173,
            3.39224683470221,
            3.357094017762678,
            3.321941200823133,
            3.2867883838835765,
            3.2516355669440076,
            3.216482750004427,
            3.1813299330648355,
            3.1461771161252323,
            3.1110242991856176,
            3.075871482245992,
            3.0407186653063545,
            3.005565848366707,
            2.970413031427048,
            2.9352602144873785,
            2.9001073975476985,
            2.864954580608008,
            2.829801763668307,
            2.7946489467285964,
            2.759496129788875,
            2.724343312849144,
            2.6891904959094033,
            2.654037678969653,
            2.6188848620298932,
            2.583732045090123,
            2.5485792281503445,
            2.5134264112105567,
            2.47827359427076,
            2.443120777330954,
            2.4079679603911397,
            2.372815143451316,
            2.337662326511484,
            2.3025095095716437,
            2.267356692631795,
            2.2322038756919387,
            2.1970510587520744,
            2.1618982418122017,
            2.1267454248723205,
            2.0915926079324327,
            2.0564397909925374,
            2.021286974052634,
            1.986134157112724,
            1.9509813401728067,
            1.9158285232328822,
            1.8806757062929509,
            1.845522889353013,
            1.8103700724130682,
            1.7752172554731172,
            1.7400644385331594,
            1.7049116215931956,
            1.6697588046532257,
            1.6346059877132497,
            1.599453170773268,
            1.5643003538332805,
            1.5291475368932872,
            1.4939947199532886,
            1.4588419030132844,
            1.4236890860732752,
            1.3885362691332608,
            1.3533834521932415,
            1.3182306352532174,
            1.2830778183131881,
            1.2479250013731547,
            1.2127721844331167,
            1.1776193674930742,
            1.1424665505530276,
            1.1073137336129768,
            1.072160916672922,
            1.0370080997328637,
            1.0018552827928013,
            0.9667024658527352,
            0.9315496489126661,
            0.8963968319725931,
            0.8612440150325175,
            0.8260911980924384,
            0.7909383811523563,
            0.7557855642122715,
            0.7206327472721838,
            0.6854799303320939,
            0.6503271133920011,
            0.6151742964519064,
            0.5800214795118092,
            0.54486866257171,
            0.5097158456316088,
            0.4745630286915058,
            0.4394102117514011,
            0.4042573948112948,
            0.369104577871187,
            0.333951760931078,
            0.2987989439909678,
            0.2636461270508564,
            0.22849331011074414,
            0.19334049317063104,
            0.1581876762305173,
            0.12303485929040288,
            0.08788204235028811,
            0.05272922541017298,
            0.017576408470057668,
            -0.017576408470057668,
            -0.05272922541017298,
            -0.08788204235028811,
            -0.12303485929040288,
            -0.1581876762305173,
            -0.19334049317063104,
            -0.22849331011074414,
            -0.2636461270508564,
            -0.2987989439909678,
            -0.333951760931078,
            -0.369104577871187,
            -0.4042573948112948,
            -0.4394102117514011,
            -0.4745630286915058,
            -0.5097158456316088,
            -0.54486866257171,
            -0.5800214795118092,
            -0.6151742964519064,
            -0.6503271133920011,
            -0.6854799303320939,
            -0.7206327472721838,
            -0.7557855642122715,
            -0.7909383811523563,
            -0.8260911980924384,
            -0.8612440150325175,
            -0.8963968319725931,
            -0.9315496489126661,
            -0.9667024658527352,
            -1.0018552827928013,
            -1.0370080997328637,
            -1.072160916672922,
            -1.1073137336129768,
            -1.1424665505530276,
            -1.1776193674930742,
            -1.2127721844331167,
            -1.2479250013731547,
            -1.2830778183131881,
            -1.3182306352532174,
            -1.3533834521932415,
            -1.3885362691332608,
            -1.4236890860732752,
            -1.4588419030132844,
            -1.4939947199532886,
            -1.5291475368932872,
            -1.5643003538332805,
            -1.599453170773268,
            -1.6346059877132497,
            -1.6697588046532257,
            -1.7049116215931956,
            -1.7400644385331594,
            -1.7752172554731172,
            -1.8103700724130682,
            -1.845522889353013,
            -1.8806757062929509,
            -1.9158285232328822,
            -1.9509813401728067,
            -1.986134157112724,
            -2.021286974052634,
            -2.0564397909925374,
            -2.0915926079324327,
            -2.1267454248723205,
            -2.1618982418122017,
            -2.1970510587520744,
            -2.2322038756919387,
            -2.267356692631795,
            -2.3025095095716437,
            -2.337662326511484,
            -2.372815143451316,
            -2.4079679603911397,
            -2.443120777330954,
            -2.47827359427076,
            -2.5134264112105567,
            -2.5485792281503445,
            -2.583732045090123,
            -2.6188848620298932,
            -2.654037678969653,
            -2.6891904959094033,
            -2.724343312849144,
            -2.759496129788875,
            -2.7946489467285964,
            -2.829801763668307,
            -2.864954580608008,
            -2.9001073975476985,
            -2.9352602144873785,
            -2.970413031427048,
            -3.005565848366707,
            -3.0407186653063545,
            -3.075871482245992,
            -3.1110242991856176,
            -3.1461771161252323,
            -3.1813299330648355,
            -3.216482750004427,
            -3.2516355669440076,
            -3.2867883838835765,
            -3.321941200823133,
            -3.357094017762678,
            -3.39224683470221,
            -3.42739965164173,
            -3.4625524685812388,
            -3.497705285520734,
            -3.5328581024602173,
            -3.568010919399687,
            -3.6031637363391447,
            -3.6383165532785893,
            -3.67346937021802,
            -3.7086221871574376,
            -3.7437750040968427,
            -3.778927821036234,
            -3.814080637975611,
            -3.8492334549149745,
            -3.8843862718543245,
            -3.91953908879366,
            -3.9546919057329823,
            -3.9898447226722893,
            -4.024997539611582,
            -4.060150356550859,
            -4.095303173490125,
            -4.130455990429373,
            -4.165608807368606,
            -4.200761624307825,
            -4.235914441247029,
            -4.271067258186218,
            -4.30622007512539,
            -4.341372892064547,
            -4.3765257090036895,
            -4.411678525942815,
            -4.446831342881925,
            -4.481984159821018,
            -4.517136976760096,
            -4.552289793699156,
            -4.587442610638202,
            -4.62259542757723,
            -4.657748244516241,
            -4.692901061455236,
            -4.728053878394213,
            -4.7632066953331735,
            -4.798359512272117,
            -4.833512329211042,
            -4.868665146149951,
            -4.903817963088841,
            -4.938970780027715,
            -4.974123596966568,
            -5.009276413905406,
            -5.044429230844225,
            -5.079582047783026,
            -5.114734864721807,
            -5.149887681660571,
            -5.185040498599316,
            -5.220193315538042,
            -5.255346132476749,
            -5.290498949415436,
            -5.325651766354105,
            -5.360804583292754,
            -5.395957400231384,
            -5.431110217169994,
            -5.466263034108586,
            -5.501415851047156,
            -5.536568667985707,
            -5.571721484924238,
            -5.606874301862748,
            -5.642027118801237,
            -5.677179935739707,
            -5.712332752678156,
            -5.747485569616583,
            -5.782638386554992,
            -5.817791203493377,
            -5.8529440204317424,
            -5.888096837370086,
            -5.923249654308408,
            -5.958402471246708,
            -5.993555288184988,
            -6.028708105123244,
            -6.06386092206148,
            -6.099013738999694,
            -6.134166555937884,
            -6.1693193728760525,
            -6.204472189814198,
            -6.239625006752321,
            -6.274777823690423,
            -6.3099306406285,
            -6.345083457566554,
            -6.380236274504585,
            -6.415389091442594,
            -6.450541908380579,
            -6.48569472531854,
            -6.520847542256478,
            -6.556000359194392,
            -6.5911531761322815,
            -6.626305993070147,
            -6.661458810007989,
            -6.696611626945806,
            -6.731764443883599,
            -6.766917260821368,
            -6.802070077759111,
            -6.837222894696829,
            -6.872375711634522,
            -6.9075285285721915,
            -6.942681345509834,
            -6.9778341624474525,
            -7.012986979385044,
            -7.04813979632261,
            -7.083292613260151,
            -7.118445430197665,
            -7.153598247135154,
            -7.188751064072616,
            -7.223903881010052,
            -7.2590566979474636,
            -7.294209514884845,
            -7.329362331822201,
            -7.364515148759531,
            -7.399667965696834,
            -7.434820782634108,
            -7.469973599571356,
            -7.505126416508576,
            -7.5402792334457684,
            -7.575432050382934,
            -7.610584867320071,
            -7.645737684257181,
            -7.680890501194263,
            -7.716043318131316,
            -7.75119613506834,
            -7.786348952005335,
            -7.821501768942302,
            -7.856654585879242,
            -7.891807402816152,
            -7.926960219753031,
            -7.962113036689885,
            -7.997265853626706,
            -8.0324186705635,
            -8.067571487500262,
            -8.102724304436997,
            -8.1378771213737,
            -8.173029938310373,
            -8.208182755247016,
            -8.24333557218363,
            -8.278488389120213,
            -8.313641206056763,
            -8.348794022993285,
            -8.383946839929775,
            -8.419099656866235,
            -8.454252473802663,
            -8.489405290739063,
            -8.524558107675428,
            -8.559710924611762,
            -8.594863741548064,
            -8.630016558484336,
            -8.665169375420575,
            -8.700322192356781,
            -8.735475009292957,
            -8.770627826229099,
            -8.805780643165209,
            -8.840933460101287,
            -8.87608627703733,
            -8.911239093973341,
            -8.946391910909322,
            -8.981544727845264,
            -9.016697544781179,
            -9.051850361717056,
            -9.087003178652902,
            -9.122155995588713,
            -9.15730881252449,
            -9.192461629460235,
            -9.227614446395943,
            -9.262767263331618,
            -9.29792008026726,
            -9.333072897202864,
            -9.368225714138436,
            -9.403378531073974,
            -9.438531348009473,
            -9.473684164944942,
            -9.508836981880373,
            -9.543989798815767,
            -9.579142615751127,
            -9.614295432686454,
            -9.649448249621742,
            -9.684601066556993,
            -9.71975388349221,
            -9.75490670042739,
            -9.790059517362534,
            -9.82521233429764,
            -9.860365151232713,
            -9.895517968167747,
            -9.930670785102743,
            -9.965823602037702,
            -10.000976418972625,
            -10.03612923590751,
            -10.071282052842356,
            -10.106434869777168,
            -10.14158768671194,
            -10.176740503646672,
            -10.211893320581368,
            -10.247046137516026,
            -10.282198954450648,
            -10.317351771385228,
            -10.35250458831977,
            -10.387657405254272,
            -10.422810222188739,
            -10.457963039123163,
            -10.493115856057548,
            -10.528268672991894,
            -10.563421489926201,
            -10.598574306860469,
            -10.633727123794698,
            -10.668879940728884,
            -10.704032757663033,
            -10.739185574597139,
            -10.774338391531206,
            -10.809491208465232,
            -10.844644025399216,
            -10.87979684233316,
            -10.914949659267066,
            -10.950102476200929,
            -10.985255293134749,
            -11.02040811006853,
            -11.055560927002269,
            -11.090713743935966,
            -11.12586656086962,
            -11.161019377803232,
            -11.196172194736803,
            -11.23132501167033,
            -11.266477828603817,
            -11.30163064553726,
            -11.33678346247066,
            -11.371936279404018,
            -11.407089096337334,
            -11.442241913270603,
            -11.477394730203832,
            -11.512547547137018,
            -11.547700364070158,
            -11.582853181003255,
            -11.618005997936308,
            -11.653158814869316,
            -11.688311631802282,
            -11.723464448735204,
            -11.758617265668079,
            -11.793770082600911,
            -11.828922899533698,
            -11.864075716466438,
            -11.899228533399137,
            -11.934381350331789,
            -11.969534167264394,
            -12.004686984196953,
            -12.03983980112947,
            -12.074992618061938,
            -12.11014543499436,
            -12.145298251926736,
            -12.180451068859067,
            -12.215603885791351,
            -12.250756702723589,
            -12.285909519655778,
            -12.321062336587921,
            -12.356215153520019,
            -12.391367970452068,
            -12.426520787384069,
            -12.461673604316024,
            -12.496826421247933,
            -12.531979238179792,
            -12.567132055111601,
            -12.602284872043365,
            -12.637437688975078,
            -12.672590505906745,
            -12.707743322838365,
            -12.742896139769933,
            -12.778048956701452,
            -12.813201773632924,
            -12.848354590564345,
            -12.883507407495717,
            -12.91866022442704,
            -12.953813041358314,
            -12.988965858289538,
            -13.024118675220711,
            -13.059271492151833,
            -13.094424309082909,
            -13.12957712601393,
            -13.164729942944902,
            -13.199882759875823,
            -13.235035576806695,
            -13.270188393737513,
            -13.305341210668281,
            -13.340494027599,
            -13.375646844529665,
            -13.41079966146028,
            -13.445952478390842,
            -13.48110529532135,
            -13.516258112251808,
            -13.551410929182213,
            -13.586563746112567,
            -13.621716563042867,
            -13.656869379973115,
            -13.692022196903311,
            -13.72717501383345,
            -13.76232783076354,
            -13.797480647693574,
            -13.832633464623557,
            -13.867786281553485,
            -13.90293909848336,
            -13.938091915413178,
            -13.973244732342945,
            -14.008397549272656,
            -14.043550366202313,
            -14.078703183131916,
            -14.113856000061462,
            -14.149008816990957,
            -14.184161633920393,
            -14.219314450849774,
            -14.254467267779102,
            -14.289620084708373,
            -14.324772901637589,
            -14.359925718566746,
            -14.395078535495852,
            -14.430231352424897,
            -14.465384169353886,
            -14.500536986282821,
            -14.535689803211701,
            -14.570842620140517,
            -14.605995437069284,
            -14.64114825399799,
            -14.676301070926637,
            -14.711453887855232,
            -14.746606704783762,
            -14.781759521712239,
            -14.816912338640655,
            -14.852065155569013,
            -14.887217972497318,
            -14.922370789425557,
            -14.957523606353742,
            -14.992676423281866,
            -15.027829240209932,
            -15.062982057137939,
            -15.098134874065886,
            -15.133287690993775,
            -15.168440507921602,
            -15.20359332484937,
            -15.238746141777076,
            -15.273898958704725,
            -15.309051775632312,
            -15.34420459255984,
            -15.379357409487309,
            -15.414510226414713,
            -15.449663043342058,
            -15.484815860269338,
            -15.51996867719656,
            -15.555121494123721,
            -15.590274311050823,
            -15.625427127977863,
            -15.660579944904834,
            -15.69573276183175,
            -15.730885578758597,
            -15.766038395685385,
            -15.801191212612112,
            -15.836344029538775,
            -15.871496846465373,
            -15.906649663391912,
            -15.941802480318382,
            -15.976955297244794,
            -16.012108114171134,
            -16.04726093109742,
            -16.082413748023637,
            -16.11756656494979,
            -16.15271938187588,
            -16.1878721988019,
            -16.223025015727863,
            -16.258177832653757,
            -16.293330649579588,
            -16.328483466505354,
            -16.363636283431052,
            -16.398789100356687,
            -16.433941917282255,
            -16.469094734207754,
            -16.50424755113319,
            -16.539400368058562,
            -16.574553184983866,
            -16.609706001909103,
            -16.64485881883427,
            -16.68001163575937,
            -16.71516445268441,
            -16.75031726960938,
            -16.78547008653428,
            -16.820622903459107,
            -16.855775720383875,
            -16.89092853730857,
            -16.9260813542332,
            -16.961234171157763,
            -16.996386988082254,
            -17.031539805006677,
            -17.06669262193103,
            -17.101845438855317,
            -17.13699825577953,
            -17.17215107270368,
            -17.20730388962776,
            -17.242456706551767,
            -17.277609523475704,
            -17.31276234039957,
            -17.347915157323367,
            -17.38306797424709,
            -17.41822079117075,
            -17.45337360809433,
            -17.48852642501785,
            -17.52367924194129,
            -17.55883205886466,
            -17.59398487578796,
            -17.629137692711186,
            -17.66429050963434,
            -17.699443326557425,
            -17.734596143480434,
            -17.769748960403373,
            -17.804901777326243,
            -17.840054594249036,
            -17.875207411171754,
            -17.9103602280944,
            -17.945513045016977,
            -17.980665861939475,
            -18.0158186788619,
            -18.050971495784253,
            -18.08612431270653,
            -18.121277129628737,
            -18.156429946550862,
            -18.191582763472923,
            -18.2267355803949,
            -18.261888397316806,
            -18.297041214238636,
            -18.332194031160387,
            -18.367346848082065,
            -18.402499665003667,
            -18.4376524819252,
            -18.472805298846644,
            -18.50795811576802,
            -18.543110932689316,
            -18.57826374961054,
            -18.61341656653168,
            -18.64856938345275,
            -18.683722200373737,
            -18.718875017294653,
            -18.75402783421548,
            -18.78918065113624,
            -18.824333468056917,
            -18.859486284977514,
            -18.89463910189804,
            -18.929791918818484,
            -18.964944735738847,
            -19.00009755265913,
            -19.03525036957934,
            -19.070403186499462,
            -19.10555600341951,
            -19.14070882033948,
            -19.175861637259366,
            -19.21101445417917,
            -19.246167271098898,
            -19.281320088018546,
            -19.31647290493811,
            -19.351625721857594,
            -19.386778538776998,
            -19.42193135569632,
            -19.45708417261556,
            -19.49223698953472,
            -19.527389806453797,
            -19.562542623372792,
            -19.597695440291705,
            -19.632848257210533,
            -19.668001074129283,
            -19.703153891047947,
            -19.738306707966526,
            -19.773459524885027,
            -19.80861234180344,
            -19.84376515872177,
            -19.878917975640018,
            -19.914070792558178,
            -19.94922360947626,
            -19.984376426394252,
            -20.019529243312164,
            -20.054682060229986,
            -20.089834877147727,
            -20.124987694065382,
            -20.16014051098295,
            -20.19529332790043,
            -20.23044614481783,
            -20.265598961735144,
            -20.300751778652366,
            -20.3359045955695,
            -20.371057412486554,
            -20.40621022940352,
            -20.441363046320397,
            -20.47651586323719,
            -20.511668680153893,
            -20.546821497070507,
            -20.581974313987033,
            -20.617127130903473,
            -20.65227994781982,
            -20.687432764736084,
            -20.722585581652258,
            -20.757738398568343,
            -20.79289121548434,
            -20.828044032400246,
            -20.86319684931606,
            -20.898349666231788,
            -20.93350248314743,
            -20.96865530006297,
            -21.00380811697843,
            -21.038960933893797,
            -21.07411375080907,
            -21.109266567724255,
            -21.14441938463935,
            -21.179572201554347,
            -21.21472501846926,
            -21.249877835384076,
            -21.285030652298797,
            -21.320183469213433,
            -21.355336286127972,
            -21.390489103042423,
            -21.425641919956778,
            -21.46079473687104,
            -21.495947553785207,
            -21.531100370699285,
            -21.566253187613263,
            -21.601406004527153,
            -21.636558821440946,
            -21.671711638354648,
            -21.70686445526825,
            -21.74201727218176,
            -21.777170089095172,
            -21.812322906008493,
            -21.84747572292172,
            -21.882628539834847,
            -21.917781356747874,
            -21.952934173660815,
            -21.988086990573652,
            -22.023239807486394,
            -22.05839262439904,
            -22.093545441311594,
            -22.12869825822404,
            -22.163851075136396,
            -22.199003892048655,
            -22.234156708960814,
            -22.269309525872877,
            -22.304462342784838,
            -22.339615159696702,
            -22.374767976608464,
            -22.409920793520133,
            -22.4450736104317,
            -22.480226427343165,
            -22.515379244254536,
            -22.5505320611658,
            -22.58568487807697,
            -22.62083769498804,
            -22.655990511899006,
            -22.691143328809872,
            -22.726296145720635,
            -22.761448962631302,
            -22.796601779541867,
            -22.831754596452324,
            -22.866907413362682,
            -22.90206023027294,
            -22.937213047183096,
            -22.972365864093145,
            -23.007518681003095,
            -23.04267149791294,
            -23.077824314822685,
            -23.11297713173232,
            -23.14812994864186,
            -23.18328276555129,
            -23.218435582460614,
            -23.253588399369836,
            -23.288741216278957,
            -23.32389403318797,
            -23.359046850096878,
            -23.39419966700568,
            -23.429352483914375,
            -23.464505300822967,
            -23.49965811773145,
            -23.53481093463983,
            -23.5699637515481,
            -23.605116568456264,
            -23.640269385364327,
            -23.675422202272276,
            -23.71057501918012,
            -23.745727836087855,
            -23.78088065299548,
            -23.816033469903,
            -23.851186286810414,
            -23.886339103717713,
            -23.92149192062491,
            -23.956644737531995,
            -23.991797554438968,
            -24.02695037134583,
            -24.062103188252593,
            -24.097256005159235,
            -24.13240882206577,
            -24.16756163897219,
            -24.202714455878503,
            -24.237867272784708,
            -24.2730200896908,
            -24.308172906596777,
            -24.343325723502645,
            -24.378478540408402,
            -24.413631357314046,
            -24.448784174219576,
            -24.483936991124992,
            -24.519089808030298,
            -24.55424262493549,
            -24.589395441840566,
            -24.62454825874553,
            -24.659701075650382,
            -24.694853892555113,
            -24.730006709459733,
            -24.76515952636424,
            -24.800312343268633,
            -24.83546516017291,
            -24.87061797707707,
            -24.905770793981116,
            -24.94092361088504,
            -24.976076427788854,
            -25.01122924469255,
            -25.046382061596127,
            -25.08153487849959,
            -25.11668769540293,
            -25.15184051230616,
            -25.18699332920927,
            -25.22214614611226,
            -25.25729896301513,
            -25.292451779917887,
            -25.32760459682052,
            -25.362757413723035,
            -25.39791023062543,
            -25.433063047527707,
            -25.468215864429865,
            -25.503368681331903,
            -25.538521498233816,
            -25.573674315135612,
            -25.608827132037288,
            -25.643979948938842,
            -25.679132765840272,
            -25.71428558274158,
            -25.74943839964277,
            -25.784591216543838,
            -25.819744033444778,
            -25.8548968503456,
            -25.8900496672463,
            -25.925202484146872,
            -25.960355301047322,
            -25.995508117947647,
            -26.03066093484785,
            -26.06581375174793,
            -26.100966568647884,
            -26.136119385547712,
            -26.171272202447415,
            -26.206425019346995,
            -26.241577836246446,
            -26.27673065314577,
            -26.311883470044968,
            -26.347036286944046,
            -26.38218910384299,
            -26.417341920741812,
            -26.452494737640507,
            -26.48764755453907,
            -26.522800371437505,
            -26.557953188335812,
            -26.59310600523399,
            -26.628258822132043,
            -26.663411639029963,
            -26.69856445592776,
            -26.73371727282542,
            -26.768870089722956,
            -26.804022906620357,
            -26.83917572351763,
            -26.874328540414773,
            -26.909481357311783,
            -26.944634174208666,
            -26.979786991105414,
            -27.014939808002026,
            -27.05009262489851,
            -27.08524544179486,
            -27.120398258691083,
            -27.15555107558717,
            -27.190703892483125,
            -27.22585670937894,
            -27.261009526274627,
            -27.29616234317018,
            -27.331315160065596,
            -27.36646797696088,
            -27.401620793856026,
            -27.43677361075104,
            -27.47192642764592,
            -27.50707924454066,
            -27.542232061435264,
            -27.57738487832973,
            -27.612537695224063,
            -27.64769051211826,
            -27.682843329012314,
            -27.717996145906234,
            -27.753148962800015,
            -27.78830177969366,
            -27.823454596587165,
            -27.85860741348053,
            -27.893760230373758,
            -27.92891304726685,
            -27.964065864159796,
            -27.999218681052607,
            -28.034371497945276,
            -28.0695243148378,
            -28.104677131730188,
            -28.139829948622435,
            -28.174982765514542,
            -28.210135582406505,
            -28.24528839929832,
            -28.280441216190003,
            -28.31559403308154,
            -28.350746849972932,
            -28.38589966686418,
            -28.421052483755282,
            -28.456205300646246,
            -28.491358117537064,
            -28.526510934427737,
            -28.561663751318267,
            -28.596816568208652,
            -28.631969385098884,
            -28.667122201988978,
            -28.70227501887892,
            -28.737427835768727,
            -28.77258065265838,
            -28.807733469547884,
            -28.842886286437242,
            -28.878039103326454,
            -28.913191920215514,
            -28.948344737104435,
            -28.983497553993203,
            -29.018650370881815,
            -29.053803187770285,
            -29.088956004658602,
            -29.124108821546777,
            -29.159261638434792,
            -29.19441445532266,
            -29.22956727221038,
            -29.264720089097946,
            -29.299872905985364,
            -29.335025722872626,
            -29.37017853975974,
            -29.4053313566467,
            -29.4404841735335,
            -29.475636990420156,
            -29.51078980730665,
            -29.545942624193,
            -29.581095441079192,
            -29.616248257965232,
            -29.651401074851115,
            -29.686553891736843,
            -29.721706708622417,
            -29.75685952550783,
            -29.792012342393093,
            -29.8271651592782,
            -29.862317976163148,
            -29.89747079304794,
            -29.932623609932573,
            -29.96777642681705,
            -30.002929243701374,
            -30.038082060585527,
            -30.07323487746953,
            -30.108387694353375,
            -30.14354051123706,
            -30.178693328120577,
            -30.213846145003945,
            -30.248998961887146,
            -30.284151778770187,
            -30.319304595653072,
            -30.35445741253579,
            -30.38961022941834,
            -30.42476304630074,
            -30.459915863182975,
            -30.495068680065046,
            -30.530221496946947,
            -30.5653743138287,
            -30.60052713071028,
            -30.635679947591697,
            -30.670832764472948,
            -30.70598558135403,
            -30.741138398234956,
            -30.776291215115712,
            -30.8114440319963,
            -30.84659684887672,
            -30.881749665756985,
            -30.916902482637067,
            -30.952055299517,
            -30.987208116396754,
            -31.022360933276335,
            -31.057513750155756,
            -31.092666567035003,
            -31.127819383914087,
            -31.162972200793,
            -31.198125017671735,
            -31.233277834550314,
            -31.268430651428705,
            -31.303583468306947,
            -31.338736285185,
            -31.373889102062883,
            -31.409041918940602,
            -31.44419473581814,
            -31.479347552695515,
            -31.514500369572712,
            -31.54965318644973,
            -31.584806003326584,
            -31.61995882020326,
            -31.655111637079763,
            -31.690264453956086,
            -31.725417270832235,
            -31.76057008770821,
            -31.795722904584004,
            -31.83087572145963,
            -31.866028538335073,
            -31.901181355210348,
            -31.93633417208543,
            -31.97148698896035,
            -32.00663980583508,
            -32.041792622709636,
            -32.076945439584016,
            -32.112098256458204,
            -32.14725107333222,
            -32.18240389020606,
            -32.21755670707972,
            -32.25270952395319,
            -32.28786234082648,
            -32.3230151576996,
            -32.358167974572524,
            -32.39332079144527,
            -32.42847360831784,
            -32.46362642519021,
            -32.49877924206241,
            -32.53393205893442,
            -32.56908487580625,
            -32.6042376926779,
            -32.63939050954935,
            -32.674543326420626,
            -32.709696143291715,
            -32.74484896016261,
            -32.78000177703332,
            -32.81515459390384,
            -32.85030741077418,
            -32.88546022764433,
            -32.92061304451429,
            -32.95576586138406,
            -32.99091867825364,
            -33.026071495123034,
            -33.061224311992234,
            -33.09637712886125,
            -33.131529945730065,
            -33.1666827625987,
            -33.20183557946714,
            -33.23698839633538,
            -33.272141213203426,
            -33.3072940300713,
            -33.34244684693896,
            -33.37759966380643,
            -33.4127524806737,
            -33.44790529754079,
            -33.483058114407676,
            -33.518210931274375,
            -33.55336374814086,
            -33.58851656500717,
            -33.62366938187327,
            -33.65882219873917,
            -33.693975015604885,
            -33.7291278324704,
            -33.76428064933571,
            -33.79943346620082,
            -33.83458628306573,
            -33.86973909993044,
            -33.90489191679496,
            -33.94004473365928,
            -33.97519755052339,
            -34.010350367387296,
            -34.045503184251004,
            -34.08065600111451,
            -34.115808817977815,
            -34.15096163484091,
            -34.18611445170381,
            -34.2212672685665,
            -34.25642008542899,
            -34.29157290229127,
            -34.326725719153345,
            -34.361878536015226,
            -34.39703135287689,
            -34.43218416973834,
            -34.46733698659959,
            -34.502489803460634,
            -34.53764262032147,
            -34.572795437182094,
            -34.60794825404251,
            -34.643101070902716,
            -34.67825388776272,
            -34.71340670462251,
            -34.74855952148208,
            -34.783712338341445,
            -34.818865155200605,
            -34.85401797205955,
            -34.88917078891827,
            -34.92432360577679,
            -34.95947642263509,
            -34.99462923949318,
            -35.02978205635106,
            -35.06493487320871,
            -35.10008769006616,
            -35.13524050692339,
            -35.170393323780395,
            -35.20554614063719,
            -35.24069895749376,
            -35.27585177435012,
            -35.31100459120626,
            -35.34615740806218,
            -35.38131022491789,
            -35.41646304177337,
            -35.45161585862863,
            -35.48676867548368,
            -35.52192149233849,
            -35.5570743091931,
            -35.59222712604747,
            -35.62737994290163,
            -35.662532759755564,
            -35.69768557660927,
            -35.73283839346276,
            -35.767991210316005,
            -35.80314402716905,
            -35.838296844021855,
            -35.87344966087443,
            -35.908602477726795,
            -35.94375529457892,
            -35.978908111430826,
            -36.0140609282825,
            -36.04921374513395,
            -36.084366561985156,
            -36.119519378836145,
            -36.1546721956869,
            -36.18982501253743,
            -36.224977829387726,
            -36.26013064623778,
            -36.29528346308762,
            -36.33043627993722,
            -36.365589096786586,
            -36.40074191363572,
            -36.43589473048463,
            -36.471047547333285,
            -36.506200364181716,
            -36.54135318102991,
            -36.57650599787787,
            -36.611658814725594,
            -36.64681163157307,
            -36.68196444842032,
            -36.717117265267326,
            -36.7522700821141,
            -36.78742289896063,
            -36.82257571580691,
            -36.85772853265296,
            -36.89288134949877,
            -36.92803416634435,
            -36.96318698318967,
            -36.99833980003475,
            -37.03349261687959,
            -37.06864543372419,
            -37.10379825056855,
            -37.138951067412656,
            -37.17410388425652,
            -37.209256701100145,
            -37.24440951794352,
            -37.27956233478663,
            -37.314715151629514,
            -37.34986796847214,
            -37.38502078531453,
            -37.42017360215666,
            -37.455326418998546,
            -37.490479235840176,
            -37.525632052681566,
            -37.56078486952269,
            -37.59593768636357,
            -37.631090503204206,
            -37.66624332004458,
            -37.7013961368847,
            -37.73654895372457,
            -37.77170177056418,
            -37.80685458740354,
            -37.842007404242636,
            -37.87716022108148,
            -37.91231303792008,
            -37.94746585475841,
            -37.982618671596484,
            -38.017771488434306,
            -38.05292430527186,
            -38.08807712210916,
            -38.123229938946196,
            -38.158382755782974,
            -38.19353557261949,
            -38.22868838945575,
            -38.26384120629173,
            -38.29899402312746,
            -38.33414683996292,
            -38.36929965679812,
            -38.40445247363306,
            -38.43960529046773,
            -38.47475810730213,
            -38.509910924136264,
            -38.545063740970136,
            -38.58021655780374,
            -38.615369374637076,
            -38.65052219147014,
            -38.68567500830294,
            -38.72082782513546,
            -38.75598064196771,
            -38.791133458799685,
            -38.82628627563141,
            -38.861439092462845,
            -38.89659190929401,
            -38.93174472612489,
            -38.966897542955515,
            -39.002050359785855,
            -39.03720317661592,
            -39.072355993445704,
            -39.10750881027522,
            -39.142661627104445,
            -39.17781444393341,
            -39.21296726076208,
            -39.24812007759048,
            -39.2832728944186,
            -39.31842571124643,
            -39.35357852807398,
            -39.38873134490127,
            -39.42388416172825,
            -39.459036978554956,
            -39.494189795381374,
            -39.52934261220752,
            -39.56449542903337,
            -39.59964824585894,
            -39.63480106268422,
            -39.669953879509215,
            -39.705106696333914,
            -39.74025951315834,
            -39.77541232998246,
            -39.8105651468063,
            -39.84571796362985,
            -39.8808707804531,
            -39.91602359727607,
            -39.95117641409874,
            -39.98632923092113,
            -40.021482047743206,
            -40.056634864565,
            -40.091787681386485,
            -40.126940498207695,
            -40.1620933150286,
            -40.1972461318492,
            -40.23239894866951,
            -40.26755176548952,
            -40.30270458230923,
            -40.337857399128644,
            -40.37301021594776,
            -40.40816303276656,
            -40.44331584958507,
            -40.47846866640327,
            -40.51362148322117,
            -40.54877430003877,
            -40.58392711685607,
            -40.61907993367305,
            -40.65423275048973,
            -40.68938556730611,
            -40.72453838412217,
            -40.75969120093792,
            -40.79484401775338,
            -40.82999683456852,
            -40.86514965138334,
            -40.900302468197864,
            -40.935455285012075,
            -40.97060810182596,
            -41.005760918639545,
            -41.04091373545281,
            -41.076066552265765,
            -41.111219369078405,
            -41.14637218589072,
            -41.18152500270273,
            -41.21667781951441,
            -41.251830636325785,
            -41.286983453136834,
            -41.32213626994756,
            -41.35728908675797,
            -41.39244190356805,
            -41.42759472037782,
            -41.462747537187255,
            -41.49790035399638,
            -41.53305317080517,
            -41.568205987613645,
            -41.60335880442178,
            -41.6385116212296,
            -41.67366443803709,
            -41.70881725484425,
            -41.743970071651084,
            -41.779122888457586,
            -41.81427570526376,
            -41.8494285220696,
            -41.88458133887511,
            -41.91973415568028,
            -41.954886972485134,
            -41.99003978928963,
            -42.02519260609381,
            -42.060345422897655,
            -42.09549823970115,
            -42.13065105650431,
            -42.16580387330714,
            -42.20095669010963,
            -42.236109506911774,
            -42.27126232371358,
            -42.30641514051504,
            -42.341567957316165,
            -42.37672077411695,
            -42.41187359091739,
            -42.447026407717466,
            -42.48217922451722,
            -42.51733204131662,
            -42.55248485811567,
            -42.587637674914376,
            -42.622790491712735,
            -42.65794330851074,
            -42.693096125308394,
            -42.7282489421057,
            -42.76340175890265,
            -42.79855457569926,
            -42.8337073924955,
            -42.86886020929139,
            -42.90401302608693,
            -42.93916584288211,
            -42.97431865967693,
            -43.009471476471404,
            -43.0446242932655,
            -43.079777110059254,
            -43.11492992685264,
            -43.15008274364566,
            -43.18523556043832,
            -43.22038837723062,
            -43.255541194022555,
            -43.29069401081413,
            -43.32584682760533,
            -43.36099964439617,
            -43.396152461186645,
            -43.43130527797675,
            -43.46645809476648,
            -43.50161091155585,
            -43.536763728344845,
            -43.57191654513346,
            -43.60706936192171,
            -43.64222217870959,
            -43.67737499549709,
            -43.712527812284215,
            -43.747680629070956,
            -43.782833445857335,
            -43.81798626264333,
            -43.85313907942894,
            -43.88829189621418,
            -43.923444712999036,
            -43.958597529783496,
            -43.993750346567595,
            -44.0289031633513,
            -44.06405598013462,
            -44.09920879691755,
            -44.13436161370011,
            -44.16951443048226,
            -44.20466724726404,
            -44.23982006404542,
            -44.27497288082641,
            -44.31012569760702,
            -44.34527851438723,
            -44.38043133116706,
            -44.41558414794647,
            -44.45073696472551,
            -44.48588978150414,
            -44.52104259828237,
            -44.556195415060216,
            -44.59134823183765,
            -44.62650104861469,
            -44.66165386539134,
            -44.69680668216757,
            -44.731959498943404,
            -44.76711231571883,
            -44.802265132493865,
            -44.837417949268485,
            -44.87257076604269,
            -44.9077235828165,
            -44.9428763995899,
            -44.97802921636289,
            -45.01318203313547,
            -45.04833484990764,
            -45.08348766667938,
            -45.118640483450726,
            -45.15379330022165,
            -45.18894611699216,
            -45.22409893376226,
            -45.25925175053193,
            -45.294404567301186,
            -45.32955738407002,
            -45.36471020083844,
            -45.399863017606435,
            -45.435015834374,
            -45.47016865114115,
            -45.50532146790787,
            -45.54047428467417,
            -45.575627101440034,
            -45.61077991820547,
            -45.64593273497048,
            -45.68108555173506,
            -45.716238368499205,
            -45.75139118526293,
            -45.78654400202622,
            -45.82169681878906,
            -45.85684963555147,
            -45.89200245231345,
            -45.92715526907498,
            -45.96230808583608,
            -45.99746090259675,
            -46.03261371935696,
            -46.06776653611673,
            -46.10291935287606,
            -46.13807216963495,
            -46.17322498639339,
            -46.208377803151386,
            -46.24353061990893,
            -46.278683436666036,
            -46.31383625342268,
            -46.348989070178874,
            -46.384141886934614,
            -46.4192947036899,
            -46.45444752044474,
            -46.489600337199114,
            -46.52475315395304,
            -46.55990597070651,
            -46.595058787459514,
            -46.63021160421206,
            -46.66536442096413,
            -46.70051723771576,
            -46.73567005446692,
            -46.770822871217604,
            -46.80597568796783,
            -46.84112850471759,
            -46.87628132146687,
            -46.9114341382157,
            -46.94658695496403,
            -46.98173977171191,
            -47.016892588459314,
            -47.05204540520624,
            -47.0871982219527,
            -47.12235103869867,
            -47.15750385544417,
            -47.19265667218917,
            -47.22780948893371,
            -47.26296230567777,
            -47.298115122421336,
            -47.33326793916443,
            -47.36842075590702,
            -47.40357357264914,
            -47.43872638939076,
            -47.4738792061319,
            -47.509032022872546,
            -47.5441848396127,
            -47.57933765635235,
            -47.614490473091514,
            -47.64964328983018,
            -47.68479610656835,
            -47.71994892330603,
            -47.7551017400432,
            -47.790254556779864,
            -47.825407373516036,
            -47.86056019025171,
            -47.89571300698687,
            -47.930865823721525,
            -47.966018640455665,
            -48.00117145718931,
            -48.036324273922446,
            -48.07147709065506,
            -48.10662990738716,
            -48.141782724118755,
            -48.176935540849826,
            -48.21208835758039,
            -48.247241174310425,
            -48.282393991039946,
            -48.31754680776895,
            -48.35269962449742,
            -48.38785244122537,
            -48.4230052579528,
            -48.45815807467971,
            -48.49331089140608,
            -48.52846370813193,
            -48.563616524857245,
            -48.59876934158202,
            -48.63392215830627,
            -48.66907497502999,
            -48.70422779175317,
            -48.73938060847581,
            -48.774533425197916,
            -48.809686241919465,
            -48.8448390586405,
            -48.87999187536097,
            -48.9151446920809,
            -48.95029750880029,
            -48.985450325519125,
            -49.02060314223742,
            -49.05575595895515,
            -49.09090877567235,
            -49.12606159238898,
            -49.16121440910506,
            -49.19636722582058,
            -49.231520042535536,
            -49.26667285924995,
            -49.3018256759638,
            -49.336978492677076,
            -49.37213130938979,
            -49.40728412610195,
            -49.442436942813536,
            -49.477589759524555,
            -49.512742576235006,
            -49.54789539294488,
            -49.58304820965418,
            -49.61820102636292,
            -49.65335384307106,
            -49.68850665977864,
            -49.723659476485636,
            -49.75881229319205,
            -49.79396510989788,
            -49.82911792660313,
            -49.8642707433078,
            -49.89942356001188,
            -49.93457637671536,
            -49.96972919341826,
            -50.004882010120575,
            -50.040034826822286,
            -50.0751876435234,
            -50.110340460223924,
            -50.145493276923844,
            -50.180646093623174,
            -50.215798910321894,
            -50.250951727020016,
            -50.28610454371753,
            -50.32125736041443,
            -50.356410177110725,
            -50.391562993806424,
            -50.426715810501506,
            -50.46186862719597,
            -50.497021443889814,
            -50.53217426058305,
            -50.567327077275664,
            -50.60247989396767,
            -50.63763271065904,
            -50.672785527349795,
            -50.70793834403992,
            -50.74309116072942,
            -50.77824397741829,
            -50.81339679410654,
            -50.848549610794144,
            -50.88370242748112,
            -50.91885524416746,
            -50.95400806085317,
            -50.98916087753823,
            -51.02431369422265,
            -51.059466510906425,
            -51.09461932758957,
            -51.12977214427206,
            -51.16492496095391,
            -51.2000777776351,
            -51.235230594315645,
            -51.270383410995535,
            -51.30553622767477,
            -51.34068904435335,
            -51.37584186103126,
            -51.41099467770852,
            -51.44614749438512,
            -51.48130031106105,
            -51.51645312773632,
            -51.55160594441091,
            -51.58675876108484,
            -51.6219115777581,
            -51.65706439443068,
            -51.69221721110258,
            -51.72737002777381,
            -51.76252284444436,
            -51.797675661114226,
            -51.832828477783416,
            -51.867981294451916,
            -51.903134111119726,
            -51.938286927786855,
            -51.97343974445329,
            -52.00859256111902,
            -52.04374537778406,
            -52.07889819444841,
            -52.11405101111207,
            -52.14920382777502,
            -52.184356644437266,
            -52.2195094610988,
            -52.254662277759635,
            -52.28981509441977,
            -52.324967911079185,
            -52.36012072773788,
            -52.39527354439586,
            -52.43042636105314,
            -52.465579177709685,
            -52.50073199436551,
            -52.53588481102062,
            -52.571037627675,
            -52.60619044432865,
            -52.64134326098158,
            -52.676496077633765,
            -52.71164889428523,
            -52.746801710935955,
            -52.78195452758594,
            -52.81710734423518,
            -52.852260160883674,
            -52.88741297753144,
            -52.92256579417845,
            -52.95771861082471,
            -52.99287142747023,
            -53.02802424411499,
            -53.063177060758996,
            -53.09832987740223,
            -53.13348269404472,
            -53.16863551068646,
            -53.20378832732741,
            -53.23894114396761,
            -53.27409396060703,
            -53.30924677724568,
            -53.344399593883566,
            -53.37955241052068,
            -53.41470522715701,
            -53.449858043792545,
            -53.48501086042732,
            -53.520163677061305,
            -53.5553164936945,
            -53.5904693103269,
            -53.62562212695852,
            -53.660774943589345,
            -53.69592776021936,
            -53.73108057684859,
            -53.76623339347702,
            -53.80138621010464,
            -53.836539026731465,
            -53.87169184335747,
            -53.90684465998267,
            -53.94199747660706,
            -53.97715029323062,
            -54.01230310985337,
            -54.047455926475315,
            -54.08260874309642,
            -54.117761559716705,
            -54.15291437633617,
            -54.18806719295479,
            -54.22322000957259,
            -54.25837282618955,
            -54.293525642805676,
            -54.328678459420956,
            -54.3638312760354,
            -54.39898409264899,
            -54.43413690926174,
            -54.469289725873644,
            -54.50444254248469,
            -54.53959535909487,
            -54.57474817570421,
            -54.60990099231268,
            -54.64505380892029,
            -54.680206625527035,
            -54.71535944213291,
            -54.75051225873791,
            -54.78566507534204,
            -54.82081789194529,
            -54.855970708547666,
            -54.89112352514916,
            -54.92627634174978,
            -54.96142915834949,
            -54.99658197494833,
            -55.03173479154627,
            -55.066887608143304,
            -55.102040424739464,
            -55.1371932413347,
            -55.17234605792905,
            -55.207498874522486,
            -55.242651691115015,
            -55.27780450770662,
            -55.31295732429732,
            -55.3481101408871,
            -55.38326295747598,
            -55.41841577406392,
            -55.45356859065093,
            -55.488721407237016,
            -55.52387422382217,
            -55.559027040406384,
            -55.59417985698967,
            -55.629332673572016,
            -55.66448549015341,
            -55.69963830673386,
            -55.734791123313364,
            -55.76994393989192,
            -55.8050967564695,
            -55.840249573046144,
            -55.875402389621826,
            -55.910555206196534,
            -55.94570802277027,
            -55.980860839343045,
            -56.01601365591485,
            -56.051166472485676,
            -56.086319289055524,
            -56.12147210562437,
            -56.15662492219225,
            -56.191777738759136,
            -56.22693055532504,
            -56.262083371889936,
            -56.29723618845384,
            -56.332389005016736,
            -56.367541821578634,
            -56.40269463813953,
            -56.4378474546994,
            -56.473000271258265,
            -56.50815308781611,
            -56.543305904372936,
            -56.578458720928744,
            -56.613611537483514,
            -56.64876435403726,
            -56.68391717058997,
            -56.719069987141644,
            -56.75422280369227,
            -56.789375620241856,
            -56.82452843679041,
            -56.85968125333789,
            -56.89483406988434,
            -56.92998688642972,
            -56.965139702974035,
            -57.00029251951729,
            -57.03544533605948,
            -57.070598152600596,
            -57.10575096914063,
            -57.1409037856796,
            -57.17605660221748,
            -57.211209418754294,
            -57.24636223529,
            -57.28151505182461,
            -57.31666786835813,
            -57.35182068489056,
            -57.38697350142187,
            -57.422126317952085,
            -57.45727913448118,
            -57.49243195100917,
            -57.52758476753605,
            -57.56273758406181,
            -57.59789040058643,
            -57.63304321710993,
            -57.6681960336323,
            -57.70334885015353,
            -57.738501666673606,
            -57.773654483192566,
            -57.80880729971035,
            -57.84396011622701,
            -57.87911293274251,
            -57.914265749256835,
            -57.94941856577002,
            -57.98457138228201,
            -58.01972419879286,
            -58.054877015302516,
            -58.090029831811,
            -58.125182648318315,
            -58.16033546482443,
            -58.195488281329354,
            -58.23064109783308,
            -58.26579391433561,
            -58.30094673083695,
            -58.33609954733708,
            -58.371252363835985,
            -58.4064051803337,
            -58.44155799683018,
            -58.47671081332545,
            -58.51186362981948,
            -58.54701644631229,
            -58.58216926280386,
            -58.61732207929418,
            -58.652474895783264,
            -58.687627712271116,
            -58.722780528757696,
            -58.757933345243046,
            -58.7930861617271,
            -58.82823897820992,
            -58.863391794691445,
            -58.89854461117173,
            -58.93369742765071,
            -58.968850244128426,
            -59.004003060604845,
            -59.03915587707997,
            -59.0743086935538,
            -59.109461510026335,
            -59.144614326497575,
            -59.1797671429675,
            -59.2149199594361,
            -59.2500727759034,
            -59.285225592369365,
            -59.32037840883401,
            -59.35553122529733,
            -59.390684041759286,
            -59.425836858219924,
            -59.46098967467922,
            -59.49614249113716,
            -59.53129530759374,
            -59.56644812404897,
            -59.60160094050283,
            -59.636753756955315,
            -59.671906573406446,
            -59.707059389856184,
            -59.742212206304536,
            -59.777365022751496,
            -59.81251783919707,
            -59.84767065564125,
            -59.88282347208403,
            -59.917976288525395,
            -59.95312910496534,
            -59.988281921403875,
            -60.02343473784099,
            -60.05858755427666,
            -60.09374037071092,
            -60.12889318714374,
            -60.1640460035751,
            -60.19919882000503,
            -60.234351636433495,
            -60.269504452860495,
            -60.30465726928603,
            -60.339810085710106,
            -60.37496290213271,
            -60.41011571855382,
            -60.44526853497346,
            -60.4804213513916,
            -60.515574167808246,
            -60.55072698422339,
            -60.58587980063704,
            -60.62103261704917,
            -60.65618543345977,
            -60.691338249868856,
            -60.7264910662764,
            -60.76164388268243,
            -60.796796699086926,
            -60.83194951548984,
            -60.86710233189123,
            -60.902255148291076,
            -60.937407964689335,
            -60.97256078108603,
            -61.00771359748115,
            -61.042866413874705,
            -61.07801923026666,
            -61.113172046657034,
            -61.14832486304581,
            -61.18347767943296,
            -61.21863049581852,
            -61.25378331220247,
            -61.288936128584794,
            -61.32408894496549,
            -61.35924176134456,
            -61.39439457772199,
            -61.42954739409777,
            -61.4647002104719,
            -61.49985302684436,
            -61.53500584321517,
            -61.57015865958431,
            -61.60531147595177,
            -61.64046429231755,
            -61.67561710868164,
            -61.71076992504404,
            -61.74592274140473,
            -61.78107555776372,
            -61.816228374120996,
            -61.851381190476545,
            -61.88653400683038,
            -61.92168682318246,
            -61.956839639532824,
            -61.99199245588144,
            -62.027145272228275,
            -62.06229808857337,
            -62.09745090491671,
            -62.132603721258256,
            -62.16775653759803,
            -62.202909353936015,
            -62.23806217027221,
            -62.273214986606604,
            -62.308367802939195,
            -62.34352061926995,
            -62.378673435598905,
            -62.413826251926025,
            -62.44897906825131,
            -62.484131884574744,
            -62.51928470089635,
            -62.55443751721608,
            -62.58959033353395,
            -62.62474314984995,
            -62.65989596616408,
            -62.69504878247631,
            -62.73020159878666,
            -62.76535441509511,
            -62.800507231401646,
            -62.83566004770627,
            -62.87081286400896,
            -62.905965680309734,
            -62.94111849660856,
            -62.97627131290544,
            -63.011424129200385,
            -63.04657694549334,
            -63.081729761784345,
            -63.11688257807337,
            -63.15203539436042,
            -63.187188210645466,
            -63.222341026928525,
            -63.257493843209566,
            -63.2926466594886,
            -63.3277994757656,
            -63.362952292040575,
            -63.39810510831351,
            -63.433257924584396,
            -63.46841074085321,
            -63.503563557119975,
            -63.53871637338468,
            -63.57386918964728,
            -63.6090220059078,
            -63.644174822166235,
            -63.679327638422556,
            -63.71448045467677,
            -63.74963327092884,
            -63.784786087178794,
            -63.81993890342659,
            -63.85509171967227,
            -63.89024453591576,
            -63.92539735215709,
            -63.96055016839627,
            -63.99570298463323,
            -64.03085580086803,
            -64.06600861710061,
            -64.10116143333097,
            -64.13631424955913,
            -64.17146706578505,
            -64.20661988200874,
            -64.24177269823016,
            -64.27692551444935,
            -64.31207833066627,
            -64.34723114688092,
            -64.38238396309325,
            -64.41753677930333,
            -64.45268959551109,
            -64.48784241171651,
            -64.52299522791964,
            -64.55814804412043,
            -64.59330086031886,
            -64.62845367651497,
            -64.6636064927087,
            -64.69875930890007,
            -64.73391212508905,
            -64.76906494127562,
            -64.80421775745981,
            -64.83937057364157,
            -64.87452338982094,
            -64.90967620599784,
            -64.94482902217233,
            -64.97998183834434,
            -65.0151346545139,
            -65.05028747068098,
            -65.0854402868456,
            -65.1205931030077,
            -65.1557459191673,
            -65.1908987353244,
            -65.22605155147896,
            -65.26120436763098,
            -65.29635718378044,
            -65.33150999992735,
            -65.36666281607171,
            -65.40181563221347,
            -65.43696844835264,
            -65.4721212644892,
            -65.50727408062315,
            -65.54242689675448,
            -65.57757971288316,
            -65.61273252900921,
            -65.64788534513258,
            -65.68303816125328,
            -65.71819097737131,
            -65.75334379348664,
            -65.78849660959925,
            -65.82364942570915,
            -65.85880224181632,
            -65.89395505792075,
            -65.92910787402242,
            -65.96426069012132,
            -65.99941350621744,
            -66.03456632231078,
            -66.0697191384013,
            -66.10487195448903,
            -66.14002477057389,
            -66.17517758665592,
            -66.21033040273511,
            -66.24548321881144,
            -66.28063603488488,
            -66.31578885095543,
            -66.35094166702306,
            -66.38609448308779,
            -66.42124729914957,
            -66.45640011520842,
            -66.49155293126431,
            -66.52670574731725,
            -66.56185856336717,
            -66.5970113794141,
            -66.63216419545803,
            -66.66731701149894,
            -66.7024698275368,
            -66.7376226435716,
            -66.77277545960334,
            -66.807928275632,
            -66.84308109165757,
            -66.87823390768001,
            -66.91338672369935,
            -66.94853953971553,
            -66.98369235572856,
            -67.01884517173846,
            -67.05399798774513,
            -67.08915080374865,
            -67.12430361974893,
            -67.159456435746,
            -67.19460925173982,
            -67.22976206773039,
            -67.26491488371768,
            -67.30006769970169,
            -67.33522051568241,
            -67.37037333165979,
            -67.40552614763385,
            -67.44067896360457,
            -67.4758317795719,
            -67.51098459553589,
            -67.54613741149646,
            -67.58129022745362,
            -67.61644304340737,
            -67.65159585935764,
            -67.6867486753045,
            -67.72190149124783,
            -67.7570543071877,
            -67.79220712312406,
            -67.8273599390569,
            -67.86251275498618,
            -67.89766557091191,
            -67.93281838683407,
            -67.96797120275262,
            -68.00312401866758,
            -68.0382768345789,
            -68.07342965048656,
            -68.10858246639056,
            -68.14373528229089,
            -68.17888809818751,
            -68.21404091408043,
            -68.24919372996959,
            -68.28434654585503,
            -68.31949936173666,
            -68.35465217761453,
            -68.3898049934886,
            -68.4249578093588,
            -68.46011062522518,
            -68.49526344108769,
            -68.53041625694631,
            -68.56556907280105,
            -68.60072188865185,
            -68.63587470449872,
            -68.6710275203416,
            -68.70618033618052,
            -68.74133315201544,
            -68.77648596784634,
            -68.81163878367319,
            -68.846791599496,
            -68.8819444153147,
            -68.91709723112932,
            -68.95225004693981,
            -68.98740286274617,
            -69.02255567854833,
            -69.05770849434634,
            -69.09286131014014,
            -69.1280141259297,
            -69.16316694171502,
            -69.19831975749605,
            -69.23347257327282,
            -69.26862538904524,
            -69.30377820481336,
            -69.3389310205771,
            -69.37408383633647,
            -69.40923665209144,
            -69.44438946784197,
            -69.47954228358806,
            -69.51469509932967,
            -69.54984791506679,
            -69.58500073079942,
            -69.62015354652748,
            -69.65530636225097,
            -69.69045917796988,
            -69.7256119936842,
            -69.76076480939385,
            -69.79591762509887,
            -69.83107044079917,
            -69.86622325649478,
            -69.90137607218566,
            -69.93652888787177,
            -69.97168170355312,
            -70.00683451922961,
            -70.04198733490132,
            -70.07714015056814,
            -70.11229296623009,
            -70.14744578188711,
            -70.1825985975392,
            -70.21775141318632,
            -70.25290422882847,
            -70.28805704446557,
            -70.32320986009763,
            -70.35836267572463,
            -70.39351549134655,
            -70.42866830696332,
            -70.46382112257493,
            -70.49897393818136,
            -70.53412675378257,
            -70.56927956937857,
            -70.60443238496927,
            -70.6395852005547,
            -70.6747380161348,
            -70.70989083170953,
            -70.74504364727889,
            -70.78019646284282,
            -70.81534927840131,
            -70.85050209395436,
            -70.88565490950188,
            -70.92080772504387,
            -70.95596054058028,
            -70.99111335611113,
            -71.02626617163634,
            -71.06141898715586,
            -71.09657180266971,
            -71.13172461817786,
            -71.16687743368026,
            -71.20203024917683,
            -71.2371830646676,
            -71.27233588015254,
            -71.30748869563158,
            -71.34264151110469,
            -71.37779432657186,
            -71.41294714203303,
            -71.44809995748821,
            -71.48325277293733,
            -71.51840558838035,
            -71.55355840381726,
            -71.58871121924801,
            -71.62386403467256,
            -71.6590168500909,
            -71.69416966550294,
            -71.72932248090869,
            -71.76447529630813,
            -71.79962811170117,
            -71.8347809270878,
            -71.86993374246799,
            -71.9050865578417,
            -71.94023937320887,
            -71.97539218856949,
            -72.0105450039235,
            -72.0456978192709,
            -72.08085063461158,
            -72.11600344994555,
            -72.15115626527279,
            -72.18630908059319,
            -72.22146189590678,
            -72.25661471121349,
            -72.29176752651328,
            -72.3269203418061,
            -72.36207315709191,
            -72.39722597237069,
            -72.43237878764238,
            -72.46753160290694,
            -72.50268441816432,
            -72.5378372334145,
            -72.57299004865742,
            -72.60814286389304,
            -72.64329567912128,
            -72.67844849434215,
            -72.71360130955557,
            -72.7487541247615,
            -72.7839069399599,
            -72.81905975515078,
            -72.85421257033397,
            -72.88936538550952,
            -72.92451820067735,
            -72.95967101583742,
            -72.99482383098967,
            -73.02997664613406,
            -73.06512946127053,
            -73.10028227639904,
            -73.13543509151957,
            -73.17058790663201,
            -73.20574072173635,
            -73.24089353683253,
            -73.2760463519205,
            -73.3111991670002,
            -73.34635198207157,
            -73.38150479713458,
            -73.41665761218917,
            -73.45181042723527,
            -73.48696324227285,
            -73.52211605730184,
            -73.5572688723222,
            -73.59242168733385,
            -73.62757450233674,
            -73.66272731733085,
            -73.69788013231607,
            -73.73303294729237,
            -73.76818576225969,
            -73.80333857721797,
            -73.83849139216716,
            -73.87364420710718,
            -73.90879702203799,
            -73.9439498369595,
            -73.97910265187167,
            -74.01425546677446,
            -74.04940828166777,
            -74.08456109655155,
            -74.11971391142576,
            -74.15486672629031,
            -74.19001954114512,
            -74.22517235599017,
            -74.26032517082535,
            -74.29547798565063,
            -74.33063080046593,
            -74.36578361527117,
            -74.4009364300663,
            -74.43608924485122,
            -74.4712420596259,
            -74.50639487439025,
            -74.54154768914422,
            -74.57670050388771,
            -74.61185331862066,
            -74.647006133343,
            -74.68215894805466,
            -74.71731176275556,
            -74.75246457744564,
            -74.78761739212479,
            -74.82277020679298,
            -74.85792302145009,
            -74.89307583609606,
            -74.92822865073084,
            -74.96338146535429,
            -74.99853427996639,
            -75.03368709456701,
            -75.06883990915613,
            -75.10399272373363,
            -75.13914553829942,
            -75.17429835285344,
            -75.20945116739557,
            -75.24460398192576,
            -75.27975679644392,
            -75.31490961094997,
            -75.35006242544378,
            -75.38521523992533,
            -75.42036805439444,
            -75.4555208688511,
            -75.4906736832952,
            -75.52582649772664,
            -75.56097931214532,
            -75.59613212655114,
            -75.63128494094406,
            -75.66643775532391,
            -75.70159056969062,
            -75.73674338404413,
            -75.7718961983843,
            -75.80704901271106,
            -75.84220182702428,
            -75.87735464132389,
            -75.91250745560974,
            -75.9476602698818,
            -75.98281308413988,
            -76.01796589838393,
            -76.05311871261388,
            -76.08827152682954,
            -76.12342434103081,
            -76.15857715521766,
            -76.1937299693899,
            -76.22888278354743,
            -76.26403559769017,
            -76.299188411818,
            -76.33434122593079,
            -76.36949404002841,
            -76.40464685411077,
            -76.43979966817773,
            -76.4749524822292,
            -76.51010529626504,
            -76.54525811028509,
            -76.5804109242893,
            -76.6155637382775,
            -76.65071655224956,
            -76.6858693662054,
            -76.7210221801448,
            -76.75617499406773,
            -76.79132780797399,
            -76.82648062186347,
            -76.86163343573602,
            -76.89678624959154,
            -76.93193906342987,
            -76.96709187725088,
            -77.0022446910544,
            -77.0373975048403,
            -77.07255031860848,
            -77.10770313235871,
            -77.14285594609095,
            -77.17800875980494,
            -77.21316157350064,
            -77.2483143871778,
            -77.2834672008363,
            -77.31862001447601,
            -77.35377282809678,
            -77.38892564169842,
            -77.42407845528075,
            -77.45923126884364,
            -77.49438408238693,
            -77.52953689591047,
            -77.56468970941404,
            -77.5998425228975,
            -77.63499533636069,
            -77.67014814980341,
            -77.7053009632255,
            -77.7404537766268,
            -77.77560659000709,
            -77.81075940336622,
            -77.84591221670398,
            -77.88106503002021,
            -77.91621784331471,
            -77.95137065658729,
            -77.98652346983775,
            -78.02167628306591,
            -78.05682909627157,
            -78.09198190945453,
            -78.1271347226146,
            -78.16228753575152,
            -78.19744034886517,
            -78.23259316195528,
            -78.26774597502167,
            -78.30289878806408,
            -78.33805160108233,
            -78.3732044140762,
            -78.40835722704547,
            -78.4435100399899,
            -78.47866285290928,
            -78.51381566580338,
            -78.54896847867194,
            -78.58412129151476,
            -78.61927410433155,
            -78.65442691712212,
            -78.68957972988625,
            -78.7247325426236,
            -78.75988535533398,
            -78.79503816801711,
            -78.83019098067278,
            -78.86534379330068,
            -78.90049660590059,
            -78.93564941847217,
            -78.9708022310152,
            -79.00595504352944,
            -79.04110785601456,
            -79.07626066847028,
            -79.11141348089633,
            -79.14656629329242,
            -79.18171910565829,
            -79.21687191799359,
            -79.25202473029802,
            -79.28717754257134,
            -79.32233035481318,
            -79.35748316702326,
            -79.39263597920124,
            -79.42778879134681,
            -79.46294160345968,
            -79.4980944155395,
            -79.53324722758587,
            -79.56840003959856,
            -79.60355285157715,
            -79.6387056635213,
            -79.67385847543072,
            -79.70901128730503,
            -79.74416409914382,
            -79.77931691094673,
            -79.81446972271347,
            -79.84962253444357,
            -79.88477534613668,
            -79.91992815779244,
            -79.95508096941045,
            -79.99023378099028,
            -80.02538659253156,
            -80.06053940403386,
            -80.09569221549675,
            -80.13084502691984,
            -80.1659978383027,
            -80.20115064964492,
            -80.23630346094598,
            -80.27145627220554,
            -80.30660908342306,
            -80.34176189459811,
            -80.37691470573023,
            -80.41206751681895,
            -80.44722032786379,
            -80.48237313886426,
            -80.51752594981987,
            -80.55267876073015,
            -80.58783157159452,
            -80.62298438241248,
            -80.65813719318359,
            -80.69329000390721,
            -80.72844281458288,
            -80.76359562521,
            -80.79874843578806,
            -80.83390124631644,
            -80.86905405679461,
            -80.904206867222,
            -80.93935967759795,
            -80.9745124879219,
            -81.00966529819324,
            -81.04481810841138,
            -81.07997091857564,
            -81.11512372868543,
            -81.15027653874006,
            -81.1854293487389,
            -81.22058215868122,
            -81.25573496856643,
            -81.29088777839375,
            -81.32604058816251,
            -81.36119339787204,
            -81.39634620752157,
            -81.43149901711035,
            -81.46665182663769,
            -81.50180463610273,
            -81.53695744550481,
            -81.572110254843,
            -81.60726306411662,
            -81.64241587332481,
            -81.67756868246677,
            -81.71272149154161,
            -81.74787430054847,
            -81.78302710948653,
            -81.81817991835484,
            -81.85333272715256,
            -81.88848553587873,
            -81.92363834453245,
            -81.95879115311274,
            -81.9939439616186,
            -82.02909677004911,
            -82.06424957840325,
            -82.09940238667998,
            -82.1345551948783,
            -82.16970800299714,
            -82.2048608110354,
            -82.24001361899201,
            -82.27516642686582,
            -82.31031923465576,
            -82.34547204236057,
            -82.38062484997923,
            -82.41577765751036,
            -82.45093046495289,
            -82.48608327230549,
            -82.5212360795669,
            -82.55638888673587,
            -82.59154169381104,
            -82.62669450079105,
            -82.66184730767465,
            -82.69700011446027,
            -82.73215292114665,
            -82.76730572773225,
            -82.80245853421565,
            -82.83761134059527,
            -82.87276414686964,
            -82.90791695303719,
            -82.9430697590963,
            -82.9782225650453,
            -83.01337537088263,
            -83.04852817660655,
            -83.08368098221528,
            -83.11883378770712,
            -83.15398659308029,
            -83.18913939833283,
            -83.224292203463,
            -83.25944500846877,
            -83.2945978133483,
            -83.32975061809955,
            -83.36490342272042,
            -83.40005622720888,
            -83.43520903156282,
            -83.47036183578001,
            -83.50551463985829,
            -83.54066744379534,
            -83.57582024758892,
            -83.61097305123657,
            -83.64612585473593,
            -83.68127865808447,
            -83.7164314612797,
            -83.75158426431902,
            -83.78673706719977,
            -83.82188986991927,
            -83.85704267247478,
            -83.89219547486339,
            -83.92734827708223,
            -83.96250107912836,
            -83.99765388099878,
            -84.03280668269028,
            -84.06795948419972,
            -84.10311228552392,
            -84.13826508665947,
            -84.17341788760294,
            -84.20857068835086,
            -84.24372348889973,
            -84.2788762892457,
            -84.3140290893852,
            -84.34918188931425,
            -84.38433468902892,
            -84.41948748852516,
            -84.45464028779887,
            -84.48979308684572,
            -84.52494588566144,
            -84.56009868424141,
            -84.59525148258108,
            -84.63040428067582,
            -84.66555707852066,
            -84.70070987611072,
            -84.73586267344089,
            -84.77101547050583,
            -84.80616826730024,
            -84.84132106381864,
            -84.87647386005523,
            -84.91162665600422,
            -84.94677945165958,
            -84.98193224701521,
            -85.01708504206468,
            -85.05223783680162,
            -85.08739063121912,
            -85.12254342531035,
            -85.15769621906828,
            -85.19284901248552,
            -85.22800180555454,
            -85.26315459826759,
            -85.2983073906168,
            -85.33346018259377,
            -85.36861297419011,
            -85.40376576539705,
            -85.43891855620564,
            -85.47407134660659,
            -85.50922413659035,
            -85.54437692614691,
            -85.57952971526623,
            -85.61468250393777,
            -85.64983529215064,
            -85.68498807989361,
            -85.72014086715508,
            -85.75529365392322,
            -85.7904464401856,
            -85.82559922592937,
            -85.86075201114136,
            -85.89590479580797,
            -85.931057579915,
            -85.96621036344783,
            -86.00136314639137,
            -86.03651592872987,
            -86.07166871044704,
            -86.10682149152628,
            -86.14197427194992,
            -86.17712705169994,
            -86.21227983075751,
            -86.24743260910334,
            -86.28258538671719,
            -86.31773816357806,
            -86.35289093966418,
            -86.38804371495281,
            -86.42319648942066,
            -86.45834926304326,
            -86.49350203579505,
            -86.52865480764962,
            -86.56380757857941,
            -86.59896034855585,
            -86.634113117549,
            -86.66926588552778,
            -86.70441865245964,
            -86.73957141831082,
            -86.77472418304592,
            -86.80987694662807,
            -86.84502970901876,
            -86.88018247017756,
            -86.91533523006262,
            -86.9504879886297,
            -86.98564074583264,
            -87.0207935016234,
            -87.05594625595114,
            -87.09109900876308,
            -87.1262517600033,
            -87.16140450961363,
            -87.19655725753272,
            -87.23171000369605,
            -87.266862748036,
            -87.30201549048122,
            -87.33716823095666,
            -87.37232096938324,
            -87.40747370567779,
            -87.44262643975223,
            -87.47777917151384,
            -87.5129319008645,
            -87.54808462770063,
            -87.58323735191227,
            -87.61839007338347,
            -87.65354279199096,
            -87.68869550760395,
            -87.7238482200839,
            -87.75900092928332,
            -87.79415363504548,
            -87.8293063372031,
            -87.86445903557853,
            -87.89961172998179,
            -87.9347644202099,
            -87.96991710604617,
            -88.00506978725831,
            -88.04022246359764,
            -88.0753751347971,
            -88.11052780056978,
            -88.14568046060737,
            -88.18083311457711,
            -88.21598576212041,
            -88.25113840284968,
            -88.28629103634599,
            -88.32144366215479,
            -88.35659627978269,
            -88.39174888869346,
            -88.42690148830275,
            -88.46205407797261,
            -88.49720665700596,
            -88.5323592246387,
            -88.56751178003105,
            -88.60266432226004,
            -88.63781685030615,
            -88.67296936304294,
            -88.70812185922085,
            -88.7432743374502,
            -88.77842679618206,
            -88.81357923368387,
            -88.84873164801114,
            -88.8838840369745,
            -88.91903639810036,
            -88.95418872858265,
            -88.98934102522412,
            -89.02449328436776,
            -89.05964550180977,
            -89.09479767269328,
            -89.12994979137731,
            -89.16510185127088,
            -89.20025384462546,
            -89.23540576226809,
            -89.27055759325799,
            -89.30570932444132,
            -89.34086093986036,
            -89.37601241996336,
            -89.41116374053657,
            -89.44631487122197,
            -89.48146577344028,
            -89.51661639740729,
            -89.55176667774543,
            -89.5869165268675,
            -89.62206582467452,
            -89.65721440193403,
            -89.69236201232401,
            -89.72750828294004,
            -89.76265262112551,
            -89.79779402499162,
            -89.83293065763206,
            -89.86805875280535,
            -89.90316920502394,
            -89.93823314562859,
            -89.97309123040117,
        ]
        return lats

    def get_precomputed_values_N1280(self):
        lats = [0] * 2560
        lats[0] = 89.946187715665616
        lats[1] = 89.876478353332288
        lats[2] = 89.806357319542244
        lats[3] = 89.736143271609578
        lats[4] = 89.6658939412157
        lats[5] = 89.595627537554492
        lats[6] = 89.525351592371393
        lats[7] = 89.45506977912261
        lats[8] = 89.3847841013921
        lats[9] = 89.314495744374256
        lats[10] = 89.24420545380525
        lats[11] = 89.173913722284126
        lats[12] = 89.103620888238879
        lats[13] = 89.033327191845927
        lats[14] = 88.96303280826325
        lats[15] = 88.892737868230952
        lats[16] = 88.822442471310097
        lats[17] = 88.752146694650691
        lats[18] = 88.681850598961759
        lats[19] = 88.611554232668382
        lats[20] = 88.541257634868515
        lats[21] = 88.470960837474877
        lats[22] = 88.40066386679355
        lats[23] = 88.330366744702559
        lats[24] = 88.26006948954614
        lats[25] = 88.189772116820762
        lats[26] = 88.119474639706425
        lats[27] = 88.049177069484486
        lats[28] = 87.978879415867283
        lats[29] = 87.908581687261687
        lats[30] = 87.838283890981543
        lats[31] = 87.767986033419561
        lats[32] = 87.697688120188062
        lats[33] = 87.627390156234085
        lats[34] = 87.557092145935584
        lats[35] = 87.486794093180748
        lats[36] = 87.416496001434894
        lats[37] = 87.346197873795816
        lats[38] = 87.275899713041966
        lats[39] = 87.205601521672108
        lats[40] = 87.135303301939786
        lats[41] = 87.065005055882821
        lats[42] = 86.994706785348129
        lats[43] = 86.924408492014166
        lats[44] = 86.854110177408927
        lats[45] = 86.783811842927179
        lats[46] = 86.713513489844246
        lats[47] = 86.643215119328573
        lats[48] = 86.572916732453024
        lats[49] = 86.502618330203831
        lats[50] = 86.432319913489792
        lats[51] = 86.362021483149363
        lats[52] = 86.291723039957418
        lats[53] = 86.221424584631109
        lats[54] = 86.151126117835304
        lats[55] = 86.080827640187209
        lats[56] = 86.010529152260403
        lats[57] = 85.940230654588888
        lats[58] = 85.869932147670127
        lats[59] = 85.799633631968391
        lats[60] = 85.729335107917464
        lats[61] = 85.659036575922883
        lats[62] = 85.588738036364362
        lats[63] = 85.518439489597966
        lats[64] = 85.448140935957483
        lats[65] = 85.377842375756586
        lats[66] = 85.307543809290152
        lats[67] = 85.237245236835548
        lats[68] = 85.16694665865414
        lats[69] = 85.09664807499216
        lats[70] = 85.026349486081983
        lats[71] = 84.95605089214304
        lats[72] = 84.885752293382765
        lats[73] = 84.81545368999717
        lats[74] = 84.745155082171991
        lats[75] = 84.674856470082915
        lats[76] = 84.604557853896708
        lats[77] = 84.534259233771479
        lats[78] = 84.463960609857125
        lats[79] = 84.393661982296322
        lats[80] = 84.323363351224444
        lats[81] = 84.253064716770425
        lats[82] = 84.18276607905679
        lats[83] = 84.112467438200326
        lats[84] = 84.042168794312317
        lats[85] = 83.971870147498763
        lats[86] = 83.901571497860914
        lats[87] = 83.831272845495249
        lats[88] = 83.760974190494011
        lats[89] = 83.690675532945292
        lats[90] = 83.620376872933264
        lats[91] = 83.550078210538487
        lats[92] = 83.479779545838113
        lats[93] = 83.409480878905782
        lats[94] = 83.339182209812321
        lats[95] = 83.268883538625232
        lats[96] = 83.198584865409657
        lats[97] = 83.128286190227698
        lats[98] = 83.057987513139125
        lats[99] = 82.987688834201322
        lats[100] = 82.917390153469313
        lats[101] = 82.84709147099602
        lats[102] = 82.77679278683226
        lats[103] = 82.706494101026948
        lats[104] = 82.63619541362705
        lats[105] = 82.56589672467787
        lats[106] = 82.495598034222837
        lats[107] = 82.425299342304029
        lats[108] = 82.355000648961692
        lats[109] = 82.284701954234833
        lats[110] = 82.214403258160871
        lats[111] = 82.144104560776
        lats[112] = 82.073805862115165
        lats[113] = 82.003507162211946
        lats[114] = 81.933208461098829
        lats[115] = 81.862909758807191
        lats[116] = 81.792611055367345
        lats[117] = 81.722312350808508
        lats[118] = 81.652013645158945
        lats[119] = 81.581714938445955
        lats[120] = 81.511416230696042
        lats[121] = 81.441117521934686
        lats[122] = 81.370818812186627
        lats[123] = 81.300520101475826
        lats[124] = 81.230221389825374
        lats[125] = 81.159922677257711
        lats[126] = 81.089623963794551
        lats[127] = 81.019325249456955
        lats[128] = 80.949026534265244
        lats[129] = 80.878727818239184
        lats[130] = 80.808429101397948
        lats[131] = 80.73813038376008
        lats[132] = 80.667831665343556
        lats[133] = 80.59753294616587
        lats[134] = 80.527234226243991
        lats[135] = 80.456935505594302
        lats[136] = 80.386636784232863
        lats[137] = 80.316338062175078
        lats[138] = 80.246039339436052
        lats[139] = 80.175740616030438
        lats[140] = 80.105441891972376
        lats[141] = 80.035143167275749
        lats[142] = 79.9648444419539
        lats[143] = 79.894545716019948
        lats[144] = 79.824246989486554
        lats[145] = 79.753948262366038
        lats[146] = 79.683649534670437
        lats[147] = 79.61335080641139
        lats[148] = 79.543052077600308
        lats[149] = 79.472753348248219
        lats[150] = 79.402454618365894
        lats[151] = 79.332155887963822
        lats[152] = 79.261857157052191
        lats[153] = 79.191558425640977
        lats[154] = 79.121259693739859
        lats[155] = 79.050960961358285
        lats[156] = 78.980662228505423
        lats[157] = 78.910363495190211
        lats[158] = 78.840064761421445
        lats[159] = 78.769766027207638
        lats[160] = 78.699467292557102
        lats[161] = 78.629168557477882
        lats[162] = 78.558869821977908
        lats[163] = 78.488571086064923
        lats[164] = 78.418272349746417
        lats[165] = 78.347973613029708
        lats[166] = 78.277674875922045
        lats[167] = 78.207376138430348
        lats[168] = 78.137077400561424
        lats[169] = 78.066778662322022
        lats[170] = 77.996479923718596
        lats[171] = 77.926181184757539
        lats[172] = 77.855882445445019
        lats[173] = 77.785583705787161
        lats[174] = 77.71528496578982
        lats[175] = 77.644986225458879
        lats[176] = 77.574687484799924
        lats[177] = 77.504388743818524
        lats[178] = 77.434090002520122
        lats[179] = 77.363791260909963
        lats[180] = 77.293492518993247
        lats[181] = 77.22319377677502
        lats[182] = 77.15289503426024
        lats[183] = 77.082596291453768
        lats[184] = 77.012297548360323
        lats[185] = 76.941998804984564
        lats[186] = 76.871700061330955
        lats[187] = 76.801401317404
        lats[188] = 76.731102573208048
        lats[189] = 76.660803828747362
        lats[190] = 76.59050508402602
        lats[191] = 76.520206339048215
        lats[192] = 76.449907593817869
        lats[193] = 76.379608848338933
        lats[194] = 76.3093101026152
        lats[195] = 76.239011356650423
        lats[196] = 76.16871261044831
        lats[197] = 76.098413864012443
        lats[198] = 76.028115117346374
        lats[199] = 75.957816370453543
        lats[200] = 75.887517623337317
        lats[201] = 75.81721887600105
        lats[202] = 75.746920128447996
        lats[203] = 75.67662138068134
        lats[204] = 75.60632263270422
        lats[205] = 75.536023884519707
        lats[206] = 75.465725136130786
        lats[207] = 75.395426387540439
        lats[208] = 75.325127638751567
        lats[209] = 75.254828889766983
        lats[210] = 75.184530140589501
        lats[211] = 75.114231391221821
        lats[212] = 75.043932641666672
        lats[213] = 74.973633891926625
        lats[214] = 74.903335142004323
        lats[215] = 74.833036391902269
        lats[216] = 74.762737641622991
        lats[217] = 74.692438891168877
        lats[218] = 74.622140140542356
        lats[219] = 74.551841389745761
        lats[220] = 74.481542638781434
        lats[221] = 74.411243887651622
        lats[222] = 74.340945136358584
        lats[223] = 74.270646384904481
        lats[224] = 74.200347633291472
        lats[225] = 74.13004888152166
        lats[226] = 74.059750129597163
        lats[227] = 73.98945137751997
        lats[228] = 73.919152625292114
        lats[229] = 73.848853872915541
        lats[230] = 73.778555120392184
        lats[231] = 73.70825636772399
        lats[232] = 73.637957614912779
        lats[233] = 73.567658861960396
        lats[234] = 73.497360108868662
        lats[235] = 73.427061355639339
        lats[236] = 73.356762602274188
        lats[237] = 73.2864638487749
        lats[238] = 73.216165095143182
        lats[239] = 73.145866341380668
        lats[240] = 73.075567587489019
        lats[241] = 73.005268833469799
        lats[242] = 72.934970079324657
        lats[243] = 72.864671325055056
        lats[244] = 72.794372570662574
        lats[245] = 72.724073816148703
        lats[246] = 72.653775061514935
        lats[247] = 72.583476306762691
        lats[248] = 72.513177551893421
        lats[249] = 72.442878796908545
        lats[250] = 72.3725800418094
        lats[251] = 72.302281286597392
        lats[252] = 72.231982531273843
        lats[253] = 72.161683775840089
        lats[254] = 72.091385020297409
        lats[255] = 72.02108626464711
        lats[256] = 71.950787508890414
        lats[257] = 71.880488753028587
        lats[258] = 71.810189997062835
        lats[259] = 71.739891240994368
        lats[260] = 71.669592484824364
        lats[261] = 71.599293728553988
        lats[262] = 71.528994972184378
        lats[263] = 71.458696215716685
        lats[264] = 71.388397459152031
        lats[265] = 71.318098702491469
        lats[266] = 71.247799945736105
        lats[267] = 71.177501188887007
        lats[268] = 71.107202431945211
        lats[269] = 71.036903674911756
        lats[270] = 70.966604917787635
        lats[271] = 70.896306160573886
        lats[272] = 70.826007403271475
        lats[273] = 70.755708645881384
        lats[274] = 70.685409888404578
        lats[275] = 70.615111130841967
        lats[276] = 70.544812373194532
        lats[277] = 70.474513615463138
        lats[278] = 70.404214857648739
        lats[279] = 70.333916099752187
        lats[280] = 70.263617341774406
        lats[281] = 70.193318583716191
        lats[282] = 70.123019825578467
        lats[283] = 70.052721067362043
        lats[284] = 69.982422309067744
        lats[285] = 69.912123550696421
        lats[286] = 69.841824792248843
        lats[287] = 69.771526033725834
        lats[288] = 69.701227275128161
        lats[289] = 69.630928516456592
        lats[290] = 69.560629757711908
        lats[291] = 69.490330998894862
        lats[292] = 69.420032240006194
        lats[293] = 69.349733481046613
        lats[294] = 69.279434722016902
        lats[295] = 69.209135962917699
        lats[296] = 69.138837203749759
        lats[297] = 69.068538444513763
        lats[298] = 68.998239685210365
        lats[299] = 68.927940925840304
        lats[300] = 68.85764216640419
        lats[301] = 68.787343406902693
        lats[302] = 68.717044647336493
        lats[303] = 68.646745887706189
        lats[304] = 68.576447128012447
        lats[305] = 68.506148368255865
        lats[306] = 68.435849608437067
        lats[307] = 68.365550848556666
        lats[308] = 68.295252088615257
        lats[309] = 68.224953328613438
        lats[310] = 68.154654568551791
        lats[311] = 68.084355808430871
        lats[312] = 68.014057048251274
        lats[313] = 67.943758288013555
        lats[314] = 67.873459527718282
        lats[315] = 67.803160767365966
        lats[316] = 67.732862006957205
        lats[317] = 67.662563246492482
        lats[318] = 67.592264485972336
        lats[319] = 67.521965725397308
        lats[320] = 67.451666964767895
        lats[321] = 67.381368204084609
        lats[322] = 67.311069443347961
        lats[323] = 67.240770682558434
        lats[324] = 67.170471921716526
        lats[325] = 67.100173160822706
        lats[326] = 67.029874399877471
        lats[327] = 66.95957563888129
        lats[328] = 66.889276877834618
        lats[329] = 66.818978116737924
        lats[330] = 66.748679355591662
        lats[331] = 66.678380594396273
        lats[332] = 66.608081833152212
        lats[333] = 66.537783071859891
        lats[334] = 66.467484310519808
        lats[335] = 66.397185549132331
        lats[336] = 66.326886787697887
        lats[337] = 66.256588026216932
        lats[338] = 66.186289264689833
        lats[339] = 66.115990503117033
        lats[340] = 66.045691741498899
        lats[341] = 65.975392979835888
        lats[342] = 65.905094218128355
        lats[343] = 65.834795456376696
        lats[344] = 65.764496694581283
        lats[345] = 65.694197932742526
        lats[346] = 65.623899170860767
        lats[347] = 65.553600408936404
        lats[348] = 65.483301646969792
        lats[349] = 65.413002884961315
        lats[350] = 65.342704122911286
        lats[351] = 65.272405360820116
        lats[352] = 65.202106598688133
        lats[353] = 65.131807836515677
        lats[354] = 65.061509074303089
        lats[355] = 64.991210312050711
        lats[356] = 64.920911549758912
        lats[357] = 64.850612787427963
        lats[358] = 64.780314025058246
        lats[359] = 64.710015262650074
        lats[360] = 64.639716500203733
        lats[361] = 64.569417737719576
        lats[362] = 64.499118975197902
        lats[363] = 64.428820212639039
        lats[364] = 64.358521450043284
        lats[365] = 64.288222687410922
        lats[366] = 64.21792392474228
        lats[367] = 64.147625162037642
        lats[368] = 64.07732639929732
        lats[369] = 64.00702763652157
        lats[370] = 63.93672887371072
        lats[371] = 63.866430110865004
        lats[372] = 63.796131347984762
        lats[373] = 63.725832585070251
        lats[374] = 63.655533822121711
        lats[375] = 63.585235059139464
        lats[376] = 63.514936296123757
        lats[377] = 63.444637533074854
        lats[378] = 63.374338769993031
        lats[379] = 63.304040006878537
        lats[380] = 63.23374124373165
        lats[381] = 63.163442480552604
        lats[382] = 63.093143717341647
        lats[383] = 63.022844954099064
        lats[384] = 62.952546190825068
        lats[385] = 62.882247427519928
        lats[386] = 62.811948664183866
        lats[387] = 62.741649900817137
        lats[388] = 62.67135113741999
        lats[389] = 62.60105237399263
        lats[390] = 62.530753610535321
        lats[391] = 62.460454847048261
        lats[392] = 62.3901560835317
        lats[393] = 62.319857319985871
        lats[394] = 62.249558556410982
        lats[395] = 62.179259792807258
        lats[396] = 62.108961029174914
        lats[397] = 62.038662265514176
        lats[398] = 61.968363501825259
        lats[399] = 61.898064738108381
        lats[400] = 61.827765974363729
        lats[401] = 61.757467210591535
        lats[402] = 61.687168446791986
        lats[403] = 61.616869682965287
        lats[404] = 61.546570919111666
        lats[405] = 61.476272155231321
        lats[406] = 61.405973391324409
        lats[407] = 61.335674627391185
        lats[408] = 61.265375863431785
        lats[409] = 61.195077099446451
        lats[410] = 61.124778335435344
        lats[411] = 61.054479571398652
        lats[412] = 60.984180807336578
        lats[413] = 60.913882043249295
        lats[414] = 60.843583279137007
        lats[415] = 60.773284514999872
        lats[416] = 60.702985750838074
        lats[417] = 60.632686986651805
        lats[418] = 60.562388222441243
        lats[419] = 60.492089458206543
        lats[420] = 60.421790693947884
        lats[421] = 60.35149192966545
        lats[422] = 60.28119316535939
        lats[423] = 60.21089440102989
        lats[424] = 60.140595636677112
        lats[425] = 60.070296872301235
        lats[426] = 59.999998107902378
        lats[427] = 59.929699343480763
        lats[428] = 59.859400579036503
        lats[429] = 59.78910181456979
        lats[430] = 59.718803050080759
        lats[431] = 59.64850428556958
        lats[432] = 59.578205521036402
        lats[433] = 59.507906756481383
        lats[434] = 59.43760799190467
        lats[435] = 59.3673092273064
        lats[436] = 59.29701046268675
        lats[437] = 59.226711698045854
        lats[438] = 59.156412933383855
        lats[439] = 59.086114168700909
        lats[440] = 59.015815403997145
        lats[441] = 58.945516639272725
        lats[442] = 58.875217874527763
        lats[443] = 58.804919109762423
        lats[444] = 58.73462034497684
        lats[445] = 58.664321580171141
        lats[446] = 58.594022815345468
        lats[447] = 58.523724050499972
        lats[448] = 58.453425285634758
        lats[449] = 58.383126520749968
        lats[450] = 58.312827755845746
        lats[451] = 58.242528990922203
        lats[452] = 58.172230225979497
        lats[453] = 58.101931461017728
        lats[454] = 58.031632696037022
        lats[455] = 57.961333931037537
        lats[456] = 57.891035166019364
        lats[457] = 57.820736400982646
        lats[458] = 57.75043763592749
        lats[459] = 57.680138870854037
        lats[460] = 57.60984010576238
        lats[461] = 57.539541340652676
        lats[462] = 57.469242575525016
        lats[463] = 57.398943810379521
        lats[464] = 57.328645045216312
        lats[465] = 57.258346280035504
        lats[466] = 57.188047514837208
        lats[467] = 57.117748749621541
        lats[468] = 57.047449984388614
        lats[469] = 56.977151219138541
        lats[470] = 56.90685245387143
        lats[471] = 56.836553688587379
        lats[472] = 56.766254923286517
        lats[473] = 56.695956157968951
        lats[474] = 56.625657392634771
        lats[475] = 56.555358627284086
        lats[476] = 56.485059861917016
        lats[477] = 56.41476109653366
        lats[478] = 56.34446233113411
        lats[479] = 56.274163565718467
        lats[480] = 56.203864800286865
        lats[481] = 56.133566034839362
        lats[482] = 56.063267269376091
        lats[483] = 55.992968503897131
        lats[484] = 55.922669738402583
        lats[485] = 55.852370972892551
        lats[486] = 55.782072207367136
        lats[487] = 55.711773441826416
        lats[488] = 55.641474676270505
        lats[489] = 55.571175910699488
        lats[490] = 55.500877145113449
        lats[491] = 55.430578379512511
        lats[492] = 55.360279613896743
        lats[493] = 55.289980848266232
        lats[494] = 55.219682082621084
        lats[495] = 55.149383316961377
        lats[496] = 55.07908455128721
        lats[497] = 55.008785785598668
        lats[498] = 54.938487019895831
        lats[499] = 54.868188254178797
        lats[500] = 54.797889488447652
        lats[501] = 54.727590722702473
        lats[502] = 54.657291956943347
        lats[503] = 54.586993191170357
        lats[504] = 54.516694425383605
        lats[505] = 54.446395659583146
        lats[506] = 54.376096893769081
        lats[507] = 54.305798127941479
        lats[508] = 54.235499362100448
        lats[509] = 54.165200596246031
        lats[510] = 54.094901830378333
        lats[511] = 54.024603064497434
        lats[512] = 53.954304298603383
        lats[513] = 53.884005532696307
        lats[514] = 53.813706766776235
        lats[515] = 53.743408000843282
        lats[516] = 53.673109234897495
        lats[517] = 53.602810468938962
        lats[518] = 53.53251170296776
        lats[519] = 53.462212936983953
        lats[520] = 53.391914170987633
        lats[521] = 53.321615404978871
        lats[522] = 53.251316638957725
        lats[523] = 53.181017872924265
        lats[524] = 53.110719106878584
        lats[525] = 53.040420340820731
        lats[526] = 52.970121574750792
        lats[527] = 52.899822808668837
        lats[528] = 52.829524042574917
        lats[529] = 52.759225276469131
        lats[530] = 52.688926510351514
        lats[531] = 52.618627744222159
        lats[532] = 52.548328978081123
        lats[533] = 52.478030211928477
        lats[534] = 52.407731445764284
        lats[535] = 52.337432679588609
        lats[536] = 52.26713391340153
        lats[537] = 52.196835147203096
        lats[538] = 52.126536380993372
        lats[539] = 52.056237614772435
        lats[540] = 51.985938848540336
        lats[541] = 51.915640082297152
        lats[542] = 51.845341316042933
        lats[543] = 51.775042549777737
        lats[544] = 51.704743783501634
        lats[545] = 51.634445017214695
        lats[546] = 51.56414625091697
        lats[547] = 51.493847484608516
        lats[548] = 51.423548718289396
        lats[549] = 51.353249951959683
        lats[550] = 51.282951185619417
        lats[551] = 51.21265241926865
        lats[552] = 51.14235365290746
        lats[553] = 51.072054886535909
        lats[554] = 51.001756120154049
        lats[555] = 50.931457353761914
        lats[556] = 50.86115858735959
        lats[557] = 50.790859820947119
        lats[558] = 50.720561054524559
        lats[559] = 50.650262288091959
        lats[560] = 50.579963521649397
        lats[561] = 50.509664755196901
        lats[562] = 50.439365988734544
        lats[563] = 50.369067222262359
        lats[564] = 50.298768455780426
        lats[565] = 50.228469689288779
        lats[566] = 50.158170922787484
        lats[567] = 50.087872156276575
        lats[568] = 50.017573389756123
        lats[569] = 49.947274623226157
        lats[570] = 49.876975856686762
        lats[571] = 49.80667709013796
        lats[572] = 49.736378323579807
        lats[573] = 49.66607955701236
        lats[574] = 49.595780790435676
        lats[575] = 49.525482023849783
        lats[576] = 49.455183257254745
        lats[577] = 49.384884490650613
        lats[578] = 49.314585724037435
        lats[579] = 49.244286957415234
        lats[580] = 49.173988190784094
        lats[581] = 49.103689424144044
        lats[582] = 49.03339065749514
        lats[583] = 48.963091890837418
        lats[584] = 48.892793124170929
        lats[585] = 48.822494357495721
        lats[586] = 48.752195590811837
        lats[587] = 48.681896824119335
        lats[588] = 48.611598057418242
        lats[589] = 48.541299290708608
        lats[590] = 48.47100052399049
        lats[591] = 48.400701757263917
        lats[592] = 48.330402990528938
        lats[593] = 48.260104223785596
        lats[594] = 48.189805457033941
        lats[595] = 48.119506690274015
        lats[596] = 48.049207923505868
        lats[597] = 47.978909156729507
        lats[598] = 47.908610389945018
        lats[599] = 47.838311623152421
        lats[600] = 47.76801285635176
        lats[601] = 47.697714089543084
        lats[602] = 47.627415322726435
        lats[603] = 47.557116555901828
        lats[604] = 47.486817789069342
        lats[605] = 47.416519022228997
        lats[606] = 47.346220255380835
        lats[607] = 47.275921488524894
        lats[608] = 47.205622721661214
        lats[609] = 47.13532395478984
        lats[610] = 47.065025187910805
        lats[611] = 46.994726421024154
        lats[612] = 46.924427654129929
        lats[613] = 46.85412888722815
        lats[614] = 46.783830120318882
        lats[615] = 46.713531353402139
        lats[616] = 46.643232586477971
        lats[617] = 46.572933819546414
        lats[618] = 46.502635052607502
        lats[619] = 46.432336285661272
        lats[620] = 46.362037518707766
        lats[621] = 46.291738751747012
        lats[622] = 46.221439984779053
        lats[623] = 46.151141217803925
        lats[624] = 46.080842450821663
        lats[625] = 46.01054368383231
        lats[626] = 45.94024491683588
        lats[627] = 45.869946149832437
        lats[628] = 45.799647382821995
        lats[629] = 45.729348615804589
        lats[630] = 45.659049848780263
        lats[631] = 45.588751081749038
        lats[632] = 45.51845231471097
        lats[633] = 45.448153547666081
        lats[634] = 45.377854780614399
        lats[635] = 45.30755601355596
        lats[636] = 45.237257246490813
        lats[637] = 45.166958479418959
        lats[638] = 45.096659712340461
        lats[639] = 45.026360945255341
        lats[640] = 44.956062178163634
        lats[641] = 44.885763411065362
        lats[642] = 44.81546464396056
        lats[643] = 44.745165876849271
        lats[644] = 44.674867109731515
        lats[645] = 44.604568342607337
        lats[646] = 44.534269575476756
        lats[647] = 44.463970808339802
        lats[648] = 44.39367204119651
        lats[649] = 44.323373274046915
        lats[650] = 44.253074506891046
        lats[651] = 44.182775739728925
        lats[652] = 44.112476972560586
        lats[653] = 44.042178205386072
        lats[654] = 43.971879438205391
        lats[655] = 43.9015806710186
        lats[656] = 43.831281903825705
        lats[657] = 43.760983136626741
        lats[658] = 43.690684369421732
        lats[659] = 43.620385602210717
        lats[660] = 43.550086834993728
        lats[661] = 43.479788067770777
        lats[662] = 43.409489300541907
        lats[663] = 43.339190533307139
        lats[664] = 43.26889176606651
        lats[665] = 43.19859299882004
        lats[666] = 43.128294231567757
        lats[667] = 43.057995464309691
        lats[668] = 42.987696697045862
        lats[669] = 42.917397929776307
        lats[670] = 42.847099162501053
        lats[671] = 42.776800395220121
        lats[672] = 42.706501627933541
        lats[673] = 42.63620286064134
        lats[674] = 42.565904093343548
        lats[675] = 42.495605326040177
        lats[676] = 42.425306558731272
        lats[677] = 42.355007791416853
        lats[678] = 42.284709024096927
        lats[679] = 42.214410256771551
        lats[680] = 42.144111489440725
        lats[681] = 42.073812722104492
        lats[682] = 42.003513954762873
        lats[683] = 41.933215187415882
        lats[684] = 41.862916420063563
        lats[685] = 41.792617652705921
        lats[686] = 41.722318885343
        lats[687] = 41.6520201179748
        lats[688] = 41.581721350601363
        lats[689] = 41.511422583222718
        lats[690] = 41.441123815838885
        lats[691] = 41.370825048449873
        lats[692] = 41.300526281055724
        lats[693] = 41.230227513656445
        lats[694] = 41.159928746252085
        lats[695] = 41.089629978842645
        lats[696] = 41.01933121142816
        lats[697] = 40.949032444008644
        lats[698] = 40.878733676584126
        lats[699] = 40.808434909154634
        lats[700] = 40.738136141720176
        lats[701] = 40.667837374280786
        lats[702] = 40.597538606836487
        lats[703] = 40.527239839387299
        lats[704] = 40.456941071933244
        lats[705] = 40.386642304474343
        lats[706] = 40.316343537010617
        lats[707] = 40.246044769542102
        lats[708] = 40.175746002068806
        lats[709] = 40.105447234590748
        lats[710] = 40.035148467107952
        lats[711] = 39.964849699620437
        lats[712] = 39.894550932128247
        lats[713] = 39.824252164631375
        lats[714] = 39.753953397129855
        lats[715] = 39.683654629623703
        lats[716] = 39.613355862112947
        lats[717] = 39.543057094597607
        lats[718] = 39.472758327077692
        lats[719] = 39.402459559553229
        lats[720] = 39.332160792024254
        lats[721] = 39.261862024490775
        lats[722] = 39.191563256952804
        lats[723] = 39.121264489410365
        lats[724] = 39.050965721863491
        lats[725] = 38.980666954312184
        lats[726] = 38.910368186756479
        lats[727] = 38.840069419196389
        lats[728] = 38.769770651631937
        lats[729] = 38.699471884063136
        lats[730] = 38.629173116490001
        lats[731] = 38.558874348912568
        lats[732] = 38.488575581330842
        lats[733] = 38.418276813744846
        lats[734] = 38.347978046154608
        lats[735] = 38.277679278560143
        lats[736] = 38.20738051096145
        lats[737] = 38.137081743358586
        lats[738] = 38.066782975751536
        lats[739] = 37.99648420814033
        lats[740] = 37.926185440524989
        lats[741] = 37.855886672905527
        lats[742] = 37.785587905281965
        lats[743] = 37.715289137654317
        lats[744] = 37.644990370022605
        lats[745] = 37.574691602386856
        lats[746] = 37.504392834747065
        lats[747] = 37.434094067103274
        lats[748] = 37.363795299455489
        lats[749] = 37.293496531803719
        lats[750] = 37.223197764147997
        lats[751] = 37.152898996488332
        lats[752] = 37.082600228824752
        lats[753] = 37.012301461157264
        lats[754] = 36.942002693485883
        lats[755] = 36.871703925810628
        lats[756] = 36.801405158131523
        lats[757] = 36.731106390448581
        lats[758] = 36.660807622761808
        lats[759] = 36.590508855071242
        lats[760] = 36.520210087376888
        lats[761] = 36.449911319678755
        lats[762] = 36.379612551976876
        lats[763] = 36.309313784271254
        lats[764] = 36.239015016561908
        lats[765] = 36.16871624884886
        lats[766] = 36.098417481132117
        lats[767] = 36.028118713411708
        lats[768] = 35.957819945687639
        lats[769] = 35.887521177959933
        lats[770] = 35.817222410228595
        lats[771] = 35.746923642493655
        lats[772] = 35.676624874755113
        lats[773] = 35.606326107012997
        lats[774] = 35.536027339267314
        lats[775] = 35.465728571518085
        lats[776] = 35.395429803765317
        lats[777] = 35.325131036009047
        lats[778] = 35.254832268249267
        lats[779] = 35.184533500486005
        lats[780] = 35.114234732719261
        lats[781] = 35.043935964949064
        lats[782] = 34.973637197175435
        lats[783] = 34.903338429398374
        lats[784] = 34.833039661617903
        lats[785] = 34.762740893834028
        lats[786] = 34.692442126046771
        lats[787] = 34.622143358256153
        lats[788] = 34.551844590462188
        lats[789] = 34.481545822664863
        lats[790] = 34.411247054864234
        lats[791] = 34.340948287060286
        lats[792] = 34.270649519253041
        lats[793] = 34.200350751442521
        lats[794] = 34.130051983628725
        lats[795] = 34.059753215811682
        lats[796] = 33.989454447991392
        lats[797] = 33.919155680167876
        lats[798] = 33.848856912341155
        lats[799] = 33.778558144511237
        lats[800] = 33.708259376678136
        lats[801] = 33.637960608841851
        lats[802] = 33.567661841002426
        lats[803] = 33.497363073159853
        lats[804] = 33.42706430531414
        lats[805] = 33.356765537465314
        lats[806] = 33.286466769613391
        lats[807] = 33.216168001758369
        lats[808] = 33.145869233900278
        lats[809] = 33.075570466039117
        lats[810] = 33.005271698174909
        lats[811] = 32.934972930307666
        lats[812] = 32.864674162437396
        lats[813] = 32.794375394564113
        lats[814] = 32.724076626687825
        lats[815] = 32.653777858808567
        lats[816] = 32.583479090926325
        lats[817] = 32.513180323041112
        lats[818] = 32.442881555152965
        lats[819] = 32.372582787261891
        lats[820] = 32.302284019367875
        lats[821] = 32.231985251470959
        lats[822] = 32.161686483571145
        lats[823] = 32.091387715668439
        lats[824] = 32.021088947762863
        lats[825] = 31.950790179854422
        lats[826] = 31.880491411943137
        lats[827] = 31.810192644029012
        lats[828] = 31.739893876112063
        lats[829] = 31.669595108192297
        lats[830] = 31.599296340269738
        lats[831] = 31.528997572344384
        lats[832] = 31.458698804416255
        lats[833] = 31.388400036485361
        lats[834] = 31.318101268551715
        lats[835] = 31.247802500615318
        lats[836] = 31.177503732676204
        lats[837] = 31.107204964734358
        lats[838] = 31.036906196789811
        lats[839] = 30.966607428842572
        lats[840] = 30.896308660892647
        lats[841] = 30.826009892940046
        lats[842] = 30.755711124984781
        lats[843] = 30.685412357026873
        lats[844] = 30.615113589066322
        lats[845] = 30.544814821103138
        lats[846] = 30.47451605313735
        lats[847] = 30.404217285168947
        lats[848] = 30.333918517197947
        lats[849] = 30.263619749224372
        lats[850] = 30.19332098124822
        lats[851] = 30.123022213269511
        lats[852] = 30.052723445288244
        lats[853] = 29.98242467730444
        lats[854] = 29.91212590931811
        lats[855] = 29.841827141329258
        lats[856] = 29.771528373337894
        lats[857] = 29.701229605344039
        lats[858] = 29.630930837347698
        lats[859] = 29.560632069348884
        lats[860] = 29.490333301347597
        lats[861] = 29.420034533343859
        lats[862] = 29.349735765337677
        lats[863] = 29.279436997329057
        lats[864] = 29.209138229318015
        lats[865] = 29.138839461304556
        lats[866] = 29.068540693288696
        lats[867] = 28.998241925270449
        lats[868] = 28.927943157249814
        lats[869] = 28.857644389226806
        lats[870] = 28.787345621201432
        lats[871] = 28.717046853173709
        lats[872] = 28.646748085143642
        lats[873] = 28.576449317111244
        lats[874] = 28.506150549076519
        lats[875] = 28.435851781039485
        lats[876] = 28.365553013000145
        lats[877] = 28.29525424495851
        lats[878] = 28.224955476914594
        lats[879] = 28.154656708868405
        lats[880] = 28.084357940819952
        lats[881] = 28.014059172769244
        lats[882] = 27.94376040471629
        lats[883] = 27.873461636661098
        lats[884] = 27.803162868603682
        lats[885] = 27.732864100544052
        lats[886] = 27.662565332482213
        lats[887] = 27.592266564418171
        lats[888] = 27.521967796351948
        lats[889] = 27.451669028283543
        lats[890] = 27.381370260212968
        lats[891] = 27.311071492140236
        lats[892] = 27.240772724065348
        lats[893] = 27.170473955988321
        lats[894] = 27.100175187909159
        lats[895] = 27.029876419827872
        lats[896] = 26.959577651744471
        lats[897] = 26.889278883658971
        lats[898] = 26.818980115571364
        lats[899] = 26.748681347481678
        lats[900] = 26.678382579389908
        lats[901] = 26.608083811296069
        lats[902] = 26.53778504320017
        lats[903] = 26.467486275102218
        lats[904] = 26.397187507002222
        lats[905] = 26.326888738900195
        lats[906] = 26.256589970796135
        lats[907] = 26.186291202690064
        lats[908] = 26.115992434581983
        lats[909] = 26.045693666471902
        lats[910] = 25.975394898359827
        lats[911] = 25.90509613024577
        lats[912] = 25.834797362129745
        lats[913] = 25.764498594011751
        lats[914] = 25.694199825891793
        lats[915] = 25.623901057769892
        lats[916] = 25.553602289646051
        lats[917] = 25.483303521520277
        lats[918] = 25.413004753392578
        lats[919] = 25.342705985262967
        lats[920] = 25.272407217131445
        lats[921] = 25.202108448998025
        lats[922] = 25.13180968086272
        lats[923] = 25.061510912725527
        lats[924] = 24.991212144586456
        lats[925] = 24.920913376445526
        lats[926] = 24.850614608302738
        lats[927] = 24.780315840158096
        lats[928] = 24.710017072011613
        lats[929] = 24.639718303863294
        lats[930] = 24.569419535713152
        lats[931] = 24.499120767561195
        lats[932] = 24.428821999407425
        lats[933] = 24.358523231251851
        lats[934] = 24.288224463094483
        lats[935] = 24.217925694935328
        lats[936] = 24.1476269267744
        lats[937] = 24.077328158611696
        lats[938] = 24.007029390447226
        lats[939] = 23.936730622281004
        lats[940] = 23.866431854113038
        lats[941] = 23.796133085943328
        lats[942] = 23.725834317771888
        lats[943] = 23.655535549598721
        lats[944] = 23.585236781423838
        lats[945] = 23.514938013247242
        lats[946] = 23.444639245068949
        lats[947] = 23.374340476888957
        lats[948] = 23.304041708707278
        lats[949] = 23.233742940523921
        lats[950] = 23.163444172338895
        lats[951] = 23.0931454041522
        lats[952] = 23.022846635963852
        lats[953] = 22.952547867773848
        lats[954] = 22.882249099582204
        lats[955] = 22.811950331388925
        lats[956] = 22.741651563194019
        lats[957] = 22.671352794997489
        lats[958] = 22.60105402679935
        lats[959] = 22.530755258599601
        lats[960] = 22.460456490398254
        lats[961] = 22.390157722195315
        lats[962] = 22.319858953990789
        lats[963] = 22.249560185784691
        lats[964] = 22.179261417577013
        lats[965] = 22.108962649367779
        lats[966] = 22.038663881156989
        lats[967] = 21.968365112944642
        lats[968] = 21.898066344730758
        lats[969] = 21.827767576515338
        lats[970] = 21.757468808298391
        lats[971] = 21.687170040079913
        lats[972] = 21.616871271859928
        lats[973] = 21.546572503638437
        lats[974] = 21.47627373541544
        lats[975] = 21.40597496719095
        lats[976] = 21.335676198964972
        lats[977] = 21.265377430737512
        lats[978] = 21.195078662508585
        lats[979] = 21.124779894278181
        lats[980] = 21.054481126046323
        lats[981] = 20.984182357813012
        lats[982] = 20.913883589578251
        lats[983] = 20.843584821342048
        lats[984] = 20.773286053104417
        lats[985] = 20.702987284865355
        lats[986] = 20.632688516624874
        lats[987] = 20.562389748382977
        lats[988] = 20.492090980139672
        lats[989] = 20.421792211894967
        lats[990] = 20.35149344364887
        lats[991] = 20.28119467540138
        lats[992] = 20.210895907152516
        lats[993] = 20.140597138902272
        lats[994] = 20.070298370650661
        lats[995] = 19.999999602397686
        lats[996] = 19.929700834143357
        lats[997] = 19.859402065887682
        lats[998] = 19.789103297630657
        lats[999] = 19.718804529372303
        lats[1000] = 19.648505761112613
        lats[1001] = 19.578206992851602
        lats[1002] = 19.507908224589269
        lats[1003] = 19.437609456325632
        lats[1004] = 19.367310688060684
        lats[1005] = 19.297011919794439
        lats[1006] = 19.226713151526898
        lats[1007] = 19.15641438325807
        lats[1008] = 19.086115614987968
        lats[1009] = 19.015816846716586
        lats[1010] = 18.945518078443939
        lats[1011] = 18.875219310170031
        lats[1012] = 18.804920541894862
        lats[1013] = 18.734621773618446
        lats[1014] = 18.664323005340787
        lats[1015] = 18.594024237061891
        lats[1016] = 18.523725468781763
        lats[1017] = 18.453426700500408
        lats[1018] = 18.383127932217832
        lats[1019] = 18.312829163934047
        lats[1020] = 18.242530395649048
        lats[1021] = 18.172231627362851
        lats[1022] = 18.101932859075458
        lats[1023] = 18.031634090786874
        lats[1024] = 17.96133532249711
        lats[1025] = 17.89103655420616
        lats[1026] = 17.820737785914044
        lats[1027] = 17.75043901762076
        lats[1028] = 17.680140249326314
        lats[1029] = 17.60984148103071
        lats[1030] = 17.539542712733962
        lats[1031] = 17.469243944436066
        lats[1032] = 17.39894517613704
        lats[1033] = 17.328646407836878
        lats[1034] = 17.258347639535586
        lats[1035] = 17.188048871233182
        lats[1036] = 17.117750102929655
        lats[1037] = 17.04745133462502
        lats[1038] = 16.977152566319283
        lats[1039] = 16.906853798012452
        lats[1040] = 16.836555029704527
        lats[1041] = 16.766256261395515
        lats[1042] = 16.69595749308542
        lats[1043] = 16.625658724774254
        lats[1044] = 16.555359956462013
        lats[1045] = 16.485061188148713
        lats[1046] = 16.41476241983435
        lats[1047] = 16.344463651518936
        lats[1048] = 16.274164883202477
        lats[1049] = 16.203866114884974
        lats[1050] = 16.133567346566434
        lats[1051] = 16.063268578246863
        lats[1052] = 15.992969809926265
        lats[1053] = 15.922671041604652
        lats[1054] = 15.852372273282016
        lats[1055] = 15.78207350495838
        lats[1056] = 15.711774736633735
        lats[1057] = 15.641475968308091
        lats[1058] = 15.571177199981456
        lats[1059] = 15.500878431653829
        lats[1060] = 15.430579663325226
        lats[1061] = 15.360280894995643
        lats[1062] = 15.289982126665089
        lats[1063] = 15.219683358333569
        lats[1064] = 15.149384590001089
        lats[1065] = 15.07908582166765
        lats[1066] = 15.008787053333259
        lats[1067] = 14.938488284997929
        lats[1068] = 14.868189516661655
        lats[1069] = 14.797890748324447
        lats[1070] = 14.727591979986309
        lats[1071] = 14.657293211647247
        lats[1072] = 14.586994443307265
        lats[1073] = 14.516695674966371
        lats[1074] = 14.446396906624567
        lats[1075] = 14.376098138281863
        lats[1076] = 14.305799369938256
        lats[1077] = 14.23550060159376
        lats[1078] = 14.165201833248371
        lats[1079] = 14.0949030649021
        lats[1080] = 14.024604296554955
        lats[1081] = 13.954305528206934
        lats[1082] = 13.884006759858046
        lats[1083] = 13.813707991508297
        lats[1084] = 13.743409223157688
        lats[1085] = 13.673110454806226
        lats[1086] = 13.602811686453919
        lats[1087] = 13.532512918100766
        lats[1088] = 13.46221414974678
        lats[1089] = 13.391915381391959
        lats[1090] = 13.32161661303631
        lats[1091] = 13.251317844679837
        lats[1092] = 13.181019076322551
        lats[1093] = 13.110720307964451
        lats[1094] = 13.040421539605545
        lats[1095] = 12.970122771245832
        lats[1096] = 12.899824002885323
        lats[1097] = 12.829525234524022
        lats[1098] = 12.759226466161934
        lats[1099] = 12.688927697799061
        lats[1100] = 12.618628929435411
        lats[1101] = 12.548330161070988
        lats[1102] = 12.478031392705796
        lats[1103] = 12.407732624339841
        lats[1104] = 12.337433855973126
        lats[1105] = 12.267135087605659
        lats[1106] = 12.196836319237443
        lats[1107] = 12.126537550868482
        lats[1108] = 12.056238782498781
        lats[1109] = 11.985940014128348
        lats[1110] = 11.915641245757183
        lats[1111] = 11.845342477385294
        lats[1112] = 11.775043709012685
        lats[1113] = 11.704744940639358
        lats[1114] = 11.634446172265324
        lats[1115] = 11.564147403890583
        lats[1116] = 11.493848635515141
        lats[1117] = 11.423549867139002
        lats[1118] = 11.35325109876217
        lats[1119] = 11.282952330384653
        lats[1120] = 11.212653562006453
        lats[1121] = 11.142354793627575
        lats[1122] = 11.072056025248026
        lats[1123] = 11.001757256867807
        lats[1124] = 10.931458488486923
        lats[1125] = 10.861159720105382
        lats[1126] = 10.790860951723188
        lats[1127] = 10.720562183340341
        lats[1128] = 10.65026341495685
        lats[1129] = 10.579964646572719
        lats[1130] = 10.509665878187954
        lats[1131] = 10.439367109802557
        lats[1132] = 10.369068341416533
        lats[1133] = 10.298769573029887
        lats[1134] = 10.228470804642624
        lats[1135] = 10.158172036254747
        lats[1136] = 10.087873267866264
        lats[1137] = 10.017574499477174
        lats[1138] = 9.9472757310874869
        lats[1139] = 9.8769769626972046
        lats[1140] = 9.8066781943063344
        lats[1141] = 9.7363794259148779
        lats[1142] = 9.6660806575228388
        lats[1143] = 9.5957818891302242
        lats[1144] = 9.5254831207370376
        lats[1145] = 9.4551843523432826
        lats[1146] = 9.3848855839489662
        lats[1147] = 9.3145868155540921
        lats[1148] = 9.2442880471586619
        lats[1149] = 9.1739892787626829
        lats[1150] = 9.1036905103661585
        lats[1151] = 9.0333917419690941
        lats[1152] = 8.963092973571495
        lats[1153] = 8.8927942051733631
        lats[1154] = 8.8224954367747017
        lats[1155] = 8.7521966683755217
        lats[1156] = 8.6818978999758194
        lats[1157] = 8.6115991315756055
        lats[1158] = 8.5413003631748801
        lats[1159] = 8.4710015947736537
        lats[1160] = 8.4007028263719228
        lats[1161] = 8.3304040579696963
        lats[1162] = 8.2601052895669778
        lats[1163] = 8.1898065211637725
        lats[1164] = 8.1195077527600841
        lats[1165] = 8.049208984355916
        lats[1166] = 7.9789102159512737
        lats[1167] = 7.9086114475461606
        lats[1168] = 7.8383126791405831
        lats[1169] = 7.7680139107345463
        lats[1170] = 7.6977151423280494
        lats[1171] = 7.6274163739210996
        lats[1172] = 7.557117605513703
        lats[1173] = 7.4868188371058624
        lats[1174] = 7.4165200686975803
        lats[1175] = 7.3462213002888648
        lats[1176] = 7.2759225318797176
        lats[1177] = 7.2056237634701441
        lats[1178] = 7.1353249950601469
        lats[1179] = 7.0650262266497315
        lats[1180] = 6.994727458238903
        lats[1181] = 6.924428689827665
        lats[1182] = 6.8541299214160212
        lats[1183] = 6.7838311530039768
        lats[1184] = 6.7135323845915353
        lats[1185] = 6.6432336161787013
        lats[1186] = 6.5729348477654792
        lats[1187] = 6.5026360793518734
        lats[1188] = 6.4323373109378874
        lats[1189] = 6.3620385425235257
        lats[1190] = 6.2917397741087928
        lats[1191] = 6.2214410056936931
        lats[1192] = 6.151142237278231
        lats[1193] = 6.0808434688624091
        lats[1194] = 6.0105447004462347
        lats[1195] = 5.9402459320297085
        lats[1196] = 5.869947163612836
        lats[1197] = 5.7996483951956233
        lats[1198] = 5.729349626778073
        lats[1199] = 5.6590508583601888
        lats[1200] = 5.5887520899419751
        lats[1201] = 5.5184533215234373
        lats[1202] = 5.4481545531045787
        lats[1203] = 5.3778557846854023
        lats[1204] = 5.3075570162659149
        lats[1205] = 5.2372582478461194
        lats[1206] = 5.1669594794260192
        lats[1207] = 5.0966607110056197
        lats[1208] = 5.0263619425849244
        lats[1209] = 4.9560631741639369
        lats[1210] = 4.8857644057426626
        lats[1211] = 4.8154656373211049
        lats[1212] = 4.7451668688992683
        lats[1213] = 4.6748681004771564
        lats[1214] = 4.6045693320547736
        lats[1215] = 4.5342705636321252
        lats[1216] = 4.4639717952092139
        lats[1217] = 4.3936730267860451
        lats[1218] = 4.3233742583626205
        lats[1219] = 4.2530754899389471
        lats[1220] = 4.1827767215150269
        lats[1221] = 4.1124779530908659
        lats[1222] = 4.0421791846664661
        lats[1223] = 3.9718804162418326
        lats[1224] = 3.90158164781697
        lats[1225] = 3.8312828793918823
        lats[1226] = 3.7609841109665734
        lats[1227] = 3.6906853425410477
        lats[1228] = 3.6203865741153085
        lats[1229] = 3.5500878056893601
        lats[1230] = 3.4797890372632065
        lats[1231] = 3.4094902688368531
        lats[1232] = 3.339191500410303
        lats[1233] = 3.2688927319835597
        lats[1234] = 3.1985939635566285
        lats[1235] = 3.1282951951295126
        lats[1236] = 3.0579964267022164
        lats[1237] = 2.9876976582747439
        lats[1238] = 2.9173988898470999
        lats[1239] = 2.8471001214192873
        lats[1240] = 2.7768013529913107
        lats[1241] = 2.7065025845631743
        lats[1242] = 2.6362038161348824
        lats[1243] = 2.5659050477064382
        lats[1244] = 2.4956062792778466
        lats[1245] = 2.4253075108491116
        lats[1246] = 2.3550087424202366
        lats[1247] = 2.2847099739912267
        lats[1248] = 2.2144112055620848
        lats[1249] = 2.1441124371328155
        lats[1250] = 2.0738136687034232
        lats[1251] = 2.0035149002739114
        lats[1252] = 1.9332161318442849
        lats[1253] = 1.8629173634145471
        lats[1254] = 1.792618594984702
        lats[1255] = 1.7223198265547539
        lats[1256] = 1.6520210581247066
        lats[1257] = 1.5817222896945646
        lats[1258] = 1.5114235212643317
        lats[1259] = 1.4411247528340119
        lats[1260] = 1.3708259844036093
        lats[1261] = 1.300527215973128
        lats[1262] = 1.2302284475425722
        lats[1263] = 1.1599296791119456
        lats[1264] = 1.0896309106812523
        lats[1265] = 1.0193321422504964
        lats[1266] = 0.949033373819682
        lats[1267] = 0.87873460538881287
        lats[1268] = 0.80843583695789356
        lats[1269] = 0.73813706852692773
        lats[1270] = 0.66783830009591949
        lats[1271] = 0.59753953166487306
        lats[1272] = 0.52724076323379232
        lats[1273] = 0.45694199480268116
        lats[1274] = 0.3866432263715438
        lats[1275] = 0.31634445794038429
        lats[1276] = 0.24604568950920663
        lats[1277] = 0.17574692107801482
        lats[1278] = 0.10544815264681295
        lats[1279] = 0.035149384215604956
        lats[1280] = -0.035149384215604956
        lats[1281] = -0.10544815264681295
        lats[1282] = -0.17574692107801482
        lats[1283] = -0.24604568950920663
        lats[1284] = -0.31634445794038429
        lats[1285] = -0.3866432263715438
        lats[1286] = -0.45694199480268116
        lats[1287] = -0.52724076323379232
        lats[1288] = -0.59753953166487306
        lats[1289] = -0.66783830009591949
        lats[1290] = -0.73813706852692773
        lats[1291] = -0.80843583695789356
        lats[1292] = -0.87873460538881287
        lats[1293] = -0.949033373819682
        lats[1294] = -1.0193321422504964
        lats[1295] = -1.0896309106812523
        lats[1296] = -1.1599296791119456
        lats[1297] = -1.2302284475425722
        lats[1298] = -1.300527215973128
        lats[1299] = -1.3708259844036093
        lats[1300] = -1.4411247528340119
        lats[1301] = -1.5114235212643317
        lats[1302] = -1.5817222896945646
        lats[1303] = -1.6520210581247066
        lats[1304] = -1.7223198265547539
        lats[1305] = -1.792618594984702
        lats[1306] = -1.8629173634145471
        lats[1307] = -1.9332161318442849
        lats[1308] = -2.0035149002739114
        lats[1309] = -2.0738136687034232
        lats[1310] = -2.1441124371328155
        lats[1311] = -2.2144112055620848
        lats[1312] = -2.2847099739912267
        lats[1313] = -2.3550087424202366
        lats[1314] = -2.4253075108491116
        lats[1315] = -2.4956062792778466
        lats[1316] = -2.5659050477064382
        lats[1317] = -2.6362038161348824
        lats[1318] = -2.7065025845631743
        lats[1319] = -2.7768013529913107
        lats[1320] = -2.8471001214192873
        lats[1321] = -2.9173988898470999
        lats[1322] = -2.9876976582747439
        lats[1323] = -3.0579964267022164
        lats[1324] = -3.1282951951295126
        lats[1325] = -3.1985939635566285
        lats[1326] = -3.2688927319835597
        lats[1327] = -3.339191500410303
        lats[1328] = -3.4094902688368531
        lats[1329] = -3.4797890372632065
        lats[1330] = -3.5500878056893601
        lats[1331] = -3.6203865741153085
        lats[1332] = -3.6906853425410477
        lats[1333] = -3.7609841109665734
        lats[1334] = -3.8312828793918823
        lats[1335] = -3.90158164781697
        lats[1336] = -3.9718804162418326
        lats[1337] = -4.0421791846664661
        lats[1338] = -4.1124779530908659
        lats[1339] = -4.1827767215150269
        lats[1340] = -4.2530754899389471
        lats[1341] = -4.3233742583626205
        lats[1342] = -4.3936730267860451
        lats[1343] = -4.4639717952092139
        lats[1344] = -4.5342705636321252
        lats[1345] = -4.6045693320547736
        lats[1346] = -4.6748681004771564
        lats[1347] = -4.7451668688992683
        lats[1348] = -4.8154656373211049
        lats[1349] = -4.8857644057426626
        lats[1350] = -4.9560631741639369
        lats[1351] = -5.0263619425849244
        lats[1352] = -5.0966607110056197
        lats[1353] = -5.1669594794260192
        lats[1354] = -5.2372582478461194
        lats[1355] = -5.3075570162659149
        lats[1356] = -5.3778557846854023
        lats[1357] = -5.4481545531045787
        lats[1358] = -5.5184533215234373
        lats[1359] = -5.5887520899419751
        lats[1360] = -5.6590508583601888
        lats[1361] = -5.729349626778073
        lats[1362] = -5.7996483951956233
        lats[1363] = -5.869947163612836
        lats[1364] = -5.9402459320297085
        lats[1365] = -6.0105447004462347
        lats[1366] = -6.0808434688624091
        lats[1367] = -6.151142237278231
        lats[1368] = -6.2214410056936931
        lats[1369] = -6.2917397741087928
        lats[1370] = -6.3620385425235257
        lats[1371] = -6.4323373109378874
        lats[1372] = -6.5026360793518734
        lats[1373] = -6.5729348477654792
        lats[1374] = -6.6432336161787013
        lats[1375] = -6.7135323845915353
        lats[1376] = -6.7838311530039768
        lats[1377] = -6.8541299214160212
        lats[1378] = -6.924428689827665
        lats[1379] = -6.994727458238903
        lats[1380] = -7.0650262266497315
        lats[1381] = -7.1353249950601469
        lats[1382] = -7.2056237634701441
        lats[1383] = -7.2759225318797176
        lats[1384] = -7.3462213002888648
        lats[1385] = -7.4165200686975803
        lats[1386] = -7.4868188371058624
        lats[1387] = -7.557117605513703
        lats[1388] = -7.6274163739210996
        lats[1389] = -7.6977151423280494
        lats[1390] = -7.7680139107345463
        lats[1391] = -7.8383126791405831
        lats[1392] = -7.9086114475461606
        lats[1393] = -7.9789102159512737
        lats[1394] = -8.049208984355916
        lats[1395] = -8.1195077527600841
        lats[1396] = -8.1898065211637725
        lats[1397] = -8.2601052895669778
        lats[1398] = -8.3304040579696963
        lats[1399] = -8.4007028263719228
        lats[1400] = -8.4710015947736537
        lats[1401] = -8.5413003631748801
        lats[1402] = -8.6115991315756055
        lats[1403] = -8.6818978999758194
        lats[1404] = -8.7521966683755217
        lats[1405] = -8.8224954367747017
        lats[1406] = -8.8927942051733631
        lats[1407] = -8.963092973571495
        lats[1408] = -9.0333917419690941
        lats[1409] = -9.1036905103661585
        lats[1410] = -9.1739892787626829
        lats[1411] = -9.2442880471586619
        lats[1412] = -9.3145868155540921
        lats[1413] = -9.3848855839489662
        lats[1414] = -9.4551843523432826
        lats[1415] = -9.5254831207370376
        lats[1416] = -9.5957818891302242
        lats[1417] = -9.6660806575228388
        lats[1418] = -9.7363794259148779
        lats[1419] = -9.8066781943063344
        lats[1420] = -9.8769769626972046
        lats[1421] = -9.9472757310874869
        lats[1422] = -10.017574499477174
        lats[1423] = -10.087873267866264
        lats[1424] = -10.158172036254747
        lats[1425] = -10.228470804642624
        lats[1426] = -10.298769573029887
        lats[1427] = -10.369068341416533
        lats[1428] = -10.439367109802557
        lats[1429] = -10.509665878187954
        lats[1430] = -10.579964646572719
        lats[1431] = -10.65026341495685
        lats[1432] = -10.720562183340341
        lats[1433] = -10.790860951723188
        lats[1434] = -10.861159720105382
        lats[1435] = -10.931458488486923
        lats[1436] = -11.001757256867807
        lats[1437] = -11.072056025248026
        lats[1438] = -11.142354793627575
        lats[1439] = -11.212653562006453
        lats[1440] = -11.282952330384653
        lats[1441] = -11.35325109876217
        lats[1442] = -11.423549867139002
        lats[1443] = -11.493848635515141
        lats[1444] = -11.564147403890583
        lats[1445] = -11.634446172265324
        lats[1446] = -11.704744940639358
        lats[1447] = -11.775043709012685
        lats[1448] = -11.845342477385294
        lats[1449] = -11.915641245757183
        lats[1450] = -11.985940014128348
        lats[1451] = -12.056238782498781
        lats[1452] = -12.126537550868482
        lats[1453] = -12.196836319237443
        lats[1454] = -12.267135087605659
        lats[1455] = -12.337433855973126
        lats[1456] = -12.407732624339841
        lats[1457] = -12.478031392705796
        lats[1458] = -12.548330161070988
        lats[1459] = -12.618628929435411
        lats[1460] = -12.688927697799061
        lats[1461] = -12.759226466161934
        lats[1462] = -12.829525234524022
        lats[1463] = -12.899824002885323
        lats[1464] = -12.970122771245832
        lats[1465] = -13.040421539605545
        lats[1466] = -13.110720307964451
        lats[1467] = -13.181019076322551
        lats[1468] = -13.251317844679837
        lats[1469] = -13.32161661303631
        lats[1470] = -13.391915381391959
        lats[1471] = -13.46221414974678
        lats[1472] = -13.532512918100766
        lats[1473] = -13.602811686453919
        lats[1474] = -13.673110454806226
        lats[1475] = -13.743409223157688
        lats[1476] = -13.813707991508297
        lats[1477] = -13.884006759858046
        lats[1478] = -13.954305528206934
        lats[1479] = -14.024604296554955
        lats[1480] = -14.0949030649021
        lats[1481] = -14.165201833248371
        lats[1482] = -14.23550060159376
        lats[1483] = -14.305799369938256
        lats[1484] = -14.376098138281863
        lats[1485] = -14.446396906624567
        lats[1486] = -14.516695674966371
        lats[1487] = -14.586994443307265
        lats[1488] = -14.657293211647247
        lats[1489] = -14.727591979986309
        lats[1490] = -14.797890748324447
        lats[1491] = -14.868189516661655
        lats[1492] = -14.938488284997929
        lats[1493] = -15.008787053333259
        lats[1494] = -15.07908582166765
        lats[1495] = -15.149384590001089
        lats[1496] = -15.219683358333569
        lats[1497] = -15.289982126665089
        lats[1498] = -15.360280894995643
        lats[1499] = -15.430579663325226
        lats[1500] = -15.500878431653829
        lats[1501] = -15.571177199981456
        lats[1502] = -15.641475968308091
        lats[1503] = -15.711774736633735
        lats[1504] = -15.78207350495838
        lats[1505] = -15.852372273282016
        lats[1506] = -15.922671041604652
        lats[1507] = -15.992969809926265
        lats[1508] = -16.063268578246863
        lats[1509] = -16.133567346566434
        lats[1510] = -16.203866114884974
        lats[1511] = -16.274164883202477
        lats[1512] = -16.344463651518936
        lats[1513] = -16.41476241983435
        lats[1514] = -16.485061188148713
        lats[1515] = -16.555359956462013
        lats[1516] = -16.625658724774254
        lats[1517] = -16.69595749308542
        lats[1518] = -16.766256261395515
        lats[1519] = -16.836555029704527
        lats[1520] = -16.906853798012452
        lats[1521] = -16.977152566319283
        lats[1522] = -17.04745133462502
        lats[1523] = -17.117750102929655
        lats[1524] = -17.188048871233182
        lats[1525] = -17.258347639535586
        lats[1526] = -17.328646407836878
        lats[1527] = -17.39894517613704
        lats[1528] = -17.469243944436066
        lats[1529] = -17.539542712733962
        lats[1530] = -17.60984148103071
        lats[1531] = -17.680140249326314
        lats[1532] = -17.75043901762076
        lats[1533] = -17.820737785914044
        lats[1534] = -17.89103655420616
        lats[1535] = -17.96133532249711
        lats[1536] = -18.031634090786874
        lats[1537] = -18.101932859075458
        lats[1538] = -18.172231627362851
        lats[1539] = -18.242530395649048
        lats[1540] = -18.312829163934047
        lats[1541] = -18.383127932217832
        lats[1542] = -18.453426700500408
        lats[1543] = -18.523725468781763
        lats[1544] = -18.594024237061891
        lats[1545] = -18.664323005340787
        lats[1546] = -18.734621773618446
        lats[1547] = -18.804920541894862
        lats[1548] = -18.875219310170031
        lats[1549] = -18.945518078443939
        lats[1550] = -19.015816846716586
        lats[1551] = -19.086115614987968
        lats[1552] = -19.15641438325807
        lats[1553] = -19.226713151526898
        lats[1554] = -19.297011919794439
        lats[1555] = -19.367310688060684
        lats[1556] = -19.437609456325632
        lats[1557] = -19.507908224589269
        lats[1558] = -19.578206992851602
        lats[1559] = -19.648505761112613
        lats[1560] = -19.718804529372303
        lats[1561] = -19.789103297630657
        lats[1562] = -19.859402065887682
        lats[1563] = -19.929700834143357
        lats[1564] = -19.999999602397686
        lats[1565] = -20.070298370650661
        lats[1566] = -20.140597138902272
        lats[1567] = -20.210895907152516
        lats[1568] = -20.28119467540138
        lats[1569] = -20.35149344364887
        lats[1570] = -20.421792211894967
        lats[1571] = -20.492090980139672
        lats[1572] = -20.562389748382977
        lats[1573] = -20.632688516624874
        lats[1574] = -20.702987284865355
        lats[1575] = -20.773286053104417
        lats[1576] = -20.843584821342048
        lats[1577] = -20.913883589578251
        lats[1578] = -20.984182357813012
        lats[1579] = -21.054481126046323
        lats[1580] = -21.124779894278181
        lats[1581] = -21.195078662508585
        lats[1582] = -21.265377430737512
        lats[1583] = -21.335676198964972
        lats[1584] = -21.40597496719095
        lats[1585] = -21.47627373541544
        lats[1586] = -21.546572503638437
        lats[1587] = -21.616871271859928
        lats[1588] = -21.687170040079913
        lats[1589] = -21.757468808298391
        lats[1590] = -21.827767576515338
        lats[1591] = -21.898066344730758
        lats[1592] = -21.968365112944642
        lats[1593] = -22.038663881156989
        lats[1594] = -22.108962649367779
        lats[1595] = -22.179261417577013
        lats[1596] = -22.249560185784691
        lats[1597] = -22.319858953990789
        lats[1598] = -22.390157722195315
        lats[1599] = -22.460456490398254
        lats[1600] = -22.530755258599601
        lats[1601] = -22.60105402679935
        lats[1602] = -22.671352794997489
        lats[1603] = -22.741651563194019
        lats[1604] = -22.811950331388925
        lats[1605] = -22.882249099582204
        lats[1606] = -22.952547867773848
        lats[1607] = -23.022846635963852
        lats[1608] = -23.0931454041522
        lats[1609] = -23.163444172338895
        lats[1610] = -23.233742940523921
        lats[1611] = -23.304041708707278
        lats[1612] = -23.374340476888957
        lats[1613] = -23.444639245068949
        lats[1614] = -23.514938013247242
        lats[1615] = -23.585236781423838
        lats[1616] = -23.655535549598721
        lats[1617] = -23.725834317771888
        lats[1618] = -23.796133085943328
        lats[1619] = -23.866431854113038
        lats[1620] = -23.936730622281004
        lats[1621] = -24.007029390447226
        lats[1622] = -24.077328158611696
        lats[1623] = -24.1476269267744
        lats[1624] = -24.217925694935328
        lats[1625] = -24.288224463094483
        lats[1626] = -24.358523231251851
        lats[1627] = -24.428821999407425
        lats[1628] = -24.499120767561195
        lats[1629] = -24.569419535713152
        lats[1630] = -24.639718303863294
        lats[1631] = -24.710017072011613
        lats[1632] = -24.780315840158096
        lats[1633] = -24.850614608302738
        lats[1634] = -24.920913376445526
        lats[1635] = -24.991212144586456
        lats[1636] = -25.061510912725527
        lats[1637] = -25.13180968086272
        lats[1638] = -25.202108448998025
        lats[1639] = -25.272407217131445
        lats[1640] = -25.342705985262967
        lats[1641] = -25.413004753392578
        lats[1642] = -25.483303521520277
        lats[1643] = -25.553602289646051
        lats[1644] = -25.623901057769892
        lats[1645] = -25.694199825891793
        lats[1646] = -25.764498594011751
        lats[1647] = -25.834797362129745
        lats[1648] = -25.90509613024577
        lats[1649] = -25.975394898359827
        lats[1650] = -26.045693666471902
        lats[1651] = -26.115992434581983
        lats[1652] = -26.186291202690064
        lats[1653] = -26.256589970796135
        lats[1654] = -26.326888738900195
        lats[1655] = -26.397187507002222
        lats[1656] = -26.467486275102218
        lats[1657] = -26.53778504320017
        lats[1658] = -26.608083811296069
        lats[1659] = -26.678382579389908
        lats[1660] = -26.748681347481678
        lats[1661] = -26.818980115571364
        lats[1662] = -26.889278883658971
        lats[1663] = -26.959577651744471
        lats[1664] = -27.029876419827872
        lats[1665] = -27.100175187909159
        lats[1666] = -27.170473955988321
        lats[1667] = -27.240772724065348
        lats[1668] = -27.311071492140236
        lats[1669] = -27.381370260212968
        lats[1670] = -27.451669028283543
        lats[1671] = -27.521967796351948
        lats[1672] = -27.592266564418171
        lats[1673] = -27.662565332482213
        lats[1674] = -27.732864100544052
        lats[1675] = -27.803162868603682
        lats[1676] = -27.873461636661098
        lats[1677] = -27.94376040471629
        lats[1678] = -28.014059172769244
        lats[1679] = -28.084357940819952
        lats[1680] = -28.154656708868405
        lats[1681] = -28.224955476914594
        lats[1682] = -28.29525424495851
        lats[1683] = -28.365553013000145
        lats[1684] = -28.435851781039485
        lats[1685] = -28.506150549076519
        lats[1686] = -28.576449317111244
        lats[1687] = -28.646748085143642
        lats[1688] = -28.717046853173709
        lats[1689] = -28.787345621201432
        lats[1690] = -28.857644389226806
        lats[1691] = -28.927943157249814
        lats[1692] = -28.998241925270449
        lats[1693] = -29.068540693288696
        lats[1694] = -29.138839461304556
        lats[1695] = -29.209138229318015
        lats[1696] = -29.279436997329057
        lats[1697] = -29.349735765337677
        lats[1698] = -29.420034533343859
        lats[1699] = -29.490333301347597
        lats[1700] = -29.560632069348884
        lats[1701] = -29.630930837347698
        lats[1702] = -29.701229605344039
        lats[1703] = -29.771528373337894
        lats[1704] = -29.841827141329258
        lats[1705] = -29.91212590931811
        lats[1706] = -29.98242467730444
        lats[1707] = -30.052723445288244
        lats[1708] = -30.123022213269511
        lats[1709] = -30.19332098124822
        lats[1710] = -30.263619749224372
        lats[1711] = -30.333918517197947
        lats[1712] = -30.404217285168947
        lats[1713] = -30.47451605313735
        lats[1714] = -30.544814821103138
        lats[1715] = -30.615113589066322
        lats[1716] = -30.685412357026873
        lats[1717] = -30.755711124984781
        lats[1718] = -30.826009892940046
        lats[1719] = -30.896308660892647
        lats[1720] = -30.966607428842572
        lats[1721] = -31.036906196789811
        lats[1722] = -31.107204964734358
        lats[1723] = -31.177503732676204
        lats[1724] = -31.247802500615318
        lats[1725] = -31.318101268551715
        lats[1726] = -31.388400036485361
        lats[1727] = -31.458698804416255
        lats[1728] = -31.528997572344384
        lats[1729] = -31.599296340269738
        lats[1730] = -31.669595108192297
        lats[1731] = -31.739893876112063
        lats[1732] = -31.810192644029012
        lats[1733] = -31.880491411943137
        lats[1734] = -31.950790179854422
        lats[1735] = -32.021088947762863
        lats[1736] = -32.091387715668439
        lats[1737] = -32.161686483571145
        lats[1738] = -32.231985251470959
        lats[1739] = -32.302284019367875
        lats[1740] = -32.372582787261891
        lats[1741] = -32.442881555152965
        lats[1742] = -32.513180323041112
        lats[1743] = -32.583479090926325
        lats[1744] = -32.653777858808567
        lats[1745] = -32.724076626687825
        lats[1746] = -32.794375394564113
        lats[1747] = -32.864674162437396
        lats[1748] = -32.934972930307666
        lats[1749] = -33.005271698174909
        lats[1750] = -33.075570466039117
        lats[1751] = -33.145869233900278
        lats[1752] = -33.216168001758369
        lats[1753] = -33.286466769613391
        lats[1754] = -33.356765537465314
        lats[1755] = -33.42706430531414
        lats[1756] = -33.497363073159853
        lats[1757] = -33.567661841002426
        lats[1758] = -33.637960608841851
        lats[1759] = -33.708259376678136
        lats[1760] = -33.778558144511237
        lats[1761] = -33.848856912341155
        lats[1762] = -33.919155680167876
        lats[1763] = -33.989454447991392
        lats[1764] = -34.059753215811682
        lats[1765] = -34.130051983628725
        lats[1766] = -34.200350751442521
        lats[1767] = -34.270649519253041
        lats[1768] = -34.340948287060286
        lats[1769] = -34.411247054864234
        lats[1770] = -34.481545822664863
        lats[1771] = -34.551844590462188
        lats[1772] = -34.622143358256153
        lats[1773] = -34.692442126046771
        lats[1774] = -34.762740893834028
        lats[1775] = -34.833039661617903
        lats[1776] = -34.903338429398374
        lats[1777] = -34.973637197175435
        lats[1778] = -35.043935964949064
        lats[1779] = -35.114234732719261
        lats[1780] = -35.184533500486005
        lats[1781] = -35.254832268249267
        lats[1782] = -35.325131036009047
        lats[1783] = -35.395429803765317
        lats[1784] = -35.465728571518085
        lats[1785] = -35.536027339267314
        lats[1786] = -35.606326107012997
        lats[1787] = -35.676624874755113
        lats[1788] = -35.746923642493655
        lats[1789] = -35.817222410228595
        lats[1790] = -35.887521177959933
        lats[1791] = -35.957819945687639
        lats[1792] = -36.028118713411708
        lats[1793] = -36.098417481132117
        lats[1794] = -36.16871624884886
        lats[1795] = -36.239015016561908
        lats[1796] = -36.309313784271254
        lats[1797] = -36.379612551976876
        lats[1798] = -36.449911319678755
        lats[1799] = -36.520210087376888
        lats[1800] = -36.590508855071242
        lats[1801] = -36.660807622761808
        lats[1802] = -36.731106390448581
        lats[1803] = -36.801405158131523
        lats[1804] = -36.871703925810628
        lats[1805] = -36.942002693485883
        lats[1806] = -37.012301461157264
        lats[1807] = -37.082600228824752
        lats[1808] = -37.152898996488332
        lats[1809] = -37.223197764147997
        lats[1810] = -37.293496531803719
        lats[1811] = -37.363795299455489
        lats[1812] = -37.434094067103274
        lats[1813] = -37.504392834747065
        lats[1814] = -37.574691602386856
        lats[1815] = -37.644990370022605
        lats[1816] = -37.715289137654317
        lats[1817] = -37.785587905281965
        lats[1818] = -37.855886672905527
        lats[1819] = -37.926185440524989
        lats[1820] = -37.99648420814033
        lats[1821] = -38.066782975751536
        lats[1822] = -38.137081743358586
        lats[1823] = -38.20738051096145
        lats[1824] = -38.277679278560143
        lats[1825] = -38.347978046154608
        lats[1826] = -38.418276813744846
        lats[1827] = -38.488575581330842
        lats[1828] = -38.558874348912568
        lats[1829] = -38.629173116490001
        lats[1830] = -38.699471884063136
        lats[1831] = -38.769770651631937
        lats[1832] = -38.840069419196389
        lats[1833] = -38.910368186756479
        lats[1834] = -38.980666954312184
        lats[1835] = -39.050965721863491
        lats[1836] = -39.121264489410365
        lats[1837] = -39.191563256952804
        lats[1838] = -39.261862024490775
        lats[1839] = -39.332160792024254
        lats[1840] = -39.402459559553229
        lats[1841] = -39.472758327077692
        lats[1842] = -39.543057094597607
        lats[1843] = -39.613355862112947
        lats[1844] = -39.683654629623703
        lats[1845] = -39.753953397129855
        lats[1846] = -39.824252164631375
        lats[1847] = -39.894550932128247
        lats[1848] = -39.964849699620437
        lats[1849] = -40.035148467107952
        lats[1850] = -40.105447234590748
        lats[1851] = -40.175746002068806
        lats[1852] = -40.246044769542102
        lats[1853] = -40.316343537010617
        lats[1854] = -40.386642304474343
        lats[1855] = -40.456941071933244
        lats[1856] = -40.527239839387299
        lats[1857] = -40.597538606836487
        lats[1858] = -40.667837374280786
        lats[1859] = -40.738136141720176
        lats[1860] = -40.808434909154634
        lats[1861] = -40.878733676584126
        lats[1862] = -40.949032444008644
        lats[1863] = -41.01933121142816
        lats[1864] = -41.089629978842645
        lats[1865] = -41.159928746252085
        lats[1866] = -41.230227513656445
        lats[1867] = -41.300526281055724
        lats[1868] = -41.370825048449873
        lats[1869] = -41.441123815838885
        lats[1870] = -41.511422583222718
        lats[1871] = -41.581721350601363
        lats[1872] = -41.6520201179748
        lats[1873] = -41.722318885343
        lats[1874] = -41.792617652705921
        lats[1875] = -41.862916420063563
        lats[1876] = -41.933215187415882
        lats[1877] = -42.003513954762873
        lats[1878] = -42.073812722104492
        lats[1879] = -42.144111489440725
        lats[1880] = -42.214410256771551
        lats[1881] = -42.284709024096927
        lats[1882] = -42.355007791416853
        lats[1883] = -42.425306558731272
        lats[1884] = -42.495605326040177
        lats[1885] = -42.565904093343548
        lats[1886] = -42.63620286064134
        lats[1887] = -42.706501627933541
        lats[1888] = -42.776800395220121
        lats[1889] = -42.847099162501053
        lats[1890] = -42.917397929776307
        lats[1891] = -42.987696697045862
        lats[1892] = -43.057995464309691
        lats[1893] = -43.128294231567757
        lats[1894] = -43.19859299882004
        lats[1895] = -43.26889176606651
        lats[1896] = -43.339190533307139
        lats[1897] = -43.409489300541907
        lats[1898] = -43.479788067770777
        lats[1899] = -43.550086834993728
        lats[1900] = -43.620385602210717
        lats[1901] = -43.690684369421732
        lats[1902] = -43.760983136626741
        lats[1903] = -43.831281903825705
        lats[1904] = -43.9015806710186
        lats[1905] = -43.971879438205391
        lats[1906] = -44.042178205386072
        lats[1907] = -44.112476972560586
        lats[1908] = -44.182775739728925
        lats[1909] = -44.253074506891046
        lats[1910] = -44.323373274046915
        lats[1911] = -44.39367204119651
        lats[1912] = -44.463970808339802
        lats[1913] = -44.534269575476756
        lats[1914] = -44.604568342607337
        lats[1915] = -44.674867109731515
        lats[1916] = -44.745165876849271
        lats[1917] = -44.81546464396056
        lats[1918] = -44.885763411065362
        lats[1919] = -44.956062178163634
        lats[1920] = -45.026360945255341
        lats[1921] = -45.096659712340461
        lats[1922] = -45.166958479418959
        lats[1923] = -45.237257246490813
        lats[1924] = -45.30755601355596
        lats[1925] = -45.377854780614399
        lats[1926] = -45.448153547666081
        lats[1927] = -45.51845231471097
        lats[1928] = -45.588751081749038
        lats[1929] = -45.659049848780263
        lats[1930] = -45.729348615804589
        lats[1931] = -45.799647382821995
        lats[1932] = -45.869946149832437
        lats[1933] = -45.94024491683588
        lats[1934] = -46.01054368383231
        lats[1935] = -46.080842450821663
        lats[1936] = -46.151141217803925
        lats[1937] = -46.221439984779053
        lats[1938] = -46.291738751747012
        lats[1939] = -46.362037518707766
        lats[1940] = -46.432336285661272
        lats[1941] = -46.502635052607502
        lats[1942] = -46.572933819546414
        lats[1943] = -46.643232586477971
        lats[1944] = -46.713531353402139
        lats[1945] = -46.783830120318882
        lats[1946] = -46.85412888722815
        lats[1947] = -46.924427654129929
        lats[1948] = -46.994726421024154
        lats[1949] = -47.065025187910805
        lats[1950] = -47.13532395478984
        lats[1951] = -47.205622721661214
        lats[1952] = -47.275921488524894
        lats[1953] = -47.346220255380835
        lats[1954] = -47.416519022228997
        lats[1955] = -47.486817789069342
        lats[1956] = -47.557116555901828
        lats[1957] = -47.627415322726435
        lats[1958] = -47.697714089543084
        lats[1959] = -47.76801285635176
        lats[1960] = -47.838311623152421
        lats[1961] = -47.908610389945018
        lats[1962] = -47.978909156729507
        lats[1963] = -48.049207923505868
        lats[1964] = -48.119506690274015
        lats[1965] = -48.189805457033941
        lats[1966] = -48.260104223785596
        lats[1967] = -48.330402990528938
        lats[1968] = -48.400701757263917
        lats[1969] = -48.47100052399049
        lats[1970] = -48.541299290708608
        lats[1971] = -48.611598057418242
        lats[1972] = -48.681896824119335
        lats[1973] = -48.752195590811837
        lats[1974] = -48.822494357495721
        lats[1975] = -48.892793124170929
        lats[1976] = -48.963091890837418
        lats[1977] = -49.03339065749514
        lats[1978] = -49.103689424144044
        lats[1979] = -49.173988190784094
        lats[1980] = -49.244286957415234
        lats[1981] = -49.314585724037435
        lats[1982] = -49.384884490650613
        lats[1983] = -49.455183257254745
        lats[1984] = -49.525482023849783
        lats[1985] = -49.595780790435676
        lats[1986] = -49.66607955701236
        lats[1987] = -49.736378323579807
        lats[1988] = -49.80667709013796
        lats[1989] = -49.876975856686762
        lats[1990] = -49.947274623226157
        lats[1991] = -50.017573389756123
        lats[1992] = -50.087872156276575
        lats[1993] = -50.158170922787484
        lats[1994] = -50.228469689288779
        lats[1995] = -50.298768455780426
        lats[1996] = -50.369067222262359
        lats[1997] = -50.439365988734544
        lats[1998] = -50.509664755196901
        lats[1999] = -50.579963521649397
        lats[2000] = -50.650262288091959
        lats[2001] = -50.720561054524559
        lats[2002] = -50.790859820947119
        lats[2003] = -50.86115858735959
        lats[2004] = -50.931457353761914
        lats[2005] = -51.001756120154049
        lats[2006] = -51.072054886535909
        lats[2007] = -51.14235365290746
        lats[2008] = -51.21265241926865
        lats[2009] = -51.282951185619417
        lats[2010] = -51.353249951959683
        lats[2011] = -51.423548718289396
        lats[2012] = -51.493847484608516
        lats[2013] = -51.56414625091697
        lats[2014] = -51.634445017214695
        lats[2015] = -51.704743783501634
        lats[2016] = -51.775042549777737
        lats[2017] = -51.845341316042933
        lats[2018] = -51.915640082297152
        lats[2019] = -51.985938848540336
        lats[2020] = -52.056237614772435
        lats[2021] = -52.126536380993372
        lats[2022] = -52.196835147203096
        lats[2023] = -52.26713391340153
        lats[2024] = -52.337432679588609
        lats[2025] = -52.407731445764284
        lats[2026] = -52.478030211928477
        lats[2027] = -52.548328978081123
        lats[2028] = -52.618627744222159
        lats[2029] = -52.688926510351514
        lats[2030] = -52.759225276469131
        lats[2031] = -52.829524042574917
        lats[2032] = -52.899822808668837
        lats[2033] = -52.970121574750792
        lats[2034] = -53.040420340820731
        lats[2035] = -53.110719106878584
        lats[2036] = -53.181017872924265
        lats[2037] = -53.251316638957725
        lats[2038] = -53.321615404978871
        lats[2039] = -53.391914170987633
        lats[2040] = -53.462212936983953
        lats[2041] = -53.53251170296776
        lats[2042] = -53.602810468938962
        lats[2043] = -53.673109234897495
        lats[2044] = -53.743408000843282
        lats[2045] = -53.813706766776235
        lats[2046] = -53.884005532696307
        lats[2047] = -53.954304298603383
        lats[2048] = -54.024603064497434
        lats[2049] = -54.094901830378333
        lats[2050] = -54.165200596246031
        lats[2051] = -54.235499362100448
        lats[2052] = -54.305798127941479
        lats[2053] = -54.376096893769081
        lats[2054] = -54.446395659583146
        lats[2055] = -54.516694425383605
        lats[2056] = -54.586993191170357
        lats[2057] = -54.657291956943347
        lats[2058] = -54.727590722702473
        lats[2059] = -54.797889488447652
        lats[2060] = -54.868188254178797
        lats[2061] = -54.938487019895831
        lats[2062] = -55.008785785598668
        lats[2063] = -55.07908455128721
        lats[2064] = -55.149383316961377
        lats[2065] = -55.219682082621084
        lats[2066] = -55.289980848266232
        lats[2067] = -55.360279613896743
        lats[2068] = -55.430578379512511
        lats[2069] = -55.500877145113449
        lats[2070] = -55.571175910699488
        lats[2071] = -55.641474676270505
        lats[2072] = -55.711773441826416
        lats[2073] = -55.782072207367136
        lats[2074] = -55.852370972892551
        lats[2075] = -55.922669738402583
        lats[2076] = -55.992968503897131
        lats[2077] = -56.063267269376091
        lats[2078] = -56.133566034839362
        lats[2079] = -56.203864800286865
        lats[2080] = -56.274163565718467
        lats[2081] = -56.34446233113411
        lats[2082] = -56.41476109653366
        lats[2083] = -56.485059861917016
        lats[2084] = -56.555358627284086
        lats[2085] = -56.625657392634771
        lats[2086] = -56.695956157968951
        lats[2087] = -56.766254923286517
        lats[2088] = -56.836553688587379
        lats[2089] = -56.90685245387143
        lats[2090] = -56.977151219138541
        lats[2091] = -57.047449984388614
        lats[2092] = -57.117748749621541
        lats[2093] = -57.188047514837208
        lats[2094] = -57.258346280035504
        lats[2095] = -57.328645045216312
        lats[2096] = -57.398943810379521
        lats[2097] = -57.469242575525016
        lats[2098] = -57.539541340652676
        lats[2099] = -57.60984010576238
        lats[2100] = -57.680138870854037
        lats[2101] = -57.75043763592749
        lats[2102] = -57.820736400982646
        lats[2103] = -57.891035166019364
        lats[2104] = -57.961333931037537
        lats[2105] = -58.031632696037022
        lats[2106] = -58.101931461017728
        lats[2107] = -58.172230225979497
        lats[2108] = -58.242528990922203
        lats[2109] = -58.312827755845746
        lats[2110] = -58.383126520749968
        lats[2111] = -58.453425285634758
        lats[2112] = -58.523724050499972
        lats[2113] = -58.594022815345468
        lats[2114] = -58.664321580171141
        lats[2115] = -58.73462034497684
        lats[2116] = -58.804919109762423
        lats[2117] = -58.875217874527763
        lats[2118] = -58.945516639272725
        lats[2119] = -59.015815403997145
        lats[2120] = -59.086114168700909
        lats[2121] = -59.156412933383855
        lats[2122] = -59.226711698045854
        lats[2123] = -59.29701046268675
        lats[2124] = -59.3673092273064
        lats[2125] = -59.43760799190467
        lats[2126] = -59.507906756481383
        lats[2127] = -59.578205521036402
        lats[2128] = -59.64850428556958
        lats[2129] = -59.718803050080759
        lats[2130] = -59.78910181456979
        lats[2131] = -59.859400579036503
        lats[2132] = -59.929699343480763
        lats[2133] = -59.999998107902378
        lats[2134] = -60.070296872301235
        lats[2135] = -60.140595636677112
        lats[2136] = -60.21089440102989
        lats[2137] = -60.28119316535939
        lats[2138] = -60.35149192966545
        lats[2139] = -60.421790693947884
        lats[2140] = -60.492089458206543
        lats[2141] = -60.562388222441243
        lats[2142] = -60.632686986651805
        lats[2143] = -60.702985750838074
        lats[2144] = -60.773284514999872
        lats[2145] = -60.843583279137007
        lats[2146] = -60.913882043249295
        lats[2147] = -60.984180807336578
        lats[2148] = -61.054479571398652
        lats[2149] = -61.124778335435344
        lats[2150] = -61.195077099446451
        lats[2151] = -61.265375863431785
        lats[2152] = -61.335674627391185
        lats[2153] = -61.405973391324409
        lats[2154] = -61.476272155231321
        lats[2155] = -61.546570919111666
        lats[2156] = -61.616869682965287
        lats[2157] = -61.687168446791986
        lats[2158] = -61.757467210591535
        lats[2159] = -61.827765974363729
        lats[2160] = -61.898064738108381
        lats[2161] = -61.968363501825259
        lats[2162] = -62.038662265514176
        lats[2163] = -62.108961029174914
        lats[2164] = -62.179259792807258
        lats[2165] = -62.249558556410982
        lats[2166] = -62.319857319985871
        lats[2167] = -62.3901560835317
        lats[2168] = -62.460454847048261
        lats[2169] = -62.530753610535321
        lats[2170] = -62.60105237399263
        lats[2171] = -62.67135113741999
        lats[2172] = -62.741649900817137
        lats[2173] = -62.811948664183866
        lats[2174] = -62.882247427519928
        lats[2175] = -62.952546190825068
        lats[2176] = -63.022844954099064
        lats[2177] = -63.093143717341647
        lats[2178] = -63.163442480552604
        lats[2179] = -63.23374124373165
        lats[2180] = -63.304040006878537
        lats[2181] = -63.374338769993031
        lats[2182] = -63.444637533074854
        lats[2183] = -63.514936296123757
        lats[2184] = -63.585235059139464
        lats[2185] = -63.655533822121711
        lats[2186] = -63.725832585070251
        lats[2187] = -63.796131347984762
        lats[2188] = -63.866430110865004
        lats[2189] = -63.93672887371072
        lats[2190] = -64.00702763652157
        lats[2191] = -64.07732639929732
        lats[2192] = -64.147625162037642
        lats[2193] = -64.21792392474228
        lats[2194] = -64.288222687410922
        lats[2195] = -64.358521450043284
        lats[2196] = -64.428820212639039
        lats[2197] = -64.499118975197902
        lats[2198] = -64.569417737719576
        lats[2199] = -64.639716500203733
        lats[2200] = -64.710015262650074
        lats[2201] = -64.780314025058246
        lats[2202] = -64.850612787427963
        lats[2203] = -64.920911549758912
        lats[2204] = -64.991210312050711
        lats[2205] = -65.061509074303089
        lats[2206] = -65.131807836515677
        lats[2207] = -65.202106598688133
        lats[2208] = -65.272405360820116
        lats[2209] = -65.342704122911286
        lats[2210] = -65.413002884961315
        lats[2211] = -65.483301646969792
        lats[2212] = -65.553600408936404
        lats[2213] = -65.623899170860767
        lats[2214] = -65.694197932742526
        lats[2215] = -65.764496694581283
        lats[2216] = -65.834795456376696
        lats[2217] = -65.905094218128355
        lats[2218] = -65.975392979835888
        lats[2219] = -66.045691741498899
        lats[2220] = -66.115990503117033
        lats[2221] = -66.186289264689833
        lats[2222] = -66.256588026216932
        lats[2223] = -66.326886787697887
        lats[2224] = -66.397185549132331
        lats[2225] = -66.467484310519808
        lats[2226] = -66.537783071859891
        lats[2227] = -66.608081833152212
        lats[2228] = -66.678380594396273
        lats[2229] = -66.748679355591662
        lats[2230] = -66.818978116737924
        lats[2231] = -66.889276877834618
        lats[2232] = -66.95957563888129
        lats[2233] = -67.029874399877471
        lats[2234] = -67.100173160822706
        lats[2235] = -67.170471921716526
        lats[2236] = -67.240770682558434
        lats[2237] = -67.311069443347961
        lats[2238] = -67.381368204084609
        lats[2239] = -67.451666964767895
        lats[2240] = -67.521965725397308
        lats[2241] = -67.592264485972336
        lats[2242] = -67.662563246492482
        lats[2243] = -67.732862006957205
        lats[2244] = -67.803160767365966
        lats[2245] = -67.873459527718282
        lats[2246] = -67.943758288013555
        lats[2247] = -68.014057048251274
        lats[2248] = -68.084355808430871
        lats[2249] = -68.154654568551791
        lats[2250] = -68.224953328613438
        lats[2251] = -68.295252088615257
        lats[2252] = -68.365550848556666
        lats[2253] = -68.435849608437067
        lats[2254] = -68.506148368255865
        lats[2255] = -68.576447128012447
        lats[2256] = -68.646745887706189
        lats[2257] = -68.717044647336493
        lats[2258] = -68.787343406902693
        lats[2259] = -68.85764216640419
        lats[2260] = -68.927940925840304
        lats[2261] = -68.998239685210365
        lats[2262] = -69.068538444513763
        lats[2263] = -69.138837203749759
        lats[2264] = -69.209135962917699
        lats[2265] = -69.279434722016902
        lats[2266] = -69.349733481046613
        lats[2267] = -69.420032240006194
        lats[2268] = -69.490330998894862
        lats[2269] = -69.560629757711908
        lats[2270] = -69.630928516456592
        lats[2271] = -69.701227275128161
        lats[2272] = -69.771526033725834
        lats[2273] = -69.841824792248843
        lats[2274] = -69.912123550696421
        lats[2275] = -69.982422309067744
        lats[2276] = -70.052721067362043
        lats[2277] = -70.123019825578467
        lats[2278] = -70.193318583716191
        lats[2279] = -70.263617341774406
        lats[2280] = -70.333916099752187
        lats[2281] = -70.404214857648739
        lats[2282] = -70.474513615463138
        lats[2283] = -70.544812373194532
        lats[2284] = -70.615111130841967
        lats[2285] = -70.685409888404578
        lats[2286] = -70.755708645881384
        lats[2287] = -70.826007403271475
        lats[2288] = -70.896306160573886
        lats[2289] = -70.966604917787635
        lats[2290] = -71.036903674911756
        lats[2291] = -71.107202431945211
        lats[2292] = -71.177501188887007
        lats[2293] = -71.247799945736105
        lats[2294] = -71.318098702491469
        lats[2295] = -71.388397459152031
        lats[2296] = -71.458696215716685
        lats[2297] = -71.528994972184378
        lats[2298] = -71.599293728553988
        lats[2299] = -71.669592484824364
        lats[2300] = -71.739891240994368
        lats[2301] = -71.810189997062835
        lats[2302] = -71.880488753028587
        lats[2303] = -71.950787508890414
        lats[2304] = -72.02108626464711
        lats[2305] = -72.091385020297409
        lats[2306] = -72.161683775840089
        lats[2307] = -72.231982531273843
        lats[2308] = -72.302281286597392
        lats[2309] = -72.3725800418094
        lats[2310] = -72.442878796908545
        lats[2311] = -72.513177551893421
        lats[2312] = -72.583476306762691
        lats[2313] = -72.653775061514935
        lats[2314] = -72.724073816148703
        lats[2315] = -72.794372570662574
        lats[2316] = -72.864671325055056
        lats[2317] = -72.934970079324657
        lats[2318] = -73.005268833469799
        lats[2319] = -73.075567587489019
        lats[2320] = -73.145866341380668
        lats[2321] = -73.216165095143182
        lats[2322] = -73.2864638487749
        lats[2323] = -73.356762602274188
        lats[2324] = -73.427061355639339
        lats[2325] = -73.497360108868662
        lats[2326] = -73.567658861960396
        lats[2327] = -73.637957614912779
        lats[2328] = -73.70825636772399
        lats[2329] = -73.778555120392184
        lats[2330] = -73.848853872915541
        lats[2331] = -73.919152625292114
        lats[2332] = -73.98945137751997
        lats[2333] = -74.059750129597163
        lats[2334] = -74.13004888152166
        lats[2335] = -74.200347633291472
        lats[2336] = -74.270646384904481
        lats[2337] = -74.340945136358584
        lats[2338] = -74.411243887651622
        lats[2339] = -74.481542638781434
        lats[2340] = -74.551841389745761
        lats[2341] = -74.622140140542356
        lats[2342] = -74.692438891168877
        lats[2343] = -74.762737641622991
        lats[2344] = -74.833036391902269
        lats[2345] = -74.903335142004323
        lats[2346] = -74.973633891926625
        lats[2347] = -75.043932641666672
        lats[2348] = -75.114231391221821
        lats[2349] = -75.184530140589501
        lats[2350] = -75.254828889766983
        lats[2351] = -75.325127638751567
        lats[2352] = -75.395426387540439
        lats[2353] = -75.465725136130786
        lats[2354] = -75.536023884519707
        lats[2355] = -75.60632263270422
        lats[2356] = -75.67662138068134
        lats[2357] = -75.746920128447996
        lats[2358] = -75.81721887600105
        lats[2359] = -75.887517623337317
        lats[2360] = -75.957816370453543
        lats[2361] = -76.028115117346374
        lats[2362] = -76.098413864012443
        lats[2363] = -76.16871261044831
        lats[2364] = -76.239011356650423
        lats[2365] = -76.3093101026152
        lats[2366] = -76.379608848338933
        lats[2367] = -76.449907593817869
        lats[2368] = -76.520206339048215
        lats[2369] = -76.59050508402602
        lats[2370] = -76.660803828747362
        lats[2371] = -76.731102573208048
        lats[2372] = -76.801401317404
        lats[2373] = -76.871700061330955
        lats[2374] = -76.941998804984564
        lats[2375] = -77.012297548360323
        lats[2376] = -77.082596291453768
        lats[2377] = -77.15289503426024
        lats[2378] = -77.22319377677502
        lats[2379] = -77.293492518993247
        lats[2380] = -77.363791260909963
        lats[2381] = -77.434090002520122
        lats[2382] = -77.504388743818524
        lats[2383] = -77.574687484799924
        lats[2384] = -77.644986225458879
        lats[2385] = -77.71528496578982
        lats[2386] = -77.785583705787161
        lats[2387] = -77.855882445445019
        lats[2388] = -77.926181184757539
        lats[2389] = -77.996479923718596
        lats[2390] = -78.066778662322022
        lats[2391] = -78.137077400561424
        lats[2392] = -78.207376138430348
        lats[2393] = -78.277674875922045
        lats[2394] = -78.347973613029708
        lats[2395] = -78.418272349746417
        lats[2396] = -78.488571086064923
        lats[2397] = -78.558869821977908
        lats[2398] = -78.629168557477882
        lats[2399] = -78.699467292557102
        lats[2400] = -78.769766027207638
        lats[2401] = -78.840064761421445
        lats[2402] = -78.910363495190211
        lats[2403] = -78.980662228505423
        lats[2404] = -79.050960961358285
        lats[2405] = -79.121259693739859
        lats[2406] = -79.191558425640977
        lats[2407] = -79.261857157052191
        lats[2408] = -79.332155887963822
        lats[2409] = -79.402454618365894
        lats[2410] = -79.472753348248219
        lats[2411] = -79.543052077600308
        lats[2412] = -79.61335080641139
        lats[2413] = -79.683649534670437
        lats[2414] = -79.753948262366038
        lats[2415] = -79.824246989486554
        lats[2416] = -79.894545716019948
        lats[2417] = -79.9648444419539
        lats[2418] = -80.035143167275749
        lats[2419] = -80.105441891972376
        lats[2420] = -80.175740616030438
        lats[2421] = -80.246039339436052
        lats[2422] = -80.316338062175078
        lats[2423] = -80.386636784232863
        lats[2424] = -80.456935505594302
        lats[2425] = -80.527234226243991
        lats[2426] = -80.59753294616587
        lats[2427] = -80.667831665343556
        lats[2428] = -80.73813038376008
        lats[2429] = -80.808429101397948
        lats[2430] = -80.878727818239184
        lats[2431] = -80.949026534265244
        lats[2432] = -81.019325249456955
        lats[2433] = -81.089623963794551
        lats[2434] = -81.159922677257711
        lats[2435] = -81.230221389825374
        lats[2436] = -81.300520101475826
        lats[2437] = -81.370818812186627
        lats[2438] = -81.441117521934686
        lats[2439] = -81.511416230696042
        lats[2440] = -81.581714938445955
        lats[2441] = -81.652013645158945
        lats[2442] = -81.722312350808508
        lats[2443] = -81.792611055367345
        lats[2444] = -81.862909758807191
        lats[2445] = -81.933208461098829
        lats[2446] = -82.003507162211946
        lats[2447] = -82.073805862115165
        lats[2448] = -82.144104560776
        lats[2449] = -82.214403258160871
        lats[2450] = -82.284701954234833
        lats[2451] = -82.355000648961692
        lats[2452] = -82.425299342304029
        lats[2453] = -82.495598034222837
        lats[2454] = -82.56589672467787
        lats[2455] = -82.63619541362705
        lats[2456] = -82.706494101026948
        lats[2457] = -82.77679278683226
        lats[2458] = -82.84709147099602
        lats[2459] = -82.917390153469313
        lats[2460] = -82.987688834201322
        lats[2461] = -83.057987513139125
        lats[2462] = -83.128286190227698
        lats[2463] = -83.198584865409657
        lats[2464] = -83.268883538625232
        lats[2465] = -83.339182209812321
        lats[2466] = -83.409480878905782
        lats[2467] = -83.479779545838113
        lats[2468] = -83.550078210538487
        lats[2469] = -83.620376872933264
        lats[2470] = -83.690675532945292
        lats[2471] = -83.760974190494011
        lats[2472] = -83.831272845495249
        lats[2473] = -83.901571497860914
        lats[2474] = -83.971870147498763
        lats[2475] = -84.042168794312317
        lats[2476] = -84.112467438200326
        lats[2477] = -84.18276607905679
        lats[2478] = -84.253064716770425
        lats[2479] = -84.323363351224444
        lats[2480] = -84.393661982296322
        lats[2481] = -84.463960609857125
        lats[2482] = -84.534259233771479
        lats[2483] = -84.604557853896708
        lats[2484] = -84.674856470082915
        lats[2485] = -84.745155082171991
        lats[2486] = -84.81545368999717
        lats[2487] = -84.885752293382765
        lats[2488] = -84.95605089214304
        lats[2489] = -85.026349486081983
        lats[2490] = -85.09664807499216
        lats[2491] = -85.16694665865414
        lats[2492] = -85.237245236835548
        lats[2493] = -85.307543809290152
        lats[2494] = -85.377842375756586
        lats[2495] = -85.448140935957483
        lats[2496] = -85.518439489597966
        lats[2497] = -85.588738036364362
        lats[2498] = -85.659036575922883
        lats[2499] = -85.729335107917464
        lats[2500] = -85.799633631968391
        lats[2501] = -85.869932147670127
        lats[2502] = -85.940230654588888
        lats[2503] = -86.010529152260403
        lats[2504] = -86.080827640187209
        lats[2505] = -86.151126117835304
        lats[2506] = -86.221424584631109
        lats[2507] = -86.291723039957418
        lats[2508] = -86.362021483149363
        lats[2509] = -86.432319913489792
        lats[2510] = -86.502618330203831
        lats[2511] = -86.572916732453024
        lats[2512] = -86.643215119328573
        lats[2513] = -86.713513489844246
        lats[2514] = -86.783811842927179
        lats[2515] = -86.854110177408927
        lats[2516] = -86.924408492014166
        lats[2517] = -86.994706785348129
        lats[2518] = -87.065005055882821
        lats[2519] = -87.135303301939786
        lats[2520] = -87.205601521672108
        lats[2521] = -87.275899713041966
        lats[2522] = -87.346197873795816
        lats[2523] = -87.416496001434894
        lats[2524] = -87.486794093180748
        lats[2525] = -87.557092145935584
        lats[2526] = -87.627390156234085
        lats[2527] = -87.697688120188062
        lats[2528] = -87.767986033419561
        lats[2529] = -87.838283890981543
        lats[2530] = -87.908581687261687
        lats[2531] = -87.978879415867283
        lats[2532] = -88.049177069484486
        lats[2533] = -88.119474639706425
        lats[2534] = -88.189772116820762
        lats[2535] = -88.26006948954614
        lats[2536] = -88.330366744702559
        lats[2537] = -88.40066386679355
        lats[2538] = -88.470960837474877
        lats[2539] = -88.541257634868515
        lats[2540] = -88.611554232668382
        lats[2541] = -88.681850598961759
        lats[2542] = -88.752146694650691
        lats[2543] = -88.822442471310097
        lats[2544] = -88.892737868230952
        lats[2545] = -88.96303280826325
        lats[2546] = -89.033327191845927
        lats[2547] = -89.103620888238879
        lats[2548] = -89.173913722284126
        lats[2549] = -89.24420545380525
        lats[2550] = -89.314495744374256
        lats[2551] = -89.3847841013921
        lats[2552] = -89.45506977912261
        lats[2553] = -89.525351592371393
        lats[2554] = -89.595627537554492
        lats[2555] = -89.6658939412157
        lats[2556] = -89.736143271609578
        lats[2557] = -89.806357319542244
        lats[2558] = -89.876478353332288
        lats[2559] = -89.946187715665616
        return lats

    def first_axis_vals(self):
        if self._resolution == 2560:
            return self.get_precomputed_values_N2560()
        if self._resolution == 1280:
            return self.get_precomputed_values_N1280()
        else:
            logging.info("Calculating grid. Not using a pre-computed grid.")
            precision = 1.0e-14
            nval = self._resolution * 2
            rad2deg = 180 / math.pi
            convval = 1 - ((2 / math.pi) * (2 / math.pi)) * 0.25
            vals = self.gauss_first_guess()
            new_vals = [0] * nval
            denom = math.sqrt(((nval + 0.5) * (nval + 0.5)) + convval)
            for jval in range(self._resolution):
                root = math.cos(vals[jval] / denom)
                conv = 1
                while abs(conv) >= precision:
                    mem2 = 1
                    mem1 = root
                    for legi in range(nval):
                        legfonc = ((2.0 * (legi + 1) - 1.0) * root * mem1 - legi * mem2) / (legi + 1)
                        mem2 = mem1
                        mem1 = legfonc
                    conv = legfonc / ((nval * (mem2 - root * legfonc)) / (1.0 - (root * root)))
                    root = root - conv
                    # add maybe a max iter here to make sure we converge at some point
                new_vals[jval] = math.asin(root) * rad2deg
                new_vals[nval - 1 - jval] = -new_vals[jval]
            return new_vals

    def map_first_axis(self, lower, upper):
        axis_lines = self._first_axis_vals
        end_idx = bisect_left_cmp(axis_lines, lower, cmp=lambda x, y: x > y) + 1
        start_idx = bisect_right_cmp(axis_lines, upper, cmp=lambda x, y: x > y)
        return_vals = axis_lines[start_idx:end_idx]
        return return_vals

    def second_axis_vals(self, first_val):
        first_axis_vals = self._first_axis_vals
        tol = 1e-10
        first_idx = bisect_left_cmp(first_axis_vals, first_val[0] - tol, cmp=lambda x, y: x > y)
        if first_idx >= self._resolution:
            first_idx = (2 * self._resolution) - 1 - first_idx
        first_idx = first_idx + 1
        npoints = 4 * first_idx + 16
        second_axis_spacing = 360 / npoints
        second_axis_vals = [i * second_axis_spacing for i in range(npoints)]
        return second_axis_vals

    def second_axis_spacing(self, first_val):
        first_axis_vals = self._first_axis_vals
        tol = 1e-10
        _first_idx = bisect_left_cmp(first_axis_vals, first_val[0] - tol, cmp=lambda x, y: x > y)
        first_idx = _first_idx
        if first_idx >= self._resolution:
            first_idx = (2 * self._resolution) - 1 - first_idx
        first_idx = first_idx + 1
        npoints = 4 * first_idx + 16
        second_axis_spacing = 360 / npoints
        return (second_axis_spacing, _first_idx + 1)

    def map_second_axis(self, first_val, lower, upper):
        second_axis_spacing, first_idx = self.second_axis_spacing(first_val)
        start_idx = int(lower / second_axis_spacing)
        end_idx = int(upper / second_axis_spacing) + 1
        return_vals = [i * second_axis_spacing for i in range(start_idx, end_idx)]
        return return_vals

    def axes_idx_to_octahedral_idx(self, first_idx, second_idx):
        # NOTE: for now this takes ~2e-4s per point, so taking significant time -> for 20k points, takes 4s
        # Would it be better to store a dictionary of first_idx with cumulative number of points on that idx?
        # Because this is what we are doing here, but we are calculating for each point...
        # But then this would only work for special grid resolutions, so need to do like a O1280 version of this

        # NOTE: OR somehow cache this for a given first_idx and then only modify the axis idx for second_idx when the
        # first_idx changes
        octa_idx = self._first_idx_map[first_idx - 1] + second_idx
        return octa_idx

    def create_first_idx_map(self):
        first_idx_list = {}
        idx = 0
        for i in range(2 * self._resolution):
            first_idx_list[i] = idx
            if i <= self._resolution - 1:
                idx += 20 + 4 * i
            else:
                i = i - self._resolution + 1
                if i == 1:
                    idx += 16 + 4 * self._resolution
                else:
                    i = i - 1
                    idx += 16 + 4 * (self._resolution - i)
        return first_idx_list

    def find_second_axis_idx(self, first_val, second_val):
        (second_axis_spacing, first_idx) = self.second_axis_spacing(first_val)
        tol = 1e-8
        if second_val[0] / second_axis_spacing > int(second_val[0] / second_axis_spacing) + 1 - tol:
            second_idx = int(second_val[0] / second_axis_spacing) + 1
        else:
            second_idx = int(second_val[0] / second_axis_spacing)
        return (first_idx, second_idx)

    def unmap(self, first_val, second_val):
        (first_idx, second_idx) = self.find_second_axis_idx(first_val, second_val)
        octahedral_index = self.axes_idx_to_octahedral_idx(first_idx, second_idx)
        return octahedral_index


# md5 grid hash in form {resolution : hash}
_md5_hash = {
    1280: "158db321ae8e773681eeb40e0a3d350f",
    2560: "b46ef646819838ead0a38749197e17a9",
}
