import logging
import math

from .....utility.list_tools import bisect_left_cmp, bisect_right_cmp
from ..datacube_mappers import DatacubeMapper


class ReducedGaussianGridMapper(DatacubeMapper):
    def __init__(self, base_axis, mapped_axes, resolution, md5_hash=None, local_area=[], axis_reversed=None):
        # TODO: if local area is not empty list, raise NotImplemented
        self._mapped_axes = mapped_axes
        self._base_axis = base_axis
        self._resolution = resolution
        self._first_axis_vals = self.first_axis_vals()
        self._second_axis_spacing = {}
        self._axis_reversed = {mapped_axes[0]: True, mapped_axes[1]: False}
        if self._axis_reversed[mapped_axes[1]]:
            raise NotImplementedError("Octahedral grid with second axis in decreasing order is not supported")
        if not self._axis_reversed[mapped_axes[0]]:
            raise NotImplementedError("Octahedral grid with first axis in increasing order is not supported")
        self.compressed_grid_axes = [self._mapped_axes[1]]
        if md5_hash is not None:
            self.md5_hash = md5_hash
        else:
            self.md5_hash = _md5_hash.get(resolution, None)

    def gauss_first_guess(self):
        i = 0
        gvals = [
            2.4048255577e0,
            5.5200781103e0,
            8.6537279129e0,
            11.7915344391e0,
            14.9309177086e0,
            18.0710639679e0,
            21.2116366299e0,
            24.3524715308e0,
            27.4934791320e0,
            30.6346064684e0,
            33.7758202136e0,
            36.9170983537e0,
            40.0584257646e0,
            43.1997917132e0,
            46.3411883717e0,
            49.4826098974e0,
            52.6240518411e0,
            55.7655107550e0,
            58.9069839261e0,
            62.0484691902e0,
            65.1899648002e0,
            68.3314693299e0,
            71.4729816036e0,
            74.6145006437e0,
            77.7560256304e0,
            80.8975558711e0,
            84.0390907769e0,
            87.1806298436e0,
            90.3221726372e0,
            93.4637187819e0,
            96.6052679510e0,
            99.7468198587e0,
            102.8883742542e0,
            106.0299309165e0,
            109.1714896498e0,
            112.3130502805e0,
            115.4546126537e0,
            118.5961766309e0,
            121.7377420880e0,
            124.8793089132e0,
            128.0208770059e0,
            131.1624462752e0,
            134.3040166383e0,
            137.4455880203e0,
            140.5871603528e0,
            143.7287335737e0,
            146.8703076258e0,
            150.0118824570e0,
            153.1534580192e0,
            156.2950342685e0,
        ]

        numVals = len(gvals)
        vals = []
        for i in range(self._resolution):
            if i < numVals:
                vals.append(gvals[i])
            else:
                vals.append(vals[i - 1] + math.pi)
        return vals

    def get_precomputed_values_N320(self):
        lats = [
            -89.78487690721863,
            -89.50620273820637,
            -89.22588284761157,
            -88.94519111831644,
            -88.66435834182293,
            -88.38345731224797,
            -88.10251813893724,
            -87.82155550711056,
            -87.54057742641116,
            -87.25958863483953,
            -86.97859211359663,
            -86.69758983192202,
            -86.41658314273641,
            -86.13557300618399,
            -85.85456012248531,
            -85.57354501429673,
            -85.29252807962695,
            -85.01150962689766,
            -84.73048989880277,
            -84.44946908892904,
            -84.16844735357665,
            -83.88742482032309,
            -83.60640159433007,
            -83.32537776305915,
            -83.04435339984514,
            -82.76332856663693,
            -82.48230331612486,
            -82.20127769340762,
            -81.92025173731213,
            -81.63922548144693,
            -81.35819895505040,
            -81.07717218367863,
            -80.79614518976737,
            -80.51511799309434,
            -80.23409061116114,
            -79.95306305951186,
            -79.67203535199914,
            -79.39100750100822,
            -79.10997951764607,
            -78.82895141190238,
            -78.54792319278644,
            -78.26689486844451,
            -77.98586644626094,
            -77.70483793294494,
            -77.42380933460623,
            -77.14278065682046,
            -76.86175190468626,
            -76.58072308287524,
            -76.29969419567588,
            -76.01866524703182,
            -75.73763624057599,
            -75.45660717966057,
            -75.17557806738355,
            -74.89454890661246,
            -74.61351970000541,
            -74.33249045002979,
            -74.05146115897912,
            -73.77043182898812,
            -73.48940246204617,
            -73.20837306000952,
            -72.92734362461226,
            -72.64631415747613,
            -72.36528466011954,
            -72.08425513396567,
            -71.80322558034980,
            -71.52219600052597,
            -71.24116639567315,
            -70.96013676690066,
            -70.67910711525340,
            -70.39807744171628,
            -70.11704774721858,
            -69.83601803263780,
            -69.55498829880318,
            -69.27395854649907,
            -68.99292877646779,
            -68.71189898941257,
            -68.43086918599992,
            -68.14983936686214,
            -67.86880953259940,
            -67.58777968378180,
            -67.30674982095123,
            -67.02571994462305,
            -66.74469005528775,
            -66.46366015341239,
            -66.18263023944205,
            -65.90160031380103,
            -65.62057037689407,
            -65.33954042910749,
            -65.05851047081022,
            -64.77748050235478,
            -64.49645052407814,
            -64.21542053630260,
            -63.93439053933660,
            -63.65336053347541,
            -63.37233051900186,
            -63.09130049618697,
            -62.81027046529061,
            -62.52924042656195,
            -62.24821038024017,
            -61.96718032655483,
            -61.68615026572643,
            -61.40512019796673,
            -61.12409012347936,
            -60.84306004246005,
            -60.56202995509710,
            -60.28099986157171,
            -59.99996976205827,
            -59.71893965672474,
            -59.43790954573291,
            -59.15687942923870,
            -58.87584930739236,
            -58.59481918033887,
            -58.31378904821798,
            -58.03275891116461,
            -57.75172876930898,
            -57.47069862277679,
            -57.18966847168948,
            -56.90863831616438,
            -56.62760815631488,
            -56.34657799225056,
            -56.06554782407743,
            -55.78451765189804,
            -55.50348747581155,
            -55.22245729591398,
            -54.94142711229829,
            -54.66039692505446,
            -54.37936673426967,
            -54.09833654002838,
            -53.81730634241245,
            -53.53627614150120,
            -53.25524593737158,
            -52.97421573009819,
            -52.69318551975338,
            -52.41215530640741,
            -52.13112509012841,
            -51.85009487098255,
            -51.56906464903404,
            -51.28803442434530,
            -51.00700419697692,
            -50.72597396698776,
            -50.44494373443509,
            -50.16391349937449,
            -49.88288326186004,
            -49.60185302194436,
            -49.32082277967856,
            -49.03979253511243,
            -48.75876228829437,
            -48.47773203927152,
            -48.19670178808972,
            -47.91567153479365,
            -47.63464127942677,
            -47.35361102203147,
            -47.07258076264900,
            -46.79155050131955,
            -46.51052023808231,
            -46.22948997297545,
            -45.94845970603621,
            -45.66742943730089,
            -45.38639916680486,
            -45.10536889458269,
            -44.82433862066802,
            -44.54330834509373,
            -44.26227806789189,
            -43.98124778909381,
            -43.70021750873003,
            -43.41918722683038,
            -43.13815694342401,
            -42.85712665853937,
            -42.57609637220426,
            -42.29506608444584,
            -42.01403579529062,
            -41.73300550476453,
            -41.45197521289295,
            -41.17094491970063,
            -40.88991462521181,
            -40.60888432945017,
            -40.32785403243889,
            -40.04682373420064,
            -39.76579343475757,
            -39.48476313413139,
            -39.20373283234334,
            -38.92270252941418,
            -38.64167222536426,
            -38.36064192021350,
            -38.07961161398138,
            -37.79858130668703,
            -37.51755099834911,
            -37.23652068898597,
            -36.95549037861555,
            -36.67446006725543,
            -36.39342975492286,
            -36.11239944163471,
            -35.83136912740756,
            -35.55033881225761,
            -35.26930849620082,
            -34.98827817925277,
            -34.70724786142878,
            -34.42621754274384,
            -34.14518722321272,
            -33.86415690284984,
            -33.58312658166939,
            -33.30209625968529,
            -33.02106593691121,
            -32.74003561336056,
            -32.45900528904648,
            -32.17797496398192,
            -31.89694463817959,
            -31.61591431165194,
            -31.33488398441122,
            -31.05385365646946,
            -30.77282332783850,
            -30.49179299852995,
            -30.21076266855524,
            -29.92973233792557,
            -29.64870200665200,
            -29.36767167474536,
            -29.08664134221633,
            -28.80561100907540,
            -28.52458067533288,
            -28.24355034099893,
            -27.96252000608353,
            -27.68148967059651,
            -27.40045933454753,
            -27.11942899794612,
            -26.83839866080164,
            -26.55736832312331,
            -26.27633798492022,
            -25.99530764620129,
            -25.71427730697534,
            -25.43324696725105,
            -25.15221662703694,
            -24.87118628634144,
            -24.59015594517284,
            -24.30912560353932,
            -24.02809526144893,
            -23.74706491890961,
            -23.46603457592919,
            -23.18500423251539,
            -22.90397388867583,
            -22.62294354441799,
            -22.34191319974931,
            -22.06088285467707,
            -21.77985250920850,
            -21.49882216335069,
            -21.21779181711067,
            -20.93676147049538,
            -20.65573112351166,
            -20.37470077616624,
            -20.09367042846582,
            -19.81264008041698,
            -19.53160973202621,
            -19.25057938329996,
            -18.96954903424457,
            -18.68851868486632,
            -18.40748833517141,
            -18.12645798516599,
            -17.84542763485610,
            -17.56439728424774,
            -17.28336693334685,
            -17.00233658215929,
            -16.72130623069085,
            -16.44027587894727,
            -16.15924552693424,
            -15.87821517465737,
            -15.59718482212223,
            -15.31615446933432,
            -15.03512411629909,
            -14.75409376302196,
            -14.47306340950825,
            -14.19203305576328,
            -13.91100270179228,
            -13.62997234760047,
            -13.34894199319299,
            -13.06791163857496,
            -12.78688128375144,
            -12.50585092872746,
            -12.22482057350799,
            -11.94379021809798,
            -11.66275986250232,
            -11.38172950672589,
            -11.10069915077350,
            -10.81966879464995,
            -10.53863843836000,
            -10.25760808190837,
            -9.97657772529974,
            -9.69554736853877,
            -9.41451701163010,
            -9.13348665457831,
            -8.85245629738798,
            -8.57142594006364,
            -8.29039558260981,
            -8.00936522503099,
            -7.72833486733163,
            -7.44730450951617,
            -7.16627415158902,
            -6.88524379355459,
            -6.60421343541725,
            -6.32318307718135,
            -6.04215271885122,
            -5.76112236043118,
            -5.48009200192553,
            -5.19906164333854,
            -4.91803128467449,
            -4.63700092593762,
            -4.35597056713217,
            -4.07494020826237,
            -3.79390984933241,
            -3.51287949034651,
            -3.23184913130884,
            -2.95081877222359,
            -2.66978841309493,
            -2.38875805392700,
            -2.10772769472398,
            -1.82669733549000,
            -1.54566697622920,
            -1.26463661694571,
            -0.98360625764368,
            -0.70257589832721,
            -0.42154553900045,
            -0.14051517966750,
            0.14051517966750,
            0.42154553900045,
            0.70257589832721,
            0.98360625764368,
            1.26463661694571,
            1.54566697622920,
            1.82669733549000,
            2.10772769472398,
            2.38875805392700,
            2.66978841309493,
            2.95081877222359,
            3.23184913130884,
            3.51287949034651,
            3.79390984933241,
            4.07494020826237,
            4.35597056713217,
            4.63700092593762,
            4.91803128467449,
            5.19906164333854,
            5.48009200192553,
            5.76112236043118,
            6.04215271885122,
            6.32318307718135,
            6.60421343541725,
            6.88524379355459,
            7.16627415158902,
            7.44730450951617,
            7.72833486733163,
            8.00936522503099,
            8.29039558260981,
            8.57142594006364,
            8.85245629738798,
            9.13348665457831,
            9.41451701163010,
            9.69554736853877,
            9.97657772529974,
            10.25760808190837,
            10.53863843836000,
            10.81966879464995,
            11.10069915077350,
            11.38172950672589,
            11.66275986250232,
            11.94379021809798,
            12.22482057350799,
            12.50585092872746,
            12.78688128375144,
            13.06791163857496,
            13.34894199319299,
            13.62997234760047,
            13.91100270179228,
            14.19203305576328,
            14.47306340950825,
            14.75409376302196,
            15.03512411629909,
            15.31615446933432,
            15.59718482212223,
            15.87821517465737,
            16.15924552693424,
            16.44027587894727,
            16.72130623069085,
            17.00233658215929,
            17.28336693334685,
            17.56439728424774,
            17.84542763485610,
            18.12645798516599,
            18.40748833517141,
            18.68851868486632,
            18.96954903424457,
            19.25057938329996,
            19.53160973202621,
            19.81264008041698,
            20.09367042846582,
            20.37470077616624,
            20.65573112351166,
            20.93676147049538,
            21.21779181711067,
            21.49882216335069,
            21.77985250920850,
            22.06088285467707,
            22.34191319974931,
            22.62294354441799,
            22.90397388867583,
            23.18500423251539,
            23.46603457592919,
            23.74706491890961,
            24.02809526144893,
            24.30912560353932,
            24.59015594517284,
            24.87118628634144,
            25.15221662703694,
            25.43324696725105,
            25.71427730697534,
            25.99530764620129,
            26.27633798492022,
            26.55736832312331,
            26.83839866080164,
            27.11942899794612,
            27.40045933454753,
            27.68148967059651,
            27.96252000608353,
            28.24355034099893,
            28.52458067533288,
            28.80561100907540,
            29.08664134221633,
            29.36767167474536,
            29.64870200665200,
            29.92973233792557,
            30.21076266855524,
            30.49179299852995,
            30.77282332783850,
            31.05385365646946,
            31.33488398441122,
            31.61591431165194,
            31.89694463817959,
            32.17797496398192,
            32.45900528904648,
            32.74003561336056,
            33.02106593691121,
            33.30209625968529,
            33.58312658166939,
            33.86415690284984,
            34.14518722321272,
            34.42621754274384,
            34.70724786142878,
            34.98827817925277,
            35.26930849620082,
            35.55033881225761,
            35.83136912740756,
            36.11239944163471,
            36.39342975492286,
            36.67446006725543,
            36.95549037861555,
            37.23652068898597,
            37.51755099834911,
            37.79858130668703,
            38.07961161398138,
            38.36064192021350,
            38.64167222536426,
            38.92270252941418,
            39.20373283234334,
            39.48476313413139,
            39.76579343475757,
            40.04682373420064,
            40.32785403243889,
            40.60888432945017,
            40.88991462521181,
            41.17094491970063,
            41.45197521289295,
            41.73300550476453,
            42.01403579529062,
            42.29506608444584,
            42.57609637220426,
            42.85712665853937,
            43.13815694342401,
            43.41918722683038,
            43.70021750873003,
            43.98124778909381,
            44.26227806789189,
            44.54330834509373,
            44.82433862066802,
            45.10536889458269,
            45.38639916680486,
            45.66742943730089,
            45.94845970603621,
            46.22948997297545,
            46.51052023808231,
            46.79155050131955,
            47.07258076264900,
            47.35361102203147,
            47.63464127942677,
            47.91567153479365,
            48.19670178808972,
            48.47773203927152,
            48.75876228829437,
            49.03979253511243,
            49.32082277967856,
            49.60185302194436,
            49.88288326186004,
            50.16391349937449,
            50.44494373443509,
            50.72597396698776,
            51.00700419697692,
            51.28803442434530,
            51.56906464903404,
            51.85009487098255,
            52.13112509012841,
            52.41215530640741,
            52.69318551975338,
            52.97421573009819,
            53.25524593737158,
            53.53627614150120,
            53.81730634241245,
            54.09833654002838,
            54.37936673426967,
            54.66039692505446,
            54.94142711229829,
            55.22245729591398,
            55.50348747581155,
            55.78451765189804,
            56.06554782407743,
            56.34657799225056,
            56.62760815631488,
            56.90863831616438,
            57.18966847168948,
            57.47069862277679,
            57.75172876930898,
            58.03275891116461,
            58.31378904821798,
            58.59481918033887,
            58.87584930739236,
            59.15687942923870,
            59.43790954573291,
            59.71893965672474,
            59.99996976205827,
            60.28099986157171,
            60.56202995509710,
            60.84306004246005,
            61.12409012347936,
            61.40512019796673,
            61.68615026572643,
            61.96718032655483,
            62.24821038024017,
            62.52924042656195,
            62.81027046529061,
            63.09130049618697,
            63.37233051900186,
            63.65336053347541,
            63.93439053933660,
            64.21542053630260,
            64.49645052407814,
            64.77748050235478,
            65.05851047081022,
            65.33954042910749,
            65.62057037689407,
            65.90160031380103,
            66.18263023944205,
            66.46366015341239,
            66.74469005528775,
            67.02571994462305,
            67.30674982095123,
            67.58777968378180,
            67.86880953259940,
            68.14983936686214,
            68.43086918599992,
            68.71189898941257,
            68.99292877646779,
            69.27395854649907,
            69.55498829880318,
            69.83601803263780,
            70.11704774721858,
            70.39807744171628,
            70.67910711525340,
            70.96013676690066,
            71.24116639567315,
            71.52219600052597,
            71.80322558034980,
            72.08425513396567,
            72.36528466011954,
            72.64631415747613,
            72.92734362461226,
            73.20837306000952,
            73.48940246204617,
            73.77043182898812,
            74.05146115897912,
            74.33249045002979,
            74.61351970000541,
            74.89454890661246,
            75.17557806738355,
            75.45660717966057,
            75.73763624057599,
            76.01866524703182,
            76.29969419567588,
            76.58072308287524,
            76.86175190468626,
            77.14278065682046,
            77.42380933460623,
            77.70483793294494,
            77.98586644626094,
            78.26689486844451,
            78.54792319278644,
            78.82895141190238,
            79.10997951764607,
            79.39100750100822,
            79.67203535199914,
            79.95306305951186,
            80.23409061116114,
            80.51511799309434,
            80.79614518976737,
            81.07717218367863,
            81.35819895505040,
            81.63922548144693,
            81.92025173731213,
            82.20127769340762,
            82.48230331612486,
            82.76332856663693,
            83.04435339984514,
            83.32537776305915,
            83.60640159433007,
            83.88742482032309,
            84.16844735357665,
            84.44946908892904,
            84.73048989880277,
            85.01150962689766,
            85.29252807962695,
            85.57354501429673,
            85.85456012248531,
            86.13557300618399,
            86.41658314273641,
            86.69758983192202,
            86.97859211359663,
            87.25958863483953,
            87.54057742641116,
            87.82155550711056,
            88.10251813893724,
            88.38345731224797,
            88.66435834182293,
            88.94519111831644,
            89.22588284761157,
            89.50620273820637,
            89.78487690721863,
        ][::-1]
        return lats

    def first_axis_vals(self):
        if self._resolution == 320:
            return self.get_precomputed_values_N320()
        else:
            logging.info("Calculating grid. Not using a pre-computed grid.")
            precision = 1.0e-14
            nval = self._resolution * 2
            rad2deg = 180 / math.pi
            convval = 1 - ((2 / math.pi) * (2 / math.pi)) * 0.25
            vals = self.gauss_first_guess()
            new_vals = [0] * nval
            denom = math.sqrt(((nval + 0.5) * (nval + 0.5)) + convval)
            for jval in range(self._resolution):
                root = math.cos(vals[jval] / denom)
                conv = 1
                while abs(conv) >= precision:
                    mem2 = 1
                    mem1 = root
                    for legi in range(nval):
                        legfonc = ((2.0 * (legi + 1) - 1.0) * root * mem1 - legi * mem2) / (legi + 1)
                        mem2 = mem1
                        mem1 = legfonc
                    conv = legfonc / ((nval * (mem2 - root * legfonc)) / (1.0 - (root * root)))
                    root = root - conv
                    # add maybe a max iter here to make sure we converge at some point
                new_vals[jval] = math.asin(root) * rad2deg
                new_vals[nval - 1 - jval] = -new_vals[jval]
            return new_vals

    def map_first_axis(self, lower, upper):
        axis_lines = self._first_axis_vals
        end_idx = bisect_left_cmp(axis_lines, lower, cmp=lambda x, y: x > y) + 1
        start_idx = bisect_right_cmp(axis_lines, upper, cmp=lambda x, y: x > y)
        return_vals = axis_lines[start_idx:end_idx]
        return return_vals

    def lon_spacing(self):
        if self._resolution == 320:
            return [
                18,
                25,
                36,
                40,
                45,
                50,
                60,
                64,
                72,
                72,
                75,
                81,
                90,
                96,
                100,
                108,
                120,
                120,
                125,
                135,
                144,
                144,
                150,
                160,
                180,
                180,
                180,
                192,
                192,
                200,
                216,
                216,
                216,
                225,
                240,
                240,
                240,
                250,
                256,
                270,
                270,
                288,
                288,
                288,
                300,
                300,
                320,
                320,
                320,
                324,
                360,
                360,
                360,
                360,
                360,
                360,
                375,
                375,
                384,
                384,
                400,
                400,
                405,
                432,
                432,
                432,
                432,
                450,
                450,
                450,
                480,
                480,
                480,
                480,
                480,
                486,
                500,
                500,
                500,
                512,
                512,
                540,
                540,
                540,
                540,
                540,
                576,
                576,
                576,
                576,
                576,
                576,
                600,
                600,
                600,
                600,
                640,
                640,
                640,
                640,
                640,
                640,
                640,
                648,
                648,
                675,
                675,
                675,
                675,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                729,
                750,
                750,
                750,
                750,
                768,
                768,
                768,
                768,
                800,
                800,
                800,
                800,
                800,
                800,
                810,
                810,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                900,
                900,
                900,
                900,
                900,
                900,
                900,
                900,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                972,
                972,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                1024,
                1024,
                1024,
                1024,
                1024,
                1024,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1215,
                1215,
                1215,
                1215,
                1215,
                1215,
                1215,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1280,
                1215,
                1215,
                1215,
                1215,
                1215,
                1215,
                1215,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1200,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1152,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1125,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1080,
                1024,
                1024,
                1024,
                1024,
                1024,
                1024,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                1000,
                972,
                972,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                960,
                900,
                900,
                900,
                900,
                900,
                900,
                900,
                900,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                864,
                810,
                810,
                800,
                800,
                800,
                800,
                800,
                800,
                768,
                768,
                768,
                768,
                750,
                750,
                750,
                750,
                729,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                720,
                675,
                675,
                675,
                675,
                648,
                648,
                640,
                640,
                640,
                640,
                640,
                640,
                640,
                600,
                600,
                600,
                600,
                576,
                576,
                576,
                576,
                576,
                576,
                540,
                540,
                540,
                540,
                540,
                512,
                512,
                500,
                500,
                500,
                486,
                480,
                480,
                480,
                480,
                480,
                450,
                450,
                450,
                432,
                432,
                432,
                432,
                405,
                400,
                400,
                384,
                384,
                375,
                375,
                360,
                360,
                360,
                360,
                360,
                360,
                324,
                320,
                320,
                320,
                300,
                300,
                288,
                288,
                288,
                270,
                270,
                256,
                250,
                240,
                240,
                240,
                225,
                216,
                216,
                216,
                200,
                192,
                192,
                180,
                180,
                180,
                160,
                150,
                144,
                144,
                135,
                125,
                120,
                120,
                108,
                100,
                96,
                90,
                81,
                75,
                72,
                72,
                64,
                60,
                50,
                45,
                40,
                36,
                25,
                18,
            ]

    def second_axis_vals(self, first_val):
        tol = 1e-8
        first_value = [i for i in self._first_axis_vals if first_val[0] - tol <= i <= first_val[0] + tol][0]
        first_idx = self._first_axis_vals.index(first_value)
        Ny = self.lon_spacing()[first_idx]
        second_spacing = 360 / Ny
        return [i * second_spacing for i in range(Ny)]

    def axes_idx_to_reduced_gaussian_idx(self, first_idx, second_idx):
        Ny_array = self.lon_spacing()
        idx = 0
        for i in range(self._resolution):
            if i != first_idx:
                idx += Ny_array[i]
            else:
                idx += second_idx
                return idx

    # def find_second_idx(self, first_val, second_val):
    #     tol = 1e-10
    #     second_axis_vals = self.second_axis_vals(first_val)
    #     second_idx = bisect.bisect_left(second_axis_vals, second_val[0] - tol)
    #     return second_idx

    # def unmap(self, first_val, second_val):
    #     (first_idx, second_idx) = self.find_second_idx(first_val, second_val)
    #     octahedral_index = self.axes_idx_to_reduced_gaussian_idx(first_idx, second_idx)
    #     return octahedral_index

    def unmap(self, first_val, second_val):
        tol = 1e-8
        first_value = [i for i in self._first_axis_vals if first_val[0] - tol <= i <= first_val[0] + tol][0]
        first_idx = self._first_axis_vals.index(first_value)
        second_val = [i for i in self.second_axis_vals(first_val) if second_val[0] - tol <= i <= second_val[0] + tol][0]
        second_idx = self.second_axis_vals(first_val).index(second_val)
        reduced_gaussian_index = self.axes_idx_to_reduced_gaussian_idx(first_idx, second_idx)
        return reduced_gaussian_index


# md5 grid hash in form {resolution : hash}
_md5_hash = {
    320: "158db321ae8e773681eeb40e0a3d350f",
}
