from typing import Any, Mapping, Sequence, Tuple

import posthog
from openfeature.evaluation_context import EvaluationContext
from openfeature.exception import TypeMismatchError
from openfeature.flag_evaluation import FlagResolutionDetails, FlagValueType, Reason
from openfeature.provider import AbstractProvider
from openfeature.provider.metadata import Metadata

DEFAULT_DISTINCT_ID = "anonymous"


class PosthogProvider(AbstractProvider):
    def __init__(
        self,
        posthog_client: posthog.Posthog,  # type: ignore[name-defined]
        *args: Any,
        default_distinct_id: str = DEFAULT_DISTINCT_ID,
        **kwargs: Any,
    ):
        super().__init__(*args, **kwargs)
        self._posthog_client = posthog_client
        self._default_distinct_id = default_distinct_id

    def get_metadata(self) -> Metadata:
        return Metadata(name="PosthogProvider")

    def resolve_boolean_details(
        self,
        flag_key: str,
        default_value: bool,
        evaluation_context: EvaluationContext | None = None,
    ) -> FlagResolutionDetails[bool]:
        distinct_id = self._get_distinct_id(evaluation_context)
        (person_properties, group_properties, groups) = self._get_attributes(
            evaluation_context
        )
        is_enabled = self._posthog_client.feature_enabled(
            flag_key,
            distinct_id,
            groups=groups,
            group_properties=group_properties,
            person_properties=person_properties,
        )
        return FlagResolutionDetails(
            value=is_enabled,
            variant=str(is_enabled),
            reason=Reason.TARGETING_MATCH if is_enabled else Reason.DISABLED,
        )

    def resolve_integer_details(
        self,
        flag_key: str,
        default_value: int,
        evaluation_context: EvaluationContext | None = None,
    ) -> FlagResolutionDetails[int]:
        flag_value = self._get_flag_value(flag_key, evaluation_context)
        try:
            int_value = int(flag_value)
            return FlagResolutionDetails(
                value=int(flag_value),
                variant=str(int_value),
                reason=Reason.TARGETING_MATCH,
            )
        except ValueError as e:
            raise TypeMismatchError("Flag value cannot be converted to integer.") from e

    def resolve_string_details(
        self,
        flag_key: str,
        default_value: str,
        evaluation_context: EvaluationContext | None = None,
    ) -> FlagResolutionDetails[str]:
        flag_value = self._get_flag_value(flag_key, evaluation_context)
        try:
            str_value = str(flag_value)
            return FlagResolutionDetails(
                value=str_value,
                variant=str_value,
                reason=Reason.TARGETING_MATCH,
            )
        except ValueError as e:
            raise TypeMismatchError("Flag value cannot be converted to string.") from e

    def resolve_float_details(
        self,
        flag_key: str,
        default_value: float,
        evaluation_context: EvaluationContext | None = None,
    ) -> FlagResolutionDetails[float]:
        flag_value = self._get_flag_value(flag_key, evaluation_context)
        try:
            float_value = float(flag_value)
            return FlagResolutionDetails(
                value=float_value,
                variant=str(float_value),
                reason=Reason.TARGETING_MATCH,
            )
        except ValueError as e:
            raise TypeMismatchError("Flag value cannot be converted to float.") from e

    def resolve_object_details(
        self,
        flag_key: str,
        default_value: Sequence[FlagValueType] | Mapping[str, FlagValueType],
        evaluation_context: EvaluationContext | None = None,
    ) -> FlagResolutionDetails[dict]:
        raise NotImplementedError

    def _get_distinct_id(
        self,
        evaluation_context: EvaluationContext | None = None,
    ) -> str:
        if (
            evaluation_context is not None
            and evaluation_context.targeting_key is not None
        ):
            return evaluation_context.targeting_key
        return self._default_distinct_id

    @staticmethod
    def _get_attributes(
        evaluation_context: EvaluationContext | None = None,
    ) -> Tuple[dict, dict, dict]:
        if evaluation_context is None or evaluation_context.attributes is None:
            return {}, {}, {}
        # All basic attributes go into person_properties
        person_properties = {
            key: value
            for key, value in evaluation_context.attributes.items()
            if isinstance(value, (str, int, float, bool))
        }

        try:
            group_properties_from_context = evaluation_context.attributes.get(
                "group_properties", {}
            )
            if not isinstance(group_properties_from_context, dict):
                group_properties = {}
            else:
                group_properties = {
                    key: value
                    for key, value in group_properties_from_context.items()
                    if not isinstance(value, (str, int, float, bool))
                }
        except (AttributeError, KeyError):
            group_properties = {}

        try:
            groups_from_context = evaluation_context.attributes.get("groups", {})
            if not isinstance(groups_from_context, dict):
                groups = {}
            else:
                groups = {
                    key: value
                    for key, value in groups_from_context.items()
                    if not isinstance(value, (str, int, float, bool))
                }
        except (AttributeError, KeyError):
            groups = {}
        return person_properties, group_properties, groups

    def _get_flag_value(
        self, flag_key: str, evaluation_context: EvaluationContext | None = None
    ):
        distinct_id = self._get_distinct_id(evaluation_context)
        (person_properties, group_properties, groups) = self._get_attributes(
            evaluation_context
        )
        return self._posthog_client.get_feature_flag(
            flag_key,
            distinct_id,
            groups=groups,
            group_properties=group_properties,
            person_properties=person_properties,
        )
