# Copyright (c) 2023 PaddlePaddle Authors. All Rights Reserved.
# Copyright 2023 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from dataclasses import dataclass
from typing import Any, Dict, List, Optional, Tuple, Union

import paddle
import paddle.nn as nn
import paddle.nn.functional as F

from ..configuration_utils import ConfigMixin, register_to_config
from ..initializer import zeros_
from ..utils import BaseOutput, logging
from .attention_processor import AttentionProcessor, AttnProcessor
from .embeddings import TimestepEmbedding, Timesteps
from .modeling_utils import ModelMixin
from .unet_2d_blocks import (
    CrossAttnDownBlock2D,
    DownBlock2D,
    UNetMidBlock2DCrossAttn,
    get_down_block, )
from .unet_2d_condition import UNet2DConditionModel

logger = logging.get_logger(__name__)  # pylint: disable=invalid-name


@dataclass
class ControlNetOutput(BaseOutput):
    down_block_res_samples: Tuple[paddle.Tensor]
    mid_block_res_sample: paddle.Tensor


class Mish(nn.Layer):
    def forward(self, hidden_states):
        return hidden_states * paddle.tanh(F.softplus(hidden_states))


class ControlNetConditioningEmbedding(nn.Layer):
    """
    "Stable Diffusion uses a pre-processing method similar to VQ-GAN [11] to convert the entire dataset of 512 × 512
    images into smaller 64 × 64 “latent images” for stabilized training. This requires ControlNets to convert
    image-based conditions to 64 × 64 feature space to match the convolution size. We use a tiny network E(·) of four
    convolution layers with 4 × 4 kernels and 2 × 2 strides (activated by ReLU, channels are 16, 32, 64, 128,
    initialized with Gaussian weights, trained jointly with the full model) to encode image-space conditions ... into
    feature maps ..."
    """

    def __init__(
            self,
            conditioning_embedding_channels: int,
            conditioning_channels: int=3,
            block_out_channels: Tuple[int]=(16, 32, 96, 256), ):
        super().__init__()

        self.conv_in = nn.Conv2D(
            conditioning_channels,
            block_out_channels[0],
            kernel_size=3,
            padding=1)

        self.blocks = nn.LayerList([])

        for i in range(len(block_out_channels) - 1):
            channel_in = block_out_channels[i]
            channel_out = block_out_channels[i + 1]
            self.blocks.append(
                nn.Conv2D(
                    channel_in, channel_in, kernel_size=3, padding=1))
            self.blocks.append(
                nn.Conv2D(
                    channel_in, channel_out, kernel_size=3, padding=1,
                    stride=2))

        self.conv_out = zero_module(
            nn.Conv2D(
                block_out_channels[-1],
                conditioning_embedding_channels,
                kernel_size=3,
                padding=1))

    def forward(self, conditioning):
        embedding = self.conv_in(conditioning)
        embedding = F.silu(embedding)

        for block in self.blocks:
            embedding = block(embedding)
            embedding = F.silu(embedding)

        embedding = self.conv_out(embedding)

        return embedding


class ControlNetModel(ModelMixin, ConfigMixin):
    _supports_gradient_checkpointing = True

    @register_to_config
    def __init__(
            self,
            in_channels: int=4,
            flip_sin_to_cos: bool=True,
            freq_shift: int=0,
            down_block_types: Tuple[str]=(
                "CrossAttnDownBlock2D",
                "CrossAttnDownBlock2D",
                "CrossAttnDownBlock2D",
                "DownBlock2D", ),
            only_cross_attention: Union[bool, Tuple[bool]]=False,
            block_out_channels: Tuple[int]=(320, 640, 1280, 1280),
            layers_per_block: int=2,
            downsample_padding: int=1,
            mid_block_scale_factor: float=1,
            act_fn: str="silu",
            norm_num_groups: Optional[int]=32,
            norm_eps: float=1e-5,
            cross_attention_dim: int=1280,
            attention_head_dim: Union[int, Tuple[int]]=8,
            use_linear_projection: bool=False,
            class_embed_type: Optional[str]=None,
            num_class_embeds: Optional[int]=None,
            upcast_attention: bool=False,
            resnet_time_scale_shift: str="default",
            projection_class_embeddings_input_dim: Optional[int]=None,
            controlnet_conditioning_channel_order: str="rgb",
            conditioning_embedding_out_channels: Optional[Tuple[int]]=(16, 32,
                                                                       96, 256),
            global_pool_conditions: bool=False,
            resnet_pre_temb_non_linearity: bool=False, ):
        super().__init__()

        # Check inputs
        if len(block_out_channels) != len(down_block_types):
            raise ValueError(
                f"Must provide the same number of `block_out_channels` as `down_block_types`. `block_out_channels`: {block_out_channels}. `down_block_types`: {down_block_types}."
            )

        if not isinstance(
                only_cross_attention,
                bool) and len(only_cross_attention) != len(down_block_types):
            raise ValueError(
                f"Must provide the same number of `only_cross_attention` as `down_block_types`. `only_cross_attention`: {only_cross_attention}. `down_block_types`: {down_block_types}."
            )

        if not isinstance(
                attention_head_dim,
                int) and len(attention_head_dim) != len(down_block_types):
            raise ValueError(
                f"Must provide the same number of `attention_head_dim` as `down_block_types`. `attention_head_dim`: {attention_head_dim}. `down_block_types`: {down_block_types}."
            )

        # input
        conv_in_kernel = 3
        conv_in_padding = (conv_in_kernel - 1) // 2
        self.conv_in = nn.Conv2D(
            in_channels,
            block_out_channels[0],
            kernel_size=conv_in_kernel,
            padding=conv_in_padding)

        # time
        time_embed_dim = block_out_channels[0] * 4

        self.time_proj = Timesteps(block_out_channels[0], flip_sin_to_cos,
                                   freq_shift)
        timestep_input_dim = block_out_channels[0]

        self.time_embedding = TimestepEmbedding(
            timestep_input_dim,
            time_embed_dim,
            act_fn=act_fn, )

        # class embedding
        if class_embed_type is None and num_class_embeds is not None:
            self.class_embedding = nn.Embedding(num_class_embeds,
                                                time_embed_dim)
        elif class_embed_type == "timestep":
            self.class_embedding = TimestepEmbedding(timestep_input_dim,
                                                     time_embed_dim)
        elif class_embed_type == "identity":
            self.class_embedding = nn.Identity(time_embed_dim, time_embed_dim)
        elif class_embed_type == "projection":
            if projection_class_embeddings_input_dim is None:
                raise ValueError(
                    "`class_embed_type`: 'projection' requires `projection_class_embeddings_input_dim` be set"
                )
            # The projection `class_embed_type` is the same as the timestep `class_embed_type` except
            # 1. the `class_labels` inputs are not first converted to sinusoidal embeddings
            # 2. it projects from an arbitrary input dimension.
            #
            # Note that `TimestepEmbedding` is quite general, being mainly linear layers and activations.
            # When used for embedding actual timesteps, the timesteps are first converted to sinusoidal embeddings.
            # As a result, `TimestepEmbedding` can be passed arbitrary vectors.
            self.class_embedding = TimestepEmbedding(
                projection_class_embeddings_input_dim, time_embed_dim)
        else:
            self.class_embedding = None

        # control net conditioning embedding
        self.controlnet_cond_embedding = ControlNetConditioningEmbedding(
            conditioning_embedding_channels=block_out_channels[0],
            block_out_channels=conditioning_embedding_out_channels, )

        self.down_blocks = nn.LayerList([])
        self.controlnet_down_blocks = nn.LayerList([])

        if isinstance(only_cross_attention, bool):
            only_cross_attention = [only_cross_attention] * len(
                down_block_types)

        if isinstance(attention_head_dim, int):
            attention_head_dim = (attention_head_dim, ) * len(down_block_types)

        # pre_temb_act_fun opt
        self.resnet_pre_temb_non_linearity = resnet_pre_temb_non_linearity
        if resnet_pre_temb_non_linearity:
            if act_fn == "swish":
                self.down_resnet_temb_nonlinearity = lambda x: F.silu(x)
            elif act_fn == "mish":
                self.down_resnet_temb_nonlinearity = nn.Mish()
            elif act_fn == "silu":
                self.down_resnet_temb_nonlinearity = nn.Silu()
            elif act_fn == "gelu":
                self.down_resnet_temb_nonlinearity = nn.GELU()

        # down
        output_channel = block_out_channels[0]

        controlnet_block = nn.Conv2D(
            output_channel, output_channel, kernel_size=1)
        controlnet_block = zero_module(controlnet_block)
        self.controlnet_down_blocks.append(controlnet_block)

        for i, down_block_type in enumerate(down_block_types):
            input_channel = output_channel
            output_channel = block_out_channels[i]
            is_final_block = i == len(block_out_channels) - 1

            down_block = get_down_block(
                down_block_type,
                num_layers=layers_per_block,
                in_channels=input_channel,
                out_channels=output_channel,
                temb_channels=time_embed_dim,
                add_downsample=not is_final_block,
                resnet_eps=norm_eps,
                resnet_act_fn=act_fn,
                resnet_groups=norm_num_groups,
                cross_attention_dim=cross_attention_dim,
                attn_num_head_channels=attention_head_dim[i],
                downsample_padding=downsample_padding,
                use_linear_projection=use_linear_projection,
                only_cross_attention=only_cross_attention[i],
                upcast_attention=upcast_attention,
                resnet_time_scale_shift=resnet_time_scale_shift,
                resnet_pre_temb_non_linearity=self.
                resnet_pre_temb_non_linearity, )
            self.down_blocks.append(down_block)

            for _ in range(layers_per_block):
                controlnet_block = nn.Conv2D(
                    output_channel, output_channel, kernel_size=1)
                controlnet_block = zero_module(controlnet_block)
                self.controlnet_down_blocks.append(controlnet_block)

            if not is_final_block:
                controlnet_block = nn.Conv2D(
                    output_channel, output_channel, kernel_size=1)
                controlnet_block = zero_module(controlnet_block)
                self.controlnet_down_blocks.append(controlnet_block)

        # mid
        mid_block_channel = block_out_channels[-1]

        controlnet_block = nn.Conv2D(
            mid_block_channel, mid_block_channel, kernel_size=1)
        controlnet_block = zero_module(controlnet_block)
        self.controlnet_mid_block = controlnet_block

        self.mid_block = UNetMidBlock2DCrossAttn(
            in_channels=mid_block_channel,
            temb_channels=time_embed_dim,
            resnet_eps=norm_eps,
            resnet_act_fn=act_fn,
            output_scale_factor=mid_block_scale_factor,
            resnet_time_scale_shift=resnet_time_scale_shift,
            cross_attention_dim=cross_attention_dim,
            attn_num_head_channels=attention_head_dim[-1],
            resnet_groups=norm_num_groups,
            use_linear_projection=use_linear_projection,
            upcast_attention=upcast_attention,
            resnet_pre_temb_non_linearity=self.resnet_pre_temb_non_linearity, )

    @classmethod
    def from_unet(
            cls,
            unet: UNet2DConditionModel,
            controlnet_conditioning_channel_order: str="rgb",
            conditioning_embedding_out_channels: Optional[Tuple[int]]=(16, 32,
                                                                       96, 256),
            load_weights_from_unet: bool=True, ):
        r"""
        Instantiate Controlnet class from UNet2DConditionModel.
        Parameters:
            unet (`UNet2DConditionModel`):
                UNet model which weights are copied to the ControlNet. Note that all configuration options are also
                copied where applicable.
        """
        controlnet = cls(
            in_channels=unet.config.in_channels,
            flip_sin_to_cos=unet.config.flip_sin_to_cos,
            freq_shift=unet.config.freq_shift,
            down_block_types=unet.config.down_block_types,
            only_cross_attention=unet.config.only_cross_attention,
            block_out_channels=unet.config.block_out_channels,
            layers_per_block=unet.config.layers_per_block,
            downsample_padding=unet.config.downsample_padding,
            mid_block_scale_factor=unet.config.mid_block_scale_factor,
            act_fn=unet.config.act_fn,
            norm_num_groups=unet.config.norm_num_groups,
            norm_eps=unet.config.norm_eps,
            cross_attention_dim=unet.config.cross_attention_dim,
            attention_head_dim=unet.config.attention_head_dim,
            use_linear_projection=unet.config.use_linear_projection,
            class_embed_type=unet.config.class_embed_type,
            num_class_embeds=unet.config.num_class_embeds,
            upcast_attention=unet.config.upcast_attention,
            resnet_time_scale_shift=unet.config.resnet_time_scale_shift,
            projection_class_embeddings_input_dim=unet.config.
            projection_class_embeddings_input_dim,
            controlnet_conditioning_channel_order=controlnet_conditioning_channel_order,
            conditioning_embedding_out_channels=conditioning_embedding_out_channels,
            resnet_pre_temb_non_linearity=unet.config.
            resnet_pre_temb_non_linearity, )

        if load_weights_from_unet:
            controlnet.conv_in.load_dict(unet.conv_in.state_dict())
            controlnet.time_proj.load_dict(unet.time_proj.state_dict())
            controlnet.time_embedding.load_dict(unet.time_embedding.state_dict(
            ))

            if controlnet.class_embedding:
                controlnet.class_embedding.load_dict(
                    unet.class_embedding.state_dict())

            controlnet.down_blocks.load_dict(unet.down_blocks.state_dict())
            controlnet.mid_block.load_dict(unet.mid_block.state_dict())

        return controlnet

    @property
    def attn_processors(self) -> Dict[str, AttentionProcessor]:
        r"""
        Returns:
            `dict` of attention processors: A dictionary containing all attention processors used in the model with
            indexed by its weight name.
        """
        # set recursively
        processors = {}

        def fn_recursive_add_processors(
                name: str,
                module: nn.Layer,
                processors: Dict[str, AttentionProcessor]):
            if hasattr(module, "set_processor"):
                processors[f"{name}.processor"] = module.processor

            for sub_name, child in module.named_children():
                fn_recursive_add_processors(f"{name}.{sub_name}", child,
                                            processors)

            return processors

        for name, module in self.named_children():
            fn_recursive_add_processors(name, module, processors)

        return processors

    def set_attn_processor(self,
                           processor: Union[AttentionProcessor, Dict[
                               str, AttentionProcessor]]):
        r"""
        Parameters:
            `processor (`dict` of `AttentionProcessor` or `AttentionProcessor`):
                The instantiated processor class or a dictionary of processor classes that will be set as the processor
                of **all** `Attention` layers.
            In case `processor` is a dict, the key needs to define the path to the corresponding cross attention processor. This is strongly recommended when setting trainable attention processors.:
        """
        count = len(self.attn_processors.keys())

        if isinstance(processor, dict) and len(processor) != count:
            raise ValueError(
                f"A dict of processors was passed, but the number of processors {len(processor)} does not match the"
                f" number of attention layers: {count}. Please make sure to pass {count} processor classes."
            )

        def fn_recursive_attn_processor(name: str, module: nn.Layer, processor):
            if hasattr(module, "set_processor"):
                if not isinstance(processor, dict):
                    module.set_processor(processor)
                else:
                    module.set_processor(processor.pop(f"{name}.processor"))

            for sub_name, child in module.named_children():
                fn_recursive_attn_processor(f"{name}.{sub_name}", child,
                                            processor)

        for name, module in self.named_children():
            fn_recursive_attn_processor(name, module, processor)

    # Copied from ppdiffusers.models.unet_2d_condition.UNet2DConditionModel.set_default_attn_processor
    def set_default_attn_processor(self):
        """
        Disables custom attention processors and sets the default attention implementation.
        """
        self.set_attn_processor(AttnProcessor())

    def set_attention_slice(self, slice_size):
        r"""
        Enable sliced attention computation.
        When this option is enabled, the attention module will split the input tensor in slices, to compute attention
        in several steps. This is useful to save some memory in exchange for a small speed decrease.
        Args:
            slice_size (`str` or `int` or `list(int)`, *optional*, defaults to `"auto"`):
                When `"auto"`, halves the input to the attention heads, so attention will be computed in two steps. If
                `"max"`, maximum amount of memory will be saved by running only one slice at a time. If a number is
                provided, uses as many slices as `attention_head_dim // slice_size`. In this case, `attention_head_dim`
                must be a multiple of `slice_size`.
        """
        sliceable_head_dims = []

        def fn_recursive_retrieve_sliceable_dims(module: nn.Layer):
            if hasattr(module, "set_attention_slice"):
                sliceable_head_dims.append(module.sliceable_head_dim)

            for child in module.children():
                fn_recursive_retrieve_sliceable_dims(child)

        # retrieve number of attention layers
        for module in self.children():
            fn_recursive_retrieve_sliceable_dims(module)

        num_sliceable_layers = len(sliceable_head_dims)

        if slice_size == "auto":
            # half the attention head size is usually a good trade-off between
            # speed and memory
            slice_size = [dim // 2 for dim in sliceable_head_dims]
        elif slice_size == "max":
            # make smallest slice possible
            slice_size = num_sliceable_layers * [1]

        slice_size = num_sliceable_layers * [slice_size] if not isinstance(
            slice_size, list) else slice_size

        if len(slice_size) != len(sliceable_head_dims):
            raise ValueError(
                f"You have provided {len(slice_size)}, but {self.config} has {len(sliceable_head_dims)} different"
                f" attention layers. Make sure to match `len(slice_size)` to be {len(sliceable_head_dims)}."
            )

        for i in range(len(slice_size)):
            size = slice_size[i]
            dim = sliceable_head_dims[i]
            if size is not None and size > dim:
                raise ValueError(
                    f"size {size} has to be smaller or equal to {dim}.")

        # Recursively walk through all the children.
        # Any children which exposes the set_attention_slice method
        # gets the message
        def fn_recursive_set_attention_slice(module: nn.Layer,
                                             slice_size: List[int]):
            if hasattr(module, "set_attention_slice"):
                module.set_attention_slice(slice_size.pop())

            for child in module.children():
                fn_recursive_set_attention_slice(child, slice_size)

        reversed_slice_size = list(reversed(slice_size))
        for module in self.children():
            fn_recursive_set_attention_slice(module, reversed_slice_size)

    def _set_gradient_checkpointing(self, module, value=False):
        if isinstance(module, (CrossAttnDownBlock2D, DownBlock2D)):
            module.gradient_checkpointing = value

    def forward(
            self,
            sample: paddle.Tensor,
            timestep: Union[paddle.Tensor, float, int],
            encoder_hidden_states: paddle.Tensor,
            controlnet_cond: paddle.Tensor,
            conditioning_scale: Union[List[float], float]=1.0,
            class_labels: Optional[paddle.Tensor]=None,
            timestep_cond: Optional[paddle.Tensor]=None,
            attention_mask: Optional[paddle.Tensor]=None,
            cross_attention_kwargs: Optional[Dict[str, Any]]=None,
            guess_mode: bool=False,
            return_dict: bool=True, ) -> Union[ControlNetOutput, Tuple]:
        # TODO junnyu, add this to support pure fp16
        sample = sample.cast(self.dtype)

        # check channel order
        channel_order = self.config.controlnet_conditioning_channel_order

        if channel_order == "rgb":
            # in rgb order by default
            ...
        elif channel_order == "bgr":
            controlnet_cond = paddle.flip(controlnet_cond, axis=[1])
        else:
            raise ValueError(
                f"unknown `controlnet_conditioning_channel_order`: {channel_order}"
            )

        # prepare attention_mask
        if attention_mask is not None:
            attention_mask = (1 - attention_mask.cast(sample.dtype)) * -10000.0
            attention_mask = attention_mask.unsqueeze(1)

        # 1. time
        timesteps = timestep
        if not paddle.is_tensor(timesteps):
            timesteps = paddle.to_tensor([timesteps], dtype="int64")
        elif len(timesteps.shape) == 0:
            timesteps = timesteps[None]

        # broadcast to batch dimension in a way that's compatible with ONNX/Core ML
        timesteps = timesteps.expand([sample.shape[0], ])

        t_emb = self.time_proj(timesteps)

        # timesteps does not contain any weights and will always return f32 tensors
        # but time_embedding might actually be running in fp16. so we need to cast here.
        # there might be better ways to encapsulate this.
        t_emb = t_emb.cast(dtype=self.dtype)

        emb = self.time_embedding(t_emb, timestep_cond)

        if self.class_embedding is not None:
            if class_labels is None:
                raise ValueError(
                    "class_labels should be provided when num_class_embeds > 0")

            # maybe cast it to float16
            class_labels = class_labels.cast(self.dtype)
            if self.config.class_embed_type == "timestep":
                class_labels = self.time_proj(class_labels)

            # maybe cast it to int64
            if isinstance(self.class_embedding, nn.Embedding):
                class_labels = class_labels.cast(paddle.int64)
            class_emb = self.class_embedding(class_labels).cast(self.dtype)
            emb = emb + class_emb

        if self.resnet_pre_temb_non_linearity:
            emb = self.down_resnet_temb_nonlinearity(emb)

        # 2. pre-process
        sample = self.conv_in(sample)

        controlnet_cond = self.controlnet_cond_embedding(controlnet_cond)

        sample += controlnet_cond

        # 3. down
        down_block_res_samples = (sample, )

        for downsample_block in self.down_blocks:
            if hasattr(downsample_block, "has_cross_attention"
                       ) and downsample_block.has_cross_attention:
                sample, res_samples = downsample_block(
                    hidden_states=sample,
                    temb=emb,
                    encoder_hidden_states=encoder_hidden_states,
                    attention_mask=attention_mask,
                    cross_attention_kwargs=cross_attention_kwargs, )
            else:
                sample, res_samples = downsample_block(
                    hidden_states=sample, temb=emb)

            down_block_res_samples += res_samples

        # 4. mid
        if self.mid_block is not None:
            sample = self.mid_block(
                sample,
                emb,
                encoder_hidden_states=encoder_hidden_states,
                attention_mask=attention_mask,
                cross_attention_kwargs=cross_attention_kwargs, )

        # 5. Control net blocks

        controlnet_down_block_res_samples = ()

        for down_block_res_sample, controlnet_block in zip(
                down_block_res_samples, self.controlnet_down_blocks):
            down_block_res_sample = controlnet_block(down_block_res_sample)
            controlnet_down_block_res_samples += (down_block_res_sample, )

        down_block_res_samples = controlnet_down_block_res_samples

        mid_block_res_sample = self.controlnet_mid_block(sample)

        # 6. scaling
        if guess_mode:
            scales = paddle.logspace(
                -1, 0, len(down_block_res_samples) + 1)  # 0.1 to 1.0
            scales *= conditioning_scale
            down_block_res_samples = [
                sample * scale
                for sample, scale in zip(down_block_res_samples, scales)
            ]
            mid_block_res_sample *= scales[-1]  # last one
        else:
            # add conditioning_scale https://github.com/huggingface/diffusers/pull/2627
            if isinstance(conditioning_scale, (float, int)):
                down_block_res_samples = [
                    sample * conditioning_scale
                    for sample in down_block_res_samples
                ]
                mid_block_res_sample *= conditioning_scale
            else:
                down_block_res_samples = [
                    sample * ccs
                    for sample, ccs in zip(down_block_res_samples,
                                           conditioning_scale[:-1])
                ]
                mid_block_res_sample *= conditioning_scale[-1]

        if self.config.global_pool_conditions:
            down_block_res_samples = [
                paddle.mean(
                    sample, axis=(2, 3), keepdim=True)
                for sample in down_block_res_samples
            ]
            mid_block_res_sample = paddle.mean(
                mid_block_res_sample, axis=(2, 3), keepdim=True)

        if not return_dict:
            return (down_block_res_samples, mid_block_res_sample)

        return ControlNetOutput(
            down_block_res_samples=down_block_res_samples,
            mid_block_res_sample=mid_block_res_sample)


def zero_module(module):
    for p in module.parameters():
        zeros_(p)
    return module
