# coding=utf-8
# Copyright 2023-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import os
import warnings
from typing import Optional

import paddle
from huggingface_hub.utils import EntryNotFoundError

from ppdiffusers.utils import smart_load
from ppdiffusers.utils.downloader import (
    bos_aistudio_hf_download,
    bos_aistudio_hf_file_exist,
)

from .other import (
    EMBEDDING_LAYER_NAMES,
    SAFETENSORS_WEIGHTS_NAME,
    TORCH_WEIGHTS_NAME,
    WEIGHTS_NAME,
    infer_device,
)
from .peft_types import PeftType


def has_valid_embedding_base_layer(layer):
    """Check if the layer has an embedding base layer"""
    return hasattr(layer, "base_layer") and isinstance(layer.base_layer, (paddle.nn.Linear, paddle.nn.Embedding))


def get_embedding_layer_name(model, layer, is_prompt_learning):
    """Get the name of the embedding module for a given layer."""
    for name, module in model.named_sublayers(include_self=True):
        if (is_prompt_learning and module == layer) or module == layer.base_layer:
            return name
    return None


def get_peft_model_state_dict(
    model,
    state_dict=None,
    adapter_name="default",
    unwrap_compiled=False,
    save_embedding_layers="auto",
    to_diffusers: bool = False,
):
    """
    Get the state dict of the Peft model.

    Args:
        model ([`PeftModel`]): The Peft model. When using paddle.nn.DistributedDataParallel, DeepSpeed or FSDP,
            the model should be the underlying model/unwrapped model (i.e. model.module).
        state_dict (`dict`, *optional*, defaults to `None`):
            The state dict of the model. If not provided, the state dict of the passed model will be used.
        adapter_name (`str`, *optional*, defaults to `"default"`):
            The name of the adapter whose state dict should be returned.
        unwrap_compiled (`bool`, *optional*, defaults to `False`):
            Whether to unwrap the model if torch.compile was used.
        save_embedding_layers (`Union[bool, str]`, , *optional*, defaults to `auto`):
            If `True`, save the embedding layers in addition to adapter weights. If `auto`, checks the common embedding
            layers `peft.utils.other.EMBEDDING_LAYER_NAMES` in config's `target_modules` when available. Based on it
            sets the boolean flag. This only works for 🤗 transformers models.
    """
    if unwrap_compiled:
        model = getattr(model, "_orig_mod", model)

    config = model.peft_config[adapter_name]
    if state_dict is None:
        state_dict = model.state_dict()

    if to_diffusers and config.peft_type in [PeftType.LORA, PeftType.P_TUNING, PeftType.PREFIX_TUNING]:
        need_transpose = []
        new_state_dict = {}
        for k, v in model.named_sublayers(include_self=True):
            if isinstance(v, paddle.nn.Linear):
                need_transpose.append(k + ".weight")
        for k, v in state_dict.items():
            if k in need_transpose:
                v = v.T
            new_state_dict[k] = v
        state_dict = new_state_dict

    if config.peft_type in (PeftType.LORA, PeftType.ADALORA):
        # to_return = lora_state_dict(model, bias=model.peft_config.bias)
        # adapted from `https://github.com/microsoft/LoRA/blob/main/loralib/utils.py`
        # to be used directly with the state dict which is necessary when using DeepSpeed or FSDP
        bias = config.bias
        if bias == "none":
            to_return = {k: state_dict[k] for k in state_dict if "lora_" in k}
        elif bias == "all":
            to_return = {k: state_dict[k] for k in state_dict if "lora_" in k or "bias" in k}
        elif bias == "lora_only":
            to_return = {}
            for k in state_dict:
                if "lora_" in k:
                    to_return[k] = state_dict[k]
                    bias_name = k.split("lora_")[0] + "bias"
                    if bias_name in state_dict:
                        to_return[bias_name] = state_dict[bias_name]
        else:
            raise NotImplementedError
        to_return = {k: v for k, v in to_return.items() if (("lora_" in k and adapter_name in k) or ("bias" in k))}
        if config.peft_type == PeftType.ADALORA:
            rank_pattern = config.rank_pattern
            if rank_pattern is not None:
                rank_pattern = {k.replace(f".{adapter_name}", ""): v for k, v in rank_pattern.items()}
                config.rank_pattern = rank_pattern
                to_return = model.resize_state_dict_by_rank_pattern(rank_pattern, to_return, adapter_name)

    elif config.peft_type == PeftType.LOHA:
        to_return = {k: state_dict[k] for k in state_dict if "hada_" in k}

    elif config.peft_type == PeftType.LOKR:
        to_return = {k: state_dict[k] for k in state_dict if "lokr_" in k}

    elif config.peft_type == PeftType.ADAPTION_PROMPT:
        to_return = {k: state_dict[k] for k in state_dict if k.split(".")[-1].startswith("adaption_")}
    elif config.is_prompt_learning:
        to_return = {}
        if config.peft_type == PeftType.MULTITASK_PROMPT_TUNING:
            to_return["prefix_task_cols"] = model.prompt_encoder[adapter_name].prefix_task_cols
            to_return["prefix_task_rows"] = model.prompt_encoder[adapter_name].prefix_task_rows
            prompt_embeddings = model.prompt_encoder[adapter_name].embedding.weight
        else:
            if config.inference_mode:
                prompt_embeddings = model.prompt_encoder[adapter_name].embedding.weight
            else:
                prompt_embeddings = model.get_prompt_embedding_to_save(adapter_name)
        to_return["prompt_embeddings"] = prompt_embeddings
    elif config.peft_type == PeftType.IA3:
        to_return = {k: state_dict[k] for k in state_dict if "ia3_" in k}
    elif config.peft_type == PeftType.OFT:
        to_return = {k: state_dict[k] for k in state_dict if "oft_" in k}
    else:
        raise NotImplementedError
    if getattr(model, "modules_to_save", None) is not None:
        for key, value in state_dict.items():
            if any(f"{module_name}.modules_to_save.{adapter_name}" in key for module_name in model.modules_to_save):
                to_return[key.replace("modules_to_save.", "")] = value

    # check the common embedding layers in `target_modules` to reset `save_embedding_layers` if necessary
    if (
        save_embedding_layers == "auto"
        and hasattr(config, "target_modules")
        and any(k in config.target_modules for k in EMBEDDING_LAYER_NAMES)
    ):
        warnings.warn("Setting `save_embedding_layers` to `True` as embedding layers found in `target_modules`.")
        save_embedding_layers = True
    elif save_embedding_layers == "auto":
        save_embedding_layers = False

    if save_embedding_layers and hasattr(model, "get_input_embeddings"):
        for layer in [model.get_input_embeddings(), model.get_output_embeddings()]:
            if config.is_prompt_learning or has_valid_embedding_base_layer(layer):
                # support from version >= 0.6.2
                embedding_module_name = get_embedding_layer_name(model, layer, config.is_prompt_learning)
                if embedding_module_name:
                    to_return.update({k: v for k, v in state_dict.items() if embedding_module_name in k})
    elif save_embedding_layers:
        warnings.warn("Could not identify embedding layer(s) because the model is not a 🤗 transformers model.")

    to_return = {k.replace(f".{adapter_name}", ""): v for k, v in to_return.items()}
    return to_return


def set_peft_model_state_dict(model, peft_model_state_dict, adapter_name="default", from_diffusers=False):
    """
    Set the state dict of the Peft model.

    Args:
        model ([`PeftModel`]): The Peft model.
        peft_model_state_dict (`dict`): The state dict of the Peft model.
    """
    config = model.peft_config[adapter_name]
    state_dict = {}
    if getattr(model, "modules_to_save", None) is not None:
        for key, value in peft_model_state_dict.items():
            if any(module_name in key for module_name in model.modules_to_save):
                for module_name in model.modules_to_save:
                    if module_name in key:
                        key = key.replace(module_name, f"{module_name}.modules_to_save.{adapter_name}")
                        break
            state_dict[key] = value
    else:
        state_dict = peft_model_state_dict

    need_transpose = []
    if from_diffusers and config.peft_type in [PeftType.LORA, PeftType.P_TUNING, PeftType.PREFIX_TUNING]:
        need_transpose = []
        for k, v in model.named_sublayers(include_self=True):
            if isinstance(v, paddle.nn.Linear):
                need_transpose.append(k + ".weight")

    if config.peft_type in (PeftType.LORA, PeftType.LOHA, PeftType.LOKR, PeftType.ADALORA, PeftType.IA3, PeftType.OFT):
        peft_model_state_dict = {}
        parameter_prefix = {
            PeftType.IA3: "ia3_",
            PeftType.LORA: "lora_",
            PeftType.ADALORA: "lora_",
            PeftType.LOHA: "hada_",
            PeftType.LOKR: "lokr_",
            PeftType.OFT: "oft_",
        }[config.peft_type]
        for k, v in state_dict.items():
            if parameter_prefix in k:
                suffix = k.split(parameter_prefix)[1]
                if "." in suffix:
                    suffix_to_replace = ".".join(suffix.split(".")[1:])
                    k = k.replace(suffix_to_replace, f"{adapter_name}.{suffix_to_replace}")
                else:
                    k = f"{k}.{adapter_name}"
                if k in need_transpose:
                    v = v.T
                peft_model_state_dict[k] = v
            else:
                if k in need_transpose:
                    v = v.T
                peft_model_state_dict[k] = v
        if config.peft_type == PeftType.ADALORA:
            rank_pattern = config.rank_pattern
            if rank_pattern is not None:
                model.resize_modules_by_rank_pattern(rank_pattern, adapter_name)
    elif config.is_prompt_learning or config.peft_type == PeftType.ADAPTION_PROMPT:
        peft_model_state_dict = state_dict
    else:
        raise NotImplementedError
    with warnings.catch_warnings():
        from ppdiffusers.models.modeling_utils import faster_set_state_dict

        # warnings.resetwarnings()
        # warnings.filterwarnings("ignore", message=r".*is not found in the provided dict.*")
        faster_set_state_dict(model, peft_model_state_dict)
        load_result = ""
        # load_result = model.set_state_dict(
        #     peft_model_state_dict,
        # )
    if config.is_prompt_learning:
        model.prompt_encoder[adapter_name].embedding.set_state_dict(
            {"weight": peft_model_state_dict["prompt_embeddings"]},
        )

    if config.peft_type == PeftType.MULTITASK_PROMPT_TUNING:
        with warnings.catch_warnings():
            warnings.resetwarnings()
            warnings.filterwarnings("ignore", message=r".*is not found in the provided dict.*")
            model.prompt_encoder[adapter_name].set_state_dict(
                peft_model_state_dict,
            )
    return load_result


def load_peft_weights(
    model_id: str, device: Optional[str] = None, from_diffusers: bool = False, **bos_aistudio_hf_hub_download_kwargs
) -> dict:
    r"""
    A helper method to load the PEFT weights from the HuggingFace Hub or locally

    Args:
        model_id (`str`):
            The local path to the adapter weights or the name of the adapter to load from the HuggingFace Hub.
        device (`str`):
            The device to load the weights onto.
        bos_aistudio_hf_hub_download_kwargs (`dict`):
            Additional arguments to pass to the `hf_hub_download` method when loading from the HuggingFace Hub.
    """
    path = (
        os.path.join(model_id, bos_aistudio_hf_hub_download_kwargs["subfolder"])
        if bos_aistudio_hf_hub_download_kwargs.get("subfolder", None) is not None
        else model_id
    )

    if device is None:
        device = infer_device()

    if from_diffusers:
        NO_SAFE_TENSORS_NAME = TORCH_WEIGHTS_NAME
    else:
        NO_SAFE_TENSORS_NAME = WEIGHTS_NAME
    if os.path.exists(os.path.join(path, SAFETENSORS_WEIGHTS_NAME)):
        filename = os.path.join(path, SAFETENSORS_WEIGHTS_NAME)
    elif os.path.exists(os.path.join(path, NO_SAFE_TENSORS_NAME)):
        filename = os.path.join(path, NO_SAFE_TENSORS_NAME)
    else:
        token = bos_aistudio_hf_hub_download_kwargs.get("token", None)
        if token is None:
            token = bos_aistudio_hf_hub_download_kwargs.get("use_auth_token", None)

        has_remote_safetensors_file = bos_aistudio_hf_file_exist(
            repo_id=model_id,
            filename=SAFETENSORS_WEIGHTS_NAME,
            revision=bos_aistudio_hf_hub_download_kwargs.get("revision", None),
            repo_type=bos_aistudio_hf_hub_download_kwargs.get("repo_type", None),
            token=token,
            from_bos=bos_aistudio_hf_hub_download_kwargs.get("from_bos", True),
            from_aistudio=bos_aistudio_hf_hub_download_kwargs.get("from_aistudio", False),
            from_hf_hub=bos_aistudio_hf_hub_download_kwargs.get("from_hf_hub", False),
        )

        if has_remote_safetensors_file:
            # Priority 1: load safetensors weights
            filename = bos_aistudio_hf_download(
                model_id,
                SAFETENSORS_WEIGHTS_NAME,
                **bos_aistudio_hf_hub_download_kwargs,
            )
        else:
            try:
                filename = bos_aistudio_hf_download(
                    model_id, NO_SAFE_TENSORS_NAME, **bos_aistudio_hf_hub_download_kwargs
                )
            except EntryNotFoundError:
                raise ValueError(
                    f"Can't find weights for {model_id} in {model_id} or in the BOS, AIstudio or Hugging Face Hub. "
                    f"Please check that the file {NO_SAFE_TENSORS_NAME} or {SAFETENSORS_WEIGHTS_NAME} is present at {model_id}."
                )
    adapters_weights = smart_load(filename, return_numpy=True, return_is_torch_weight=True)
    from_diffusers = adapters_weights.pop("is_torch_weight", False)
    return adapters_weights, from_diffusers
