#
# A test of `processing.Pool` class
#

from processing import Pool, TimeoutError
from processing import cpuCount, currentProcess, freezeSupport

import time, random, sys

#
# Functions used by test code
#

def calculate(func, args):
    result = func(*args)
    return '%s says that %s%s = %s' % \
        (currentProcess().getName(), func.__name__, args, result)

def calculatestar(args):
    return calculate(*args)

def mul(a, b):
    time.sleep(0.5*random.random())
    return a * b

def plus(a, b):
    time.sleep(0.5*random.random())
    return a + b

def f(x):
    return 1.0 / (x-5.0)

def pow3(x):
    return x**3

def noop(x):
    pass

#
# Test code
#

def test():
    print 'cpuCount() = %d\n' % cpuCount()
    
    #
    # Create pool
    #
    
    PROCESSES = 4
    print 'Creating pool with %d processes\n' % PROCESSES
    pool = Pool(PROCESSES)    

    #
    # Tests
    #

    TASKS = [(mul, (i, 7)) for i in range(10)] + \
            [(plus, (i, 8)) for i in range(10)]

    results = [pool.apply_async(calculate, t) for t in TASKS]
    imap_it = pool.imap(calculatestar, TASKS)
    imap_unordered_it = pool.imap_unordered(calculatestar, TASKS)

    print 'Ordered results using pool.apply():'
    for r in results:
        print '\t', r.get()
    print

    print 'Ordered results using pool.imap():'        
    for x in imap_it:
        print '\t', x
    print

    print 'Unordered results using pool.imap_unordered():'
    for x in imap_unordered_it:
        print '\t', x
    print

    print 'Ordered results using pool.map() --- will block till complete:'
    for x in pool.map(calculatestar, TASKS):
        print '\t', x
    print    

    #
    # Simple benchmarks
    #

    N = 100000
    print 'def pow3(x): return x**3'
    
    t = time.time()
    A = map(pow3, xrange(N))
    print '\tmap(pow3, xrange(%d)):\t%s seconds' % (N, time.time() - t)
    
    t = time.time()
    B = pool.map(pow3, xrange(N))
    print '\tpool.map(pow3, xrange(%d)):\t%s seconds' % (N, time.time() - t)
    
    assert A == B, (len(A), len(B))
    print
    
    L = [None] * 1000000
    print 'def noop(x): pass'
    print 'L = [None] * 1000000'
    
    t = time.time()
    A = map(noop, L)
    print '\tmap(noop, L):\t\t\t%s seconds' % (time.time() - t)
    
    t = time.time()
    B = pool.map(noop, L)
    print '\tpool.map(noop, L):\t\t%s seconds' % (time.time() - t)

    assert A == B, (len(A), len(B))
    print    

    #
    # Test error handling
    #

    print 'Testing error handling:'

    try:
        print pool.apply(f, (5,))
        raise AssertionError, 'expected ZeroDivisionError'
    except ZeroDivisionError:
        print '\tGot ZeroDivisionError as expected from pool.apply()'

    try:
        print pool.map(f, range(10))
        raise AssertionError, 'expected ZeroDivisionError'
    except ZeroDivisionError:
        print '\tGot ZeroDivisionError as expected from pool.map()'

    try:
        print list(pool.imap(f, range(10)))
        raise AssertionError, 'expected ZeroDivisionError'
    except ZeroDivisionError:
        print '\tGot ZeroDivisionError as expected from list(pool.imap())'

    it = pool.imap(f, range(10))
    for i in range(10):
        try:
            x = it.next()
            if i == 5:
                raise AssertionError, 'expected ZeroDivisionError'
        except ZeroDivisionError:
            if i == 5:
                pass
        except StopIteration:
            break
    assert i == 9
    print '\tGot ZeroDivisionError as expected from IMapIterator.next()'
    print
    
    #
    # Testing timeouts
    #
    
    print 'Testing ApplyResult.get() with timeout:',
    res = pool.apply_async(calculate, TASKS[0])
    while 1:
        sys.stdout.flush()
        try:
            sys.stdout.write('\n\t%s' % res.get(0.02))
            break
        except TimeoutError:
            sys.stdout.write('.')
    print
    print

    print 'Testing IMapIterator.next() with timeout:',
    it = pool.imap(calculatestar, TASKS)
    while 1:
        sys.stdout.flush()
        try:
            sys.stdout.write('\n\t%s' % it.next(0.02))
        except StopIteration:
            break
        except TimeoutError:
            sys.stdout.write('.')
    print
    print
            
    #
    # Assert nothing left over
    #
    
    assert not pool._cache


if __name__ == '__main__':
    freezeSupport()
    test()
