import decimal
import time
from time import sleep

from boto3.dynamodb.conditions import Key, Attr
from botocore.exceptions import ClientError

from properly_util_python.helper_utils import dynamo_to_dict
from properly_util_python.settings import dynamodb_resource as dynamodb


def save_to_dynamo(table_name, item: dict, *args, **kwargs):
    item = add_dynamo_metadata(item)
    item['id'] = str(item['id'])
    if 'namespace' in kwargs:
        item['id'] = kwargs['namespace'] + '-' + item['id']

    table = dynamodb.Table(table_name)
    response = table.get_item(
        TableName=table_name,
        Key={
            'id': item['id']
        }
    )
    if not response.get('Item') or kwargs.get('overwrite_existing'):

        if response.get('Item'):
            print('overwrite existing dynamo id:', item['id'])

        response = table.put_item(Item=dict_to_dynamo(item))

        print('Saved to dynamo table: %s id: %s' % (table_name, item['id']))
    else:
        print('Skipping, this item already exists in dynamo id:', item['id'])

    return response


def dict_to_dynamo(raw_dict):
    dynamo_dict = {}

    # print(json.dumps(raw_dict, indent=4))
    for key, val in raw_dict.items():
        if isinstance(val, float):
            dynamo_dict[key] = decimal.Decimal(str(val))
        elif isinstance(val, dict):
            dynamo_dict[key] = dict_to_dynamo(raw_dict[key])

        # turn empty strings to null
        elif val == 0:
            dynamo_dict[key] = decimal.Decimal('0')
        elif not val:
            dynamo_dict[key] = None
        else:
            dynamo_dict[key] = val
    return dynamo_dict


def add_dynamo_metadata(data):
    data['dynamoUploadTimeStamp'] = int(round(time.time() * 1000))

    return data


def get_dynamo_items(table_name, table_filter=None, *args, **kwargs):
    RETRY_EXCEPTIONS = ('ProvisionedThroughputExceededException',
                        'ThrottlingException')
    table = dynamodb.Table(table_name)

    response = table.scan(**table_filter)
    items = response['Items']

    # todo watch for memory overflow in batching all the items as once
    # https://gist.github.com/shentonfreude/8d26ca1fc93fdb801b2c
    # https://github.com/boto/boto3/issues/597#issuecomment-323982159
    retries = 0
    max_retry = 3
    while 'LastEvaluatedKey' in response and retries < max_retry:
        try:

            response = table.scan(ExclusiveStartKey=response['LastEvaluatedKey'], **table_filter)
            items.extend(response['Items'])
            retries = 0
        except ClientError as err:
            if err.response['Error']['Code'] not in RETRY_EXCEPTIONS:
                raise
            print('WHOA, too fast, slow it down retries={}, items={}'.format(retries, len(items)), )
            sleep(2 ** retries)
            retries += 1  # TODO max limit

    return dynamo_to_dict(items)


def get_dynamo_item(table_name, item_value, item_key='id', ):
    """
    Return the dict representation of an object from dynamo
    :param table_name:
    :param item_value:
    :param item_key:
    :return: item: dict
    """
    deal_table = dynamodb.Table(table_name)
    resp = deal_table.get_item(
        TableName=table_name,
        Key={
            item_key: item_value
        }
    )
    item = resp.get('Item')

    if not item:
        return item

    return dynamo_to_dict(item)


def query_table(table_name, filter_key=None, filter_value=None, filter_type='eq'):
    """
    Perform a query operation on the table. Can specify filter_key (col name) and its value to be filtered.
    Returns the response.
    """

    # todo replace filter if-statements with, getattr filter statement

    if filter_type == 'not_exists':
        table_filter = {
            'FilterExpression': Attr(filter_key).not_exists()
        }
    elif filter_type == 'lt':
        table_filter = {
            'FilterExpression': Attr(filter_key).lt(filter_value)
        }
    elif filter_type == 'contains':
        table_filter = {
            'FilterExpression': Attr(filter_key).contains(filter_value)
        }

    else:
        table_filter = {
            'FilterExpression': Attr(filter_key).eq(filter_value)
        }

    items = get_dynamo_items(table_name, table_filter)
    return items


def get_last_key(config_id, table_name, META_KEY_MAP):
    # remove "dev" constant when moving to environment variables
    meta_table = dynamodb.Table(table_name)

    response = meta_table.get_item(
        Key={
            META_KEY_MAP["META_ID"]: config_id,
        })
    config = response.get('Item', None)
    if config is None:
        return None

    last_evaluated_key = config.get(META_KEY_MAP["META_CONFIG_VAL"])

    return last_evaluated_key


def set_last_key(last_eval_config, table_name, val, META_KEY_MAP):
    # remove "dev" constant when moving to environment variables
    meta_table = dynamodb.Table(table_name)

    now_seconds_epoch_utc = int(time.time())

    config_object = last_eval_config.copy()
    config_object[META_KEY_MAP["META_CONFIG_VAL"]] = val
    config_object[META_KEY_MAP["TABLE_UPDATED_AT"]] = now_seconds_epoch_utc

    meta_table.put_item(Item=config_object)


#############
# Query for next set of items

def query_some_items(table_name, META_CONFIG, META_KEY_MAP, limit=20):
    meta_table_name = "{0}-{1}".format("dev", META_KEY_MAP["META_TABLE"])

    last_evaluated = get_last_key(META_CONFIG[META_KEY_MAP["META_ID"]], meta_table_name, META_KEY_MAP)
    table = dynamodb.Table(table_name)

    # performance improved by querying for the reserved value, no retries, if throughput exceeded will pick up next time
    if last_evaluated:
        response = table.query(
            IndexName=META_CONFIG['keyName'] + '-index',
            KeyConditionExpression=Key(META_CONFIG['keyName']).eq(-1),
            Limit=limit,
            ExclusiveStartKey=last_evaluated,
        )
    else:
        response = table.query(
            IndexName=META_CONFIG['keyName'] + '-index',
            KeyConditionExpression=Key(META_CONFIG['keyName']).eq(-1),
            Limit=limit)

    last_evaluated = response.get('LastEvaluatedKey', None)
    set_last_key(META_CONFIG, meta_table_name, last_evaluated, META_KEY_MAP)

    items_to_clean = response.get('Items')
    items_to_clean = dynamo_to_dict(items_to_clean)

    return items_to_clean


def scan_some_items(table_name, META_CONFIG, META_KEY_MAP, limit=20):
    meta_table_name = "{0}-{1}".format("dev", META_KEY_MAP["META_TABLE"])

    last_evaluated = get_last_key(META_CONFIG[META_KEY_MAP["META_ID"]], meta_table_name, META_KEY_MAP)
    table = dynamodb.Table(table_name)

    # performance improved by querying for the reserved value, no retries, if throughput exceeded will pick up next time
    if last_evaluated:
        response = table.scan(
            Limit=limit,
            ExclusiveStartKey=last_evaluated,
        )
    else:
        response = table.scan(
            Limit=limit)

    last_evaluated = response.get('LastEvaluatedKey', None)
    set_last_key(META_CONFIG, meta_table_name, last_evaluated, META_KEY_MAP)

    items_to_clean = response.get('Items')
    items_to_clean = dynamo_to_dict(items_to_clean)

    return items_to_clean
