import pytest
import requests
from pulp_python.tests.functional.constants import (
    PYTHON_EGG_FILENAME,
    PYTHON_EGG_URL,
    PYTHON_WHEEL_FILENAME,
    PYTHON_WHEEL_URL,
    PYTHON_EGG_SHA256,
    PYTHON_WHEEL_SHA256,
)
from pulp_python.tests.functional.utils import ensure_metadata
from urllib.parse import urljoin


@pytest.mark.parametrize(
    "pkg_filename, pkg_url",
    [(PYTHON_WHEEL_FILENAME, PYTHON_WHEEL_URL), (PYTHON_EGG_FILENAME, PYTHON_EGG_URL)],
)
def test_synchronous_package_upload(
    delete_orphans_pre, download_python_file, gen_user, python_bindings, pkg_filename, pkg_url
):
    """
    Test synchronously uploading a Python package with labels.
    """
    python_file = download_python_file(pkg_filename, pkg_url)

    # Upload a unit with labels
    with gen_user(model_roles=["python.python_package_uploader"]):
        labels = {"key_1": "value_1"}
        content_body = {"file": python_file, "pulp_labels": labels}
        package = python_bindings.ContentPackagesApi.upload(**content_body)
        assert package.pulp_labels == labels
        assert package.name == "shelf-reader"
        assert package.filename == pkg_filename

    # Check that uploading the same unit again with different (or same) labels has no effect
    with gen_user(model_roles=["python.python_package_uploader"]):
        labels_2 = {"key_2": "value_2"}
        content_body_2 = {"file": python_file, "pulp_labels": labels_2}
        duplicate_package = python_bindings.ContentPackagesApi.upload(**content_body_2)
        assert duplicate_package.pulp_href == package.pulp_href
        assert duplicate_package.pulp_labels == package.pulp_labels
        assert duplicate_package.pulp_labels != labels_2

    # Check that the upload fails if the user does not have the required permissions
    with gen_user(model_roles=[]):
        with pytest.raises(python_bindings.ApiException) as ctx:
            python_bindings.ContentPackagesApi.upload(**content_body)
        assert ctx.value.status == 403


@pytest.mark.parallel
def test_synchronous_package_upload_with_metadata(
    download_python_file,
    monitor_task,
    pulp_content_url,
    python_bindings,
    python_distribution_factory,
    python_repo,
):
    """
    Test that the synchronous upload of a Python wheel package creates a metadata artifact.
    """
    python_file = download_python_file(PYTHON_WHEEL_FILENAME, PYTHON_WHEEL_URL)
    content_body = {"file": python_file}
    content = python_bindings.ContentPackagesApi.upload(**content_body)

    body = {"add_content_units": [content.pulp_href]}
    monitor_task(python_bindings.RepositoriesPythonApi.modify(python_repo.pulp_href, body).task)
    distro = python_distribution_factory(repository=python_repo)

    # Test that metadata is accessible
    ensure_metadata(pulp_content_url, distro.base_path, PYTHON_WHEEL_FILENAME)


@pytest.mark.parallel
def test_legacy_upload_invalid_protocol_version(
    python_empty_repo_distro, python_package_dist_directory
):
    _, egg_file, _ = python_package_dist_directory
    _, distro = python_empty_repo_distro()
    url = urljoin(distro.base_url, "legacy/")
    with open(egg_file, "rb") as f:
        response = requests.post(
            url,
            data={"sha256_digest": PYTHON_EGG_SHA256, "protocol_version": 2},
            files={"content": f},
            auth=("admin", "password"),
        )
    assert response.status_code == 400
    assert response.json()["protocol_version"] == ['"2" is not a valid choice.']

    with open(egg_file, "rb") as f:
        response = requests.post(
            url,
            data={"sha256_digest": PYTHON_EGG_SHA256, "protocol_version": 0},
            files={"content": f},
            auth=("admin", "password"),
        )
    assert response.status_code == 400
    assert response.json()["protocol_version"] == ['"0" is not a valid choice.']


@pytest.mark.parallel
def test_legacy_upload_invalid_filetype(python_empty_repo_distro, python_package_dist_directory):
    _, egg_file, wheel_file = python_package_dist_directory
    _, distro = python_empty_repo_distro()
    url = urljoin(distro.base_url, "legacy/")
    with open(egg_file, "rb") as f:
        response = requests.post(
            url,
            data={"sha256_digest": PYTHON_EGG_SHA256, "filetype": "bdist_wheel"},
            files={"content": f},
            auth=("admin", "password"),
        )
    assert response.status_code == 400
    assert response.json()["filetype"] == [
        "filetype bdist_wheel does not match found filetype sdist for file shelf-reader-0.1.tar.gz"
    ]

    with open(wheel_file, "rb") as f:
        response = requests.post(
            url,
            data={"sha256_digest": PYTHON_WHEEL_SHA256, "filetype": "sdist"},
            files={"content": f},
            auth=("admin", "password"),
        )
    assert response.status_code == 400
    assert response.json()["filetype"] == [
        "filetype sdist does not match found filetype bdist_wheel for file shelf_reader-0.1-py2-none-any.whl"  # noqa: E501
    ]


@pytest.mark.parallel
def test_legacy_upload_invalid_metadata_version(
    python_empty_repo_distro, python_package_dist_directory
):
    _, egg_file, _ = python_package_dist_directory
    _, distro = python_empty_repo_distro()
    url = urljoin(distro.base_url, "legacy/")
    with open(egg_file, "rb") as f:
        response = requests.post(
            url,
            data={"sha256_digest": PYTHON_EGG_SHA256, "metadata_version": "3.0"},
            files={"content": f},
            auth=("admin", "password"),
        )
    assert response.status_code == 400
    assert response.json()["metadata_version"] == ['"3.0" is not a valid choice.']
