import os
import shutil
import zipfile

from datetime import datetime

from pyf.componentized.components import Component
from pyf.dataflow import component
from pyf.services.core.storage import get_storage


class FilesPostHandler(Component):
    """Copy files generated by the process in a specific output folder, and
    optionnally group them as a Zip file.

    This node is to be used post process, not during the process.

    It is controled by two configuration keys:
      - output_folder: specify where the files will be copied
      - to_zip: if True, then files will be added to a zip archive and this zip
      will be moved to the output folder, not the files.

    An example of usage is:
    <node type="consumer" pluginname="files_post_handler" name="zip_output_files">
      <output_folder>/tmp</output_folder>
      <to_zip>True</to_zip>
    </node>
    """
    name = "files_post_handler"

    def __init__(self, config_node, process_name):
        #super(ReportsCopier, self).__init__(config_node, name)
        self.config_node = config_node
        self.process_name = process_name

        self.storage = get_storage('output')

        self.output_folder = self.get_config_key('output_folder')

        if not self.output_folder:
            raise ValueError(
                    'You must set an output folder to copy output files')

        if not os.path.exists(self.output_folder):
            raise ValueError(
                    'Output directory %s does not exist' % self.output_folder)

        self.to_zip = self.__eval_bool(self.get_config_key('to_zip', ''))

    def __eval_bool(self, s):
        return s.lower() in ('y', 'yes', 'true')

    def __get_zip_target_filename(self, event):
        date_iso = datetime.strftime(datetime.now(), '%Y-%m-%dT%H-%M-%S')

        if event.variant_name is not None:
            target = "%s_%s_%s_%s.zip" % (
                event.tube.name, event.variant_name, event.id, date_iso)
        else:
            target = "%s_%s_%s.zip" % (
                event.tube.name, event.id, date_iso)

        return target

    @component('IN', 'OUT')
    def launch(self, values, out):
        for event in values:
            if event.output_files:
                if self.to_zip:
                    zip_target_filename = os.path.join(
                            self.output_folder,
                            self.__get_zip_target_filename(event))
                    zarchive = zipfile.ZipFile(
                            zip_target_filename,
                            mode='w',
                            compression=zipfile.ZIP_DEFLATED)

                for output_file in event.output_files:
                    storage_filename = self.storage.get_filename(
                            output_file.file_uuid)

                    if self.to_zip:
                        zarchive.write(storage_filename, output_file.filename)

                    else:
                        target_filename = os.path.join(
                                self.output_folder, output_file.filename)
                        shutil.copy(storage_filename, target_filename)

                    yield True

                if self.to_zip:
                    zarchive.close()
                    self.message_callback(
                            "Files were added to the %s archive" % zip_target_filename,
                            message_type="success")

                else:
                    self.message_callback(
                            "Files were copied to the %s folder" % self.output_folder,
                            message_type="success")
            
            yield True

