# =================================================================
#
# Author: Benjamin Webb <bwebb@lincolninst.edu>
#
# Copyright (c) 2025 Center for Geospatial Solutions
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.
#
# =================================================================

import logging
from requests import get

from pygeofilter.parsers.cql_json import parse as parse_cql_json

from pygeoapi.config import get_config
from pygeoapi.plugin import load_plugin
from pygeoapi.process.base import BaseProcessor, ProcessorExecuteError
from pygeoapi.util import get_provider_default, filter_dict_by_key_value


LOGGER = logging.getLogger(__name__)

CONFIG = get_config()
COLLECTIONS = filter_dict_by_key_value(CONFIG['resources'], 'type', 'collection')
# TODO: Filter collections for those that support CQL


PROCESS_DEF = CONFIG['resources']['intersector']
PROCESS_DEF.update(
    {
        'version': '0.1.0',
        'id': 'intersector',
        'title': 'Intersector',
        'description': (
            'A process that takes a URL of a geoJSON, and returns all intersections.'
        ),
        'links': [
            {
                'type': 'text/html',
                'rel': 'about',
                'title': 'information',
                'href': 'https://example.org/process',
                'hreflang': 'en-US',
            }
        ],
        'inputs': {
            'url': {
                'title': {'en': 'Feature URL'},
                'description': {'en': 'URL of valid feature geoJSON'},
                'keywords': {'en': ['geojson', 'feature', 'url']},
                'schema': {'type': 'string', 'default': None},
                'minOccurs': 1,
                'maxOccurs': 1,
                'metadata': None,  # TODO how to use?
            },
            'collection': {
                'title': {'en': 'Feature Collection'},
                'description': {'en': 'Feature Collection'},
                'keywords': {'en': ['OGC API', 'collection']},
                'schema': {
                    'type': 'string',
                    'example': next(iter(COLLECTIONS)),
                    'enum': list(COLLECTIONS),
                },
                'minOccurs': 1,
                'maxOccurs': 1,
                'metadata': None,  # TODO how to use?
            },
            'geom_field': {
                'title': {'en': 'Geometry Field'},
                'description': {'en': 'Geometry field of valid feature intersect'},
                'keywords': {'en': ['geometry', 'intersect', 'field']},
                'schema': {'type': 'string', 'default': 'geom'},
                'minOccurs': 0,
                'maxOccurs': 1,
                'metadata': None,  # TODO how to use?
            },
        },
        'outputs': {
            'path': {
                'title': {'en': 'FeatureCollection'},
                'description': {
                    'en': 'A geoJSON FeatureCollection of the '
                    'path generated by the intersection process'
                },
                'schema': {'type': 'object', 'contentMediaType': 'application/json'},
            }
        },
        'example': {
            'inputs': {
                'url': 'https://demo.pygeoapi.io/master/collections/obs/items/238',
                'collection': next(iter(COLLECTIONS)),
            }
        },
    }
)


class IntersectionProcessor(BaseProcessor):
    """Intersection Processor"""

    def __init__(self, processor_def):
        """
        Initialize object

        :param processor_def: provider definition

        :returns: pygeoapi.process.intersect.IntersectionProcessor
        """
        LOGGER.debug('IntersectionProcesser init')
        super().__init__(processor_def, PROCESS_DEF)

    def execute(self, data, outputs=None):
        """
        Execute Intersection Process

        :param data: processor arguments
        :param outputs: processor outputs

        :returns: 'application/json'
        """
        mimetype = 'application/json'

        if not data.get('url') or not data.get('collection'):
            raise ProcessorExecuteError(f'Invalid input: {data.items()}')
        feature_url = data['url']
        geom_field = data.get('geom_field')
        collection = data['collection']

        LOGGER.debug(f'Fetching {feature_url}')
        params = {'f': 'json'}
        feature = get(feature_url, params=params).json()
        if not feature.get('geometry'):
            raise ProcessorExecuteError(f'Invalid geoJSON: {feature.items()}')

        for cname, c in COLLECTIONS.items():
            if str(collection) != cname:
                continue

            p = get_provider_default(c['providers'])
            provider = load_plugin('provider', p)
            if 'geom_field' in p:
                geom_field = p['geom_field']
                LOGGER.debug(f'Using geom from configuration: {geom_field}')
            else:
                LOGGER.debug(f'Using provided geom field: {geom_field}')
            if not geom_field:
                msg = f'Invalid geom_field: {data.items()}'
                raise ProcessorExecuteError(msg)

            LOGGER.debug(f'Intesecting {cname} with backend {provider}')
            outputs = self._intersect(feature, geom_field, provider)

        LOGGER.debug('Returning response')
        return mimetype, outputs

    def _intersect(self, feature, geom_field, provider):
        """
        Private Function: Use feature to run CQL intersection

        :param feature: feature
        :param geom_field: feature geomtry field
        :param provider: OGC API Provider definition

        :returns: List of GeoJSON Features
        """
        filter_ = parse_cql_json(
            {'intersects': [{'property': geom_field}, feature['geometry']]}
        )
        LOGGER.debug(f'Making CQL query: {filter_}')
        _ = provider.query(resulttype='hits', filterq=filter_)
        fc = provider.query(limit=_['numberMatched'], filterq=filter_)

        LOGGER.info(f'Returning {fc["numberReturned"]} intersections')
        return fc

    def __repr__(self):
        return f'<IntersectionProcessor> {self.name}'
