############################ Copyrights and license ############################
#                                                                              #
# Copyright 2012 Christopher Gilbert <christopher.john.gilbert@gmail.com>      #
# Copyright 2012 Steve English <steve.english@navetas.com>                     #
# Copyright 2012 Vincent Jacques <vincent@vincent-jacques.net>                 #
# Copyright 2012 Zearin <zearin@gonk.net>                                      #
# Copyright 2013 AKFish <akfish@gmail.com>                                     #
# Copyright 2013 Adrian Petrescu <adrian.petrescu@maluuba.com>                 #
# Copyright 2013 Cameron White <cawhite@pdx.edu>                               #
# Copyright 2013 David Farr <david.farr@sap.com>                               #
# Copyright 2013 Mark Roddy <markroddy@gmail.com>                              #
# Copyright 2013 Vincent Jacques <vincent@vincent-jacques.net>                 #
# Copyright 2013 martinqt <m.ki2@laposte.net>                                  #
# Copyright 2014 Vincent Jacques <vincent@vincent-jacques.net>                 #
# Copyright 2015 Aaron Levine <allevin@sandia.gov>                             #
# Copyright 2015 Christopher Wilcox <git@crwilcox.com>                         #
# Copyright 2015 Dan Vanderkam <danvdk@gmail.com>                              #
# Copyright 2015 Ed Holland <eholland@alertlogic.com>                          #
# Copyright 2015 Enix Yu <enix223@163.com>                                     #
# Copyright 2015 Jay <ja.geb@me.com>                                           #
# Copyright 2015 Jimmy Zelinskie <jimmyzelinskie@gmail.com>                    #
# Copyright 2015 Jonathan Debonis <jon@ip-172-20-10-5.ec2.internal>            #
# Copyright 2015 Kevin Lewandowski <kevinsl@gmail.com>                         #
# Copyright 2015 Kyle Hornberg <khornberg@users.noreply.github.com>            #
# Copyright 2015 edhollandAL <eholland@alertlogic.com>                         #
# Copyright 2016 @tmshn <tmshn@r.recruit.co.jp>                                #
# Copyright 2016 Dustin Spicuzza <dustin@virtualroadside.com>                  #
# Copyright 2016 Enix Yu <enix223@163.com>                                     #
# Copyright 2016 Jannis Gebauer <ja.geb@me.com>                                #
# Copyright 2016 Per Øyvind Karlsen <proyvind@moondrake.org>                   #
# Copyright 2016 Peter Buckley <dx-pbuckley@users.noreply.github.com>          #
# Copyright 2016 Sylvus <Sylvus@users.noreply.github.com>                      #
# Copyright 2016 fukatani <nannyakannya@gmail.com>                             #
# Copyright 2016 ghfan <gavintofan@gmail.com>                                  #
# Copyright 2017 Andreas Lutro <anlutro@gmail.com>                             #
# Copyright 2017 Ben Firshman <ben@firshman.co.uk>                             #
# Copyright 2017 Chris McBride <thehighlander@users.noreply.github.com>        #
# Copyright 2017 Hugo <hugovk@users.noreply.github.com>                        #
# Copyright 2017 Jannis Gebauer <ja.geb@me.com>                                #
# Copyright 2017 Jason White <jasonwhite@users.noreply.github.com>             #
# Copyright 2017 Jimmy Zelinskie <jimmy.zelinskie+git@gmail.com>               #
# Copyright 2017 Nhomar Hernández [Vauxoo] <nhomar@vauxoo.com>                 #
# Copyright 2017 Simon <spam@esemi.ru>                                         #
# Copyright 2018 Aaron L. Levine <allevin@sandia.gov>                          #
# Copyright 2018 AetherDeity <aetherdeity+github@gmail.com>                    #
# Copyright 2018 Alice GIRARD <bouhahah@gmail.com>                             #
# Copyright 2018 Andrew Smith <espadav8@gmail.com>                             #
# Copyright 2018 Benoit Latinier <benoit@latinier.fr>                          #
# Copyright 2018 Brian Torres-Gil <btorres-gil@paloaltonetworks.com>           #
# Copyright 2018 Hayden Fuss <wifu1234@gmail.com>                              #
# Copyright 2018 Ilya Konstantinov <ilya.konstantinov@gmail.com>               #
# Copyright 2018 Jacopo Notarstefano <jacopo.notarstefano@gmail.com>           #
# Copyright 2018 John Hui <j-hui@users.noreply.github.com>                     #
# Copyright 2018 Justin Kufro <jkufro@andrew.cmu.edu>                          #
# Copyright 2018 Mateusz Loskot <mateusz@loskot.net>                           #
# Copyright 2018 Michael Behrisch <oss@behrisch.de>                            #
# Copyright 2018 Nicholas Buse <NicholasBuse@users.noreply.github.com>         #
# Copyright 2018 Philip May <eniak.info@gmail.com>                             #
# Copyright 2018 Raihaan <31362124+res0nance@users.noreply.github.com>         #
# Copyright 2018 Shinichi TAMURA <shnch.tmr@gmail.com>                         #
# Copyright 2018 Steve Kowalik <steven@wedontsleep.org>                        #
# Copyright 2018 Vinay Hegde <hegde.vi@husky.neu.edu>                          #
# Copyright 2018 Wan Liuyang <tsfdye@gmail.com>                                #
# Copyright 2018 Will Yardley <wyardley@users.noreply.github.com>              #
# Copyright 2018 Yossarian King <yggy@blackbirdinteractive.com>                #
# Copyright 2018 per1234 <accounts@perglass.com>                               #
# Copyright 2018 sechastain <sechastain@gmail.com>                             #
# Copyright 2018 sfdye <tsfdye@gmail.com>                                      #
# Copyright 2019 Adam Baratz <adam.baratz@gmail.com>                           #
# Copyright 2019 Alex <alexmusa@users.noreply.github.com>                      #
# Copyright 2019 Kevin LaFlamme <k@lamfl.am>                                   #
# Copyright 2019 Olof-Joachim Frahm (欧雅福) <olof@macrolet.net>                  #
# Copyright 2019 Steve Kowalik <steven@wedontsleep.org>                        #
# Copyright 2019 Tim Gates <tim.gates@iress.com>                               #
# Copyright 2019 Wan Liuyang <tsfdye@gmail.com>                                #
# Copyright 2019 Will Li <cuichen.li94@gmail.com>                              #
# Copyright 2020 Alice GIRARD <bouhahah@gmail.com>                             #
# Copyright 2020 Anuj Bansal <bansalanuj1996@gmail.com>                        #
# Copyright 2020 Chris de Graaf <chrisadegraaf@gmail.com>                      #
# Copyright 2020 Dhruv Manilawala <dhruvmanila@gmail.com>                      #
# Copyright 2020 Dominic Davis-Foster <dominic@davis-foster.co.uk>             #
# Copyright 2020 Florent Clarret <florent.clarret@gmail.com>                   #
# Copyright 2020 Glenn McDonald <testworksau@users.noreply.github.com>         #
# Copyright 2020 Huw Jones <huwcbjones@outlook.com>                            #
# Copyright 2020 Mark Bromell <markbromell.business@gmail.com>                 #
# Copyright 2020 Max Wittig <max.wittig@siemens.com>                           #
# Copyright 2020 Pascal Hofmann <mail@pascalhofmann.de>                        #
# Copyright 2020 Steve Kowalik <steven@wedontsleep.org>                        #
# Copyright 2020 Tim Gates <tim.gates@iress.com>                               #
# Copyright 2020 Victor Zeng <zacker150@users.noreply.github.com>              #
# Copyright 2020 ton-katsu <sakamoto.yoshihisa@gmail.com>                      #
# Copyright 2021 Chris Keating <christopherkeating@gmail.com>                  #
# Copyright 2021 Floyd Hightower <floyd.hightower27@gmail.com>                 #
# Copyright 2021 Mark Walker <mark.walker@realbuzz.com>                        #
# Copyright 2021 Steve Kowalik <steven@wedontsleep.org>                        #
# Copyright 2021 Tanner <51724788+lightningboltemoji@users.noreply.github.com> #
# Copyright 2021 xmo-odoo <xmo@odoo.com>                                       #
# Copyright 2022 Aleksei Fedotov <lexa@cfotr.com>                              #
# Copyright 2022 Eric Nieuwland <eric.nieuwland@gmail.com>                     #
# Copyright 2022 Ibrahim Hussaini <ibrahimhussainialias@outlook.com>           #
# Copyright 2022 KimSia Sim <245021+simkimsia@users.noreply.github.com>        #
# Copyright 2022 Marco Köpcke <hello@parakoopa.de>                             #
# Copyright 2022 Alson van der Meulen <alson.vandermeulen@dearhealth.com>      #
# Copyright 2023 Jonathan Leitschuh <Jonathan.Leitschuh@gmail.com>             #
# Copyright 2023 Sol Redfern <59831933+Tsuesun@users.noreply.github.com>       #
# Copyright 2023 Mikhail f. Shiryaev <mr.felixoid@gmail.com>                   #
#                                                                              #
# This file is part of PyGithub.                                               #
# http://pygithub.readthedocs.io/                                              #
#                                                                              #
# PyGithub is free software: you can redistribute it and/or modify it under    #
# the terms of the GNU Lesser General Public License as published by the Free  #
# Software Foundation, either version 3 of the License, or (at your option)    #
# any later version.                                                           #
#                                                                              #
# PyGithub is distributed in the hope that it will be useful, but WITHOUT ANY  #
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS    #
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more #
# details.                                                                     #
#                                                                              #
# You should have received a copy of the GNU Lesser General Public License     #
# along with PyGithub. If not, see <http://www.gnu.org/licenses/>.             #
#                                                                              #
################################################################################

import collections
import typing
import urllib.parse
from base64 import b64encode
from datetime import date, datetime, timezone

from deprecated import deprecated

import github.Artifact
import github.Autolink
import github.Branch
import github.CheckRun
import github.CheckSuite
import github.Clones
import github.CodeScanAlert
import github.Commit
import github.CommitComment
import github.Comparison
import github.ContentFile
import github.Deployment
import github.Download
import github.Environment
import github.EnvironmentDeploymentBranchPolicy
import github.EnvironmentProtectionRule
import github.EnvironmentProtectionRuleReviewer
import github.Event
import github.GitBlob
import github.GitCommit
import github.GithubObject
import github.GitRef
import github.GitRelease
import github.GitReleaseAsset
import github.GitTag
import github.GitTree
import github.Hook
import github.HookDelivery
import github.Invitation
import github.Issue
import github.IssueEvent
import github.Label
import github.Milestone
import github.NamedUser
import github.Organization
import github.PaginatedList
import github.Path
import github.Permissions
import github.Project
import github.PublicKey
import github.PullRequest
import github.Referrer
import github.Repository
import github.RepositoryAdvisory
import github.RepositoryAdvisoryCredit
import github.RepositoryAdvisoryVulnerability
import github.RepositoryKey
import github.RepositoryPreferences
import github.SelfHostedActionsRunner
import github.SourceImport
import github.Stargazer
import github.StatsCodeFrequency
import github.StatsCommitActivity
import github.StatsContributor
import github.StatsParticipation
import github.StatsPunchCard
import github.Tag
import github.Team
import github.View
import github.Workflow
import github.WorkflowRun

from . import Consts


class Repository(github.GithubObject.CompletableGithubObject):
    """
    This class represents Repositories. The reference can be found here https://docs.github.com/en/rest/reference/repos
    """

    def __repr__(self):
        return self.get__repr__({"full_name": self._full_name.value})

    @property
    def allow_auto_merge(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_auto_merge)
        return self._allow_auto_merge.value

    @property
    def allow_forking(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_forking)
        return self._allow_forking.value

    @property
    def allow_merge_commit(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_merge_commit)
        return self._allow_merge_commit.value

    @property
    def allow_rebase_merge(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_rebase_merge)
        return self._allow_rebase_merge.value

    @property
    def allow_squash_merge(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_squash_merge)
        return self._allow_squash_merge.value

    @property
    def allow_update_branch(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._allow_update_branch)
        return self._allow_update_branch.value

    @property
    def archived(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._archived)
        return self._archived.value

    @property
    def archive_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._archive_url)
        return self._archive_url.value

    @property
    def assignees_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._assignees_url)
        return self._assignees_url.value

    @property
    def blobs_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._blobs_url)
        return self._blobs_url.value

    @property
    def branches_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._branches_url)
        return self._branches_url.value

    @property
    def clone_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._clone_url)
        return self._clone_url.value

    @property
    def collaborators_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._collaborators_url)
        return self._collaborators_url.value

    @property
    def comments_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._comments_url)
        return self._comments_url.value

    @property
    def commits_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._commits_url)
        return self._commits_url.value

    @property
    def compare_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._compare_url)
        return self._compare_url.value

    @property
    def contents_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._contents_url)
        return self._contents_url.value

    @property
    def contributors_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._contributors_url)
        return self._contributors_url.value

    @property
    def created_at(self):
        """
        :type: datetime
        """
        self._completeIfNotSet(self._created_at)
        return self._created_at.value

    @property
    def default_branch(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._default_branch)
        return self._default_branch.value

    @property
    def delete_branch_on_merge(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._delete_branch_on_merge)
        return self._delete_branch_on_merge.value

    @property
    def deployments_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._deployments_url)
        return self._deployments_url.value

    @property
    def description(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._description)
        return self._description.value

    @property
    def downloads_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._downloads_url)
        return self._downloads_url.value

    @property
    def events_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._events_url)
        return self._events_url.value

    @property
    def fork(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._fork)
        return self._fork.value

    @property
    def forks(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._forks)
        return self._forks.value

    @property
    def forks_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._forks_count)
        return self._forks_count.value

    @property
    def forks_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._forks_url)
        return self._forks_url.value

    @property
    def full_name(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._full_name)
        return self._full_name.value

    @property
    def git_commits_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._git_commits_url)
        return self._git_commits_url.value

    @property
    def git_refs_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._git_refs_url)
        return self._git_refs_url.value

    @property
    def git_tags_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._git_tags_url)
        return self._git_tags_url.value

    @property
    def git_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._git_url)
        return self._git_url.value

    @property
    def has_downloads(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._has_downloads)
        return self._has_downloads.value

    @property
    def has_issues(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._has_issues)
        return self._has_issues.value

    @property
    def has_pages(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._has_pages)
        return self._has_pages.value

    @property
    def has_projects(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._has_projects)
        return self._has_projects.value

    @property
    def has_wiki(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._has_wiki)
        return self._has_wiki.value

    @property
    def homepage(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._homepage)
        return self._homepage.value

    @property
    def hooks_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._hooks_url)
        return self._hooks_url.value

    @property
    def html_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._html_url)
        return self._html_url.value

    @property
    def id(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._id)
        return self._id.value

    @property
    def is_template(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._is_template)
        return self._is_template.value

    @property
    def issue_comment_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._issue_comment_url)
        return self._issue_comment_url.value

    @property
    def issue_events_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._issue_events_url)
        return self._issue_events_url.value

    @property
    def issues_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._issues_url)
        return self._issues_url.value

    @property
    def keys_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._keys_url)
        return self._keys_url.value

    @property
    def labels_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._labels_url)
        return self._labels_url.value

    @property
    def language(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._language)
        return self._language.value

    @property
    def languages_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._languages_url)
        return self._languages_url.value

    @property
    def master_branch(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._master_branch)
        return self._master_branch.value

    @property
    def merges_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._merges_url)
        return self._merges_url.value

    @property
    def milestones_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._milestones_url)
        return self._milestones_url.value

    @property
    def mirror_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._mirror_url)
        return self._mirror_url.value

    @property
    def name(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._name)
        return self._name.value

    @property
    def network_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._network_count)
        return self._network_count.value

    @property
    def notifications_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._notifications_url)
        return self._notifications_url.value

    @property
    def open_issues(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._open_issues)
        return self._open_issues.value

    @property
    def open_issues_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._open_issues_count)
        return self._open_issues_count.value

    @property
    def organization(self):
        """
        :type: :class:`github.Organization.Organization`
        """
        self._completeIfNotSet(self._organization)
        return self._organization.value

    @property
    def owner(self):
        """
        :type: :class:`github.NamedUser.NamedUser`
        """
        self._completeIfNotSet(self._owner)
        return self._owner.value

    @property
    def parent(self):
        """
        :type: :class:`github.Repository.Repository`
        """
        self._completeIfNotSet(self._parent)
        return self._parent.value

    @property
    def permissions(self):
        """
        :type: :class:`github.Permissions.Permissions`
        """
        self._completeIfNotSet(self._permissions)
        return self._permissions.value

    @property
    def private(self):
        """
        :type: bool
        """
        self._completeIfNotSet(self._private)
        return self._private.value

    @property
    def pulls_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._pulls_url)
        return self._pulls_url.value

    @property
    def pushed_at(self):
        """
        :type: datetime
        """
        self._completeIfNotSet(self._pushed_at)
        return self._pushed_at.value

    @property
    def releases_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._releases_url)
        return self._releases_url.value

    @property
    def size(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._size)
        return self._size.value

    @property
    def source(self):
        """
        :type: :class:`github.Repository.Repository`
        """
        self._completeIfNotSet(self._source)
        return self._source.value

    @property
    def ssh_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._ssh_url)
        return self._ssh_url.value

    @property
    def stargazers_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(
            self._stargazers_count
        )  # pragma no cover (Should be covered)
        return self._stargazers_count.value  # pragma no cover (Should be covered)

    @property
    def stargazers_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._stargazers_url)
        return self._stargazers_url.value

    @property
    def statuses_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._statuses_url)
        return self._statuses_url.value

    @property
    def subscribers_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._subscribers_url)
        return self._subscribers_url.value

    @property
    def subscribers_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._subscribers_count)
        return self._subscribers_count.value

    @property
    def subscription_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._subscription_url)
        return self._subscription_url.value

    @property
    def svn_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._svn_url)
        return self._svn_url.value

    @property
    def tags_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._tags_url)
        return self._tags_url.value

    @property
    def teams_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._teams_url)
        return self._teams_url.value

    @property
    def topics(self):
        """
        :type: list of strings
        """
        self._completeIfNotSet(self._topics)
        return self._topics.value

    @property
    def trees_url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._trees_url)
        return self._trees_url.value

    @property
    def updated_at(self):
        """
        :type: datetime
        """
        self._completeIfNotSet(self._updated_at)
        return self._updated_at.value

    @property
    def url(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._url)
        return self._url.value

    @property
    def visibility(self):
        """
        :type: string
        """
        self._completeIfNotSet(self._visibility)
        return self._visibility.value

    @property
    def watchers(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._watchers)
        return self._watchers.value

    @property
    def watchers_count(self):
        """
        :type: integer
        """
        self._completeIfNotSet(self._watchers_count)
        return self._watchers_count.value

    def add_to_collaborators(self, collaborator, permission=github.GithubObject.NotSet):
        """
        :calls: `PUT /repos/{owner}/{repo}/collaborators/{user} <https://docs.github.com/en/rest/reference/repos#collaborators>`_
        :param collaborator: string or :class:`github.NamedUser.NamedUser`
        :param permission: string 'pull', 'push' or 'admin'
        :rtype: None
        """
        assert isinstance(collaborator, github.NamedUser.NamedUser) or isinstance(
            collaborator, str
        ), collaborator
        assert permission is github.GithubObject.NotSet or isinstance(
            permission, str
        ), permission

        if isinstance(collaborator, github.NamedUser.NamedUser):
            collaborator = collaborator._identity

        if permission is not github.GithubObject.NotSet:
            put_parameters = {"permission": permission}
        else:
            put_parameters = None

        headers, data = self._requester.requestJsonAndCheck(
            "PUT", f"{self.url}/collaborators/{collaborator}", input=put_parameters
        )
        # return an invitation object if there's data returned by the API. If data is empty
        # there's a pending invitation for the given user.
        return (
            github.Invitation.Invitation(self._requester, headers, data, completed=True)
            if data is not None
            else None
        )

    def get_collaborator_permission(self, collaborator):
        """
        :calls: `GET /repos/{owner}/{repo}/collaborators/{username}/permission <https://docs.github.com/en/rest/reference/repos#collaborators>`_
        :param collaborator: string or :class:`github.NamedUser.NamedUser`
        :rtype: string
        """
        assert isinstance(collaborator, github.NamedUser.NamedUser) or isinstance(
            collaborator, str
        ), collaborator
        if isinstance(collaborator, github.NamedUser.NamedUser):
            collaborator = collaborator._identity
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/collaborators/{collaborator}/permission",
        )
        return data["permission"]

    def get_pending_invitations(self):
        """
        :calls: `GET /repos/{owner}/{repo}/invitations <https://docs.github.com/en/rest/reference/repos#invitations>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Invitation.Invitation`
        """
        return github.PaginatedList.PaginatedList(
            github.Invitation.Invitation,
            self._requester,
            f"{self.url}/invitations",
            None,
        )

    def remove_invitation(self, invite_id):
        """
        :calls: `DELETE /repos/{owner}/{repo}/invitations/{invitation_id} <https://docs.github.com/en/rest/reference/repos#invitations>`_
        :rtype: None
        """
        assert isinstance(invite_id, int), invite_id

        headers, data = self._requester.requestJsonAndCheck(
            "DELETE", f"{self.url}/invitations/{invite_id}"
        )

    def compare(self, base, head):
        """
        :calls: `GET /repos/{owner}/{repo}/compare/{base...:head} <https://docs.github.com/en/rest/reference/repos#commits>`_
        :param base: string
        :param head: string
        :rtype: :class:`github.Comparison.Comparison`
        """
        assert isinstance(base, str), base
        assert isinstance(head, str), head
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/compare/{base}...{head}"
        )
        return github.Comparison.Comparison(
            self._requester, headers, data, completed=True
        )

    def create_autolink(self, key_prefix, url_template):
        """
        :calls: `POST /repos/{owner}/{repo}/autolinks <http://docs.github.com/en/rest/reference/repos>`_
        :param key_prefix: string
        :param url_template: string
        :rtype: :class:`github.Autolink.Autolink`
        """
        assert isinstance(key_prefix, str), key_prefix
        assert isinstance(url_template, str), url_template

        post_parameters = {"key_prefix": key_prefix, "url_template": url_template}
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/autolinks", input=post_parameters
        )
        return github.Autolink.Autolink(self._requester, headers, data, completed=True)

    def create_git_blob(self, content, encoding):
        """
        :calls: `POST /repos/{owner}/{repo}/git/blobs <https://docs.github.com/en/rest/reference/git#blobs>`_
        :param content: string
        :param encoding: string
        :rtype: :class:`github.GitBlob.GitBlob`
        """
        assert isinstance(content, str), content
        assert isinstance(encoding, str), encoding
        post_parameters = {
            "content": content,
            "encoding": encoding,
        }
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/git/blobs", input=post_parameters
        )
        return github.GitBlob.GitBlob(self._requester, headers, data, completed=True)

    def create_git_commit(
        self,
        message,
        tree,
        parents,
        author=github.GithubObject.NotSet,
        committer=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/git/commits <https://docs.github.com/en/rest/reference/git#commits>`_
        :param message: string
        :param tree: :class:`github.GitTree.GitTree`
        :param parents: list of :class:`github.GitCommit.GitCommit`
        :param author: :class:`github.InputGitAuthor.InputGitAuthor`
        :param committer: :class:`github.InputGitAuthor.InputGitAuthor`
        :rtype: :class:`github.GitCommit.GitCommit`
        """
        assert isinstance(message, str), message
        assert isinstance(tree, github.GitTree.GitTree), tree
        assert all(
            isinstance(element, github.GitCommit.GitCommit) for element in parents
        ), parents
        assert author is github.GithubObject.NotSet or isinstance(
            author, github.InputGitAuthor
        ), author
        assert committer is github.GithubObject.NotSet or isinstance(
            committer, github.InputGitAuthor
        ), committer
        post_parameters = {
            "message": message,
            "tree": tree._identity,
            "parents": [element._identity for element in parents],
        }
        if author is not github.GithubObject.NotSet:
            post_parameters["author"] = author._identity
        if committer is not github.GithubObject.NotSet:
            post_parameters["committer"] = committer._identity
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/git/commits", input=post_parameters
        )
        return github.GitCommit.GitCommit(
            self._requester, headers, data, completed=True
        )

    def create_git_ref(self, ref, sha):
        """
        :calls: `POST /repos/{owner}/{repo}/git/refs <https://docs.github.com/en/rest/reference/git#references>`_
        :param ref: string
        :param sha: string
        :rtype: :class:`github.GitRef.GitRef`
        """
        assert isinstance(ref, str), ref
        assert isinstance(sha, str), sha
        post_parameters = {
            "ref": ref,
            "sha": sha,
        }
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/git/refs", input=post_parameters
        )
        return github.GitRef.GitRef(self._requester, headers, data, completed=True)

    def create_git_tag_and_release(
        self,
        tag,
        tag_message,
        release_name,
        release_message,
        object,
        type,
        tagger=github.GithubObject.NotSet,
        draft=False,
        prerelease=False,
        generate_release_notes=False,
    ):
        """
        Convenience function that calls :meth:`Repository.create_git_tag` and
        :meth:`Repository.create_git_release`.
        :param tag: string
        :param tag_message: string
        :param release_name: string
        :param release_message: string
        :param object: string
        :param type: string
        :param tagger: :class:github.InputGitAuthor.InputGitAuthor
        :param draft: bool
        :param prerelease: bool
        :param generate_release_notes: bool
        :rtype: :class:`github.GitRelease.GitRelease`
        """
        self.create_git_tag(tag, tag_message, object, type, tagger)
        return self.create_git_release(
            tag,
            release_name,
            release_message,
            draft,
            prerelease,
            generate_release_notes,
            target_commitish=object,
        )

    def create_git_release(
        self,
        tag,
        name,
        message,
        draft=False,
        prerelease=False,
        generate_release_notes=False,
        target_commitish=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/releases <https://docs.github.com/en/rest/reference/repos#releases>`_
        :param tag: string
        :param name: string
        :param message: string
        :param draft: bool
        :param prerelease: bool
        :param generate_release_notes: bool
        :param target_commitish: string or :class:`github.Branch.Branch` or :class:`github.Commit.Commit` or :class:`github.GitCommit.GitCommit`
        :rtype: :class:`github.GitRelease.GitRelease`
        """
        assert isinstance(tag, str), tag
        assert isinstance(name, str), name
        assert isinstance(message, str), message
        assert isinstance(draft, bool), draft
        assert isinstance(prerelease, bool), prerelease
        assert isinstance(generate_release_notes, bool), generate_release_notes
        assert target_commitish is github.GithubObject.NotSet or isinstance(
            target_commitish,
            (
                str,
                github.Branch.Branch,
                github.Commit.Commit,
                github.GitCommit.GitCommit,
            ),
        ), target_commitish
        post_parameters = {
            "tag_name": tag,
            "name": name,
            "body": message,
            "draft": draft,
            "prerelease": prerelease,
            "generate_release_notes": generate_release_notes,
        }
        if isinstance(target_commitish, str):
            post_parameters["target_commitish"] = target_commitish
        elif isinstance(target_commitish, github.Branch.Branch):
            post_parameters["target_commitish"] = target_commitish.name
        elif isinstance(
            target_commitish, (github.Commit.Commit, github.GitCommit.GitCommit)
        ):
            post_parameters["target_commitish"] = target_commitish.sha
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/releases", input=post_parameters
        )
        return github.GitRelease.GitRelease(
            self._requester, headers, data, completed=True
        )

    def create_git_tag(
        self, tag, message, object, type, tagger=github.GithubObject.NotSet
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/git/tags <https://docs.github.com/en/rest/reference/git#tags>`_
        :param tag: string
        :param message: string
        :param object: string
        :param type: string
        :param tagger: :class:`github.InputGitAuthor.InputGitAuthor`
        :rtype: :class:`github.GitTag.GitTag`
        """
        assert isinstance(tag, str), tag
        assert isinstance(message, str), message
        assert isinstance(object, str), object
        assert isinstance(type, str), type
        assert tagger is github.GithubObject.NotSet or isinstance(
            tagger, github.InputGitAuthor
        ), tagger
        post_parameters = {
            "tag": tag,
            "message": message,
            "object": object,
            "type": type,
        }
        if tagger is not github.GithubObject.NotSet:
            post_parameters["tagger"] = tagger._identity
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/git/tags", input=post_parameters
        )
        return github.GitTag.GitTag(self._requester, headers, data, completed=True)

    def create_git_tree(self, tree, base_tree=github.GithubObject.NotSet):
        """
        :calls: `POST /repos/{owner}/{repo}/git/trees <https://docs.github.com/en/rest/reference/git#trees>`_
        :param tree: list of :class:`github.InputGitTreeElement.InputGitTreeElement`
        :param base_tree: :class:`github.GitTree.GitTree`
        :rtype: :class:`github.GitTree.GitTree`
        """
        assert all(
            isinstance(element, github.InputGitTreeElement) for element in tree
        ), tree
        assert base_tree is github.GithubObject.NotSet or isinstance(
            base_tree, github.GitTree.GitTree
        ), base_tree
        post_parameters = {
            "tree": [element._identity for element in tree],
        }
        if base_tree is not github.GithubObject.NotSet:
            post_parameters["base_tree"] = base_tree._identity
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/git/trees", input=post_parameters
        )
        return github.GitTree.GitTree(self._requester, headers, data, completed=True)

    def create_hook(
        self,
        name,
        config,
        events=github.GithubObject.NotSet,
        active=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/hooks <https://docs.github.com/en/rest/reference/repos#webhooks>`_
        :param name: string
        :param config: dict
        :param events: list of string
        :param active: bool
        :rtype: :class:`github.Hook.Hook`
        """
        assert isinstance(name, str), name
        assert isinstance(config, dict), config
        assert events is github.GithubObject.NotSet or all(
            isinstance(element, str) for element in events
        ), events
        assert active is github.GithubObject.NotSet or isinstance(active, bool), active
        post_parameters = {
            "name": name,
            "config": config,
        }
        if events is not github.GithubObject.NotSet:
            post_parameters["events"] = events
        if active is not github.GithubObject.NotSet:
            post_parameters["active"] = active
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/hooks", input=post_parameters
        )
        return github.Hook.Hook(self._requester, headers, data, completed=True)

    def create_issue(
        self,
        title,
        body=github.GithubObject.NotSet,
        assignee=github.GithubObject.NotSet,
        milestone=github.GithubObject.NotSet,
        labels=github.GithubObject.NotSet,
        assignees=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/issues <https://docs.github.com/en/rest/reference/issues>`_
        :param title: string
        :param body: string
        :param assignee: string or :class:`github.NamedUser.NamedUser`
        :param assignees: list of string or :class:`github.NamedUser.NamedUser`
        :param milestone: :class:`github.Milestone.Milestone`
        :param labels: list of :class:`github.Label.Label`
        :rtype: :class:`github.Issue.Issue`
        """
        assert isinstance(title, str), title
        assert body is github.GithubObject.NotSet or isinstance(body, str), body
        assert (
            assignee is github.GithubObject.NotSet
            or isinstance(assignee, github.NamedUser.NamedUser)
            or isinstance(assignee, str)
        ), assignee
        assert assignees is github.GithubObject.NotSet or all(
            isinstance(element, github.NamedUser.NamedUser) or isinstance(element, str)
            for element in assignees
        ), assignees
        assert milestone is github.GithubObject.NotSet or isinstance(
            milestone, github.Milestone.Milestone
        ), milestone
        assert labels is github.GithubObject.NotSet or all(
            isinstance(element, github.Label.Label) or isinstance(element, str)
            for element in labels
        ), labels

        post_parameters = {
            "title": title,
        }
        if body is not github.GithubObject.NotSet:
            post_parameters["body"] = body
        if assignee is not github.GithubObject.NotSet:
            if isinstance(assignee, str):
                post_parameters["assignee"] = assignee
            else:
                post_parameters["assignee"] = assignee._identity
        if assignees is not github.GithubObject.NotSet:
            post_parameters["assignees"] = [
                element._identity
                if isinstance(element, github.NamedUser.NamedUser)
                else element
                for element in assignees
            ]
        if milestone is not github.GithubObject.NotSet:
            post_parameters["milestone"] = milestone._identity
        if labels is not github.GithubObject.NotSet:
            post_parameters["labels"] = [
                element.name if isinstance(element, github.Label.Label) else element
                for element in labels
            ]
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/issues", input=post_parameters
        )
        return github.Issue.Issue(self._requester, headers, data, completed=True)

    def create_key(self, title, key, read_only=False):
        """
        :calls: `POST /repos/{owner}/{repo}/keys <https://docs.github.com/en/rest/reference/repos#deploy-keys>`_
        :param title: string
        :param key: string
        :param read_only: bool
        :rtype: :class:`github.RepositoryKey.RepositoryKey`
        """
        assert isinstance(title, str), title
        assert isinstance(key, str), key
        assert isinstance(read_only, bool), read_only
        post_parameters = {
            "title": title,
            "key": key,
            "read_only": read_only,
        }
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/keys", input=post_parameters
        )
        return github.RepositoryKey.RepositoryKey(
            self._requester, headers, data, completed=True
        )

    def create_label(self, name, color, description=github.GithubObject.NotSet):
        """
        :calls: `POST /repos/{owner}/{repo}/labels <https://docs.github.com/en/rest/reference/issues#labels>`_
        :param name: string
        :param color: string
        :param description: string
        :rtype: :class:`github.Label.Label`
        """
        assert isinstance(name, str), name
        assert isinstance(color, str), color
        assert description is github.GithubObject.NotSet or isinstance(
            description, str
        ), description
        post_parameters = {
            "name": name,
            "color": color,
        }
        if description is not github.GithubObject.NotSet:
            post_parameters["description"] = description
        headers, data = self._requester.requestJsonAndCheck(
            "POST",
            f"{self.url}/labels",
            input=post_parameters,
            headers={"Accept": Consts.mediaTypeLabelDescriptionSearchPreview},
        )
        return github.Label.Label(self._requester, headers, data, completed=True)

    def create_milestone(
        self,
        title,
        state=github.GithubObject.NotSet,
        description=github.GithubObject.NotSet,
        due_on=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/milestones <https://docs.github.com/en/rest/reference/issues#milestones>`_
        :param title: string
        :param state: string
        :param description: string
        :param due_on: datetime
        :rtype: :class:`github.Milestone.Milestone`
        """
        assert isinstance(title, str), title
        assert state is github.GithubObject.NotSet or isinstance(state, str), state
        assert description is github.GithubObject.NotSet or isinstance(
            description, str
        ), description
        assert due_on is github.GithubObject.NotSet or isinstance(
            due_on, (datetime, date)
        ), due_on
        post_parameters = {
            "title": title,
        }
        if state is not github.GithubObject.NotSet:
            post_parameters["state"] = state
        if description is not github.GithubObject.NotSet:
            post_parameters["description"] = description
        if due_on is not github.GithubObject.NotSet:
            if isinstance(due_on, date):
                post_parameters["due_on"] = due_on.strftime("%Y-%m-%dT%H:%M:%SZ")
            else:
                post_parameters["due_on"] = due_on.isoformat()
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/milestones", input=post_parameters
        )
        return github.Milestone.Milestone(
            self._requester, headers, data, completed=True
        )

    def create_project(self, name, body=github.GithubObject.NotSet):
        """
        :calls: `POST /repos/{owner}/{repo}/projects <https://docs.github.com/en/rest/reference/projects#create-a-repository-project>`_
        :param name: string
        :param body: string
        :rtype: :class:`github.Project.Project`
        """
        assert isinstance(name, str), name
        assert body is github.GithubObject.NotSet or isinstance(body, str), body
        post_parameters = {
            "name": name,
        }
        import_header = {"Accept": Consts.mediaTypeProjectsPreview}
        if body is not github.GithubObject.NotSet:
            post_parameters["body"] = body
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/projects", headers=import_header, input=post_parameters
        )
        return github.Project.Project(self._requester, headers, data, completed=True)

    def create_pull(self, *args, **kwds):
        """
        :calls: `POST /repos/{owner}/{repo}/pulls <https://docs.github.com/en/rest/reference/pulls>`_
        :param title: string
        :param body: string
        :param issue: :class:`github.Issue.Issue`
        :param base: string
        :param head: string
        :param draft: bool
        :param maintainer_can_modify: bool
        :rtype: :class:`github.PullRequest.PullRequest`
        """
        if len(args) + len(kwds) >= 4:
            return self.__create_pull_1(*args, **kwds)
        else:
            return self.__create_pull_2(*args, **kwds)

    def __create_pull_1(
        self,
        title,
        body,
        base,
        head,
        maintainer_can_modify=github.GithubObject.NotSet,
        draft=False,
    ):
        assert isinstance(title, str), title
        assert isinstance(body, str), body
        assert isinstance(base, str), base
        assert isinstance(head, str), head
        assert maintainer_can_modify is github.GithubObject.NotSet or isinstance(
            maintainer_can_modify, bool
        ), maintainer_can_modify
        assert isinstance(draft, bool), draft
        if maintainer_can_modify is not github.GithubObject.NotSet:
            return self.__create_pull(
                title=title,
                body=body,
                base=base,
                head=head,
                maintainer_can_modify=maintainer_can_modify,
                draft=draft,
            )
        else:
            return self.__create_pull(
                title=title, body=body, base=base, head=head, draft=draft
            )

    def __create_pull_2(self, issue, base, head):
        assert isinstance(issue, github.Issue.Issue), issue
        assert isinstance(base, str), base
        assert isinstance(head, str), head
        return self.__create_pull(issue=issue._identity, base=base, head=head)

    def __create_pull(self, **kwds):
        post_parameters = kwds

        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/pulls", input=post_parameters
        )
        return github.PullRequest.PullRequest(
            self._requester, headers, data, completed=True
        )

    def create_repository_advisory(
        self,
        summary: str,
        description: str,
        severity_or_cvss_vector_string: str,
        cve_id: typing.Optional[str] = None,
        vulnerabilities: typing.Optional[
            typing.Iterable[
                github.RepositoryAdvisoryVulnerability.AdvisoryVulnerability
            ]
        ] = None,
        cwe_ids: typing.Optional[typing.Iterable[str]] = None,
        credits: typing.Optional[
            typing.Iterable[github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit]
        ] = None,
    ) -> github.RepositoryAdvisory.RepositoryAdvisory:
        """
        :calls: `POST /repos/{owner}/{repo}/security-advisories <https://docs.github.com/en/rest/security-advisories/repository-advisories>`_
        :param summary: string
        :param description: string
        :param severity_or_cvss_vector_string: string
        :param cve_id: string
        :param vulnerabilities: iterable of :class:`github.RepositoryAdvisoryVulnerability.AdvisoryVulnerability`
        :param cwe_ids: iterable of string
        :param credits: iterable of :class:`github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit`
        :rtype: :class:`github.RepositoryAdvisory.RepositoryAdvisory`
        """
        return self.__create_repository_advisory(
            summary=summary,
            description=description,
            severity_or_cvss_vector_string=severity_or_cvss_vector_string,
            cve_id=cve_id,
            vulnerabilities=vulnerabilities,
            cwe_ids=cwe_ids,
            credits=credits,
            private_vulnerability_reporting=False,
        )

    def report_security_vulnerability(
        self,
        summary: str,
        description: str,
        severity_or_cvss_vector_string: str,
        cve_id: typing.Optional[str] = None,
        vulnerabilities: typing.Optional[
            typing.Iterable[
                github.RepositoryAdvisoryVulnerability.AdvisoryVulnerability
            ]
        ] = None,
        cwe_ids: typing.Optional[typing.Iterable[str]] = None,
        credits: typing.Optional[
            typing.Iterable[github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit]
        ] = None,
    ) -> github.RepositoryAdvisory.RepositoryAdvisory:
        """
        :calls: `POST /repos/{owner}/{repo}/security-advisories/reports <https://docs.github.com/en/rest/security-advisories/repository-advisories#privately-report-a-security-vulnerability>`_
        :param summary: string
        :param description: string
        :param severity_or_cvss_vector_string: string
        :param cve_id: string
        :param vulnerabilities: iterable of :class:`github.RepositoryAdvisoryVulnerability.AdvisoryVulnerability`
        :param cwe_ids: iterable of string
        :param credits: iterable of :class:`github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit`
        :rtype: :class:`github.RepositoryAdvisory.RepositoryAdvisory`
        """
        return self.__create_repository_advisory(
            summary=summary,
            description=description,
            severity_or_cvss_vector_string=severity_or_cvss_vector_string,
            cve_id=cve_id,
            vulnerabilities=vulnerabilities,
            cwe_ids=cwe_ids,
            credits=credits,
            private_vulnerability_reporting=True,
        )

    def __create_repository_advisory(
        self,
        summary: str,
        description: str,
        severity_or_cvss_vector_string: str,
        cve_id: typing.Optional[str],
        vulnerabilities: typing.Optional[
            typing.Iterable[
                github.RepositoryAdvisoryVulnerability.AdvisoryVulnerability
            ]
        ],
        cwe_ids: typing.Optional[typing.Iterable[str]],
        credits: typing.Optional[
            typing.Iterable[github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit]
        ],
        private_vulnerability_reporting: bool,
    ) -> github.RepositoryAdvisory.RepositoryAdvisory:
        if vulnerabilities is None:
            vulnerabilities = []
        if cwe_ids is None:
            cwe_ids = []
        assert isinstance(summary, str), summary
        assert isinstance(description, str), description
        assert isinstance(
            severity_or_cvss_vector_string, str
        ), severity_or_cvss_vector_string
        assert isinstance(cve_id, (str, type(None))), cve_id
        assert isinstance(vulnerabilities, typing.Iterable), vulnerabilities
        for vulnerability in vulnerabilities:
            # noinspection PyProtectedMember
            github.RepositoryAdvisoryVulnerability.RepositoryAdvisoryVulnerability._validate_vulnerability(
                vulnerability
            )
        assert isinstance(cwe_ids, typing.Iterable), cwe_ids
        assert all(isinstance(element, str) for element in cwe_ids), cwe_ids
        assert isinstance(credits, (typing.Iterable, type(None))), credits
        if credits is not None:
            for credit in credits:
                # noinspection PyProtectedMember
                github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit._validate_credit(
                    credit
                )
        # noinspection PyProtectedMember
        post_parameters = {
            "summary": summary,
            "description": description,
            "vulnerabilities": [
                github.RepositoryAdvisoryVulnerability.RepositoryAdvisoryVulnerability._to_github_dict(
                    vulnerability
                )
                for vulnerability in vulnerabilities
            ],
            "cwe_ids": list(cwe_ids),
        }
        if cve_id is not None:
            post_parameters["cve_id"] = cve_id
        if credits is not None:
            # noinspection PyProtectedMember
            post_parameters["credits"] = [
                github.RepositoryAdvisoryCredit.RepositoryAdvisoryCredit._to_github_dict(
                    credit
                )
                for credit in credits
            ]
        if severity_or_cvss_vector_string.startswith("CVSS:"):
            post_parameters["cvss_vector_string"] = severity_or_cvss_vector_string
        else:
            post_parameters["severity"] = severity_or_cvss_vector_string
        if private_vulnerability_reporting:
            headers, data = self._requester.requestJsonAndCheck(
                "POST", f"{self.url}/security-advisories/reports", input=post_parameters
            )
        else:
            headers, data = self._requester.requestJsonAndCheck(
                "POST", f"{self.url}/security-advisories", input=post_parameters
            )
        return github.RepositoryAdvisory.RepositoryAdvisory(
            self._requester, headers, data, completed=True
        )

    def create_repository_dispatch(
        self, event_type, client_payload=github.GithubObject.NotSet
    ):
        """
        :calls: POST /repos/{owner}/{repo}/dispatches <https://docs.github.com/en/rest/reference/repos#create-a-repository-dispatch-event>
        :param event_type: string
        :param client_payload: dict
        :rtype: bool
        """
        assert isinstance(event_type, str), event_type
        assert client_payload is github.GithubObject.NotSet or isinstance(
            client_payload, dict
        ), client_payload
        post_parameters = {"event_type": event_type}
        if client_payload is not github.GithubObject.NotSet:
            post_parameters["client_payload"] = client_payload
        status, headers, data = self._requester.requestJson(
            "POST", f"{self.url}/dispatches", input=post_parameters
        )
        return status == 204

    def create_secret(self, secret_name, unencrypted_value):
        """
        :calls: `PUT /repos/{owner}/{repo}/actions/secrets/{secret_name} <https://docs.github.com/en/rest/reference/actions#get-a-repository-secret>`_
        :param secret_name: string
        :param unencrypted_value: string
        :rtype: bool
        """
        assert isinstance(secret_name, str), secret_name
        assert isinstance(unencrypted_value, str), unencrypted_value
        public_key = self.get_public_key()
        payload = public_key.encrypt(unencrypted_value)
        put_parameters = {
            "key_id": public_key.key_id,
            "encrypted_value": payload,
        }
        status, headers, data = self._requester.requestJson(
            "PUT", f"{self.url}/actions/secrets/{secret_name}", input=put_parameters
        )
        return status == 201

    def delete_secret(self, secret_name):
        """
        :calls: `DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name} <https://docs.github.com/en/rest/reference/actions#delete-a-repository-secret>`_
        :param secret_name: string
        :rtype: bool
        """
        assert isinstance(secret_name, str), secret_name
        status, headers, data = self._requester.requestJson(
            "DELETE", f"{self.url}/actions/secrets/{secret_name}"
        )
        return status == 204

    def create_source_import(
        self,
        vcs,
        vcs_url,
        vcs_username=github.GithubObject.NotSet,
        vcs_password=github.GithubObject.NotSet,
    ):
        """
        :calls: `PUT /repos/{owner}/{repo}/import <https://docs.github.com/en/rest/reference/migrations#start-an-import>`_
        :param vcs: string
        :param vcs_url: string
        :param vcs_username: string
        :param vcs_password: string
        :rtype: :class:`github.SourceImport.SourceImport`
        """
        assert isinstance(vcs, str), vcs
        assert isinstance(vcs_url, str), vcs_url
        assert vcs_username is github.GithubObject.NotSet or isinstance(
            vcs_username, str
        ), vcs_username
        assert vcs_password is github.GithubObject.NotSet or isinstance(
            vcs_password, str
        ), vcs_password
        put_parameters = {"vcs": vcs, "vcs_url": vcs_url}

        if vcs_username is not github.GithubObject.NotSet:
            put_parameters["vcs_username"] = vcs_username

        if vcs_password is not github.GithubObject.NotSet:
            put_parameters["vcs_password"] = vcs_password

        import_header = {"Accept": Consts.mediaTypeImportPreview}

        headers, data = self._requester.requestJsonAndCheck(
            "PUT", f"{self.url}/import", headers=import_header, input=put_parameters
        )

        return github.SourceImport.SourceImport(
            self._requester, headers, data, completed=False
        )

    def delete(self):
        """
        :calls: `DELETE /repos/{owner}/{repo} <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: None
        """
        headers, data = self._requester.requestJsonAndCheck("DELETE", self.url)

    def edit(
        self,
        name=None,
        description=github.GithubObject.NotSet,
        homepage=github.GithubObject.NotSet,
        private=github.GithubObject.NotSet,
        has_issues=github.GithubObject.NotSet,
        has_projects=github.GithubObject.NotSet,
        has_wiki=github.GithubObject.NotSet,
        has_downloads=github.GithubObject.NotSet,
        default_branch=github.GithubObject.NotSet,
        allow_auto_merge=github.GithubObject.NotSet,
        allow_forking=github.GithubObject.NotSet,
        allow_squash_merge=github.GithubObject.NotSet,
        allow_merge_commit=github.GithubObject.NotSet,
        allow_rebase_merge=github.GithubObject.NotSet,
        delete_branch_on_merge=github.GithubObject.NotSet,
        allow_update_branch=github.GithubObject.NotSet,
        archived=github.GithubObject.NotSet,
    ):
        """
        :calls: `PATCH /repos/{owner}/{repo} <https://docs.github.com/en/rest/reference/repos>`_
        :param name: string
        :param description: string
        :param homepage: string
        :param private: bool
        :param has_issues: bool
        :param has_projects: bool
        :param has_wiki: bool
        :param has_downloads: bool
        :param default_branch: string
        :param allow_forking: bool
        :param allow_squash_merge: bool
        :param allow_merge_commit: bool
        :param allow_rebase_merge: bool
        :param delete_branch_on_merge: bool
        :param allow_update_branch: bool
        :param archived: bool
        :rtype: None
        """
        if name is None:
            name = self.name
        assert isinstance(name, str), name
        assert description is github.GithubObject.NotSet or isinstance(
            description, str
        ), description
        assert homepage is github.GithubObject.NotSet or isinstance(
            homepage, str
        ), homepage
        assert private is github.GithubObject.NotSet or isinstance(
            private, bool
        ), private
        assert has_issues is github.GithubObject.NotSet or isinstance(
            has_issues, bool
        ), has_issues
        assert has_projects is github.GithubObject.NotSet or isinstance(
            has_projects, bool
        ), has_projects
        assert has_wiki is github.GithubObject.NotSet or isinstance(
            has_wiki, bool
        ), has_wiki
        assert has_downloads is github.GithubObject.NotSet or isinstance(
            has_downloads, bool
        ), has_downloads
        assert default_branch is github.GithubObject.NotSet or isinstance(
            default_branch, str
        ), default_branch
        assert allow_auto_merge is github.GithubObject.NotSet or isinstance(
            allow_auto_merge, bool
        ), allow_auto_merge
        assert allow_forking is github.GithubObject.NotSet or isinstance(
            allow_forking, bool
        ), allow_forking
        assert allow_squash_merge is github.GithubObject.NotSet or isinstance(
            allow_squash_merge, bool
        ), allow_squash_merge
        assert allow_merge_commit is github.GithubObject.NotSet or isinstance(
            allow_merge_commit, bool
        ), allow_merge_commit
        assert allow_rebase_merge is github.GithubObject.NotSet or isinstance(
            allow_rebase_merge, bool
        ), allow_rebase_merge
        assert delete_branch_on_merge is github.GithubObject.NotSet or isinstance(
            delete_branch_on_merge, bool
        ), delete_branch_on_merge
        assert allow_update_branch is github.GithubObject.NotSet or isinstance(
            allow_update_branch, bool
        ), allow_update_branch
        assert archived is github.GithubObject.NotSet or isinstance(
            archived, bool
        ), archived
        post_parameters = {
            "name": name,
        }
        if description is not github.GithubObject.NotSet:
            post_parameters["description"] = description
        if homepage is not github.GithubObject.NotSet:
            post_parameters["homepage"] = homepage
        if private is not github.GithubObject.NotSet:
            post_parameters["private"] = private
        if has_issues is not github.GithubObject.NotSet:
            post_parameters["has_issues"] = has_issues
        if has_projects is not github.GithubObject.NotSet:
            post_parameters["has_projects"] = has_projects
        if has_wiki is not github.GithubObject.NotSet:
            post_parameters["has_wiki"] = has_wiki
        if has_downloads is not github.GithubObject.NotSet:
            post_parameters["has_downloads"] = has_downloads
        if default_branch is not github.GithubObject.NotSet:
            post_parameters["default_branch"] = default_branch
        if allow_squash_merge is not github.GithubObject.NotSet:
            post_parameters["allow_squash_merge"] = allow_squash_merge
        if allow_auto_merge is not github.GithubObject.NotSet:
            post_parameters["allow_auto_merge"] = allow_auto_merge
        if allow_forking is not github.GithubObject.NotSet:
            post_parameters["allow_forking"] = allow_forking
        if allow_merge_commit is not github.GithubObject.NotSet:
            post_parameters["allow_merge_commit"] = allow_merge_commit
        if allow_rebase_merge is not github.GithubObject.NotSet:
            post_parameters["allow_rebase_merge"] = allow_rebase_merge
        if delete_branch_on_merge is not github.GithubObject.NotSet:
            post_parameters["delete_branch_on_merge"] = delete_branch_on_merge
        if allow_update_branch is not github.GithubObject.NotSet:
            post_parameters["allow_update_branch"] = allow_update_branch
        if archived is not github.GithubObject.NotSet:
            post_parameters["archived"] = archived
        headers, data = self._requester.requestJsonAndCheck(
            "PATCH", self.url, input=post_parameters
        )
        self._useAttributes(data)

    def get_archive_link(self, archive_format, ref=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/{archive_format}/{ref} <https://docs.github.com/en/rest/reference/repos#contents>`_
        :param archive_format: string
        :param ref: string
        :rtype: string
        """
        assert isinstance(archive_format, str), archive_format
        assert ref is github.GithubObject.NotSet or isinstance(ref, str), ref
        url = f"{self.url}/{archive_format}"
        if ref is not github.GithubObject.NotSet:
            url += f"/{ref}"
        headers, data = self._requester.requestJsonAndCheck("GET", url)
        return headers["location"]

    def get_assignees(self):
        """
        :calls: `GET /repos/{owner}/{repo}/assignees <https://docs.github.com/en/rest/reference/issues#assignees>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """
        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser, self._requester, f"{self.url}/assignees", None
        )

    def get_branch(self, branch):
        """
        :calls: `GET /repos/{owner}/{repo}/branches/{branch} <https://docs.github.com/en/rest/reference/repos#get-a-branch>`_
        :param branch: string
        :rtype: :class:`github.Branch.Branch`
        """
        assert isinstance(branch, str), branch
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/branches/{branch}"
        )
        return github.Branch.Branch(self._requester, headers, data, completed=True)

    def rename_branch(self, branch, new_name):
        """
        :calls: `POST /repos/{owner}/{repo}/branches/{branch}/rename <https://docs.github.com/en/rest/reference/repos#branches>`_
        :param branch: :class:`github.Branch.Branch` or string
        :param new_name: string
        :rtype: bool

        NOTE: This method does not return the branch since it may take some
        time to fully complete server-side.
        """
        is_branch = isinstance(branch, github.Branch.Branch)
        assert isinstance(branch, str) or is_branch, branch
        assert isinstance(new_name, str), new_name
        if is_branch:
            branch = branch.name
        parameters = {"new_name": new_name}
        status, _, _ = self._requester.requestJson(
            "POST", f"{self.url}/branches/{branch}/rename", input=parameters
        )
        return status == 201

    def get_branches(self):
        """
        :calls: `GET /repos/{owner}/{repo}/branches <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Branch.Branch`
        """
        return github.PaginatedList.PaginatedList(
            github.Branch.Branch, self._requester, f"{self.url}/branches", None
        )

    def get_collaborators(self, affiliation=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/collaborators <https://docs.github.com/en/rest/reference/repos#collaborators>`_
        :param affiliation: string
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """

        url_parameters = dict()
        allowed_affiliations = ["outside", "direct", "all"]
        if affiliation is not github.GithubObject.NotSet:
            assert isinstance(affiliation, str), affiliation
            assert (
                affiliation in allowed_affiliations
            ), f"Affiliation can be one of {', '.join(allowed_affiliations)}"
            url_parameters["affiliation"] = affiliation

        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser,
            self._requester,
            f"{self.url}/collaborators",
            url_parameters,
        )

    def get_comment(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/comments/{id} <https://docs.github.com/en/rest/reference/repos#comments>`_
        :param id: integer
        :rtype: :class:`github.CommitComment.CommitComment`
        """
        assert isinstance(id, int), id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/comments/{id}"
        )
        return github.CommitComment.CommitComment(
            self._requester, headers, data, completed=True
        )

    def get_comments(self):
        """
        :calls: `GET /repos/{owner}/{repo}/comments <https://docs.github.com/en/rest/reference/repos#comments>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.CommitComment.CommitComment`
        """
        return github.PaginatedList.PaginatedList(
            github.CommitComment.CommitComment,
            self._requester,
            f"{self.url}/comments",
            None,
        )

    def get_commit(self, sha):
        """
        :calls: `GET /repos/{owner}/{repo}/commits/{sha} <https://docs.github.com/en/rest/reference/repos#commits>`_
        :param sha: string
        :rtype: :class:`github.Commit.Commit`
        """
        assert isinstance(sha, str), sha
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/commits/{sha}"
        )
        return github.Commit.Commit(self._requester, headers, data, completed=True)

    def get_commits(
        self,
        sha=github.GithubObject.NotSet,
        path=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
        until=github.GithubObject.NotSet,
        author=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/commits <https://docs.github.com/en/rest/reference/repos#commits>`_
        :param sha: string
        :param path: string
        :param since: datetime
        :param until: datetime
        :param author: string or :class:`github.NamedUser.NamedUser` or :class:`github.AuthenticatedUser.AuthenticatedUser`
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Commit.Commit`
        """
        assert sha is github.GithubObject.NotSet or isinstance(sha, str), sha
        assert path is github.GithubObject.NotSet or isinstance(path, str), path
        assert since is github.GithubObject.NotSet or isinstance(since, datetime), since
        assert until is github.GithubObject.NotSet or isinstance(until, datetime), until
        assert author is github.GithubObject.NotSet or isinstance(
            author,
            (
                str,
                github.NamedUser.NamedUser,
                github.AuthenticatedUser.AuthenticatedUser,
            ),
        ), author
        url_parameters = dict()
        if sha is not github.GithubObject.NotSet:
            url_parameters["sha"] = sha
        if path is not github.GithubObject.NotSet:
            url_parameters["path"] = path
        if since is not github.GithubObject.NotSet:
            url_parameters["since"] = since.strftime("%Y-%m-%dT%H:%M:%SZ")
        if until is not github.GithubObject.NotSet:
            url_parameters["until"] = until.strftime("%Y-%m-%dT%H:%M:%SZ")
        if author is not github.GithubObject.NotSet:
            if isinstance(
                author,
                (
                    github.NamedUser.NamedUser,
                    github.AuthenticatedUser.AuthenticatedUser,
                ),
            ):
                url_parameters["author"] = author.login
            else:
                url_parameters["author"] = author
        return github.PaginatedList.PaginatedList(
            github.Commit.Commit, self._requester, f"{self.url}/commits", url_parameters
        )

    def get_contents(self, path, ref=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/contents/{path} <https://docs.github.com/en/rest/reference/repos#contents>`_
        :param path: string
        :param ref: string
        :rtype: :class:`github.ContentFile.ContentFile` or a list of them
        """
        assert isinstance(path, str), path
        assert ref is github.GithubObject.NotSet or isinstance(ref, str), ref
        # Path of '/' should be the empty string.
        if path == "/":
            path = ""
        url_parameters = dict()
        if ref is not github.GithubObject.NotSet:
            url_parameters["ref"] = ref
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/contents/{urllib.parse.quote(path)}",
            parameters=url_parameters,
        )

        # Handle 302 redirect response
        if headers.get("status") == "302 Found" and headers.get("location"):
            headers, data = self._requester.requestJsonAndCheck(
                "GET", headers["location"], parameters=url_parameters
            )

        if isinstance(data, list):
            return [
                # Lazy completion only makes sense for files. See discussion
                # here: https://github.com/jacquev6/PyGithub/issues/140#issuecomment-13481130
                github.ContentFile.ContentFile(
                    self._requester, headers, item, completed=(item["type"] != "file")
                )
                for item in data
            ]
        return github.ContentFile.ContentFile(
            self._requester, headers, data, completed=True
        )

    def get_deployments(
        self,
        sha=github.GithubObject.NotSet,
        ref=github.GithubObject.NotSet,
        task=github.GithubObject.NotSet,
        environment=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/deployments <https://docs.github.com/en/rest/reference/repos#deployments>`_
        :param: sha: string
        :param: ref: string
        :param: task: string
        :param: environment: string
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Deployment.Deployment`
        """
        assert sha is github.GithubObject.NotSet or isinstance(sha, str), sha
        assert ref is github.GithubObject.NotSet or isinstance(ref, str), ref
        assert task is github.GithubObject.NotSet or isinstance(task, str), task
        assert environment is github.GithubObject.NotSet or isinstance(
            environment, str
        ), environment
        parameters = {}
        if sha is not github.GithubObject.NotSet:
            parameters["sha"] = sha
        if ref is not github.GithubObject.NotSet:
            parameters["ref"] = ref
        if task is not github.GithubObject.NotSet:
            parameters["task"] = task
        if environment is not github.GithubObject.NotSet:
            parameters["environment"] = environment
        return github.PaginatedList.PaginatedList(
            github.Deployment.Deployment,
            self._requester,
            f"{self.url}/deployments",
            parameters,
            headers={"Accept": Consts.deploymentEnhancementsPreview},
        )

    def get_deployment(self, id_):
        """
        :calls: `GET /repos/{owner}/{repo}/deployments/{deployment_id} <https://docs.github.com/en/rest/reference/repos#deployments>`_
        :param: id_: int
        :rtype: :class:`github.Deployment.Deployment`
        """
        assert isinstance(id_, int), id_
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/deployments/{id_}",
            headers={"Accept": Consts.deploymentEnhancementsPreview},
        )
        return github.Deployment.Deployment(
            self._requester, headers, data, completed=True
        )

    def create_deployment(
        self,
        ref,
        task=github.GithubObject.NotSet,
        auto_merge=github.GithubObject.NotSet,
        required_contexts=github.GithubObject.NotSet,
        payload=github.GithubObject.NotSet,
        environment=github.GithubObject.NotSet,
        description=github.GithubObject.NotSet,
        transient_environment=github.GithubObject.NotSet,
        production_environment=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/deployments <https://docs.github.com/en/rest/reference/repos#deployments>`_
        :param: ref: string
        :param: task: string
        :param: auto_merge: bool
        :param: required_contexts: list of status contexts
        :param: payload: dict
        :param: environment: string
        :param: description: string
        :param: transient_environment: bool
        :param: production_environment: bool
        :rtype: :class:`github.Deployment.Deployment`
        """
        assert isinstance(ref, str), ref
        assert task is github.GithubObject.NotSet or isinstance(task, str), task
        assert auto_merge is github.GithubObject.NotSet or isinstance(
            auto_merge, bool
        ), auto_merge
        assert required_contexts is github.GithubObject.NotSet or isinstance(
            required_contexts, list
        ), required_contexts  # need to do better checking here
        assert payload is github.GithubObject.NotSet or isinstance(
            payload, dict
        ), payload
        assert environment is github.GithubObject.NotSet or isinstance(
            environment, str
        ), environment
        assert description is github.GithubObject.NotSet or isinstance(
            description, str
        ), description
        assert transient_environment is github.GithubObject.NotSet or isinstance(
            transient_environment, bool
        ), transient_environment
        assert production_environment is github.GithubObject.NotSet or isinstance(
            production_environment, bool
        ), production_environment

        post_parameters = {"ref": ref}
        if task is not github.GithubObject.NotSet:
            post_parameters["task"] = task
        if auto_merge is not github.GithubObject.NotSet:
            post_parameters["auto_merge"] = auto_merge
        if required_contexts is not github.GithubObject.NotSet:
            post_parameters["required_contexts"] = required_contexts
        if payload is not github.GithubObject.NotSet:
            post_parameters["payload"] = payload
        if environment is not github.GithubObject.NotSet:
            post_parameters["environment"] = environment
        if description is not github.GithubObject.NotSet:
            post_parameters["description"] = description
        if transient_environment is not github.GithubObject.NotSet:
            post_parameters["transient_environment"] = transient_environment
        if production_environment is not github.GithubObject.NotSet:
            post_parameters["production_environment"] = production_environment

        headers, data = self._requester.requestJsonAndCheck(
            "POST",
            f"{self.url}/deployments",
            input=post_parameters,
            headers={"Accept": Consts.deploymentEnhancementsPreview},
        )

        return github.Deployment.Deployment(
            self._requester, headers, data, completed=True
        )

    def get_top_referrers(self):
        """
        :calls: `GET /repos/{owner}/{repo}/traffic/popular/referrers <https://docs.github.com/en/rest/reference/repos#traffic>`_
        :rtype: :class:`list` of :class:`github.Referrer.Referrer`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/traffic/popular/referrers"
        )
        if isinstance(data, list):
            return [
                github.Referrer.Referrer(self._requester, headers, item, completed=True)
                for item in data
            ]

    def get_top_paths(self):
        """
        :calls: `GET /repos/{owner}/{repo}/traffic/popular/paths <https://docs.github.com/en/rest/reference/repos#traffic>`_
        :rtype: :class:`list` of :class:`github.Path.Path`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/traffic/popular/paths"
        )
        if isinstance(data, list):
            return [
                github.Path.Path(self._requester, headers, item, completed=True)
                for item in data
            ]

    def get_views_traffic(self, per=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/traffic/views <https://docs.github.com/en/rest/reference/repos#traffic>`_
        :param per: string, must be one of day or week, day by default
        :rtype: None or list of :class:`github.View.View`
        """
        assert per is github.GithubObject.NotSet or (
            isinstance(per, str) and (per == "day" or per == "week")
        ), "per must be day or week, day by default"
        url_parameters = dict()
        if per is not github.GithubObject.NotSet:
            url_parameters["per"] = per
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/traffic/views", parameters=url_parameters
        )
        if (
            (isinstance(data, dict))
            and ("views" in data)
            and (isinstance(data["views"], list))
        ):
            data["views"] = [
                github.View.View(self._requester, headers, item, completed=True)
                for item in data["views"]
            ]
            return data

    def get_clones_traffic(self, per=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/traffic/clones <https://docs.github.com/en/rest/reference/repos#traffic>`_
        :param per: string, must be one of day or week, day by default
        :rtype: None or list of :class:`github.Clones.Clones`
        """
        assert per is github.GithubObject.NotSet or (
            isinstance(per, str) and (per == "day" or per == "week")
        ), "per must be day or week, day by default"
        url_parameters = dict()
        if per is not github.GithubObject.NotSet:
            url_parameters["per"] = per
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/traffic/clones", parameters=url_parameters
        )
        if (
            (isinstance(data, dict))
            and ("clones" in data)
            and (isinstance(data["clones"], list))
        ):
            data["clones"] = [
                github.Clones.Clones(self._requester, headers, item, completed=True)
                for item in data["clones"]
            ]
            return data

    def get_projects(self, state=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/projects <https://docs.github.com/en/rest/reference/projects#list-repository-projects>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Project.Project`
        :param state: string
        """

        url_parameters = dict()
        if state is not github.GithubObject.NotSet:
            url_parameters["state"] = state

        return github.PaginatedList.PaginatedList(
            github.Project.Project,
            self._requester,
            f"{self.url}/projects",
            url_parameters,
            {"Accept": Consts.mediaTypeProjectsPreview},
        )

    def get_autolinks(self):
        """
        :calls: `GET /repos/{owner}/{repo}/autolinks <http://docs.github.com/en/rest/reference/repos>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Autolink.Autolink`
        """
        return github.PaginatedList.PaginatedList(
            github.Autolink.Autolink, self._requester, f"{self.url}/autolinks", None
        )

    def create_file(
        self,
        path,
        message,
        content,
        branch=github.GithubObject.NotSet,
        committer=github.GithubObject.NotSet,
        author=github.GithubObject.NotSet,
    ):
        """Create a file in this repository.

        :calls: `PUT /repos/{owner}/{repo}/contents/{path} <https://docs.github.com/en/rest/reference/repos#create-or-update-file-contents>`_
        :param path: string, (required), path of the file in the repository
        :param message: string, (required), commit message
        :param content: string, (required), the actual data in the file
        :param branch: string, (optional), branch to create the commit on. Defaults to the default branch of the repository
        :param committer: InputGitAuthor, (optional), if no information is given the authenticated user's information will be used. You must specify both a name and email.
        :param author: InputGitAuthor, (optional), if omitted this will be filled in with committer information. If passed, you must specify both a name and email.
        :rtype: {
            'content': :class:`ContentFile <github.ContentFile.ContentFile>`:,
            'commit': :class:`Commit <github.Commit.Commit>`}
        """
        assert isinstance(path, str)
        assert isinstance(message, str)
        assert isinstance(content, (str, bytes))
        assert branch is github.GithubObject.NotSet or isinstance(branch, str)
        assert author is github.GithubObject.NotSet or isinstance(
            author, github.InputGitAuthor
        )
        assert committer is github.GithubObject.NotSet or isinstance(
            committer, github.InputGitAuthor
        )

        if not isinstance(content, bytes):
            content = content.encode("utf-8")
        content = b64encode(content).decode("utf-8")
        put_parameters = {"message": message, "content": content}

        if branch is not github.GithubObject.NotSet:
            put_parameters["branch"] = branch
        if author is not github.GithubObject.NotSet:
            put_parameters["author"] = author._identity
        if committer is not github.GithubObject.NotSet:
            put_parameters["committer"] = committer._identity

        headers, data = self._requester.requestJsonAndCheck(
            "PUT",
            f"{self.url}/contents/{urllib.parse.quote(path)}",
            input=put_parameters,
        )

        return {
            "content": github.ContentFile.ContentFile(
                self._requester, headers, data["content"], completed=False
            ),
            "commit": github.Commit.Commit(
                self._requester, headers, data["commit"], completed=True
            ),
        }

    def get_repository_advisories(
        self,
    ) -> "github.PaginatedList.PaginatedList[github.RepositoryAdvisory.RepositoryAdvisory]":
        """
        :calls: `GET /repos/{owner}/{repo}/security-advisories <https://docs.github.com/en/rest/security-advisories/repository-advisories>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.RepositoryAdvisory.RepositoryAdvisory`
        """
        return github.PaginatedList.PaginatedList(
            github.RepositoryAdvisory.RepositoryAdvisory,
            self._requester,
            f"{self.url}/security-advisories",
            None,
        )

    def get_repository_advisory(
        self, ghsa: str
    ) -> github.RepositoryAdvisory.RepositoryAdvisory:
        """
        :calls: `GET /repos/{owner}/{repo}/security-advisories/{ghsa} <https://docs.github.com/en/rest/security-advisories/repository-advisories>`_
        :param ghsa: string
        :rtype: :class:`github.RepositoryAdvisory.RepositoryAdvisory`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/security-advisories/{ghsa}"
        )
        return github.RepositoryAdvisory.RepositoryAdvisory(
            self._requester, headers, data, completed=True
        )

    def update_file(
        self,
        path,
        message,
        content,
        sha,
        branch=github.GithubObject.NotSet,
        committer=github.GithubObject.NotSet,
        author=github.GithubObject.NotSet,
    ):
        """This method updates a file in a repository

        :calls: `PUT /repos/{owner}/{repo}/contents/{path} <https://docs.github.com/en/rest/reference/repos#create-or-update-file-contents>`_
        :param path: string, Required. The content path.
        :param message: string, Required. The commit message.
        :param content: string, Required. The updated file content, either base64 encoded, or ready to be encoded.
        :param sha: string, Required. The blob SHA of the file being replaced.
        :param branch: string. The branch name. Default: the repository’s default branch (usually master)
        :param committer: InputGitAuthor, (optional), if no information is given the authenticated user's information will be used. You must specify both a name and email.
        :param author: InputGitAuthor, (optional), if omitted this will be filled in with committer information. If passed, you must specify both a name and email.
        :rtype: {
            'content': :class:`ContentFile <github.ContentFile.ContentFile>`:,
            'commit': :class:`Commit <github.Commit.Commit>`}
        """
        assert isinstance(path, str)
        assert isinstance(message, str)
        assert isinstance(content, (str, bytes))
        assert isinstance(sha, str)
        assert branch is github.GithubObject.NotSet or isinstance(branch, str)
        assert author is github.GithubObject.NotSet or isinstance(
            author, github.InputGitAuthor
        )
        assert committer is github.GithubObject.NotSet or isinstance(
            committer, github.InputGitAuthor
        )

        if not isinstance(content, bytes):
            content = content.encode("utf-8")
        content = b64encode(content).decode("utf-8")

        put_parameters = {"message": message, "content": content, "sha": sha}

        if branch is not github.GithubObject.NotSet:
            put_parameters["branch"] = branch
        if author is not github.GithubObject.NotSet:
            put_parameters["author"] = author._identity
        if committer is not github.GithubObject.NotSet:
            put_parameters["committer"] = committer._identity

        headers, data = self._requester.requestJsonAndCheck(
            "PUT",
            f"{self.url}/contents/{urllib.parse.quote(path)}",
            input=put_parameters,
        )

        return {
            "commit": github.Commit.Commit(
                self._requester, headers, data["commit"], completed=True
            ),
            "content": github.ContentFile.ContentFile(
                self._requester, headers, data["content"], completed=False
            ),
        }

    def delete_file(
        self,
        path,
        message,
        sha,
        branch=github.GithubObject.NotSet,
        committer=github.GithubObject.NotSet,
        author=github.GithubObject.NotSet,
    ):
        """This method deletes a file in a repository

        :calls: `DELETE /repos/{owner}/{repo}/contents/{path} <https://docs.github.com/en/rest/reference/repos#delete-a-file>`_
        :param path: string, Required. The content path.
        :param message: string, Required. The commit message.
        :param sha: string, Required. The blob SHA of the file being replaced.
        :param branch: string. The branch name. Default: the repository’s default branch (usually master)
        :param committer: InputGitAuthor, (optional), if no information is given the authenticated user's information will be used. You must specify both a name and email.
        :param author: InputGitAuthor, (optional), if omitted this will be filled in with committer information. If passed, you must specify both a name and email.
        :rtype: {
            'content': :class:`null <github.GithubObject.NotSet>`:,
            'commit': :class:`Commit <github.Commit.Commit>`}
        """
        assert isinstance(path, str), "path must be str/unicode object"
        assert isinstance(message, str), "message must be str/unicode object"
        assert isinstance(sha, str), "sha must be a str/unicode object"
        assert branch is github.GithubObject.NotSet or isinstance(
            branch, str
        ), "branch must be a str/unicode object"
        assert author is github.GithubObject.NotSet or isinstance(
            author, github.InputGitAuthor
        ), "author must be a github.InputGitAuthor object"
        assert committer is github.GithubObject.NotSet or isinstance(
            committer, github.InputGitAuthor
        ), "committer must be a github.InputGitAuthor object"

        url_parameters = {"message": message, "sha": sha}
        if branch is not github.GithubObject.NotSet:
            url_parameters["branch"] = branch
        if author is not github.GithubObject.NotSet:
            url_parameters["author"] = author._identity
        if committer is not github.GithubObject.NotSet:
            url_parameters["committer"] = committer._identity

        headers, data = self._requester.requestJsonAndCheck(
            "DELETE",
            f"{self.url}/contents/{urllib.parse.quote(path)}",
            input=url_parameters,
        )

        return {
            "commit": github.Commit.Commit(
                self._requester, headers, data["commit"], completed=True
            ),
            "content": github.GithubObject.NotSet,
        }

    @deprecated(
        reason="""
        Repository.get_dir_contents() is deprecated, use
        Repository.get_contents() instead.
        """
    )
    def get_dir_contents(self, path, ref=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/contents/{path} <https://docs.github.com/en/rest/reference/repos#contents>`_
        :param path: string
        :param ref: string
        :rtype: list of :class:`github.ContentFile.ContentFile`
        """
        return self.get_contents(path, ref=ref)

    def get_contributors(self, anon=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/contributors <https://docs.github.com/en/rest/reference/repos>`_
        :param anon: string
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """
        url_parameters = dict()
        if anon is not github.GithubObject.NotSet:
            url_parameters["anon"] = anon

        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser,
            self._requester,
            f"{self.url}/contributors",
            url_parameters,
        )

    def get_download(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/downloads/{id} <https://docs.github.com/en/rest/reference/repos>`_
        :param id: integer
        :rtype: :class:`github.Download.Download`
        """
        assert isinstance(id, int), id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/downloads/{id}"
        )
        return github.Download.Download(self._requester, headers, data, completed=True)

    def get_downloads(self):
        """
        :calls: `GET /repos/{owner}/{repo}/downloads <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Download.Download`
        """
        return github.PaginatedList.PaginatedList(
            github.Download.Download, self._requester, f"{self.url}/downloads", None
        )

    def get_events(self):
        """
        :calls: `GET /repos/{owner}/{repo}/events <https://docs.github.com/en/rest/reference/activity#events>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Event.Event`
        """
        return github.PaginatedList.PaginatedList(
            github.Event.Event, self._requester, f"{self.url}/events", None
        )

    def get_forks(self):
        """
        :calls: `GET /repos/{owner}/{repo}/forks <https://docs.github.com/en/rest/reference/repos#forks>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Repository.Repository`
        """
        return github.PaginatedList.PaginatedList(
            Repository, self._requester, f"{self.url}/forks", None
        )

    def create_fork(
        self,
        organization=github.GithubObject.NotSet,
        name=github.GithubObject.NotSet,
        default_branch_only=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/forks <https://docs.github.com/en/rest/reference/repos#forks>`_
        :param organization: :class:`github.Organization.Organization` or string
        :param name: string
        :param default_branch_only: bool
        :rtype: :class:`github.Repository.Repository`
        """
        post_parameters = {}
        if isinstance(organization, github.Organization.Organization):
            post_parameters["organization"] = organization.login
        elif isinstance(organization, str):
            post_parameters["organization"] = organization
        else:
            assert organization is github.GithubObject.NotSet, organization
        assert name is github.GithubObject.NotSet or isinstance(name, str), name
        assert default_branch_only is github.GithubObject.NotSet or isinstance(
            default_branch_only, bool
        ), default_branch_only
        if name is not github.GithubObject.NotSet:
            post_parameters["name"] = name
        if default_branch_only is not github.GithubObject.NotSet:
            post_parameters["default_branch_only"] = default_branch_only
        headers, data = self._requester.requestJsonAndCheck(
            "POST",
            f"{self.url}/forks",
            input=post_parameters,
        )
        return Repository(self._requester, headers, data, completed=True)

    def get_git_blob(self, sha):
        """
        :calls: `GET /repos/{owner}/{repo}/git/blobs/{sha} <https://docs.github.com/en/rest/reference/git#blobs>`_
        :param sha: string
        :rtype: :class:`github.GitBlob.GitBlob`
        """
        assert isinstance(sha, str), sha
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/git/blobs/{sha}"
        )
        return github.GitBlob.GitBlob(self._requester, headers, data, completed=True)

    def get_git_commit(self, sha):
        """
        :calls: `GET /repos/{owner}/{repo}/git/commits/{sha} <https://docs.github.com/en/rest/reference/git#commits>`_
        :param sha: string
        :rtype: :class:`github.GitCommit.GitCommit`
        """
        assert isinstance(sha, str), sha
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/git/commits/{sha}"
        )
        return github.GitCommit.GitCommit(
            self._requester, headers, data, completed=True
        )

    def get_git_ref(self, ref):
        """
        :calls: `GET /repos/{owner}/{repo}/git/refs/{ref} <https://docs.github.com/en/rest/reference/git#references>`_
        :param ref: string
        :rtype: :class:`github.GitRef.GitRef`
        """
        prefix = "/git/refs/"
        if not self._requester.FIX_REPO_GET_GIT_REF:
            prefix = "/git/"
        assert isinstance(ref, str), ref
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}{prefix}{ref}"
        )
        return github.GitRef.GitRef(self._requester, headers, data, completed=True)

    def get_git_refs(self):
        """
        :calls: `GET /repos/{owner}/{repo}/git/refs <https://docs.github.com/en/rest/reference/git#references>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.GitRef.GitRef`
        """
        return github.PaginatedList.PaginatedList(
            github.GitRef.GitRef, self._requester, f"{self.url}/git/refs", None
        )

    def get_git_matching_refs(self, ref):
        """
        :calls: `GET /repos/{owner}/{repo}/git/matching-refs/{ref} <https://docs.github.com/en/rest/reference/git#list-matching-references>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.GitRef.GitRef`
        """
        assert isinstance(ref, str), ref
        return github.PaginatedList.PaginatedList(
            github.GitRef.GitRef,
            self._requester,
            f"{self.url}/git/matching-refs/{ref}",
            None,
        )

    def get_git_tag(self, sha):
        """
        :calls: `GET /repos/{owner}/{repo}/git/tags/{sha} <https://docs.github.com/en/rest/reference/git#tags>`_
        :param sha: string
        :rtype: :class:`github.GitTag.GitTag`
        """
        assert isinstance(sha, str), sha
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/git/tags/{sha}"
        )
        return github.GitTag.GitTag(self._requester, headers, data, completed=True)

    def get_git_tree(self, sha, recursive=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/git/trees/{sha} <https://docs.github.com/en/rest/reference/git#trees>`_
        :param sha: string
        :param recursive: bool
        :rtype: :class:`github.GitTree.GitTree`
        """
        assert isinstance(sha, str), sha
        assert recursive is github.GithubObject.NotSet or isinstance(
            recursive, bool
        ), recursive
        url_parameters = dict()
        if recursive is not github.GithubObject.NotSet and recursive:
            # GitHub API requires the recursive parameter be set to 1.
            url_parameters["recursive"] = 1
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/git/trees/{sha}", parameters=url_parameters
        )
        return github.GitTree.GitTree(self._requester, headers, data, completed=True)

    def get_hook(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/hooks/{id} <https://docs.github.com/en/rest/reference/repos#webhooks>`_
        :param id: integer
        :rtype: :class:`github.Hook.Hook`
        """
        assert isinstance(id, int), id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/hooks/{id}"
        )
        return github.Hook.Hook(self._requester, headers, data, completed=True)

    def get_hooks(self):
        """
        :calls: `GET /repos/{owner}/{repo}/hooks <https://docs.github.com/en/rest/reference/repos#webhooks>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Hook.Hook`
        """
        return github.PaginatedList.PaginatedList(
            github.Hook.Hook, self._requester, f"{self.url}/hooks", None
        )

    def get_hook_delivery(
        self, hook_id: int, delivery_id: int
    ) -> github.HookDelivery.HookDelivery:
        """
        :calls: `GET /repos/{owner}/{repo}/hooks/{hook_id}/deliveries/{delivery_id} <https://docs.github.com/en/rest/webhooks/repo-deliveries>`_
        :param hook_id: integer
        :param delivery_id: integer
        :rtype: :class:`github.HookDelivery.HookDelivery`
        """
        assert isinstance(hook_id, int), hook_id
        assert isinstance(delivery_id, int), delivery_id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/hooks/{hook_id}/deliveries/{delivery_id}"
        )
        return github.HookDelivery.HookDelivery(
            self._requester, headers, data, completed=True
        )

    def get_hook_deliveries(
        self, hook_id: int
    ) -> github.PaginatedList.PaginatedList[github.HookDelivery.HookDeliverySummary]:
        """
        :calls: `GET /repos/{owner}/{repo}/hooks/{hook_id}/deliveries <https://docs.github.com/en/rest/webhooks/repo-deliveries>`_
        :param hook_id: integer
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.HookDelivery.HookDeliverySummary`
        """
        assert isinstance(hook_id, int), hook_id

        return github.PaginatedList.PaginatedList(
            github.HookDelivery.HookDeliverySummary,
            self._requester,
            f"{self.url}/hooks/{hook_id}/deliveries",
            None,
        )

    def get_issue(self, number):
        """
        :calls: `GET /repos/{owner}/{repo}/issues/{number} <https://docs.github.com/en/rest/reference/issues>`_
        :param number: integer
        :rtype: :class:`github.Issue.Issue`
        """
        assert isinstance(number, int), number
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/issues/{number}"
        )
        return github.Issue.Issue(self._requester, headers, data, completed=True)

    def get_issues(
        self,
        milestone=github.GithubObject.NotSet,
        state=github.GithubObject.NotSet,
        assignee=github.GithubObject.NotSet,
        mentioned=github.GithubObject.NotSet,
        labels=github.GithubObject.NotSet,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
        creator=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/issues <https://docs.github.com/en/rest/reference/issues>`_
        :param milestone: :class:`github.Milestone.Milestone` or "none" or "*"
        :param state: string. `open`, `closed`, or `all`. If this is not set the GitHub API default behavior will be used. At the moment this is to return only open issues. This might change anytime on GitHub API side and it could be clever to explicitly specify the state value.
        :param assignee: string or :class:`github.NamedUser.NamedUser` or "none" or "*"
        :param mentioned: :class:`github.NamedUser.NamedUser`
        :param labels: list of string or :class:`github.Label.Label`
        :param sort: string
        :param direction: string
        :param since: datetime
        :param creator: string or :class:`github.NamedUser.NamedUser`
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Issue.Issue`
        """
        assert (
            milestone is github.GithubObject.NotSet
            or milestone == "*"
            or milestone == "none"
            or isinstance(milestone, github.Milestone.Milestone)
        ), milestone
        assert state is github.GithubObject.NotSet or isinstance(state, str), state
        assert (
            assignee is github.GithubObject.NotSet
            or isinstance(assignee, github.NamedUser.NamedUser)
            or isinstance(assignee, str)
        ), assignee
        assert mentioned is github.GithubObject.NotSet or isinstance(
            mentioned, github.NamedUser.NamedUser
        ), mentioned
        assert labels is github.GithubObject.NotSet or all(
            isinstance(element, github.Label.Label) or isinstance(element, str)
            for element in labels
        ), labels
        assert sort is github.GithubObject.NotSet or isinstance(sort, str), sort
        assert direction is github.GithubObject.NotSet or isinstance(
            direction, str
        ), direction
        assert since is github.GithubObject.NotSet or isinstance(since, datetime), since
        assert (
            creator is github.GithubObject.NotSet
            or isinstance(creator, github.NamedUser.NamedUser)
            or isinstance(creator, str)
        ), creator
        url_parameters = dict()
        if milestone is not github.GithubObject.NotSet:
            if isinstance(milestone, str):
                url_parameters["milestone"] = milestone
            else:
                url_parameters["milestone"] = milestone._identity
        if state is not github.GithubObject.NotSet:
            url_parameters["state"] = state
        if assignee is not github.GithubObject.NotSet:
            if isinstance(assignee, str):
                url_parameters["assignee"] = assignee
            else:
                url_parameters["assignee"] = assignee._identity
        if mentioned is not github.GithubObject.NotSet:
            url_parameters["mentioned"] = mentioned._identity
        if labels is not github.GithubObject.NotSet:
            url_parameters["labels"] = ",".join(
                [
                    label.name if isinstance(label, github.Label.Label) else label
                    for label in labels
                ]
            )
        if sort is not github.GithubObject.NotSet:
            url_parameters["sort"] = sort
        if direction is not github.GithubObject.NotSet:
            url_parameters["direction"] = direction
        if since is not github.GithubObject.NotSet:
            url_parameters["since"] = since.strftime("%Y-%m-%dT%H:%M:%SZ")
        if creator is not github.GithubObject.NotSet:
            if isinstance(creator, str):
                url_parameters["creator"] = creator
            else:
                url_parameters["creator"] = creator._identity
        return github.PaginatedList.PaginatedList(
            github.Issue.Issue, self._requester, f"{self.url}/issues", url_parameters
        )

    def get_issues_comments(
        self,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/issues/comments <https://docs.github.com/en/rest/reference/issues#comments>`_
        :param sort: string
        :param direction: string
        :param since: datetime
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.IssueComment.IssueComment`
        """
        assert sort is github.GithubObject.NotSet or isinstance(sort, str), sort
        assert direction is github.GithubObject.NotSet or isinstance(
            direction, str
        ), direction
        assert since is github.GithubObject.NotSet or isinstance(since, datetime), since
        url_parameters = dict()
        if sort is not github.GithubObject.NotSet:
            url_parameters["sort"] = sort
        if direction is not github.GithubObject.NotSet:
            url_parameters["direction"] = direction
        if since is not github.GithubObject.NotSet:
            url_parameters["since"] = since.strftime("%Y-%m-%dT%H:%M:%SZ")
        return github.PaginatedList.PaginatedList(
            github.IssueComment.IssueComment,
            self._requester,
            f"{self.url}/issues/comments",
            url_parameters,
        )

    def get_issues_event(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/issues/events/{id} <https://docs.github.com/en/rest/reference/issues#events>`_
        :param id: integer
        :rtype: :class:`github.IssueEvent.IssueEvent`
        """
        assert isinstance(id, int), id
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/issues/events/{id}",
            headers={"Accept": Consts.mediaTypeLockReasonPreview},
        )
        return github.IssueEvent.IssueEvent(
            self._requester, headers, data, completed=True
        )

    def get_issues_events(self):
        """
        :calls: `GET /repos/{owner}/{repo}/issues/events <https://docs.github.com/en/rest/reference/issues#events>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.IssueEvent.IssueEvent`
        """
        return github.PaginatedList.PaginatedList(
            github.IssueEvent.IssueEvent,
            self._requester,
            f"{self.url}/issues/events",
            None,
            headers={"Accept": Consts.mediaTypeLockReasonPreview},
        )

    def get_key(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/keys/{id} <https://docs.github.com/en/rest/reference/repos#deploy-keys>`_
        :param id: integer
        :rtype: :class:`github.RepositoryKey.RepositoryKey`
        """
        assert isinstance(id, int), id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/keys/{id}"
        )
        return github.RepositoryKey.RepositoryKey(
            self._requester, headers, data, completed=True
        )

    def get_keys(self):
        """
        :calls: `GET /repos/{owner}/{repo}/keys <https://docs.github.com/en/rest/reference/repos#deploy-keys>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.RepositoryKey.RepositoryKey`
        """
        return github.PaginatedList.PaginatedList(
            github.RepositoryKey.RepositoryKey,
            self._requester,
            f"{self.url}/keys",
            None,
        )

    def get_label(self, name):
        """
        :calls: `GET /repos/{owner}/{repo}/labels/{name} <https://docs.github.com/en/rest/reference/issues#labels>`_
        :param name: string
        :rtype: :class:`github.Label.Label`
        """
        assert isinstance(name, str), name
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/labels/{urllib.parse.quote(name)}"
        )
        return github.Label.Label(self._requester, headers, data, completed=True)

    def get_labels(self):
        """
        :calls: `GET /repos/{owner}/{repo}/labels <https://docs.github.com/en/rest/reference/issues#labels>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Label.Label`
        """
        return github.PaginatedList.PaginatedList(
            github.Label.Label, self._requester, f"{self.url}/labels", None
        )

    def get_languages(self):
        """
        :calls: `GET /repos/{owner}/{repo}/languages <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: dict of string to integer
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/languages"
        )
        return data

    def get_license(self):
        """
        :calls: `GET /repos/{owner}/{repo}/license <https://docs.github.com/en/rest/reference/licenses>`_
        :rtype: :class:`github.ContentFile.ContentFile`
        """

        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/license"
        )
        return github.ContentFile.ContentFile(
            self._requester, headers, data, completed=True
        )

    def get_milestone(self, number):
        """
        :calls: `GET /repos/{owner}/{repo}/milestones/{number} <https://docs.github.com/en/rest/reference/issues#milestones>`_
        :param number: integer
        :rtype: :class:`github.Milestone.Milestone`
        """
        assert isinstance(number, int), number
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/milestones/{number}"
        )
        return github.Milestone.Milestone(
            self._requester, headers, data, completed=True
        )

    def get_milestones(
        self,
        state=github.GithubObject.NotSet,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/milestones <https://docs.github.com/en/rest/reference/issues#milestones>`_
        :param state: string
        :param sort: string
        :param direction: string
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Milestone.Milestone`
        """
        assert state is github.GithubObject.NotSet or isinstance(state, str), state
        assert sort is github.GithubObject.NotSet or isinstance(sort, str), sort
        assert direction is github.GithubObject.NotSet or isinstance(
            direction, str
        ), direction
        url_parameters = dict()
        if state is not github.GithubObject.NotSet:
            url_parameters["state"] = state
        if sort is not github.GithubObject.NotSet:
            url_parameters["sort"] = sort
        if direction is not github.GithubObject.NotSet:
            url_parameters["direction"] = direction
        return github.PaginatedList.PaginatedList(
            github.Milestone.Milestone,
            self._requester,
            f"{self.url}/milestones",
            url_parameters,
        )

    def get_network_events(self):
        """
        :calls: `GET /networks/{owner}/{repo}/events <https://docs.github.com/en/rest/reference/activity#events>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Event.Event`
        """
        return github.PaginatedList.PaginatedList(
            github.Event.Event,
            self._requester,
            f"/networks/{self.owner.login}/{self.name}/events",
            None,
        )

    def get_public_key(self):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/secrets/public-key <https://docs.github.com/en/rest/reference/actions#get-a-repository-public-key>`_
        :rtype: :class:`github.PublicKey.PublicKey`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/actions/secrets/public-key"
        )
        return github.PublicKey.PublicKey(
            self._requester, headers, data, completed=True
        )

    def get_pull(self, number):
        """
        :calls: `GET /repos/{owner}/{repo}/pulls/{number} <https://docs.github.com/en/rest/reference/pulls>`_
        :param number: integer
        :rtype: :class:`github.PullRequest.PullRequest`
        """
        assert isinstance(number, int), number
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/pulls/{number}"
        )
        return github.PullRequest.PullRequest(
            self._requester, headers, data, completed=True
        )

    def get_pulls(
        self,
        state=github.GithubObject.NotSet,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
        base=github.GithubObject.NotSet,
        head=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/pulls <https://docs.github.com/en/rest/reference/pulls>`_
        :param state: string
        :param sort: string
        :param direction: string
        :param base: string
        :param head: string
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.PullRequest.PullRequest`
        """
        assert state is github.GithubObject.NotSet or isinstance(state, str), state
        assert sort is github.GithubObject.NotSet or isinstance(sort, str), sort
        assert direction is github.GithubObject.NotSet or isinstance(
            direction, str
        ), direction
        assert base is github.GithubObject.NotSet or isinstance(base, str), base
        assert head is github.GithubObject.NotSet or isinstance(head, str), head
        url_parameters = dict()
        if state is not github.GithubObject.NotSet:
            url_parameters["state"] = state
        if sort is not github.GithubObject.NotSet:
            url_parameters["sort"] = sort
        if direction is not github.GithubObject.NotSet:
            url_parameters["direction"] = direction
        if base is not github.GithubObject.NotSet:
            url_parameters["base"] = base
        if head is not github.GithubObject.NotSet:
            url_parameters["head"] = head
        return github.PaginatedList.PaginatedList(
            github.PullRequest.PullRequest,
            self._requester,
            f"{self.url}/pulls",
            url_parameters,
        )

    def get_pulls_comments(
        self,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/pulls/comments <https://docs.github.com/en/rest/reference/pulls#comments>`_
        :param sort: string
        :param direction: string
        :param since: datetime
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.PullRequestComment.PullRequestComment`
        """
        return self.get_pulls_review_comments(sort, direction, since)

    def get_pulls_review_comments(
        self,
        sort=github.GithubObject.NotSet,
        direction=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/pulls/comments <https://docs.github.com/en/rest/reference/pulls#review-comments>`_
        :param sort: string 'created', 'updated', 'created_at'
        :param direction: string 'asc' or 'desc'
        :param since: datetime
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.PullRequestComment.PullRequestComment`
        """
        assert sort is github.GithubObject.NotSet or isinstance(sort, str), sort
        assert direction is github.GithubObject.NotSet or isinstance(
            direction, str
        ), direction
        assert since is github.GithubObject.NotSet or isinstance(since, datetime), since
        url_parameters = dict()
        if sort is not github.GithubObject.NotSet:
            url_parameters["sort"] = sort
        if direction is not github.GithubObject.NotSet:
            url_parameters["direction"] = direction
        if since is not github.GithubObject.NotSet:
            url_parameters["since"] = since.strftime("%Y-%m-%dT%H:%M:%SZ")
        return github.PaginatedList.PaginatedList(
            github.PullRequestComment.PullRequestComment,
            self._requester,
            f"{self.url}/pulls/comments",
            url_parameters,
        )

    def get_readme(self, ref=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/readme <https://docs.github.com/en/rest/reference/repos#contents>`_
        :param ref: string
        :rtype: :class:`github.ContentFile.ContentFile`
        """
        assert ref is github.GithubObject.NotSet or isinstance(ref, str), ref
        url_parameters = dict()
        if ref is not github.GithubObject.NotSet:
            url_parameters["ref"] = ref
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/readme", parameters=url_parameters
        )
        return github.ContentFile.ContentFile(
            self._requester, headers, data, completed=True
        )

    def get_self_hosted_runner(self, runner_id):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/runners/{id} <https://docs.github.com/en/rest/reference/actions#get-a-self-hosted-runner-for-a-repository>`_
        :param runner_id: int
        :rtype: :class:`github.SelfHostedActionsRunner.SelfHostedActionsRunner`
        """
        assert isinstance(runner_id, int), runner_id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/actions/runners/{runner_id}"
        )
        return github.SelfHostedActionsRunner.SelfHostedActionsRunner(
            self._requester, headers, data, completed=True
        )

    def get_self_hosted_runners(self):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/runners <https://docs.github.com/en/rest/reference/actions#list-self-hosted-runners-for-a-repository>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.SelfHostedActionsRunner.SelfHostedActionsRunner`
        """
        return github.PaginatedList.PaginatedList(
            github.SelfHostedActionsRunner.SelfHostedActionsRunner,
            self._requester,
            f"{self.url}/actions/runners",
            None,
            list_item="runners",
        )

    def get_source_import(self):
        """
        :calls: `GET /repos/{owner}/{repo}/import <https://docs.github.com/en/rest/reference/migrations#source-imports>`_
        :rtype: :class:`github.SourceImport.SourceImport`
        """
        import_header = {"Accept": Consts.mediaTypeImportPreview}
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/import",
            headers=import_header,
        )
        if not data:
            return None
        else:
            return github.SourceImport.SourceImport(
                self._requester, headers, data, completed=True
            )

    def get_stargazers(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stargazers <https://docs.github.com/en/rest/reference/activity#starring>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """
        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser, self._requester, f"{self.url}/stargazers", None
        )

    def get_stargazers_with_dates(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stargazers <https://docs.github.com/en/rest/reference/activity#starring>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Stargazer.Stargazer`
        """
        return github.PaginatedList.PaginatedList(
            github.Stargazer.Stargazer,
            self._requester,
            f"{self.url}/stargazers",
            None,
            headers={"Accept": Consts.mediaTypeStarringPreview},
        )

    def get_stats_contributors(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stats/contributors <https://docs.github.com/en/rest/reference/repos#get-all-contributor-commit-activity>`_
        :rtype: None or list of :class:`github.StatsContributor.StatsContributor`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/stats/contributors"
        )
        if not data:
            return None
        else:
            return [
                github.StatsContributor.StatsContributor(
                    self._requester, headers, attributes, completed=True
                )
                for attributes in data
            ]

    def get_stats_commit_activity(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stats/commit_activity <https://docs.github.com/en/rest/reference/repos#get-the-last-year-of-commit-activity>`_
        :rtype: None or list of :class:`github.StatsCommitActivity.StatsCommitActivity`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/stats/commit_activity"
        )
        if not data:
            return None
        else:
            return [
                github.StatsCommitActivity.StatsCommitActivity(
                    self._requester, headers, attributes, completed=True
                )
                for attributes in data
            ]

    def get_stats_code_frequency(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stats/code_frequency <https://docs.github.com/en/rest/reference/repos#get-the-weekly-commit-activity>`_
        :rtype: None or list of :class:`github.StatsCodeFrequency.StatsCodeFrequency`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/stats/code_frequency"
        )
        if not data:
            return None
        else:
            return [
                github.StatsCodeFrequency.StatsCodeFrequency(
                    self._requester, headers, attributes, completed=True
                )
                for attributes in data
            ]

    def get_stats_participation(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stats/participation <https://docs.github.com/en/rest/reference/repos#get-the-weekly-commit-count>`_
        :rtype: None or :class:`github.StatsParticipation.StatsParticipation`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/stats/participation"
        )
        if not data:
            return None
        else:
            return github.StatsParticipation.StatsParticipation(
                self._requester, headers, data, completed=True
            )

    def get_stats_punch_card(self):
        """
        :calls: `GET /repos/{owner}/{repo}/stats/punch_card <https://docs.github.com/en/rest/reference/repos#get-the-hourly-commit-count-for-each-day>`_
        :rtype: None or :class:`github.StatsPunchCard.StatsPunchCard`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/stats/punch_card"
        )
        if not data:
            return None
        else:
            return github.StatsPunchCard.StatsPunchCard(
                self._requester, headers, data, completed=True
            )

    def get_subscribers(self):
        """
        :calls: `GET /repos/{owner}/{repo}/subscribers <https://docs.github.com/en/rest/reference/activity#watching>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """
        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser, self._requester, f"{self.url}/subscribers", None
        )

    def get_tags(self):
        """
        :calls: `GET /repos/{owner}/{repo}/tags <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Tag.Tag`
        """
        return github.PaginatedList.PaginatedList(
            github.Tag.Tag, self._requester, f"{self.url}/tags", None
        )

    def get_releases(self):
        """
        :calls: `GET /repos/{owner}/{repo}/releases <https://docs.github.com/en/rest/reference/repos#list-releases>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.GitRelease.GitRelease`
        """
        return github.PaginatedList.PaginatedList(
            github.GitRelease.GitRelease, self._requester, f"{self.url}/releases", None
        )

    def get_release(self, id):
        """
        :calls: `GET /repos/{owner}/{repo}/releases/{id} <https://docs.github.com/en/rest/reference/repos#get-a-release>`_
        :param id: int (release id), str (tag name)
        :rtype: None or :class:`github.GitRelease.GitRelease`
        """
        if isinstance(id, int):
            headers, data = self._requester.requestJsonAndCheck(
                "GET", f"{self.url}/releases/{id}"
            )
            return github.GitRelease.GitRelease(
                self._requester, headers, data, completed=True
            )
        elif isinstance(id, str):
            headers, data = self._requester.requestJsonAndCheck(
                "GET", f"{self.url}/releases/tags/{id}"
            )
            return github.GitRelease.GitRelease(
                self._requester, headers, data, completed=True
            )

    def get_latest_release(self):
        """
        :calls: `GET /repos/{owner}/{repo}/releases/latest <https://docs.github.com/en/rest/reference/repos#get-the-latest-release>`_
        :rtype: :class:`github.GitRelease.GitRelease`
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/releases/latest"
        )
        return github.GitRelease.GitRelease(
            self._requester, headers, data, completed=True
        )

    def get_teams(self):
        """
        :calls: `GET /repos/{owner}/{repo}/teams <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Team.Team`
        """
        return github.PaginatedList.PaginatedList(
            github.Team.Team, self._requester, f"{self.url}/teams", None
        )

    def get_topics(self):
        """
        :calls: `GET /repos/{owner}/{repo}/topics <https://docs.github.com/en/rest/reference/repos#replace-all-repository-topics>`_
        :rtype: list of strings
        """
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/topics",
            headers={"Accept": Consts.mediaTypeTopicsPreview},
        )
        return data["names"]

    def get_watchers(self):
        """
        :calls: `GET /repos/{owner}/{repo}/watchers <https://docs.github.com/en/rest/reference/activity#starring>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.NamedUser.NamedUser`
        """
        return github.PaginatedList.PaginatedList(
            github.NamedUser.NamedUser, self._requester, f"{self.url}/watchers", None
        )

    def get_workflows(self):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/workflows <https://docs.github.com/en/rest/reference/actions#workflows>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Workflow.Workflow`
        """
        return github.PaginatedList.PaginatedList(
            github.Workflow.Workflow,
            self._requester,
            f"{self.url}/actions/workflows",
            None,
            list_item="workflows",
        )

    def get_workflow(self, id_or_name):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/workflows/{workflow_id} <https://docs.github.com/en/rest/reference/actions#workflows>`_
        :param id_or_name: int or string

        :rtype: :class:`github.Workflow.Workflow`
        """
        assert isinstance(id_or_name, int) or isinstance(id_or_name, str), id_or_name
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/actions/workflows/{id_or_name}"
        )
        return github.Workflow.Workflow(self._requester, headers, data, completed=True)

    def get_workflow_runs(
        self,
        actor=github.GithubObject.NotSet,
        branch=github.GithubObject.NotSet,
        event=github.GithubObject.NotSet,
        status=github.GithubObject.NotSet,
        exclude_pull_requests=github.GithubObject.NotSet,
        head_sha=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/runs <https://docs.github.com/en/rest/reference/actions#list-workflow-runs-for-a-repository>`_
        :param actor: :class:`github.NamedUser.NamedUser` or string
        :param branch: :class:`github.Branch.Branch` or string
        :param event: string
        :param status: string `queued`, `in_progress`, `completed`, `success`, `failure`, `neutral`, `cancelled`, `skipped`, `timed_out`, or `action_required`
        :param exclude_pull_requests: bool
        :param head_sha: string

        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.WorkflowRun.WorkflowRun`
        """
        assert (
            actor is github.GithubObject.NotSet
            or isinstance(actor, github.NamedUser.NamedUser)
            or isinstance(actor, str)
        ), actor
        assert (
            branch is github.GithubObject.NotSet
            or isinstance(branch, github.Branch.Branch)
            or isinstance(branch, str)
        ), branch
        assert event is github.GithubObject.NotSet or isinstance(event, str), event
        assert status is github.GithubObject.NotSet or isinstance(status, str), status
        assert exclude_pull_requests is github.GithubObject.NotSet or isinstance(
            exclude_pull_requests, bool
        ), exclude_pull_requests
        assert head_sha is github.GithubObject.NotSet or isinstance(
            head_sha, str
        ), head_sha

        url_parameters = dict()
        if actor is not github.GithubObject.NotSet:
            if isinstance(actor, github.NamedUser.NamedUser):
                url_parameters["actor"] = actor._identity
            else:
                url_parameters["actor"] = actor
        if branch is not github.GithubObject.NotSet:
            if isinstance(branch, github.Branch.Branch):
                url_parameters["branch"] = branch.name
            else:
                url_parameters["branch"] = branch
        if event is not github.GithubObject.NotSet:
            url_parameters["event"] = event
        if status is not github.GithubObject.NotSet:
            url_parameters["status"] = status
        if (
            exclude_pull_requests is not github.GithubObject.NotSet
            and exclude_pull_requests
        ):
            url_parameters["exclude_pull_requests"] = 1
        if head_sha is not github.GithubObject.NotSet:
            url_parameters["head_sha"] = head_sha

        return github.PaginatedList.PaginatedList(
            github.WorkflowRun.WorkflowRun,
            self._requester,
            f"{self.url}/actions/runs",
            url_parameters,
            list_item="workflow_runs",
        )

    def get_workflow_run(self, id_):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/runs/{run_id} <https://docs.github.com/en/rest/reference/actions#workflow-runs>`_
        :param id_: int

        :rtype: :class:`github.WorkflowRun.WorkflowRun`
        """
        assert isinstance(id_, int)
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/actions/runs/{id_}"
        )
        return github.WorkflowRun.WorkflowRun(
            self._requester, headers, data, completed=True
        )

    def has_in_assignees(self, assignee):
        """
        :calls: `GET /repos/{owner}/{repo}/assignees/{assignee} <https://docs.github.com/en/rest/reference/issues#assignees>`_
        :param assignee: string or :class:`github.NamedUser.NamedUser`
        :rtype: bool
        """
        assert isinstance(assignee, github.NamedUser.NamedUser) or isinstance(
            assignee, str
        ), assignee

        if isinstance(assignee, github.NamedUser.NamedUser):
            assignee = assignee._identity

        status, headers, data = self._requester.requestJson(
            "GET", f"{self.url}/assignees/{assignee}"
        )
        return status == 204

    def has_in_collaborators(self, collaborator):
        """
        :calls: `GET /repos/{owner}/{repo}/collaborators/{user} <https://docs.github.com/en/rest/reference/repos#collaborators>`_
        :param collaborator: string or :class:`github.NamedUser.NamedUser`
        :rtype: bool
        """
        assert isinstance(collaborator, github.NamedUser.NamedUser) or isinstance(
            collaborator, str
        ), collaborator

        if isinstance(collaborator, github.NamedUser.NamedUser):
            collaborator = collaborator._identity

        status, headers, data = self._requester.requestJson(
            "GET", f"{self.url}/collaborators/{collaborator}"
        )
        return status == 204

    def _legacy_convert_issue(self, attributes):
        convertedAttributes = {
            "number": attributes["number"],
            "url": f"/repos{urllib.parse.urlparse(attributes['html_url']).path}",
            "user": {
                "login": attributes["user"],
                "url": f"/users/{attributes['user']}",
            },
        }
        if "labels" in attributes:  # pragma no branch
            convertedAttributes["labels"] = [
                {"name": label} for label in attributes["labels"]
            ]
        for attr in ("title", "created_at", "comments", "body", "updated_at", "state"):
            if attr in attributes:  # pragma no branch
                convertedAttributes[attr] = attributes[attr]
        return convertedAttributes

    def legacy_search_issues(self, state, keyword):
        """
        :calls: `GET /legacy/issues/search/{owner}/{repository}/{state}/{keyword} <https://docs.github.com/en/rest/reference/search>`_
        :param state: "open" or "closed"
        :param keyword: string
        :rtype: List of :class:`github.Issue.Issue`
        """
        assert state in ["open", "closed"], state
        assert isinstance(keyword, str), keyword
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"/legacy/issues/search/{self.owner.login}/{self.name}/{state}/{urllib.parse.quote(keyword)}",
        )
        return [
            github.Issue.Issue(
                self._requester,
                headers,
                self._legacy_convert_issue(element),
                completed=False,
            )
            for element in data["issues"]
        ]

    def get_notifications(
        self,
        all=github.GithubObject.NotSet,
        participating=github.GithubObject.NotSet,
        since=github.GithubObject.NotSet,
        before=github.GithubObject.NotSet,
    ):
        """
        :calls: `GET /repos/{owner}/{repo}/notifications <https://docs.github.com/en/rest/reference/activity#notifications>`_
        :param all: bool
        :param participating: bool
        :param since: datetime
        :param before: datetime
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Notification.Notification`
        """

        assert all is github.GithubObject.NotSet or isinstance(all, bool), all
        assert participating is github.GithubObject.NotSet or isinstance(
            participating, bool
        ), participating
        assert since is github.GithubObject.NotSet or isinstance(since, datetime), since
        assert before is github.GithubObject.NotSet or isinstance(
            before, datetime
        ), before

        params = dict()
        if all is not github.GithubObject.NotSet:
            params["all"] = all
        if participating is not github.GithubObject.NotSet:
            params["participating"] = participating
        if since is not github.GithubObject.NotSet:
            params["since"] = since.strftime("%Y-%m-%dT%H:%M:%SZ")
        if before is not github.GithubObject.NotSet:
            params["before"] = before.strftime("%Y-%m-%dT%H:%M:%SZ")

        return github.PaginatedList.PaginatedList(
            github.Notification.Notification,
            self._requester,
            f"{self.url}/notifications",
            params,
        )

    def mark_notifications_as_read(self, last_read_at=datetime.now(timezone.utc)):
        """
        :calls: `PUT /repos/{owner}/{repo}/notifications <https://docs.github.com/en/rest/reference/activity#notifications>`_
        :param last_read_at: datetime
        """
        assert isinstance(last_read_at, datetime)
        put_parameters = {"last_read_at": last_read_at.strftime("%Y-%m-%dT%H:%M:%SZ")}

        headers, data = self._requester.requestJsonAndCheck(
            "PUT", f"{self.url}/notifications", input=put_parameters
        )

    def merge(self, base, head, commit_message=github.GithubObject.NotSet):
        """
        :calls: `POST /repos/{owner}/{repo}/merges <https://docs.github.com/en/rest/reference/repos#merging>`_
        :param base: string
        :param head: string
        :param commit_message: string
        :rtype: :class:`github.Commit.Commit`
        """
        assert isinstance(base, str), base
        assert isinstance(head, str), head
        assert commit_message is github.GithubObject.NotSet or isinstance(
            commit_message, str
        ), commit_message
        post_parameters = {
            "base": base,
            "head": head,
        }
        if commit_message is not github.GithubObject.NotSet:
            post_parameters["commit_message"] = commit_message
        headers, data = self._requester.requestJsonAndCheck(
            "POST", f"{self.url}/merges", input=post_parameters
        )
        if data is None:
            return None
        else:
            return github.Commit.Commit(self._requester, headers, data, completed=True)

    def replace_topics(self, topics):
        """
        :calls: `PUT /repos/{owner}/{repo}/topics <https://docs.github.com/en/rest/reference/repos>`_
        :param topics: list of strings
        :rtype: None
        """
        post_parameters = {"names": topics}
        headers, data = self._requester.requestJsonAndCheck(
            "PUT",
            f"{self.url}/topics",
            headers={"Accept": Consts.mediaTypeTopicsPreview},
            input=post_parameters,
        )

    def get_vulnerability_alert(self):
        """
        :calls: `GET /repos/{owner}/{repo}/vulnerability-alerts <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: bool
        """
        status, _, _ = self._requester.requestJson(
            "GET",
            f"{self.url}/vulnerability-alerts",
            headers={"Accept": Consts.vulnerabilityAlertsPreview},
        )
        return status == 204

    def enable_vulnerability_alert(self):
        """
        :calls: `PUT /repos/{owner}/{repo}/vulnerability-alerts <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: bool
        """
        status, _, _ = self._requester.requestJson(
            "PUT",
            f"{self.url}/vulnerability-alerts",
            headers={"Accept": Consts.vulnerabilityAlertsPreview},
        )
        return status == 204

    def disable_vulnerability_alert(self):
        """
        :calls: `DELETE /repos/{owner}/{repo}/vulnerability-alerts <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: bool
        """
        status, _, _ = self._requester.requestJson(
            "DELETE",
            f"{self.url}/vulnerability-alerts",
            headers={"Accept": Consts.vulnerabilityAlertsPreview},
        )
        return status == 204

    def enable_automated_security_fixes(self):
        """
        :calls: `PUT /repos/{owner}/{repo}/automated-security-fixes <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: bool
        """
        status, _, _ = self._requester.requestJson(
            "PUT",
            f"{self.url}/automated-security-fixes",
            headers={"Accept": Consts.automatedSecurityFixes},
        )
        return status == 204

    def disable_automated_security_fixes(self):
        """
        :calls: `DELETE /repos/{owner}/{repo}/automated-security-fixes <https://docs.github.com/en/rest/reference/repos>`_
        :rtype: bool
        """
        status, _, _ = self._requester.requestJson(
            "DELETE",
            f"{self.url}/automated-security-fixes",
            headers={"Accept": Consts.automatedSecurityFixes},
        )
        return status == 204

    def remove_from_collaborators(self, collaborator):
        """
        :calls: `DELETE /repos/{owner}/{repo}/collaborators/{user} <https://docs.github.com/en/rest/reference/repos#collaborators>`_
        :param collaborator: string or :class:`github.NamedUser.NamedUser`
        :rtype: None
        """
        assert isinstance(collaborator, github.NamedUser.NamedUser) or isinstance(
            collaborator, str
        ), collaborator

        if isinstance(collaborator, github.NamedUser.NamedUser):
            collaborator = collaborator._identity

        headers, data = self._requester.requestJsonAndCheck(
            "DELETE", f"{self.url}/collaborators/{collaborator}"
        )

    def remove_self_hosted_runner(self, runner):
        """
        :calls: `DELETE /repos/{owner}/{repo}/actions/runners/{runner_id} <https://docs.github.com/en/rest/reference/actions#delete-a-self-hosted-runner-from-a-repository>`_
        :param runner: int or :class:`github.SelfHostedActionsRunner.SelfHostedActionsRunner`
        :rtype: bool
        """
        assert isinstance(
            runner, github.SelfHostedActionsRunner.SelfHostedActionsRunner
        ) or isinstance(runner, int), runner

        if isinstance(runner, github.SelfHostedActionsRunner.SelfHostedActionsRunner):
            runner = runner.id

        status, _, _ = self._requester.requestJson(
            "DELETE", f"{self.url}/actions/runners/{runner}"
        )
        return status == 204

    def remove_autolink(self, autolink):
        """
        :calls: `DELETE /repos/{owner}/{repo}/autolinks/{id} <https://docs.github.com/en/rest/reference/repos>`_
        :param autolink: int or :class:`github.Autolink.Autolink`
        :rtype: None
        """
        is_autolink = isinstance(autolink, github.Autolink.Autolink)
        assert is_autolink or isinstance(autolink, int), autolink

        status, _, _ = self._requester.requestJson(
            "DELETE", f"{self.url}/autolinks/{autolink.id if is_autolink else autolink}"
        )
        return status == 204

    def subscribe_to_hub(self, event, callback, secret=github.GithubObject.NotSet):
        """
        :calls: `POST /hub <https://docs.github.com/en/rest/reference/repos#pubsubhubbub>`_
        :param event: string
        :param callback: string
        :param secret: string
        :rtype: None
        """
        return self._hub("subscribe", event, callback, secret)

    def unsubscribe_from_hub(self, event, callback):
        """
        :calls: `POST /hub <https://docs.github.com/en/rest/reference/repos#pubsubhubbub>`_
        :param event: string
        :param callback: string
        :param secret: string
        :rtype: None
        """
        return self._hub("unsubscribe", event, callback, github.GithubObject.NotSet)

    def create_check_suite(self, head_sha):
        """
        :calls: `POST /repos/{owner}/{repo}/check-suites <https://docs.github.com/en/rest/reference/checks#create-a-check-suite>`_
        :param head_sha: string
        :rtype: :class:`github.CheckSuite.CheckSuite`
        """
        assert isinstance(head_sha, str), head_sha
        headers, data = self._requester.requestJsonAndCheck(
            "POST",
            f"{self.url}/check-suites",
            input={"head_sha": head_sha},
        )
        return github.CheckSuite.CheckSuite(
            self._requester, headers, data, completed=True
        )

    def get_check_suite(self, check_suite_id):
        """
        :calls: `GET /repos/{owner}/{repo}/check-suites/{check_suite_id} <https://docs.github.com/en/rest/reference/checks#get-a-check-suite>`_
        :param check_suite_id: int
        :rtype: :class:`github.CheckSuite.CheckSuite`
        """
        assert isinstance(check_suite_id, int), check_suite_id
        requestHeaders = {"Accept": "application/vnd.github.v3+json"}
        headers, data = self._requester.requestJsonAndCheck(
            "GET",
            f"{self.url}/check-suites/{check_suite_id}",
            headers=requestHeaders,
        )
        return github.CheckSuite.CheckSuite(
            self._requester, headers, data, completed=True
        )

    def update_check_suites_preferences(self, auto_trigger_checks):
        """
        :calls: `PATCH /repos/{owner}/{repo}/check-suites/preferences <https://docs.github.com/en/rest/reference/checks#update-repository-preferences-for-check-suites>`_
        :param auto_trigger_checks: list of dict
        :rtype: :class:`github.RepositoryPreferences.RepositoryPreferences`
        """
        assert all(
            isinstance(element, dict) for element in auto_trigger_checks
        ), auto_trigger_checks
        headers, data = self._requester.requestJsonAndCheck(
            "PATCH",
            f"{self.url}/check-suites/preferences",
            input={"auto_trigger_checks": auto_trigger_checks},
        )
        return github.RepositoryPreferences.RepositoryPreferences(
            self._requester, headers, data, completed=True
        )

    def _hub(self, mode, event, callback, secret):
        assert isinstance(mode, str), mode
        assert isinstance(event, str), event
        assert isinstance(callback, str), callback
        assert secret is github.GithubObject.NotSet or isinstance(secret, str), secret

        post_parameters = collections.OrderedDict()
        post_parameters["hub.callback"] = callback
        post_parameters[
            "hub.topic"
        ] = f"https://github.com/{self.full_name}/events/{event}"
        post_parameters["hub.mode"] = mode
        if secret is not github.GithubObject.NotSet:
            post_parameters["hub.secret"] = secret

        headers, output = self._requester.requestMultipartAndCheck(
            "POST", "/hub", input=post_parameters
        )

    @property
    def _identity(self):
        return f"{self.owner.login}/{self.name}"

    def get_release_asset(self, id):
        assert isinstance(id, (int)), id

        resp_headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/releases/assets/{id}"
        )
        return github.GitReleaseAsset.GitReleaseAsset(
            self._requester, resp_headers, data, completed=True
        )

    def create_check_run(
        self,
        name,
        head_sha,
        details_url=github.GithubObject.NotSet,
        external_id=github.GithubObject.NotSet,
        status=github.GithubObject.NotSet,
        started_at=github.GithubObject.NotSet,
        conclusion=github.GithubObject.NotSet,
        completed_at=github.GithubObject.NotSet,
        output=github.GithubObject.NotSet,
        actions=github.GithubObject.NotSet,
    ):
        """
        :calls: `POST /repos/{owner}/{repo}/check-runs <https://docs.github.com/en/rest/reference/checks#create-a-check-run>`_
        :param name: string
        :param head_sha: string
        :param details_url: string
        :param external_id: string
        :param status: string
        :param started_at: datetime
        :param conclusion: string
        :param completed_at: datetime
        :param output: dict
        :param actions: list of dict
        :rtype: :class:`github.CheckRun.CheckRun`
        """
        assert isinstance(name, str), name
        assert isinstance(head_sha, str), head_sha
        assert details_url is github.GithubObject.NotSet or isinstance(
            details_url, str
        ), details_url
        assert external_id is github.GithubObject.NotSet or isinstance(
            external_id, str
        ), external_id
        assert status is github.GithubObject.NotSet or isinstance(status, str), status
        assert started_at is github.GithubObject.NotSet or isinstance(
            started_at, datetime
        ), started_at
        assert conclusion is github.GithubObject.NotSet or isinstance(
            conclusion, str
        ), conclusion
        assert completed_at is github.GithubObject.NotSet or isinstance(
            completed_at, datetime
        ), completed_at
        assert output is github.GithubObject.NotSet or isinstance(output, dict), output
        assert actions is github.GithubObject.NotSet or all(
            isinstance(element, dict) for element in actions
        ), actions

        post_parameters = {
            "name": name,
            "head_sha": head_sha,
        }
        if details_url is not github.GithubObject.NotSet:
            post_parameters["details_url"] = details_url
        if external_id is not github.GithubObject.NotSet:
            post_parameters["external_id"] = external_id
        if status is not github.GithubObject.NotSet:
            post_parameters["status"] = status
        if started_at is not github.GithubObject.NotSet:
            post_parameters["started_at"] = started_at.strftime("%Y-%m-%dT%H:%M:%SZ")
        if completed_at is not github.GithubObject.NotSet:
            post_parameters["completed_at"] = completed_at.strftime(
                "%Y-%m-%dT%H:%M:%SZ"
            )
        if conclusion is not github.GithubObject.NotSet:
            post_parameters["conclusion"] = conclusion
        if output is not github.GithubObject.NotSet:
            post_parameters["output"] = output
        if actions is not github.GithubObject.NotSet:
            post_parameters["actions"] = actions

        headers, data = self._requester.requestJsonAndCheck(
            "POST",
            f"{self.url}/check-runs",
            input=post_parameters,
        )
        return github.CheckRun.CheckRun(self._requester, headers, data, completed=True)

    def get_check_run(self, check_run_id):
        """
        :calls: `GET /repos/{owner}/{repo}/check-runs/{check_run_id} <https://docs.github.com/en/rest/reference/checks#get-a-check-run>`_
        :param check_run_id: int
        :rtype: :class:`github.CheckRun.CheckRun`
        """
        assert isinstance(check_run_id, int), check_run_id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/check-runs/{check_run_id}"
        )
        return github.CheckRun.CheckRun(self._requester, headers, data, completed=True)

    def get_artifacts(self, name=github.GithubObject.NotSet):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/artifacts <https://docs.github.com/en/rest/actions/artifacts#list-artifacts-for-a-repository>`_
        :param name: str
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Artifact.Artifact`
        """

        assert name is github.GithubObject.NotSet or isinstance(name, str), name

        param = {
            key: value
            for key, value in {"name": name}.items()
            if value is not github.GithubObject.NotSet
        }

        return github.PaginatedList.PaginatedList(
            github.Artifact.Artifact,
            self._requester,
            f"{self.url}/actions/artifacts",
            firstParams=param,
            list_item="artifacts",
        )

    def get_artifact(self, artifact_id):
        """
        :calls: `GET /repos/{owner}/{repo}/actions/artifacts/{artifact_id} <https://docs.github.com/en/rest/actions/artifacts#get-an-artifact>`_
        :param artifact_id: int
        :rtype: :class:`github.Artifact.Artifact`
        """
        assert isinstance(artifact_id, int), artifact_id
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/actions/artifacts/{artifact_id}"
        )

        return github.Artifact.Artifact(self._requester, headers, data, completed=True)

    def get_codescan_alerts(self):
        """
        :calls: `GET https://api.github.com/repos/{owner}/{repo}/code-scanning/alerts <https://docs.github.com/en/rest/reference/code-scanning#list-code-scanning-alerts-for-a-repository>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.CodeScanAlert.CodeScanAlert`
        """
        return github.PaginatedList.PaginatedList(
            github.CodeScanAlert.CodeScanAlert,
            self._requester,
            f"{self.url}/code-scanning/alerts",
            None,
        )

    def get_environments(self):
        """
        :calls: `GET /repos/{owner}/{repo}/environments <https://docs.github.com/en/rest/reference/deployments#get-all-environments>`_
        :rtype: :class:`github.PaginatedList.PaginatedList` of :class:`github.Environment.Environment`
        """
        return github.PaginatedList.PaginatedList(
            github.Environment.Environment,
            self._requester,
            f"{self.url}/environments",
            None,
            list_item="environments",
        )

    def get_environment(self, environment_name):
        """
        :calls: `GET /repos/{owner}/{repo}/environments/{environment_name} <https://docs.github.com/en/rest/reference/deployments#get-an-environment>`_
        :rtype: :class:`github.Environment.Environment`
        """
        assert isinstance(environment_name, str), environment_name
        headers, data = self._requester.requestJsonAndCheck(
            "GET", f"{self.url}/environments/{environment_name}"
        )
        return github.Environment.Environment(
            self._requester, headers, data, completed=True
        )

    def create_environment(
        self,
        environment_name,
        wait_timer=0,
        reviewers=[],
        deployment_branch_policy=None,
    ):
        """
        :calls: `PUT /repos/{owner}/{repo}/environments/{environment_name} <https://docs.github.com/en/rest/reference/deployments#create-or-update-an-environment>`_
        :param environment_name: string
        :param wait_timer: int
        :param reviews: List[:class:github.EnvironmentDeploymentBranchPolicy.EnvironmentDeploymentBranchPolicyParams]
        :param deployment_branch_policy: Optional[:class:github.EnvironmentDeploymentBranchPolicy.EnvironmentDeploymentBranchPolicyParams`]
        :rtype: :class:`github.Environment.Environment`
        """
        assert isinstance(environment_name, str), environment_name
        assert isinstance(wait_timer, int)
        assert isinstance(reviewers, list)
        assert all(
            [
                isinstance(
                    reviewer, github.EnvironmentProtectionRuleReviewer.ReviewerParams
                )
                for reviewer in reviewers
            ]
        )
        assert (
            isinstance(
                deployment_branch_policy,
                github.EnvironmentDeploymentBranchPolicy.EnvironmentDeploymentBranchPolicyParams,
            )
            or deployment_branch_policy is None
        )

        put_parameters = {
            "wait_timer": wait_timer,
            "reviewers": [reviewer._asdict() for reviewer in reviewers],
            "deployment_branch_policy": deployment_branch_policy._asdict()
            if deployment_branch_policy
            else None,
        }

        headers, data = self._requester.requestJsonAndCheck(
            "PUT", f"{self.url}/environments/{environment_name}", input=put_parameters
        )
        return github.Environment.Environment(
            self._requester, headers, data, completed=True
        )

    update_environment = create_environment

    def delete_environment(self, environment_name):
        """
        :calls: `DELETE /repos/{owner}/{repo}/environments/{environment_name} <https://docs.github.com/en/rest/reference/deployments#delete-an-environment>`_
        :param environment_name: string
        :rtype: None
        """
        assert isinstance(environment_name, str), environment_name

        headers, data = self._requester.requestJsonAndCheck(
            "DELETE", f"{self.url}/environments/{environment_name}"
        )

    def _initAttributes(self):
        self._allow_auto_merge = github.GithubObject.NotSet
        self._allow_forking = github.GithubObject.NotSet
        self._allow_merge_commit = github.GithubObject.NotSet
        self._allow_rebase_merge = github.GithubObject.NotSet
        self._allow_squash_merge = github.GithubObject.NotSet
        self._allow_update_branch = github.GithubObject.NotSet
        self._archived = github.GithubObject.NotSet
        self._archive_url = github.GithubObject.NotSet
        self._assignees_url = github.GithubObject.NotSet
        self._blobs_url = github.GithubObject.NotSet
        self._branches_url = github.GithubObject.NotSet
        self._clone_url = github.GithubObject.NotSet
        self._collaborators_url = github.GithubObject.NotSet
        self._comments_url = github.GithubObject.NotSet
        self._commits_url = github.GithubObject.NotSet
        self._compare_url = github.GithubObject.NotSet
        self._contents_url = github.GithubObject.NotSet
        self._contributors_url = github.GithubObject.NotSet
        self._created_at = github.GithubObject.NotSet
        self._default_branch = github.GithubObject.NotSet
        self._delete_branch_on_merge = github.GithubObject.NotSet
        self._deployments_url = github.GithubObject.NotSet
        self._description = github.GithubObject.NotSet
        self._downloads_url = github.GithubObject.NotSet
        self._events_url = github.GithubObject.NotSet
        self._fork = github.GithubObject.NotSet
        self._forks = github.GithubObject.NotSet
        self._forks_count = github.GithubObject.NotSet
        self._forks_url = github.GithubObject.NotSet
        self._full_name = github.GithubObject.NotSet
        self._git_commits_url = github.GithubObject.NotSet
        self._git_refs_url = github.GithubObject.NotSet
        self._git_tags_url = github.GithubObject.NotSet
        self._git_url = github.GithubObject.NotSet
        self._has_downloads = github.GithubObject.NotSet
        self._has_issues = github.GithubObject.NotSet
        self._has_pages = github.GithubObject.NotSet
        self._has_projects = github.GithubObject.NotSet
        self._has_wiki = github.GithubObject.NotSet
        self._homepage = github.GithubObject.NotSet
        self._hooks_url = github.GithubObject.NotSet
        self._html_url = github.GithubObject.NotSet
        self._id = github.GithubObject.NotSet
        self._is_template = github.GithubObject.NotSet
        self._issue_comment_url = github.GithubObject.NotSet
        self._issue_events_url = github.GithubObject.NotSet
        self._issues_url = github.GithubObject.NotSet
        self._keys_url = github.GithubObject.NotSet
        self._labels_url = github.GithubObject.NotSet
        self._language = github.GithubObject.NotSet
        self._languages_url = github.GithubObject.NotSet
        self._master_branch = github.GithubObject.NotSet
        self._merges_url = github.GithubObject.NotSet
        self._milestones_url = github.GithubObject.NotSet
        self._mirror_url = github.GithubObject.NotSet
        self._name = github.GithubObject.NotSet
        self._network_count = github.GithubObject.NotSet
        self._notifications_url = github.GithubObject.NotSet
        self._open_issues = github.GithubObject.NotSet
        self._open_issues_count = github.GithubObject.NotSet
        self._organization = github.GithubObject.NotSet
        self._owner = github.GithubObject.NotSet
        self._parent = github.GithubObject.NotSet
        self._permissions = github.GithubObject.NotSet
        self._private = github.GithubObject.NotSet
        self._pulls_url = github.GithubObject.NotSet
        self._pushed_at = github.GithubObject.NotSet
        self._releases_url = github.GithubObject.NotSet
        self._size = github.GithubObject.NotSet
        self._source = github.GithubObject.NotSet
        self._ssh_url = github.GithubObject.NotSet
        self._stargazers_count = github.GithubObject.NotSet
        self._stargazers_url = github.GithubObject.NotSet
        self._statuses_url = github.GithubObject.NotSet
        self._subscribers_url = github.GithubObject.NotSet
        self._subscribers_count = github.GithubObject.NotSet
        self._subscription_url = github.GithubObject.NotSet
        self._svn_url = github.GithubObject.NotSet
        self._tags_url = github.GithubObject.NotSet
        self._teams_url = github.GithubObject.NotSet
        self._topics = github.GithubObject.NotSet
        self._trees_url = github.GithubObject.NotSet
        self._updated_at = github.GithubObject.NotSet
        self._url = github.GithubObject.NotSet
        self._visibility = github.GithubObject.NotSet
        self._watchers = github.GithubObject.NotSet
        self._watchers_count = github.GithubObject.NotSet

    def _useAttributes(self, attributes):
        if "allow_auto_merge" in attributes:  # pragma no branch
            self._allow_auto_merge = self._makeBoolAttribute(
                attributes["allow_auto_merge"]
            )
        if "allow_forking" in attributes:  # pragma no branch
            self._allow_forking = self._makeBoolAttribute(attributes["allow_forking"])
        if "allow_merge_commit" in attributes:  # pragma no branch
            self._allow_merge_commit = self._makeBoolAttribute(
                attributes["allow_merge_commit"]
            )
        if "allow_rebase_merge" in attributes:  # pragma no branch
            self._allow_rebase_merge = self._makeBoolAttribute(
                attributes["allow_rebase_merge"]
            )
        if "allow_squash_merge" in attributes:  # pragma no branch
            self._allow_squash_merge = self._makeBoolAttribute(
                attributes["allow_squash_merge"]
            )
        if "allow_update_branch" in attributes:  # pragma no branch
            self._allow_update_branch = self._makeBoolAttribute(
                attributes["allow_update_branch"]
            )
        if "archived" in attributes:  # pragma no branch
            self._archived = self._makeBoolAttribute(attributes["archived"])
        if "archive_url" in attributes:  # pragma no branch
            self._archive_url = self._makeStringAttribute(attributes["archive_url"])
        if "assignees_url" in attributes:  # pragma no branch
            self._assignees_url = self._makeStringAttribute(attributes["assignees_url"])
        if "blobs_url" in attributes:  # pragma no branch
            self._blobs_url = self._makeStringAttribute(attributes["blobs_url"])
        if "branches_url" in attributes:  # pragma no branch
            self._branches_url = self._makeStringAttribute(attributes["branches_url"])
        if "clone_url" in attributes:  # pragma no branch
            self._clone_url = self._makeStringAttribute(attributes["clone_url"])
        if "collaborators_url" in attributes:  # pragma no branch
            self._collaborators_url = self._makeStringAttribute(
                attributes["collaborators_url"]
            )
        if "comments_url" in attributes:  # pragma no branch
            self._comments_url = self._makeStringAttribute(attributes["comments_url"])
        if "commits_url" in attributes:  # pragma no branch
            self._commits_url = self._makeStringAttribute(attributes["commits_url"])
        if "compare_url" in attributes:  # pragma no branch
            self._compare_url = self._makeStringAttribute(attributes["compare_url"])
        if "contents_url" in attributes:  # pragma no branch
            self._contents_url = self._makeStringAttribute(attributes["contents_url"])
        if "contributors_url" in attributes:  # pragma no branch
            self._contributors_url = self._makeStringAttribute(
                attributes["contributors_url"]
            )
        if "created_at" in attributes:  # pragma no branch
            self._created_at = self._makeDatetimeAttribute(attributes["created_at"])
        if "default_branch" in attributes:  # pragma no branch
            self._default_branch = self._makeStringAttribute(
                attributes["default_branch"]
            )
        if "delete_branch_on_merge" in attributes:  # pragma no branch
            self._delete_branch_on_merge = self._makeBoolAttribute(
                attributes["delete_branch_on_merge"]
            )
        if "deployments_url" in attributes:  # pragma no branch
            self._deployments_url = self._makeStringAttribute(
                attributes["deployments_url"]
            )
        if "description" in attributes:  # pragma no branch
            self._description = self._makeStringAttribute(attributes["description"])
        if "downloads_url" in attributes:  # pragma no branch
            self._downloads_url = self._makeStringAttribute(attributes["downloads_url"])
        if "events_url" in attributes:  # pragma no branch
            self._events_url = self._makeStringAttribute(attributes["events_url"])
        if "fork" in attributes:  # pragma no branch
            self._fork = self._makeBoolAttribute(attributes["fork"])
        if "forks" in attributes:  # pragma no branch
            self._forks = self._makeIntAttribute(attributes["forks"])
        if "forks_count" in attributes:  # pragma no branch
            self._forks_count = self._makeIntAttribute(attributes["forks_count"])
        if "forks_url" in attributes:  # pragma no branch
            self._forks_url = self._makeStringAttribute(attributes["forks_url"])
        if "full_name" in attributes:  # pragma no branch
            self._full_name = self._makeStringAttribute(attributes["full_name"])
        if "git_commits_url" in attributes:  # pragma no branch
            self._git_commits_url = self._makeStringAttribute(
                attributes["git_commits_url"]
            )
        if "git_refs_url" in attributes:  # pragma no branch
            self._git_refs_url = self._makeStringAttribute(attributes["git_refs_url"])
        if "git_tags_url" in attributes:  # pragma no branch
            self._git_tags_url = self._makeStringAttribute(attributes["git_tags_url"])
        if "git_url" in attributes:  # pragma no branch
            self._git_url = self._makeStringAttribute(attributes["git_url"])
        if "has_downloads" in attributes:  # pragma no branch
            self._has_downloads = self._makeBoolAttribute(attributes["has_downloads"])
        if "has_issues" in attributes:  # pragma no branch
            self._has_issues = self._makeBoolAttribute(attributes["has_issues"])
        if "has_pages" in attributes:  # pragma no branch
            self._has_pages = self._makeBoolAttribute(attributes["has_pages"])
        if "has_projects" in attributes:  # pragma no branch
            self._has_projects = self._makeBoolAttribute(attributes["has_projects"])
        if "has_wiki" in attributes:  # pragma no branch
            self._has_wiki = self._makeBoolAttribute(attributes["has_wiki"])
        if "homepage" in attributes:  # pragma no branch
            self._homepage = self._makeStringAttribute(attributes["homepage"])
        if "hooks_url" in attributes:  # pragma no branch
            self._hooks_url = self._makeStringAttribute(attributes["hooks_url"])
        if "html_url" in attributes:  # pragma no branch
            self._html_url = self._makeStringAttribute(attributes["html_url"])
        if "id" in attributes:  # pragma no branch
            self._id = self._makeIntAttribute(attributes["id"])
        if "is_template" in attributes:  # pragma no branch
            self._is_template = self._makeBoolAttribute(attributes["is_template"])
        if "issue_comment_url" in attributes:  # pragma no branch
            self._issue_comment_url = self._makeStringAttribute(
                attributes["issue_comment_url"]
            )
        if "issue_events_url" in attributes:  # pragma no branch
            self._issue_events_url = self._makeStringAttribute(
                attributes["issue_events_url"]
            )
        if "issues_url" in attributes:  # pragma no branch
            self._issues_url = self._makeStringAttribute(attributes["issues_url"])
        if "keys_url" in attributes:  # pragma no branch
            self._keys_url = self._makeStringAttribute(attributes["keys_url"])
        if "labels_url" in attributes:  # pragma no branch
            self._labels_url = self._makeStringAttribute(attributes["labels_url"])
        if "language" in attributes:  # pragma no branch
            self._language = self._makeStringAttribute(attributes["language"])
        if "languages_url" in attributes:  # pragma no branch
            self._languages_url = self._makeStringAttribute(attributes["languages_url"])
        if "master_branch" in attributes:  # pragma no branch
            self._master_branch = self._makeStringAttribute(attributes["master_branch"])
        if "merges_url" in attributes:  # pragma no branch
            self._merges_url = self._makeStringAttribute(attributes["merges_url"])
        if "milestones_url" in attributes:  # pragma no branch
            self._milestones_url = self._makeStringAttribute(
                attributes["milestones_url"]
            )
        if "mirror_url" in attributes:  # pragma no branch
            self._mirror_url = self._makeStringAttribute(attributes["mirror_url"])
        if "name" in attributes:  # pragma no branch
            self._name = self._makeStringAttribute(attributes["name"])
        if "network_count" in attributes:  # pragma no branch
            self._network_count = self._makeIntAttribute(attributes["network_count"])
        if "notifications_url" in attributes:  # pragma no branch
            self._notifications_url = self._makeStringAttribute(
                attributes["notifications_url"]
            )
        if "open_issues" in attributes:  # pragma no branch
            self._open_issues = self._makeIntAttribute(attributes["open_issues"])
        if "open_issues_count" in attributes:  # pragma no branch
            self._open_issues_count = self._makeIntAttribute(
                attributes["open_issues_count"]
            )
        if "organization" in attributes:  # pragma no branch
            self._organization = self._makeClassAttribute(
                github.Organization.Organization, attributes["organization"]
            )
        if "owner" in attributes:  # pragma no branch
            self._owner = self._makeClassAttribute(
                github.NamedUser.NamedUser, attributes["owner"]
            )
        if "parent" in attributes:  # pragma no branch
            self._parent = self._makeClassAttribute(Repository, attributes["parent"])
        if "permissions" in attributes:  # pragma no branch
            self._permissions = self._makeClassAttribute(
                github.Permissions.Permissions, attributes["permissions"]
            )
        if "private" in attributes:  # pragma no branch
            self._private = self._makeBoolAttribute(attributes["private"])
        if "pulls_url" in attributes:  # pragma no branch
            self._pulls_url = self._makeStringAttribute(attributes["pulls_url"])
        if "pushed_at" in attributes:  # pragma no branch
            self._pushed_at = self._makeDatetimeAttribute(attributes["pushed_at"])
        if "releases_url" in attributes:  # pragma no branch
            self._releases_url = self._makeStringAttribute(attributes["releases_url"])
        if "size" in attributes:  # pragma no branch
            self._size = self._makeIntAttribute(attributes["size"])
        if "source" in attributes:  # pragma no branch
            self._source = self._makeClassAttribute(Repository, attributes["source"])
        if "ssh_url" in attributes:  # pragma no branch
            self._ssh_url = self._makeStringAttribute(attributes["ssh_url"])
        if "stargazers_count" in attributes:  # pragma no branch
            self._stargazers_count = self._makeIntAttribute(
                attributes["stargazers_count"]
            )
        if "stargazers_url" in attributes:  # pragma no branch
            self._stargazers_url = self._makeStringAttribute(
                attributes["stargazers_url"]
            )
        if "statuses_url" in attributes:  # pragma no branch
            self._statuses_url = self._makeStringAttribute(attributes["statuses_url"])
        if "subscribers_url" in attributes:  # pragma no branch
            self._subscribers_url = self._makeStringAttribute(
                attributes["subscribers_url"]
            )
        if "subscribers_count" in attributes:  # pragma no branch
            self._subscribers_count = self._makeIntAttribute(
                attributes["subscribers_count"]
            )
        if "subscription_url" in attributes:  # pragma no branch
            self._subscription_url = self._makeStringAttribute(
                attributes["subscription_url"]
            )
        if "svn_url" in attributes:  # pragma no branch
            self._svn_url = self._makeStringAttribute(attributes["svn_url"])
        if "tags_url" in attributes:  # pragma no branch
            self._tags_url = self._makeStringAttribute(attributes["tags_url"])
        if "teams_url" in attributes:  # pragma no branch
            self._teams_url = self._makeStringAttribute(attributes["teams_url"])
        if "trees_url" in attributes:  # pragma no branch
            self._trees_url = self._makeStringAttribute(attributes["trees_url"])
        if "topics" in attributes:  # pragma no branch
            self._topics = self._makeListOfStringsAttribute(attributes["topics"])
        if "updated_at" in attributes:  # pragma no branch
            self._updated_at = self._makeDatetimeAttribute(attributes["updated_at"])
        if "url" in attributes:  # pragma no branch
            self._url = self._makeStringAttribute(attributes["url"])
        if "visibility" in attributes:  # pragma no branch
            self._visibility = self._makeStringAttribute(attributes["visibility"])
        if "watchers" in attributes:  # pragma no branch
            self._watchers = self._makeIntAttribute(attributes["watchers"])
        if "watchers_count" in attributes:  # pragma no branch
            self._watchers_count = self._makeIntAttribute(attributes["watchers_count"])
