# Pyrogram - Telegram MTProto API Client Library for Python
# Copyright (C) 2017-2020 Dan <https://github.com/delivrance>
#
# This file is part of Pyrogram.
#
# Pyrogram is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Pyrogram is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from io import BytesIO

from pyrogram.api.core import *


class SetCallRating(TLObject):
    """Attributes:
        LAYER: ``112``

    Attributes:
        ID: ``0x59ead627``

    Parameters:
        peer: :obj:`InputPhoneCall <pyrogram.api.types.InputPhoneCall>`
        rating: ``int`` ``32-bit``
        comment: ``str``
        user_initiative (optional): ``bool``

    Returns:
        Either :obj:`UpdatesTooLong <pyrogram.api.types.UpdatesTooLong>`, :obj:`UpdateShortMessage <pyrogram.api.types.UpdateShortMessage>`, :obj:`UpdateShortChatMessage <pyrogram.api.types.UpdateShortChatMessage>`, :obj:`UpdateShort <pyrogram.api.types.UpdateShort>`, :obj:`UpdatesCombined <pyrogram.api.types.UpdatesCombined>`, :obj:`Update <pyrogram.api.types.Update>` or :obj:`UpdateShortSentMessage <pyrogram.api.types.UpdateShortSentMessage>`
    """

    __slots__ = ["peer", "rating", "comment", "user_initiative"]

    ID = 0x59ead627
    QUALNAME = "functions.phone.SetCallRating"

    def __init__(self, *, peer, rating: int, comment: str, user_initiative: bool = None):
        self.user_initiative = user_initiative  # flags.0?true
        self.peer = peer  # InputPhoneCall
        self.rating = rating  # int
        self.comment = comment  # string

    @staticmethod
    def read(b: BytesIO, *args) -> "SetCallRating":
        flags = Int.read(b)
        
        user_initiative = True if flags & (1 << 0) else False
        peer = TLObject.read(b)
        
        rating = Int.read(b)
        
        comment = String.read(b)
        
        return SetCallRating(peer=peer, rating=rating, comment=comment, user_initiative=user_initiative)

    def write(self) -> bytes:
        b = BytesIO()
        b.write(Int(self.ID, False))

        flags = 0
        flags |= (1 << 0) if self.user_initiative is not None else 0
        b.write(Int(flags))
        
        b.write(self.peer.write())
        
        b.write(Int(self.rating))
        
        b.write(String(self.comment))
        
        return b.getvalue()
