# Pyrogram - Telegram MTProto API Client Library for Python
# Copyright (C) 2017-2018 Dan Tès <https://github.com/delivrance>
#
# This file is part of Pyrogram.
#
# Pyrogram is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Pyrogram is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from io import BytesIO

from pyrogram.api.core import *


class SendPaymentForm(Object):
    """
    Attributes:
        ID (:obj:`int`): ``0x2b8879b3``

    Args:
        msg_id: :obj:`int`
        credentials: :class:`pyrogram.api.types.InputPaymentCredentials`
        requested_info_id: :obj:`string` (optional)
        shipping_option_id: :obj:`string` (optional)

    Returns:
        :class:`pyrogram.api.types.payments.PaymentResult` | :class:`pyrogram.api.types.payments.PaymentVerficationNeeded`

    Raises:
        :class:`pyrogram.Error`
    """
    ID = 0x2b8879b3

    def __init__(self, msg_id, credentials, requested_info_id=None, shipping_option_id=None):
        self.msg_id = msg_id  # int
        self.requested_info_id = requested_info_id  # flags.0?string
        self.shipping_option_id = shipping_option_id  # flags.1?string
        self.credentials = credentials  # InputPaymentCredentials

    @staticmethod
    def read(b: BytesIO, *args) -> "SendPaymentForm":
        flags = Int.read(b)
        
        msg_id = Int.read(b)
        
        requested_info_id = String.read(b) if flags & (1 << 0) else None
        shipping_option_id = String.read(b) if flags & (1 << 1) else None
        credentials = Object.read(b)
        
        return SendPaymentForm(msg_id, credentials, requested_info_id, shipping_option_id)

    def write(self) -> bytes:
        b = BytesIO()
        b.write(Int(self.ID, False))

        flags = 0
        flags |= (1 << 0) if self.requested_info_id is not None else 0
        flags |= (1 << 1) if self.shipping_option_id is not None else 0
        b.write(Int(flags))
        
        b.write(Int(self.msg_id))
        
        if self.requested_info_id is not None:
            b.write(String(self.requested_info_id))
        
        if self.shipping_option_id is not None:
            b.write(String(self.shipping_option_id))
        
        b.write(self.credentials.write())
        
        return b.getvalue()
