# Pyrogram - Telegram MTProto API Client Library for Python
# Copyright (C) 2017-2018 Dan Tès <https://github.com/delivrance>
#
# This file is part of Pyrogram.
#
# Pyrogram is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Pyrogram is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from io import BytesIO

from pyrogram.api.core import *


class PhoneCallWaiting(Object):
    """
    Attributes:
        ID (:obj:`int`): ``0x1b8f4ad1``

    Args:
        id: :obj:`int`
        access_hash: :obj:`int`
        date: :obj:`int`
        admin_id: :obj:`int`
        participant_id: :obj:`int`
        protocol: :class:`pyrogram.api.types.PhoneCallProtocol`
        receive_date: :obj:`int` (optional)
    """
    ID = 0x1b8f4ad1

    def __init__(self, id, access_hash, date, admin_id, participant_id, protocol, receive_date=None):
        self.id = id  # long
        self.access_hash = access_hash  # long
        self.date = date  # int
        self.admin_id = admin_id  # int
        self.participant_id = participant_id  # int
        self.protocol = protocol  # PhoneCallProtocol
        self.receive_date = receive_date  # flags.0?int

    @staticmethod
    def read(b: BytesIO, *args) -> "PhoneCallWaiting":
        flags = Int.read(b)
        
        id = Long.read(b)
        
        access_hash = Long.read(b)
        
        date = Int.read(b)
        
        admin_id = Int.read(b)
        
        participant_id = Int.read(b)
        
        protocol = Object.read(b)
        
        receive_date = Int.read(b) if flags & (1 << 0) else None
        return PhoneCallWaiting(id, access_hash, date, admin_id, participant_id, protocol, receive_date)

    def write(self) -> bytes:
        b = BytesIO()
        b.write(Int(self.ID, False))

        flags = 0
        flags |= (1 << 0) if self.receive_date is not None else 0
        b.write(Int(flags))
        
        b.write(Long(self.id))
        
        b.write(Long(self.access_hash))
        
        b.write(Int(self.date))
        
        b.write(Int(self.admin_id))
        
        b.write(Int(self.participant_id))
        
        b.write(self.protocol.write())
        
        if self.receive_date is not None:
            b.write(Int(self.receive_date))
        
        return b.getvalue()
