from builtins import super

from fnmatch import fnmatch

from ..tecutil import _tecutil
from ..constant import *
from ..exception import *
from .. import session
from ..legend import LineLegend
from ..tecutil import Index, IndexSet, flatten_args, lock, log_setattr, sv
from .axes import (Cartesian2DFieldAxes, Cartesian3DFieldAxes, SketchAxes,
                   PolarLineAxes, XYLineAxes)
from .contour import ContourGroup
from .linemap import PolarLinemap, XYLinemap
from .fieldmap import Cartesian2DFieldmap, Cartesian3DFieldmap
from .scatter import Scatter
from .vector import Vector2D, Vector3D
from .view import Cartesian2DView, Cartesian3DView, LineView, PolarView


class Plot(session.Style):
    def __init__(self, frame, *svargs):
        self.frame = frame
        super().__init__(*svargs, uniqueid=frame.uid)

    def __eq__(self, that):
        return isinstance(that, type(self)) and self.frame == that.frame

    def __ne__(self, that):
        return not (self == that)


class SketchPlot(Plot):
    """A plot space with no data attached.

    .. code-block:: python
        :emphasize-lines: 5,8-9

        import tecplot as tp
        from tecplot.constant import PlotType

        frame = tp.active_frame()
        plot = frame.plot(PlotType.Sketch)

        frame.add_text('Hello, World!', (36, 50), size=34)
        plot.axes.x_axis.show = True
        plot.axes.y_axis.show = True

        tp.export_image('plot_sketch.png')

    ..  figure:: /_static/images/plot_sketch.png
        :width: 300px
        :figwidth: 300px
    """

    def activate(self):
        """Make this the active plot type on the parent frame.

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> plot = frame.plot(PlotType.Sketch)
            >>> plot.activate()
        """
        self.frame.plot_type = PlotType.Sketch

    @property
    def axes(self):
        """Axes (x and y) for the sketch plot.

        :type: `SketchAxes`

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> frame.plot_type = PlotType.Sketch
            >>> frame.plot().axes.x_axis.show = True
        """
        return SketchAxes(self)


class FieldPlot(Plot):
    """Plot containing data associated with style through fieldmaps.

    .. code-block:: python
        :emphasize-lines: 8-12

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '3D', 'jet_surface.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        plot = frame.plot(PlotType.Cartesian3D)
        plot.activate()
        plot.show_contour = True
        plot.show_translucency = True
        plot.contour(0).variable = dataset.variable('S')

        # save image to file
        tp.export_image('plot_field.png')

    ..  figure:: /_static/images/plot_field.png
        :width: 300px
        :figwidth: 300px
    """
    def __init__(self, frame):
        super().__init__(frame, sv.FIELDLAYERS)

    def contour(self, index):
        """Plot-local `ContourGroup` style control.

        :type: `ContourGroup`

        Example usage::

            >>> contour = frame.plot().contour(0)
            >>> contour.colormap_name = 'Magma'
        """
        return ContourGroup(index, self)

    @property
    def scatter(self):
        """Plot-local `Scatter` style control.

        :type: `Scatter`

        Example usage::

            >>> scatter = frame.plot().scatter
            >>> scatter.variable = dataset.variable('P')
        """
        return Scatter(self)

    @property
    def show_contour(self):
        """Enable contours for this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_contour = True
        """
        return self._get_style(bool, sv.SHOWCONTOUR)

    @show_contour.setter
    def show_contour(self, show):
        self._set_style(bool(show), sv.SHOWCONTOUR)

    @property
    def show_edge(self):
        """Enable zone edge lines for this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_edge = True
        """
        return self._get_style(bool, sv.SHOWEDGE)

    @show_edge.setter
    def show_edge(self, show):
        self._set_style(bool(show), sv.SHOWEDGE)

    @property
    def show_mesh(self):
        """Enable mesh lines for this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_mesh = True
        """
        return self._get_style(bool, sv.SHOWMESH)

    @show_mesh.setter
    def show_mesh(self, show):
        self._set_style(bool(show), sv.SHOWMESH)

    @property
    def show_scatter(self):
        """Enable scatter symbols for this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_scatter = True
        """
        return self._get_style(bool, sv.SHOWSCATTER)

    @show_scatter.setter
    def show_scatter(self, show):
        self._set_style(bool(show), sv.SHOWSCATTER)

    @property
    def show_shade(self):
        """Enable surface shading effect for this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_shade = True
        """
        return self._get_style(bool, sv.SHOWSHADE)

    @show_shade.setter
    def show_shade(self, show):
        self._set_style(bool(show), sv.SHOWSHADE)

    @property
    def show_streamtraces(self):
        """Enable drawing streamtraces on this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_streamtraces = True
        """
        with self.frame.activated():
            return session.get_style(bool, sv.STREAMTRACELAYERS, sv.SHOW,
                                     uniqueid=self.frame.uid)

    @show_streamtraces.setter
    def show_streamtraces(self, show):
        with self.frame.activated():
            session.set_style(bool(show), sv.STREAMTRACELAYERS, sv.SHOW,
                              uniqueid=self.frame.uid)

    @property
    def show_lighting_effect(self):
        """Enable lighting effect for all objects within this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_lighting_effect = True
        """
        return self._get_style(bool, sv.USELIGHTINGEFFECT)

    @show_lighting_effect.setter
    def show_lighting_effect(self, value):
        self._set_style(bool(value), sv.USELIGHTINGEFFECT)

    @property
    def show_translucency(self):
        """Enable translucent effect for all objects within this plot.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_translucency = True
        """
        return self._get_style(bool, sv.USETRANSLUCENCY)

    @show_translucency.setter
    def show_translucency(self, show):
        self._set_style(bool(show), sv.USETRANSLUCENCY)

    @property
    def show_vector(self):
        """Enable drawing of vectors.

        :type: `bool`

        Example usage::

            >>> frame.plot().show_vector = True
        """
        return self._get_style(bool, sv.SHOWVECTOR)

    @show_vector.setter
    def show_vector(self, show):
        self._set_style(bool(show), sv.SHOWVECTOR)

    @property
    def num_fieldmaps(self):
        """Number of all fieldmaps in this plot.

        :type: `integer <int>`

        Example usage::

            >>> print(frame.plot().num_fieldmaps)
            3
        """
        return _tecutil.FieldMapGetCountForFrame(self.frame.uid)

    def fieldmap_index(self, zone):
        """Returns the index of the fieldmap associated with a `Zone`.

        Parameter:
            zone (`Zone`): The `Zone` object that belongs to the `Dataset`
                associated with this plot.

        Returns:
            `Index`

        Example usage::

            >>> fmap_index = plot.fieldmap_index(dataset.zone('Zone'))
            >>> plot.fieldmap(fmap_index).show_mesh = True
        """
        return Index(_tecutil.ZoneGetFieldMap(zone.index + 1) - 1)

    @property
    def fieldmaps(self):
        """All fieldmaps in this plot.

        Yields: `Cartesian2DFieldmap` or `Cartesian3DFieldmap`

        Example usage::

            >>> for fmap in plot.fieldmaps:
            ...     fmap.mesh.show = True
        """
        for i in range(self.num_fieldmaps):
            yield self.fieldmap(i)

    @property
    def active_fieldmap_indices(self):
        """Set of active fieldmaps by index.

        :type: `set`

        Example usage::

            >>> for i in plot.active_fieldmap_indices:
            ...     plot.fieldmap(i).scatter.show = True
        """
        return session.get_style(set, sv.ACTIVEFIELDMAPS,
                                 uniqueid=self.frame.uid)

    @property
    def active_fieldmaps(self):
        """Active fieldmaps in this plot.

        Yields: `Cartesian2DFieldmap` or `Cartesian3DFieldmap`

        Example usage::

            >>> for fmap in plot.active_fieldmaps:
            ...     fmap.vector.show = True
        """
        for i in sorted(self.active_fieldmap_indices):
            yield self.fieldmap(i)


class Cartesian2DFieldPlot(FieldPlot):
    """2D plot containing field data associated with style through fieldmaps.

    .. code-block:: python
        :emphasize-lines: 10-13,15

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '2D', 'intflow.lpk')
        dataset = tp.load_layout(infile)

        frame = tp.active_frame()
        plot = frame.plot(PlotType.Cartesian2D)
        plot.activate()
        plot.show_edge = False
        plot.show_mesh = False

        plot.contour(0).colormap_name = 'Sequential - Yellow/Green/Blue'

        # save image to file
        tp.export_image('plot_field2d.png')

    ..  figure:: /_static/images/plot_field2d.png
        :width: 300px
        :figwidth: 300px
    """

    def activate(self):
        """Make this the active plot type on the parent frame.

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> plot = frame.plot(PlotType.Cartesian2D)
            >>> plot.activate()
        """
        self.frame.plot_type = PlotType.Cartesian2D

    @property
    def axes(self):
        """Axes style control for this plot.

        :type: `Cartesian2DFieldAxes`

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> frame.plot_type = PlotType.Cartesian2D
            >>> axes = frame.plot().axes
            >>> axes.x_axis.variable = dataset.variable('U')
            >>> axes.y_axis.variable = dataset.variable('V')
        """
        return Cartesian2DFieldAxes(self)

    def fieldmap(self, key):
        """Returns a `Cartesian2DFieldmap` by `Zone` or index.

        Parameter:
            key (`Zone` or `integer <int>`): The `Zone` must be in the `Dataset`
                attached to the assocated frame of this plot.

        Example usage::

            >>> fmap = plot.fieldmap(dataset.zone(0))
            >>> fmap.scatter.show = True
        """
        index = key if isinstance(key, int) else self.fieldmap_index(key)
        return Cartesian2DFieldmap(index, self)

    @property
    def vector(self):
        """Vector variable and style control for this plot.

        :type: `Vector2D`

        Example usage::

            >>> plot.vector.u_variable = dataset.variable('U')
        """
        return Vector2D(self)

    @property
    def draw_order(self):
        """The order in which objects are drawn to the screen.

        :type: `TwoDDrawOrder`

        Possible values: `TwoDDrawOrder.ByZone`, `TwoDDrawOrder.ByLayer`.

        The order is either by `Zone` or by visual layer (contour, mesh, etc.)::

            >>> plot.draw_order = TwoDDrawOrder.ByZone
        """
        return self._get_style(TwoDDrawOrder, sv.TWODDRAWORDER)

    @draw_order.setter
    def draw_order(self, order):
        self._set_style(TwoDDrawOrder(order), sv.TWODDRAWORDER)

    @property
    def view(self):
        """Axes orientation and limits adjustments.

        :type: `Cartesian2DView`

        Example usage::

            >>> plot.view.fit()
        """
        return Cartesian2DView(self)


class Cartesian3DFieldPlot(FieldPlot):
    """3D plot containing field data associated with style through fieldmaps.

    .. code-block:: python
        :emphasize-lines: 10-13,15

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '3D', 'spaceship.lpk')
        dataset = tp.load_layout(infile)

        frame = tp.active_frame()
        plot = frame.plot(PlotType.Cartesian3D)
        plot.activate()
        plot.show_lighting_effect = False
        plot.show_streamtraces = False

        # save image to file
        tp.export_image('plot_field3d.png')

    ..  figure:: /_static/images/plot_field3d.png
        :width: 300px
        :figwidth: 300px
    """

    def activate(self):
        """Make this the active plot type on the parent frame.

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> plot = frame.plot(PlotType.Cartesian3D)
            >>> plot.activate()
        """
        self.frame.plot_type = PlotType.Cartesian3D

    @property
    def axes(self):
        """Axes style control for this plot.

        :type: `Cartesian3DFieldAxes`

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> frame.plot_type = PlotType.Cartesian3D
            >>> axes = frame.plot().axes
            >>> axes.x_axis.variable = dataset.variable('U')
            >>> axes.y_axis.variable = dataset.variable('V')
            >>> axes.z_axis.variable = dataset.variable('W')
        """
        return Cartesian3DFieldAxes(self)

    def fieldmap(self, key):
        """Returns a `Cartesian3DFieldmap` by `Zone` or index.

        Parameter:
            key (`Zone` or `integer <int>`): The `Zone` must be in the `Dataset`
                attached to the assocated frame of this plot.

        Example usage::

            >>> fmap = plot.fieldmap(dataset.zone(0))
            >>> fmap.scatter.show = True
        """
        index = key if isinstance(key, int) else self.fieldmap_index(key)
        return Cartesian3DFieldmap(index, self)

    @property
    def vector(self):
        """Vector variable and style control for this plot.

        :type: `Vector3D`

        Example usage::

            >>> plot.vector.u_variable = dataset.variable('U')
        """
        return Vector3D(self)

    @property
    def view(self):
        """Viewport, axes orientation and limits adjustments.

        :type: `Cartesian3DView`

        Example usage::

            >>> plot.view.fit()
        """
        return Cartesian3DView(self)


class LinePlot(Plot):
    """Plot with line data and associated style through linemaps.

    .. code-block:: python
        :emphasize-lines: 10-12

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType, Color, LinePattern

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, 'XY', 'rainfall.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        frame.plot_type = PlotType.XYLine
        plot = frame.plot()
        plot.show_symbols = True

        # save image to file
        tp.export_image('plot_line.png')

    ..  figure:: /_static/images/plot_line.png
        :width: 300px
        :figwidth: 300px
    """

    def __init__(self, frame):
        super().__init__(frame, sv.LINEPLOTLAYERS)

    @lock()
    def delete_linemaps(self, *linemaps):
        r"""Clear all linemaps within this plot.

        Parameter:
            \*linemaps (`Linemap`, `integer <int>` or `string <str>`): One or
                more of the following: `Linemap` objects, linemap indices
                (zero-based) or linemap names. If none are given, all linemaps
                will be deleted.

        Example usage::

            >>> plot.delete_linemaps()
            >>> print(plot.num_linemaps)
            0
        """
        if not linemaps:
            return _tecutil.LineMapDelete(None)
        with IndexSet() as indices:
            for lmap in flatten_args(*linemaps):
                try:
                    # try as a Linemap object
                    indices.append(lmap.index)
                except (AttributeError, TypeError):
                    try:
                        # try as a linemap index
                        indices.append(lmap)
                    except TypeError:
                        # assume name pattern
                        for submap in self.linemaps(lmap):
                            indices.append(submap.index)
            return _tecutil.LineMapDelete(indices)

    @property
    def num_linemaps(self):
        """Number of linemaps held by this plot.

        :type: `integer <int>`

        Example usage::

            >>> print(plot.num_linemaps)
            3
        """
        return _tecutil.LineMapGetCountForFrame(self.frame.uid)

    def linemaps(self, pattern=None):
        """Yields linemaps matching the given pattern

        Parameter:
            pattern (`string <str>`, optional): A name pattern to match. If no
                pattern is given, all linemaps are yielded.

        Yields:
            `XYLinemap` or `PolarLine` objects.

        Example usage::

            >>> for lmap in plot.linemaps():
            ...     lmap.show = True
        """
        with self.frame.activated():
            for i in range(self.num_linemaps):
                success, name = _tecutil.LineMapGetName(i + 1)
                if success:
                    if pattern is None or fnmatch(name, pattern):
                        yield self.linemap(i)

    def _linemap_unique_id(self, index):
        with self.frame.activated():
            return _tecutil.LineMapGetUniqueID(index + 1)

    @lock()
    def _add_linemap(self, name, zone, show=True):
        with self.frame.activated():
            new_linemap_index = self.num_linemaps
            if not _tecutil.LineMapCreate():
                raise TecplotSystemError()
            linemap = self.linemap(new_linemap_index)
            linemap.name = name
            linemap.zone = zone
            linemap.show = show
            return linemap

    @property
    def legend(self):
        """Line legend style and placement control.

        :type: `LineLegend`

        Example usage::

            >>> plot.legend.show = True
        """
        return LineLegend(self)

    @property
    def active_linemap_indices(self):
        """`set` of all active linemaps by index.

        :type: `set` of `integers <int>`

        Numbers are zero-based indices to the linemaps::

            >>> active_indices = plot.active_linemap_indices
            >>> active_lmaps = [plot.linemap(i) for i in active_indices]
        """
        return session.get_style(set, sv.ACTIVELINEMAPS,
                                 uniqueid=self.frame.uid)

    @property
    def active_linemaps(self):
        """Yields all active linemaps.

        Yields: `XYLinemap` or `PolarLinemap`

        Example usage::

            >>> from tecplot.constant import Color
            >>> for lmap in plot.active_linemaps:
            ...     lmap.line.color = Color.Blue
        """
        for i in sorted(self.active_linemap_indices):
            yield self.linemap(i)

    @property
    def show_lines(self):
        """Enable lines for this plot.

        :type: `boolean <bool>`

        Example usage::

            >>> plot.show_lines = True
        """
        return self._get_style(bool, sv.SHOWLINES)

    @show_lines.setter
    def show_lines(self, value):
        return self._set_style(bool(value), sv.SHOWLINES)

    @property
    def show_symbols(self):
        """Enable symbols at line vertices for this plot.

        :type: `boolean <bool>`

        Example usage::

            >>> plot.show_symbols = True
        """
        return self._get_style(bool, sv.SHOWSYMBOLS)

    @show_symbols.setter
    def show_symbols(self, value):
        return self._set_style(bool(value), sv.SHOWSYMBOLS)


class XYLinePlot(LinePlot):
    """Cartesian plot with line data and associated style through linemaps.

    .. code-block:: python
        :emphasize-lines: 10-14

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType, FillMode

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '2D', 'random.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        plot = frame.plot(PlotType.XYLine)
        plot.activate()
        plot.show_symbols = True
        plot.linemap(0).line.show = False
        plot.linemap(0).symbols.fill_color = FillMode.UseLineColor

        # save image to file
        tp.export_image('plot_xyline.png')

    ..  figure:: /_static/images/plot_xyline.png
        :width: 300px
        :figwidth: 300px
    """

    def activate(self):
        """Make this the active plot type on the parent frame.

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> plot = frame.plot(PlotType.XYLine)
            >>> plot.activate()
        """
        self.frame.plot_type = PlotType.XYLine

    def linemap(self, pattern):
        """Returns a specific linemap within this plot.

        Parameter:
            pattern (`integer <int>` or `string <str>`): This is either a
                zero-based index or a glob-like pattern which is matched to the
                linemap names. If more than one linemap shares the same name,
                the lowest indexed linemap will be returned.

        Returns:
            `XYLinemap`

        Example usage::

            >>> plot.linemap(0).error_bar.show = True
        """
        if isinstance(pattern, int):
            return XYLinemap(self._linemap_unique_id(pattern), self)
        else:
            return next(self.linemaps(pattern))

    def add_linemap(self, name, zone, x, y, show=True):
        """Add a linemap using the specified zone and variables.

        Parameters:
            name (`string <str>`): Name of the linemap which can be used for
                retrieving with `XYLinePlot.linemap`.
            zone (`Zone`): The data to be used when drawing this linemap.
            x (`Variable`): The ``x`` variable which must be from the same
                `Dataset` as ``y`` and ``zone``.
            y (`Variable`): The ``y`` variable which must be from the same
                `Dataset` as ``x`` and ``zone``.
            show (`boolean <bool>`, optional): Enable this linemap as soon as
                it's added. (default: `True`)

        Returns:
            `XYLinemap`

        Example usage::

            >>> lmap = plot.add_linemap('Line 1', dataset.zone('Zone'),
            ...                         dataset.variable('X'),
            ...                         dataset.variable('Y'))
            >>> lmap.line.line_thickness = 0.8
        """
        lmap = self._add_linemap(name, zone, show)
        lmap.x_variable = x
        lmap.y_variable = y
        return lmap

    @property
    def axes(self):
        """Axes style control for this plot.

        :type: `XYLineAxes`

        Example usage::

            >>> from tecplot.constant import PlotType, AxisMode
            >>> frame.plot_type = PlotType.XYLine
            >>> axes = frame.plot().axes
            >>> axes.axis_mode = AxisMode.XYDependent
            >>> axes.xy_ratio = 2
        """
        return XYLineAxes(self)

    @property
    def show_bars(self):
        """Enable bar chart drawing mode for this plot.

        :type: `boolean <bool>`

        Example usage::

            >>> plot.show_bars = True
        """
        return self._get_style(bool, sv.SHOWBARCHARTS)

    @show_bars.setter
    def show_bars(self, value):
        return self._set_style(bool(value), sv.SHOWBARCHARTS)

    @property
    def show_error_bars(self):
        """Enable error bars for this plot.

        :type: `boolean <bool>`

        The variable to be used for error bars must be set first on at least
        one linemap within this plot::

            >>> plot.linemap(0).error_bars.variable = dataset.variable('E')
            >>> plot.show_error_bars = True
        """
        return self._get_style(bool, sv.SHOWERRORBARS)

    @show_error_bars.setter
    def show_error_bars(self, value):
        return self._set_style(bool(value), sv.SHOWERRORBARS)

    @property
    def view(self):
        """View control of the plot relative to the frame.

        :type: `LineView`

        Example usage::

            >>> plot.view.fit()
        """
        return LineView(self)


class PolarLinePlot(LinePlot):
    """Polar plot with line data and associated style through linemaps.

    .. code-block:: python
        :emphasize-lines: 16-22,25

        import numpy as np
        import tecplot as tp
        from tecplot.constant import PlotType, ThetaMode

        frame = tp.active_frame()

        npoints = 300
        r = np.linspace(0, 2000, npoints)
        theta = np.linspace(0, 10, npoints)

        dataset = frame.create_dataset('Data', ['R', 'Theta'])
        zone = dataset.add_ordered_zone('Zone', (300,))
        zone.variable('R')[:] = r
        zone.variable('Theta')[:] = theta

        plot = frame.plot(PlotType.PolarLine)
        plot.activate()
        plot.axes.r_axis.max = r.max()
        plot.axes.theta_axis.mode = ThetaMode.Radians
        plot.delete_linemaps()
        lmap = plot.add_linemap('Linemap', zone, dataset.variable('R'),
                                dataset.variable('Theta'))
        lmap.line.line_thickness = 0.8

        plot.view.fit()

        tp.export_image('plot_polar.png')

    ..  figure:: /_static/images/plot_polar.png
        :width: 300px
        :figwidth: 300px
    """

    def activate(self):
        """Make this the active plot type on the parent frame.

        Example usage::

            >>> from tecplot.constant import PlotType
            >>> plot = frame.plot(PlotType.PolarLine)
            >>> plot.activate()
        """
        self.frame.plot_type = PlotType.PolarLine

    @property
    def axes(self):
        """Axes style control for this plot.

        :type: `PolarLineAxes`

        Example usage::

            >>> from tecplot.constant import PlotType, ThetaMode
            >>> frame.plot_type = PlotType.PolarLine
            >>> axes = frame.plot().axes
            >>> axes.theta_mode = ThetaMode.Radians
        """
        return PolarLineAxes(self)

    @property
    def view(self):
        """View control of the plot relative to the frame.

        :type: `PolarView`

        Example usage::

            >>> plot.view.fit()
        """
        return PolarView(self)

    def add_linemap(self, name, zone, r, theta, show=True):
        """Add a linemap using the specified zone and variables.

        Parameters:
            name (`string <str>`): Name of the linemap which can be used for
                retrieving with `PolarLinePlot.linemap`.
            zone (`Zone`): The data to be used when drawing this linemap.
            r (`Variable`): The ``r`` variable which must be from the same
                `Dataset` as ``theta`` and ``zone``.
            theta (`Variable`): The ``theta`` variable which must be from the
                same `Dataset` as ``r`` and ``zone``.
            show (`boolean <bool>`, optional): Enable this linemap as soon as
                it's added. (default: `True`)

        Returns:
            `PolarLinemap`

        Example usage::

            >>> lmap = plot.add_linemap('Line 1', dataset.zone('Zone'),
            ...                         dataset.variable('R'),
            ...                         dataset.variable('Theta'))
            >>> lmap.line.line_thickness = 0.8
        """
        lmap = self._add_linemap(name, zone, show)
        lmap.r_variable = r
        lmap.theta_variable = theta
        return lmap

    def linemap(self, pattern):
        """Returns a specific linemap within this plot.

        Parameter:
            pattern (`integer <int>` or `string <str>`): This is either a
                zero-based index or a glob-like pattern which is matched to the
                linemap names. If more than one linemap shares the same name,
                the lowest indexed linemap will be returned.

        Returns:
            `PolarLinemap`

        Example usage::

            >>> plot.linemap(0).error_bar.show = True
        """
        if isinstance(pattern, int):
            return PolarLinemap(self._linemap_unique_id(pattern), self)
        else:
            return next(self.linemaps(pattern))
