from builtins import super

from ..constant import *
from ..exception import *
from ..session import Style
from ..tecutil import color_spec, log_setattr, sv, Index

class Scatter(Style):
    """Plot-local scatter style settings.

    This class controls the style of drawn scatter points on a specific plot.
    """
    def __init__(self, plot):
        self.plot = plot
        super().__init__(sv.GLOBALSCATTER, uniqueid=self.plot.frame.uid)

    @property
    def variable_index(self):
        """Zero-based index of the `Variable` used for size of scatter symbols.

        .. code-block:: python

            >>> plot.scatter.variable_index = dataset.variable('P').index
            >>> plot.fieldmap(0).scatter.size_by_variable = True

        The `Dataset` attached to this contour group's `Frame` is used, and
        the variable itself can be obtained through it::

            >>> scatter = plot.scatter
            >>> scatter_var = dataset.variable(scatter.variable_index)
            >>> scatter_var.index == scatter.variable_index
            True
        """
        return self._get_style(Index, sv.VAR)
    @variable_index.setter
    def variable_index(self, index):
        self._set_style(Index(index), sv.VAR)

    @property
    def variable(self):
        """The `Variable` to be used when sizing scatter symbols.

        The variable must belong to the `Dataset` attached to the `Frame`
        that holds this `ContourGroup`. Example usage::

            >>> plot.scatter.variable = dataset.variable('P')
            >>> plot.fieldmap(0).scatter.size_by_variable = True
        """
        return self.plot.frame.dataset.variable(self.variable_index)
    @variable.setter
    def variable(self, variable):
        self.variable_index = variable.index

@log_setattr
class Symbol(object):
    def __init__(self, parent):
        self.parent = parent
        self._sv = [sv.SYMBOLSHAPE]

    def _get_style(self, rettype, *svargs):
        return self.parent._get_style(rettype, *(self._sv + list(svargs)))
    def _set_style(self, value, *svargs):
        self.parent._set_style(value, *(self._sv + list(svargs)))

    @property
    def _symbol_type(self):
        if self._get_style(bool, sv.ISASCII):
            return SymbolType.Text
        else:
            return SymbolType.Geometry
    @_symbol_type.setter
    def _symbol_type(self, value):
        value = SymbolType(value)
        if value is SymbolType.Text:
            self._set_style(True, sv.ISASCII)
        else:
            self._set_style(False, sv.ISASCII)


class TextSymbol(Symbol):
    """Text character for linemap symbols.

    Only a single character can be used.

    .. code-block:: python
        :emphasize-lines: 27

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType, Color, SymbolType

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, 'XY', 'rainfall.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        frame.plot_type = PlotType.XYLine
        plot = frame.plot()
        plot.show_symbols = True

        cols = [Color.DeepRed, Color.Blue, Color.Fern]
        chars = ['S','D','M']

        for lmap,color,character in zip(plot.linemaps(), cols, chars):
            lmap.show = True
            lmap.line.color = color

            syms = lmap.symbols
            syms.show = True
            syms.symbol_type = SymbolType.Text
            syms.size = 2.5
            syms.color = Color.White
            syms.fill_color = color
            syms.symbol().text = character

        plot.view.fit()

        # save image to file
        tp.export_image('linemap_symbols.png')

    .. figure:: /_static/images/linemap_symbols_text.png
        :width: 300px
        :figwidth: 300px
    """
    def __init__(self, parent):
        super().__init__(parent)
        self._sv += [sv.ASCIISHAPE]

    @property
    def text(self):
        """The ASCII character to use as the symbol to show

        .. note:: This is limited to a single character.

        Example usage::

            >>> from tecplot.constant import SymbolType
            >>> symbols = plot.linemap(0).symbols
            >>> symbols.symbol_type = SymbolType.Text
            >>> symbols.symbol().text = 'X'
        """
        return self._get_style(str, sv.ASCIICHAR)
    @text.setter
    def text(self, value):
        self._set_style(str(value)[0], sv.ASCIICHAR)

    @property
    def typeface(self):
        """Type face to use when rendering text-based symbols.

        :type: `Font.Greek`, `Font.Math`, Font.UserDefined` or Python's
            built-in `None` (default).

        If `None` (the default), the plot's top-level scatter settings will
        determine the font used.

        Example usage::

            >>> from tecplot.constant import SymbolType, Font
            >>> symbols = plot.linemap(0).symbols
            >>> symbols.symbol_type = SymbolType.Text
            >>> symbols.symbol().typeface = Font.Greek
        """
        if self._get_style(bool, sv.USEBASEFONT):
            return None
        else:
            return self._get_style(Font, sv.FONTOVERRIDE)
    @typeface.setter
    def typeface(self, value):
        if __debug__:
            valid_typefaces = [None, Font.Greek, Font.Math, Font.UserDefined]
            if value not in valid_typefaces:
                msg = 'typeface must be one of: '
                msg += ' '.join(str(x) for x in valid_typefaces)
                raise TecplotLogicError(msg)
        if value is None:
            self._set_style(True, sv.USEBASEFONT)
        else:
            self._set_style(False, sv.USEBASEFONT)
            self._set_style(Font(value), sv.FONTOVERRIDE)


class TextScatterSymbol(TextSymbol):
    """Text character for scatter plots.

    Only a single character can be used.

    .. code-block:: python
        :emphasize-lines: 13,22-29

        from os import path
        import tecplot as tp
        from tecplot.constant import (Color, PlotType, PointsToPlot, SymbolType,
                                      GeomShape)

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '2D', 'exchng.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        frame.plot_type = PlotType.Cartesian2D
        plot = frame.plot()
        plot.show_scatter = True

        zones = list(dataset.zones())

        for i,zone in enumerate(zones):
            points = plot.fieldmap(zone).points
            points.points_to_plot = PointsToPlot.SurfaceCellCenters
            points.step = (4,4)

            scatter = plot.fieldmap(zone).scatter
            scatter.color = Color((i%4)+13)
            scatter.fill_color = Color.Yellow
            scatter.size = 3
            scatter.symbol_type = SymbolType.Text
            symbol = scatter.symbol()
            symbol.text = hex(i)[-1]

        tp.export_image('fieldmap_scatter_text.png')

    .. figure:: /_static/images/fieldmap_scatter_text.png
        :width: 300px
        :figwidth: 300px
    """

    @property
    def text(self):
        """The ASCII character to use as the symbol to show

        .. note:: This is limited to a single character.

        Example usage::

            >>> from tecplot.constant import SymbolType
            >>> scatter = plot.fieldmap(0).scatter
            >>> scatter.symbol_type = SymbolType.Text
            >>> scatter.symbol().text = 'X'
        """
        return super().text
    @text.setter
    def text(self, value):
        super(type(self), self.__class__).text.fset(self,value)

    @property
    def typeface(self):
        """Type face to use when rendering text-based scatter.

        :type: `Font.Greek`, `Font.Math`, `Font.UserDefined` or `None`
            (default).

        If `None` (the default), the plot's top-level scatter settings will
        determine the font used.

        Example usage::

            >>> from tecplot.constant import SymbolType, Font
            >>> scatter = plot.fieldmap(0).scatter
            >>> scatter.symbol_type = SymbolType.Text
            >>> scatter.symbol().typeface = Font.Greek
        """
        return super().typeface
    @typeface.setter
    def typeface(self, value):
        super(type(self), self.__class__).typeface.fset(self,value)

class GeometrySymbol(Symbol):
    """Geometric shape for linemap symbols.

    .. code-block:: python
        :emphasize-lines: 27

        from os import path
        import tecplot as tp
        from tecplot.constant import PlotType, Color, GeomShape, SymbolType

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, 'XY', 'rainfall.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        frame.plot_type = PlotType.XYLine
        plot = frame.plot()
        plot.show_symbols = True

        cols = [Color.DeepRed, Color.Blue, Color.Fern]
        shapes = [GeomShape.Square, GeomShape.Circle, GeomShape.Del]

        for lmap,color,shape in zip(plot.linemaps(), cols, shapes):
            lmap.show = True
            lmap.line.color = color

            symbols = lmap.symbols
            symbols.show = True
            symbols.size = 4.5
            symbols.color = color
            symbols.fill_color = color
            symbols.symbol_type = SymbolType.Geometry
            symbols.symbol().shape = shape

        plot.view.fit()

        # save image to file
        tp.export_image('linemap_symbols_geometry.png')

    .. figure:: /_static/images/linemap_symbols_geometry.png
        :width: 300px
        :figwidth: 300px
    """

    @property
    def shape(self):
        """Geometric shape to use when plotting linemap symbols.

        :type: `GeomShape`

        Possible values: `Square <GeomShape.Square>`, `Del`, `Grad`, `RTri`,
        `LTri`, `Diamond`, `Circle <GeomShape.Circle>`, `Cube`, `Sphere`,
        `Octahedron`, `Point <GeomShape.Point>`.

        Example usage::

            >>> from tecplot.constant import SymbolType, GeomShape
            >>> symbols = plot.linemap(0).symbols
            >>> symbols.symbol_type = SymbolType.Geometry
            >>> symbols.symbol().shape = GeomShape.Diamond
        """
        return self._get_style(GeomShape, sv.GEOMSHAPE)
    @shape.setter
    def shape(self, value):
        self._set_style(GeomShape(value), sv.GEOMSHAPE)


class GeometryScatterSymbol(GeometrySymbol):
    """Geometric shape for scatter plots.

    .. code-block:: python
        :emphasize-lines: 13,22-29

        from os import path
        import tecplot as tp
        from tecplot.constant import (Color, PlotType, PointsToPlot, SymbolType,
                                      GeomShape)

        examples_dir = tp.session.tecplot_examples_directory()
        infile = path.join(examples_dir, '2D', 'exchng.plt')
        dataset = tp.data.load_tecplot(infile)

        frame = tp.active_frame()
        frame.plot_type = PlotType.Cartesian2D
        plot = frame.plot()
        plot.show_scatter = True

        zones = list(dataset.zones())

        for i,zone in enumerate(zones):
            points = plot.fieldmap(zone).points
            points.points_to_plot = PointsToPlot.SurfaceCellCenters
            points.step = (2,2)

            scatter = plot.fieldmap(zone).scatter
            scatter.color = Color(i)
            scatter.fill_color = Color(i+len(zones))
            scatter.size = 2
            scatter.line_thickness = 0.5
            scatter.symbol_type = SymbolType.Geometry
            scatter.symbol().shape = GeomShape(i%7)

        tp.export_image('fieldmap_scatter_geometry.png')

    .. figure:: /_static/images/fieldmap_scatter_geometry.png
        :width: 300px
        :figwidth: 300px
    """

    @property
    def shape(self):
        """Geometric shape to use when plotting scatter points.

        :type: `GeomShape`

        Possible values: `Square <GeomShape.Square>`, `Del`, `Grad`, `RTri`,
        `LTri`, `Diamond`, `Circle <GeomShape.Circle>`, `Cube`, `Sphere`,
        `Octahedron`, `Point <GeomShape.Point>`.

        Example usage::

            >>> from tecplot.constant import SymbolType, GeomShape
            >>> scatter = plot.fieldmap(0).scatter
            >>> scatter.symbol_type = SymbolType.Geometry
            >>> scatter.symbol().shape = GeomShape.Diamond
        """
        return super().shape
    @shape.setter
    def shape(self, value):
        super(type(self), self.__class__).shape.fset(self,value)
