from __future__ import unicode_literals
from builtins import super
import unittest
from .test_annotation import AnnotationTestAbstract, AnnotationIteratorTest
from tecplot.constant import *
from tecplot.annotation.text import Text
from tecplot.annotation.annotation import Annotation
import tecplot
from tecplot.exception import TecplotLogicError


class TestTextIterator(AnnotationIteratorTest):
    def setUp(self):
        super().setUp()

    def test_empty_iterator(self):
        self.run_test_empty_iterator(
            Annotation._Iterator(Text, tecplot.active_frame()))

    def test_non_empty_iterator(self):
        # Add a text, see if we get it back in the iterator
        text = tecplot.active_frame().add_text('testing 1,2,3')
        text_objects = [
            T for T in Annotation._Iterator(Text, tecplot.active_frame())]
        self.assertEqual(1, len(text_objects))
        self.assertTrue(text == text_objects[0])

    def test_frame_switch(self):
        for text_obj in range(3):
            tecplot.active_frame().add_text('testing 1,2,3')

        self.run_test_frame_switch(
            Annotation._Iterator(Text, tecplot.active_frame()))

    def test___str__(self):
        text = tecplot.active_frame().add_text('abc')
        self.assertEqual(str(text), 'Text: "abc"')

    def test___eq__(self):
        text_1 = tecplot.active_frame().add_text('abc')
        text_2 = tecplot.active_frame().add_text('abc')
        self.assertEqual(text_1, text_1)
        self.assertNotEqual(text_1, text_2)


class TestTextRoundTrip(AnnotationTestAbstract):
    def setUp(self):
        tecplot.new_layout()
        self.text = tecplot.active_frame().add_text('test_annotation_text')

    def annotation_object(self):
        return self.text

    def annotation_class(self):
        return Text

    def test_round_trip(self):
        for api, value_or_type in [
            ('font_family', 'Times'),
            ('bold', bool),
            ('italic', bool),
            ('size', float),
            ('anchor', TextAnchor),
            ('angle', float),
            ('attached', bool),
            ('line_spacing', float),
            ('text_string', 'abc')
                           ]:
            self.internal_test_round_trip(api, value_or_type)


class TestTextBoxRoundTrip(AnnotationTestAbstract):
    def setUp(self):
        super().setUp()
        tecplot.new_layout()
        self.text = tecplot.active_frame().add_text('abc')

    def annotation_object(self):
        return self.text.text_box

    def annotation_class(self):
        return tecplot.annotation.text.TextBox

    def test_round_trip(self):
        for api, value_or_type in [
            ('margin', float),
            ('text_box_type', tecplot.constant.TextBox),
            ('color', Color),
            ('fill_color', Color),
            ('line_thickness', float)
           ]:
            self.internal_test_round_trip(api, value_or_type)

    def test_textbox_position(self):
        # position is read-only.
        position = self.text.text_box.position
        for value in position:
            self.assertIsInstance(value, float)


class TestText(unittest.TestCase):
    def setUp(self):
        tecplot.new_layout()
        self.text = tecplot.active_frame().add_text('abc')

    def test_size_units_and_coord_sys(self):
        self.text.position_coordinate_system = CoordSys.Frame
        self.assertEqual(self.text.position_coordinate_system, CoordSys.Frame)
        self.text.size_units = Units.Point
        self.assertEqual(self.text.size_units, Units.Point)
        self.text.size_units = Units.Grid
        self.assertEqual(self.text.position_coordinate_system, CoordSys.Grid)

        self.text.position_coordinate_system = CoordSys.Frame
        self.assertEqual(self.text.size_units, Units.Frame)

    def test_invalid_text_box_line_thickness(self):
        if __debug__:
            self.text.text_box.text_box_type = TextBox.Filled
            with self.assertRaises(TecplotLogicError):
                self.text.text_box.line_thickness = 0.0

            with self.assertRaises(TecplotLogicError):
                self.text.text_box.line_thickness = -1.0

    def test_invalid_text_box_margin(self):
        if __debug__:
            self.text.text_box.text_box_type = TextBox.Filled
            with self.assertRaises(TecplotLogicError):
                self.text.text_box.margin = -1.0
