# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_cache_assert', 'pytest_cache_assert._check_assert']

package_data = \
{'': ['*']}

install_requires = \
['attrs-strict>=0.2',
 'beartype>=0.9.0',
 'dictdiffer>=0.9',
 'pendulum>=2.1.2,<3.0.0',
 'pytest>=5']

entry_points = \
{'pytest11': ['pytest_cache_assert = pytest_cache_assert.plugin']}

setup_kwargs = {
    'name': 'pytest-cache-assert',
    'version': '0.1.0',
    'description': 'Cache assertion data to simplify regression testing of complex serializable data',
    'long_description': '# pytest_cache_assert\n\nCache assertion data to simplify regression testing of complex serializable data\n\n## Installation\n\n`poetry add pytest_assert_check --dev`\n\n## Quick Start\n\nThe primary use case of this package is regression testing of large dictionaries. You may have some parameterized test cases where you need to assert that a resulting dictionary is the same, but you don’t want to manually generate the expected fields and values and couple the test case to the source code. Instead you can cache or “record” the expected serialized data structure, check the expected dictionary into version control, then regenerate on changes\n\nThis package can minimize test case logic, while improving test thoroughness\n\nThis project was heavily inspired by the excellent [pytest-recording](https://github.com/kiwicom/pytest-recording)\n\n### Basic Example\n\nSituation: You\'ve created a new project with poetry called `package_a` with one file `source_file.py` and test `tests/test_file.py`\n\n```py\n"""package_a/source_file.py"""\n\nimport sys\nfrom datetime import datetime\nfrom typing import Any, Dict, List, Optional\n\nfrom beartype import beartype\nfrom pydantic import BaseModel\n\n\nclass User(BaseModel):  # noqa: H601\n    """Example from pydantic documentation."""\n\n    id: int  # noqa: A003,VNE003\n    name = \'John Doe\'\n    signup_ts: Optional[datetime] = None\n    friends: List[int] = []\n\n\n@beartype\ndef create_data(name: str) -> Dict[str, Any]:\n    """Arbitrary function that returns a dictionary.\n\n    This demonstration uses pydantic, but any dictionary can be tested!\n\n    """\n    return User(id=sys.maxsize, name=name).dict()\n```\n\n```py\n"""tests/test_file.py"""\n\nimport pytest\n\n\n@pytest.mark.parametrize(\'name\', [\'Test Name 1\', \'Test Name 2\'])\ndef test_create_data(name, assert_against_cache):\n    """Basic test of create_data()."""\n    result = create_data(name=name)\n\n    # One could manually create the expected dictionary\n    cache = {\'id\': 9223372036854775807, \'signup_ts\': None, \'friends\': [], \'name\': name}\n    assert result == cache\n    # ----------------------------------------------------------------------------------\n    # Or utilize the pytest_cache_assert fixture to compare against the last cached version\n    assert_against_cache(result)\n```\n\n`pytest_cache_assert` will automatically create: `tests/cache-assert/source_file/test_file/test_create_data-[Test Name 1].json` (and `test_create_data[Test Name 2].json`) for each of the parameters when first run by caching the `result`. Below is the example for `test_create_data-[Test Name 1].json`. The cached files should be checked into version control. They can be manually edited if needed or regenerated by deleting the file and re-running the tests suite.\n\n```json\n{\n  "_info": {\n    "func_args": {\n      "name": "Test Name 1"\n    },\n    "test_file": ".../tests/test_file.py",\n    "test_name": "test_create_data"\n  },\n  "_json": {\n    "friends": [],\n    "id": 9223372036854775807,\n    "name": "Test Name 1",\n    "signup_ts": null\n  }\n}\n\n```\n\n### More Examples\n\nIn your cached dictionary, you may have variable values with more complex logic to verify, such as dates, UUIDs, etc. These can be selectively ignored, matched-if-null, or some other user-specified check:\n\n```py\nfrom uuid import uuid4\nfrom datetime import datetime\n\nfrom pytest_cache_assert import assert_against_cache\n\n# TODO: Implement this example! Add as red test, then implement!\n\ndef test_variable_cache(assert_against_cache):\n    """Demonstration of specifying custom assertion rules."""\n    # TODO: support list indexing on top of jmespath...\n    result = {"date": datetime.now(), {"nested": {"uuid": uuid4()}}}\n\n    assert_against_cache(result, ...)\n```\n\nOr you may want to write your own custom checks against the serialized data, such as with Cerberus or another library. This is possible with the `validator` callable. The default is a no-op and that may be replaced with a custom function that raises an Exception on error.\n\n```py\n# TODO: Add example for custom cerberus checks with `validator()`\n```\n\n### Even More Example\n\nFor more examples, see [Scripts](https://github.com/kyleking/pytest_cache_assert/scripts) or [Tests](https://github.com/kyleking/pytest_cache_assert/tests)\n\n## Global Configuration Options\n\n- `custom directory` (set in pytest fixture like pytest-record)\n    - Default is test directory/cache-assert/\n- `record_rule`: regenerate on failure (re-raises assert, but updates cache). Default is to record once\n\n```py\nimport pytest\n\n\n# TODO: Not currently used...\n@pytest.fixture(scope=\'module\')\ndef cache_assert_config():\n    return {\n        \'rel_path_cache_dir\': \'..........\',\n        \'record_mode\': \'rewrite\',  # TODO: Consider record mode\n        \'filter_headers\': [\'authorization\'],  # TODO: Consider filters\n    }\n```\n\n## Roadmap\n\nSee the `Open Issues` and `Milestones` for current status and [./docs/CODE_TAG_SUMMARY.md](./docs/CODE_TAG_SUMMARY.md) for annotations in the source code.\n\nFor release history, see the [./docs/CHANGELOG.md](./docs/CHANGELOG.md)\n\n## Contributing\n\nSee the Developer Guide, Contribution Guidelines, etc\n\n- [./docs/DEVELOPER_GUIDE.md](./docs/DEVELOPER_GUIDE.md)\n- [./docs/STYLE_GUIDE.md](./docs/STYLE_GUIDE.md)\n- [./docs/CONTRIBUTING.md](./docs/CONTRIBUTING.md)\n- [./docs/CODE_OF_CONDUCT.md](./docs/CODE_OF_CONDUCT.md)\n- [./docs/SECURITY.md](./docs/SECURITY.md)\n\n## License\n\n[LICENSE](https://github.com/kyleking/pytest_cache_assert/LICENSE)\n',
    'author': 'Kyle King',
    'author_email': 'dev.act.kyle@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kyleking/pytest_cache_assert',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
