"""
Python A2A - Agent-to-Agent Protocol

A Python library for implementing Google's Agent-to-Agent (A2A) protocol.
"""

__version__ = "0.4.0"  # Updated version number

# Import core models
from .models import (
    Message,
    MessageRole,
    Conversation,
    TextContent,
    FunctionParameter,
    FunctionCallContent,
    FunctionResponseContent,
    ErrorContent,
    Metadata,
    ContentType,
    # Add new models
    AgentCard,
    AgentSkill,
    Task,
    TaskStatus,
    TaskState
)

# Import clients with new components
from .client import (
    BaseA2AClient,
    A2AClient,
    OpenAIA2AClient,
    AnthropicA2AClient,
    # Add new client components
    AgentNetwork,
    AIAgentRouter,
    StreamingClient
)

# Import servers
from .server import (
    BaseA2AServer,
    A2AServer,
    run_server,
    OpenAIA2AServer,
    AnthropicA2AServer,
    BedrockA2AServer
)

# Import workflow system
from .workflow import (
    Flow,
    WorkflowContext,
    WorkflowStep,
    QueryStep,
    AutoRouteStep,
    FunctionStep,
    ConditionalBranch,
    ConditionStep,
    ParallelStep,
    ParallelBuilder,
    StepType
)

# Import utility functions
from .utils import (
    # Formatting utilities
    format_message_as_text,
    format_conversation_as_text,
    pretty_print_message,
    pretty_print_conversation,

    # Validation utilities
    validate_message,
    validate_conversation,
    is_valid_message,
    is_valid_conversation,

    # Conversion utilities
    create_text_message,
    create_function_call,
    create_function_response,
    create_error_message,
    format_function_params,
    conversation_to_messages,

    # New decorators
    skill,
    agent
)

# Import documentation utilities - now that we have stubs
from .docs import (
    generate_a2a_docs,
    generate_html_docs
)

# Import exceptions
from .exceptions import (
    A2AError,
    A2AImportError,
    A2AConnectionError,
    A2AResponseError,
    A2ARequestError,
    A2AValidationError,
    A2AAuthenticationError,
    A2AConfigurationError
)

# Expose command-line interface
from .cli import main as cli_main

# Import MCP integration with improved error handling
try:
    # MCP client
    from .mcp.client import (
        MCPClient,
        MCPError,
        MCPConnectionError,
        MCPTimeoutError,
        MCPToolError,
        MCPTools
    )

    # MCP agent integration
    from .mcp.agent import MCPEnabledAgent

    # FastMCP implementation
    from .mcp.fastmcp import (
        FastMCP,
        MCPResponse,
        text_response,
        error_response,
        image_response,
        multi_content_response,
        ContentType as MCPContentType
    )

    # Improved agent integration
    from .mcp.integration import (
        FastMCPAgent,
        A2AMCPAgent
    )

    # Proxy functionality
    from .mcp.proxy import create_proxy_server

    # Transport for easy imports
    from .mcp.transport import create_fastapi_app

    HAS_MCP = True
except ImportError as e:
    # Print more detailed error information to help diagnose import issues
    import sys
    print(f"Warning: MCP module could not be imported: {e}", file=sys.stderr)
    HAS_MCP = False

# Base package exports
__all__ = [
    # Version
    '__version__',

    # Models
    'Message',
    'MessageRole',
    'Conversation',
    'TextContent',
    'FunctionParameter',
    'FunctionCallContent',
    'FunctionResponseContent',
    'ErrorContent',
    'Metadata',
    'ContentType',
    # New models
    'AgentCard',
    'AgentSkill',
    'Task',
    'TaskStatus',
    'TaskState',

    # Clients
    'BaseA2AClient',
    'A2AClient',
    'OpenAIA2AClient',
    'AnthropicA2AClient',
    'AgentNetwork',
    'AIAgentRouter',
    'StreamingClient',

    # Servers
    'BaseA2AServer',
    'A2AServer',
    'run_server',
    'OpenAIA2AServer',
    'AnthropicA2AServer',
    'BedrockA2AServer',

    # Workflow system
    'Flow',
    'WorkflowContext',
    'WorkflowStep',
    'QueryStep',
    'AutoRouteStep',
    'FunctionStep',
    'ConditionalBranch',
    'ConditionStep',
    'ParallelStep',
    'ParallelBuilder',
    'StepType',

    # Utilities
    'format_message_as_text',
    'format_conversation_as_text',
    'pretty_print_message',
    'pretty_print_conversation',
    'validate_message',
    'validate_conversation',
    'is_valid_message',
    'is_valid_conversation',
    'create_text_message',
    'create_function_call',
    'create_function_response',
    'create_error_message',
    'format_function_params',
    'conversation_to_messages',
    'skill',
    'agent',

    # Documentation
    'generate_a2a_docs',
    'generate_html_docs',

    # Exceptions
    'A2AError',
    'A2AImportError',
    'A2AConnectionError',
    'A2AResponseError',
    'A2ARequestError',
    'A2AValidationError',
    'A2AAuthenticationError',
    'A2AConfigurationError',

    # CLI
    'cli_main',
]

# Add MCP classes if available
if HAS_MCP:
    # Add to __all__ list
    __all__.extend([
        # MCP client
        'MCPClient',
        'MCPError',
        'MCPConnectionError',
        'MCPTimeoutError',
        'MCPToolError',
        'MCPTools',

        # MCP agent integration
        'MCPEnabledAgent',

        # FastMCP implementation
        'FastMCP',
        'MCPResponse',
        'text_response',
        'error_response',
        'image_response',
        'multi_content_response',
        'MCPContentType',

        # Improved agent integration
        'FastMCPAgent',
        'A2AMCPAgent',

        # Proxy functionality
        'create_proxy_server',

        # Transport
        'create_fastapi_app'
    ])