"""SAR ADC models"""

import numpy as np
from adc_eval.adcs.basic import ADC


class SAR(ADC):
    """
    SAR ADC Class.

    Parameters
    ----------
    nbits : int, optional
        Number of bits for the ADC. The default is 8.
    fs : float, optional
        Sample rate for the ADC in Hz. The default is 1Hz.
    vref : float, optional
        Reference level of the ADC in Volts ([0, +vref] conversion range). The default is 1.
    seed : int, optional
        Seed for random variable generation. The default is 1.
    **kwargs
        Extra arguments.
        weights : list, optional
            List of weights for SAR capacitors. Must be >= nbits. Defaults to binary weights.
            MSB weight should be in index 0.

    Attributes
    -------
    vin : float
        Sets or returns the current input voltage level. Assumed +/-vref/2 input
    vlsb : float
        LSB voltage of the converter. vref/2^nbits
    noise : float, default=0
        Sets or returns the stdev of the noise generated by the converter.
    weights : list
        Sets or returns the capacitor weighting of the array. Default is binary weighting.
    mismatch : float
        Sets or returns the stdev of the mismatch of the converter. Default is no mismatch.
    comp_noise : float
        Sets or returns the stdev of the comparator noise. Default is no noise.
    offset : tuple of float
        Sets the (mean, stdev) of the offset of the converter. Default is no offset.
    gain_error : tuple of float
        Sets the (mean, stdev) of the gain error of the converter. Default is no gain error.
    distortion : list of float
        Sets the harmonic distortion values with index=0 corresponding to HD1.
        Example: For unity gain and only -30dB of HD3, input is [1, 0, 0.032]
    dout : int
        Digital output code for current vin value.

    Methods
    -------
    run_step

    """

    def __init__(self, nbits=8, fs=1, vref=1, seed=1, **kwargs):
        """Initialization function for Generic ADC."""
        super().__init__(nbits, fs, vref, seed)

        self._mismatch = None
        self._comp_noise = 0

        # Get keyword arguments
        self._weights = None
        self.weights = kwargs.get("weights", self.weights)

    @property
    def weights(self):
        """Returns capacitor unit weights."""
        if self._weights is None:
            self._weights = np.flip(2 ** np.linspace(0, self.nbits - 1, self.nbits))
        return np.array(self._weights)

    @weights.setter
    def weights(self, values):
        """Sets the capacitor unit weights."""
        self._weights = np.array(values)
        self.dbits = np.zeros(len(values))
        if self._weights.size < self.nbits:
            print(
                f"WARNING: Capacitor weight array size is {self._weights.size} for {self.nbits}-bit ADC."
            )
        self.mismatch = self.err["mismatch"]

    @property
    def mismatch(self):
        """Return noise stdev."""
        if self._mismatch is None:
            self._mismatch = np.zeros(self.weights.size)
        return self._mismatch

    @mismatch.setter
    def mismatch(self, stdev):
        """Sets mismatch stdev."""
        self.err["mismatch"] = stdev
        self._mismatch = np.random.normal(0, stdev, self.weights.size)
        self._mismatch /= np.sqrt(self.weights)

    @property
    def comp_noise(self):
        """Returns the noise of the comparator."""
        return self._comp_noise

    @comp_noise.setter
    def comp_noise(self, value):
        """Sets the noise of the comparator."""
        self._comp_noise = value

    def run_step(self):
        """Run a single ADC step."""
        vinx = self.vin

        cweights = self.weights * (1 + self.mismatch)
        cdenom = sum(cweights) + 1

        comp_noise = np.random.normal(0, self.comp_noise, cweights.size)

        # Bit cycling
        vdac = vinx
        for n, _ in enumerate(cweights):
            vcomp = vdac - self.vref / 2 + comp_noise[n]
            compout = vcomp * 1e6
            compout = -1 if compout <= 0 else 1
            self.dbits[n] = max(0, compout)
            vdac -= compout * self.vref / 2 * cweights[n] / cdenom

        # Re-scale the data
        scalar = 2**self.nbits / cdenom
        self.dval = min(2**self.nbits - 1, scalar * sum(self.weights * self.dbits))
