from ai_kit.shared_console import shared_console
from ai_kit.core.basic_fetcher import BasicFetcher, BatchFetcherResult
from ai_kit.core.duckduckgo import DuckDuckGoSearch
from ai_kit.core.llms.litellm_client import (
    RerankClient,
    RerankCompletionResult,
)
from ai_kit.core.llms.google_genai_client import GoogleGenAI
from typing import List
from ai_kit.utils import print_stream

async def summarize_markdown(query: str, parsed_results: list[BatchFetcherResult]) -> str:
    """Summarize the markdown content."""
    client = GoogleGenAI(model="gemini-2.0-flash")

    markdown_string = ""
    for result in parsed_results:
        markdown_string += f"Title: {result['title']}\n"
        markdown_string += f"Source: {result['href']}\n"
        markdown_string += f"Content: {result['parsed_page_content']}\n\n"

    PROMPT = f"""
    You are an expert research analyst tasked with extracting, analyzing, and organizing information from various sources. Your goal is to provide a comprehensive, well-structured analysis that is immediately useful to the reader. Adapt your response format based on the type of query and available content.

    Here is the content to analyze:
    {markdown_string}

    Here is the query:
    {query}

    Important Instructions:
    1. Focus on extracting valuable information from the accessible content
    2. If some sources are inaccessible, note this and focus on what is available
    3. Identify gaps in the information and what might need additional research
    4. Synthesize information across sources to provide a complete picture
    5. Highlight any conflicting information or disagreements between sources
    6. Include direct quotes when they add significant value
    7. Note the recency and relevance of the information
    8. Distinguish between established facts and emerging trends
    9. When showing code examples, use plain text without any special formatting characters

    Respond using the most appropriate sections for the query type. Choose from:

    FOR TECHNICAL CONTENT:
    - QUICK START (requirements, installation, basic usage)
    - DETAILED IMPLEMENTATION (architecture, components, security)
    - CODE EXAMPLES (complete, documented examples)
    - CONFIGURATION (options, customization, tuning)
    - ADVANCED USAGE (patterns, integrations, scaling)
    - TROUBLESHOOTING (errors, limitations, solutions)
    - API DETAILS (methods, parameters, responses)

    FOR RESEARCH CONTENT:
    - RESEARCH CONTEXT (field overview, current state)
    - FINDINGS SYNTHESIS (key findings, themes, patterns)
    - METHODOLOGY ANALYSIS (approaches, techniques, validity)
    - PRACTICAL IMPLICATIONS (applications, impact)
    - CRITICAL EVALUATION (strengths, limitations, future work)

    FOR GENERAL TOPICS:
    - CURRENT STATE (overview, concepts, developments)
    - DETAILED ANALYSIS (components, trends, comparisons)
    - PRACTICAL APPLICATIONS (implementations, benefits, risks)
    - FUTURE OUTLOOK (trends, opportunities, recommendations)
    - SUPPORTING EVIDENCE (statistics, examples, references)

    Start with an INFORMATION QUALITY NOTICE describing:
    - Number and types of sources analyzed
    - Any access limitations or missing data
    - Potential biases or gaps
    - Date range of the information

    End with:
    - Summary of key points
    - Identified gaps or limitations
    - Suggestions for further research
    - Actionable next steps

    Do not include:
    - Irrelevant or redundant information
    - Marketing content or promotional material
    - Unsubstantiated claims
    - Basic or obvious information unless specifically requested
    - Navigation elements or structural content
    - Special formatting characters in code examples

    Focus on providing comprehensive, accurate, and actionable information that directly addresses the query while acknowledging any limitations in the available data.
    """

    return await client.chat_completion(
        messages=[{"role": "user", "content": PROMPT}], stream=True
    )


def pprint(parsed_page: dict):
    shared_console.print(f"[bold cyan]{parsed_page['title']}[/bold cyan]")
    shared_console.print(f"[bold green]Source: {parsed_page['href']}[/bold green]")
    shared_console.print("[bold yellow]Content:[/bold yellow]")
    shared_console.print(parsed_page["parsed_page_content"])
    shared_console.print("\n")


async def research_command(query: str, max_results: int = 10):
    """Research a topic."""
    try:
        from duckduckgo_search import DDGS
    except ImportError:
        shared_console.print(
            "[red]Error: duckduckgo_search is not installed. "
            "Please install it using 'pip install duckduckgo-search'[/red]"
        )
        return []

    with shared_console.status(
        "[bold cyan]Okay, I'm searching the web...[/bold cyan]"
    ) as status:
        # 1. Search the web for information
        ddgs = DuckDuckGoSearch()  # for search
        # Since we're reranking, we need to search for more than max_results then trim down to max_results
        buffer = 10
        results = await ddgs.search(query, max_results + buffer)  # search

        status.update(f"[green]Found {len(results)} results[/green]")
        DuckDuckGoSearch.pprint(results)  # print the search results

        # 2. Find the most relevant pages
        status.update(f"[green]Reranking results...[/green]")
        reranker = RerankClient()
        reranked_results: List[RerankCompletionResult] = (
            await reranker.rerank_completion(
                query,
                [
                    {
                        "text": result["body"],
                        "metadata": {
                            "title": result["title"],
                            "url": result["href"],
                        },
                    }
                    for result in results
                ],
            )
        )
        reranked_results = reranked_results[
            :max_results
        ]  # now we trim down to max_results

        # 3. fetch the content
        status.update(f"[green]Fetching content...[/green]")
        async with BasicFetcher() as fetcher:
            try:
                parsed: list[BatchFetcherResult] = await fetcher.batch_fetch_and_parse_pages(results)
                successful_fetches = len([p for p in parsed if p.get('parsed_page_content')])
                failed_fetches = len(results) - successful_fetches
                
                if successful_fetches == 0:
                    shared_console.print("[red]Error: Could not fetch any content. Please try a different query or check your internet connection.[/red]")
                    return []
                
                if failed_fetches > 0:
                    shared_console.print(f"[yellow]Note: {failed_fetches} sources were inaccessible. Results will be based on {successful_fetches} accessible sources.[/yellow]")

            except Exception as e:
                shared_console.print(f"[red]Error fetching content: {str(e)}[/red]")
                return []

        # 4. summarize the content
        status.update(f"[green]Summarizing content from {successful_fetches} sources...[/green]")
        summary = await summarize_markdown(query, parsed)
        
        shared_console.print(f"[bold green]Here's what I found from {successful_fetches} sources:[/bold green]")

        # 5. print
        await print_stream(summary)

    return parsed
