from pathlib import Path
from ai_kit.shared_console import shared_console

def handle_cursorrules(source: Path, dest: Path) -> None:
    """Handle cursor rules file copying with special merge logic.
    
    Copies all cursor rule files from system_prompts/cursorrules to .cursor/rules directory.
    """
    # Ensure .cursor/rules directory exists
    cursor_dir = Path('.cursor/rules')
    if not cursor_dir.exists():
        cursor_dir.mkdir(parents=True, exist_ok=True)
    
    # Get source directory (the cursorrules directory)
    source_dir = source.parent / "cursorrules"
    if not source_dir.exists():
        shared_console.print("[red]Error: Could not find cursor rules directory[/red]")
        return
        
    # Process each .mdc file in the source directory
    for source_file in source_dir.glob("*.mdc"):
        dest_file = cursor_dir / source_file.name
        
        # Read source content
        source_content = source_file.read_text().splitlines()
        
        # If destination doesn't exist, just copy the source
        if not dest_file.exists():
            dest_file.write_text('\n'.join(source_content) + '\n')
            shared_console.print(f"[green]✓ Created cursor rule at {dest_file}[/green]")
            continue
            
        # Read existing content
        dest_content = dest_file.read_text().splitlines()
        
        # Add our entries if they don't exist
        added = False
        for line in source_content:
            if line and line not in dest_content:
                if not added:
                    # Add a blank line and comment if this is our first addition
                    if dest_content and dest_content[-1] != '':
                        dest_content.append('')
                    dest_content.append('# Added by AI Kit')
                    added = True
                dest_content.append(line)
        
        # Write back if we made changes
        if added:
            dest_file.write_text('\n'.join(dest_content) + '\n')
            shared_console.print(f"[green]✓ Updated cursor rule at {dest_file}[/green]") 