"""
Favorites Operations Client - Level 3: Favorites-Specific Operations

This module provides favorites-specific operations within the Core API.
Part of the three-tier V1.1 architecture with complete 4-pattern detailed functions.
"""

import asyncio
from typing import Optional, List, Union, Any
from httpx import Response

# Import from Level 3 (operation-specific models)
from .models import FavoritesResponse, FavoritesListResponse, CreateFavoritesRequest

# Import raw operations
try:
    from ....raw_clients.alfresco_core_client.core_client.api.favorites import (
            create_favorite as _create_favorite,
            create_site_favorite as _create_site_favorite,
            delete_favorite as _delete_favorite,
            delete_site_favorite as _delete_site_favorite,
            get_favorite as _get_favorite,
            get_favorite_site as _get_favorite_site,
            list_favorites as _list_favorites,
            list_favorite_sites_for_person as _list_favorite_sites_for_person
    )
    RAW_OPERATIONS_AVAILABLE = True
except ImportError:
    RAW_OPERATIONS_AVAILABLE = False


class FavoritesClient:
    """
    Favorites operations client with 4-pattern detailed functions.
    
    Provides high-level methods for favorites management operations
    that are essential for MCP servers and favorites workflows.
    
    Each operation has 4 variants for maximum flexibility:
    - Basic sync/async for simple use cases
    - Detailed sync/async for full HTTP response access
    """
    
    def __init__(self, client_factory):
        """Initialize with client factory for raw client access."""
        self._client_factory = client_factory
        self._raw_client = None
        
        # Store raw operation references
        if RAW_OPERATIONS_AVAILABLE:
            self._create_favorite = _create_favorite
            self._create_site_favorite = _create_site_favorite
            self._delete_favorite = _delete_favorite
            self._delete_site_favorite = _delete_site_favorite
            self._get_favorite = _get_favorite
            self._get_favorite_site = _get_favorite_site
            self._list_favorites = _list_favorites
            self._list_favorite_sites_for_person = _list_favorite_sites_for_person
    
    def _get_raw_client(self):
        """Get or create the raw client."""
        if self._raw_client is None:
            # Import the raw core client directly
            from ....raw_clients.alfresco_core_client.core_client.client import AuthenticatedClient
            
            # Create the raw client with same auth setup
            self._raw_client = AuthenticatedClient(
                base_url=f"{self._client_factory.base_url}/alfresco/api/-default-/public/alfresco/versions/1",
                token=self._client_factory.auth.get_auth_token(),
                prefix=self._client_factory.auth.get_auth_prefix(),
                verify_ssl=self._client_factory.verify_ssl
            )
        return self._raw_client
    
    def get_httpx_client(self):
        """
        Get direct access to raw httpx client for advanced operations.
        
        Perfect for MCP servers that need raw HTTP access.
        """
        return self._get_raw_client().get_httpx_client()
    
    # Placeholder for favorites operations - will be populated from the original file
    def __repr__(self) -> str:
        """String representation for debugging."""
        base_url = getattr(self._client_factory, 'base_url', 'unknown')
        return f"FavoritesClient(base_url='{base_url}')" 