"""
Groups Operations Client - Level 3: Groups-Specific Operations

This module provides groups-specific operations within the Core API.
Part of the three-tier V1.1 architecture with complete 4-pattern detailed functions.
"""

import asyncio
from typing import Optional, List, Union, Any
from httpx import Response

# Import from Level 3 (operation-specific models)
from .models import GroupsResponse, GroupsListResponse, CreateGroupsRequest

# Import raw operations
try:
    from ....raw_clients.alfresco_core_client.core_client.api.groups import (
            create_group as _create_group,
            create_group_membership as _create_group_membership,
            delete_group as _delete_group,
            delete_group_membership as _delete_group_membership,
            get_group as _get_group,
            list_group_memberships as _list_group_memberships,
            list_groups as _list_groups,
            update_group as _update_group
    )
    RAW_OPERATIONS_AVAILABLE = True
except ImportError:
    RAW_OPERATIONS_AVAILABLE = False


class GroupsClient:
    """
    Groups operations client with 4-pattern detailed functions.
    
    Provides high-level methods for groups management operations
    that are essential for MCP servers and groups workflows.
    
    Each operation has 4 variants for maximum flexibility:
    - Basic sync/async for simple use cases
    - Detailed sync/async for full HTTP response access
    """
    
    def __init__(self, client_factory):
        """Initialize with client factory for raw client access."""
        self._client_factory = client_factory
        self._raw_client = None
        
        # Store raw operation references
        if RAW_OPERATIONS_AVAILABLE:
            self._create_group = _create_group
            self._create_group_membership = _create_group_membership
            self._delete_group = _delete_group
            self._delete_group_membership = _delete_group_membership
            self._get_group = _get_group
            self._list_group_memberships = _list_group_memberships
            self._list_groups = _list_groups
            self._update_group = _update_group
    
    def _get_raw_client(self):
        """Get or create the raw client."""
        if self._raw_client is None:
            # Import the raw core client directly
            from ....raw_clients.alfresco_core_client.core_client.client import AuthenticatedClient
            
            # Create the raw client with same auth setup
            self._raw_client = AuthenticatedClient(
                base_url=f"{self._client_factory.base_url}/alfresco/api/-default-/public/alfresco/versions/1",
                token=self._client_factory.auth.get_auth_token(),
                prefix=self._client_factory.auth.get_auth_prefix(),
                verify_ssl=self._client_factory.verify_ssl
            )
        return self._raw_client
    
    def get_httpx_client(self):
        """
        Get direct access to raw httpx client for advanced operations.
        
        Perfect for MCP servers that need raw HTTP access.
        """
        return self._get_raw_client().get_httpx_client()
    
    # Placeholder for groups operations - will be populated from the original file
    def __repr__(self) -> str:
        """String representation for debugging."""
        base_url = getattr(self._client_factory, 'base_url', 'unknown')
        return f"AlfrescoGroupsClient(base_url='{base_url}')" 