"""
Level 3: Preferences Operation Models - Specific to Preferences Operations

This module contains models that are specific to preferences operations
within the Core API.

Three-Tier Architecture:
- Level 1: Global models shared across ALL APIs  
- Level 2: Core API models shared within Core API
- Level 3 (This file): Preferences operation-specific models

Benefits:
- Perfect locality (preferences models exactly where preferences operations are)
- Clean imports (from .models import PreferencesResponse)
- Logical organization (operation-specific grouping)
"""

from typing import Optional, List, Dict, Any, Annotated
from datetime import datetime
from pydantic import BaseModel, Field, ConfigDict

# Import from Level 1 (global)
from ...models import BaseEntry, ContentInfo, UserInfo

# Import from Level 2 (Core API)
from ..models import CoreResponse


class PreferencesResponse(CoreResponse):
    """Response wrapper for preferences operations."""
    entry: BaseEntry = Field(..., description="Preferences data")


class PreferencesListResponse(BaseModel):
    """Response wrapper for preferences list operations."""
    model_config = ConfigDict(extra='forbid')
    
    list: Dict[str, Any] = Field(..., description="List data with pagination")


# Operation-specific models will be added here based on analysis
# Example models for common patterns:


class CreatePreferencesRequest(BaseModel):
    """Request model for creating preferences."""
    model_config = ConfigDict(extra='forbid')
    
    name: Annotated[str, Field(
        description="Preferences name",
        min_length=1,
        max_length=255
    )]
    
    properties: Annotated[Optional[Dict[str, Any]], Field(
        description="Custom properties",
        default=None
    )]


class UpdatePreferencesRequest(BaseModel):
    """Request model for updating preferences."""
    model_config = ConfigDict(extra='forbid')
    
    name: Annotated[Optional[str], Field(
        description="Updated name",
        min_length=1,
        max_length=255,
        default=None
    )]
    
    properties: Annotated[Optional[Dict[str, Any]], Field(
        description="Updated properties",
        default=None
    )]


# Export all models
__all__ = [
    'PreferencesResponse', 
    'PreferencesListResponse',
    'CreatePreferencesRequest',
    'UpdatePreferencesRequest'
]