//! Parsing of the [alpm-pkgbuild-bridge] script output, which prints package metadata contained
//! in [PKGBUILD] files.
//!
//! This module handles the parsing of output generated by the [alpm-pkgbuild-bridge] script,
//! which is shipped as a separate package. At this step, we **do not** yet ensure the correctness
//! of the parsed data, instead we only parse the data structure of the output and bring it into an
//! easy to handle struct.
//! You can think of it as an equivalent of serde's intermediate `Value`.
//!
//! The output of the script has a well-known structure and provides some guarantees that we'll use
//! to keep the parsers simple.
//!
//! Overall format specification:
//! 1. Variables are declared one-line per variable.
//! 1. All variables declaration lines begin with `VAR`
//! 1. All variable values are surrounded by double quotes `"`. If a value contains double quotes,
//!    those double quotes are escaped with a backslash: `\"`.
//! 1. If a variable is an array, the array's values are represented as space separated items on the
//!    same line. Each value is surrounded by double quotes.
//! 1. Variables **cannot** be declared twice.
//! 1. pkgbase variable declarations look as follows:
//!   
//!    ```text
//!    VAR GLOBAL [ARRAY|STRING] $KEY $VALUES
//!    Examples:
//!    VAR GLOBAL STRING pkgrel "6"
//!    VAR GLOBAL ARRAY license "GPL-2.0-or-later" "CCPL"
//!    ```
//! 1. package function variable declarations look as follows:
//!   
//!    ```txt
//!    VAR FUNCTION $PACKAGE_NAME [ARRAY|STRING] $KEY $VALUES
//!    Examples:
//!    VAR FUNCTION package STRING pkgdesc "My normal package"
//!    VAR FUNCTION package_my-split-package STRING pkgdesc "My special split package"
//!    VAR FUNCTION package_my-split-package ARRAY depends "cargo" "glibc"
//!    ```
//! 1. Package overrides that clear a value/array simply don't have any string values.
//!   
//!    Example:
//!   
//!    ```text
//!    VAR FUNCTION package_my-split-package ARRAY depends
//!    ```
//! 1. Split-/Package functions are declared as: `FUNCTION $PACKAGE_NAME`
//!   
//!    Example:
//!    
//!    ```text
//!    FUNCTION package_my-split-package
//!    ```
//!    
//!    Note that single-package PKGBUILD files don't necessarily need to have the `package_$NAME`
//!    format, but rather may just have a single `package` function.
//!
//! Structure guarantees:
//!
//! - There are no empty lines.
//! - Global variables (those that apply to all packages defined in a [PKGBUILD]) are declared
//!   first.
//! - Variables that are declared in package functions of the [PKGBUILD] are always returned after
//!   the global variables and are grouped by package.
//! - The declarations of package function names are always returned at the end (after all package
//!   function variables are returned).
//!
//! [PKGBUILD]: https://man.archlinux.org/man/PKGBUILD.5
//! [alpm-pkgbuild-bridge]: https://gitlab.archlinux.org/archlinux/alpm/alpm-pkgbuild-bridge

use std::{collections::HashMap, fmt::Display, path::Path, str::FromStr};

use alpm_parsers::iter_str_context;
#[cfg(doc)]
use alpm_types::Architecture;
use strum::{EnumString, VariantNames};
use winnow::{
    ModalResult,
    Parser,
    ascii::{alphanumeric1, space0, space1},
    combinator::{alt, cut_err, delimited, eof, opt, preceded, repeat, terminated, trace},
    error::{StrContext, StrContextValue},
    token::{none_of, one_of, take_till, take_until},
};

use crate::{bridge::run_bridge_script, error::Error};

/// A single value or a list of values declared in the output of the `alpm-pkgbuild-bridge` script.
///
/// Both parser functions ensure that at least one value exists.
#[derive(Clone, Debug)]
pub enum Value {
    /// A single value.
    Single(String),
    /// An array of values.
    Array(Vec<String>),
}

impl Value {
    /// Returns the values of `&self` in vector representation.
    ///
    /// This is useful for values that may be available as both single values and arrays.
    pub fn as_vec(&self) -> Vec<&String> {
        match self {
            Value::Single(item) => vec![&item],
            Value::Array(items) => Vec::from_iter(items.iter()),
        }
    }

    /// Returns the values of `self` in vector representation.
    ///
    /// This is useful for values that may be available as both single values and arrays.
    pub fn as_owned_vec(self) -> Vec<String> {
        match self {
            Value::Single(item) => vec![item],
            Value::Array(items) => items,
        }
    }

    /// Checks whether this holds a value.
    ///
    /// Returns `true` if `self` is [`Value::Single`] (they always have a value set by definition),
    /// or if `self` is [`Value::Array`] and contains at least one element.
    /// Returns `false` in all other cases.
    pub fn has_value(&self) -> bool {
        match self {
            Value::Single(_) => true,
            Value::Array(items) => !items.is_empty(),
        }
    }

    /// Recognizes a [`Value::Single`] in a string slice while handling the surroundings.
    ///
    /// Consumes leading and trailing spaces while also consuming newlines.
    /// Also allows variables at the end of the file where no newline is found.
    /// Calls [`Value::parse_next_value`] for variable parsing.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`Value::Single`] can be recognized in `input`.
    fn single_till_newline(input: &mut &str) -> ModalResult<Self> {
        cut_err(delimited(
            space1,
            Self::parse_next_value,
            (space0, alt(("\n", eof))),
        ))
        .context(StrContext::Label("variable"))
        .map(Value::Single)
        .parse_next(input)
    }

    /// Recognizes a [`Value::Array`] in a string slice while handling the surroundings.
    ///
    /// Consumes leading and trailing spaces while also consuming newlines.
    /// Also allows variables at the end of the file where no newline is found.
    /// Calls [`Self::parse_next_value`] for variable parsing.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`Value::Array`] is found in `input`.
    fn list_till_newline(input: &mut &str) -> ModalResult<Self> {
        let values = repeat(0.., preceded(space1, Value::parse_next_value))
            .map(Value::Array)
            .parse_next(input)?;

        // Make sure we find the end of the line after the last element in the list.
        cut_err(preceded(space0, alt(("\n", eof))))
            .context(StrContext::Label("character"))
            .context(StrContext::Expected(StrContextValue::Description(
                "end of line or end of file.",
            )))
            .parse_next(input)?;

        Ok(values)
    }

    /// Recognizes a [`Value::Single`] in a string slice.
    ///
    /// Calls [`Value::variable_character`] to handle escaped characters in `input`.
    ///
    /// # Examples
    ///
    /// ```text
    /// "This is a string value\" with escaped \\ characters"
    /// ```
    ///
    /// # Errors
    ///
    /// Returns an error if no [`Value::Single`] is found in `input`.
    pub fn parse_next_value(input: &mut &str) -> ModalResult<String> {
        // We expect a value surrounded with double quotes (") that may contain escaped characters.
        // As we resolve the escaping, we cannot just `.take()`. Instead we have to accumulate in
        // a `String`.
        let string_value = trace(
            "variable",
            preceded(
                '"',
                cut_err(
                    terminated(
                        repeat(0.., Self::variable_character).fold(String::new, |mut string, c| {
                            string.push(c);
                            string
                        }),
                        '"',
                    )
                    .context(StrContext::Label("variable"))
                    .context(StrContext::Expected(
                        StrContextValue::Description("A string surrounded by double quotes"),
                    )),
                ),
            ),
        )
        .parse_next(input)?;

        Ok(string_value)
    }

    /// Recognizes a single [`char`] in a string slice.
    ///
    /// This allows for detecting escaped characters, such as `\"` or `\\`.
    ///
    /// Consumes a single character and returns afterwards.
    ///
    /// # Errors
    ///
    /// Returns an error if a stand-alone `"` or `\` is encountered.
    pub fn variable_character(input: &mut &str) -> ModalResult<char> {
        let c = none_of('"').parse_next(input)?;
        if c == '\\' {
            cut_err(one_of(['"', '\\']))
                .context(StrContext::Label("escaped sequence"))
                .context(StrContext::Expected(StrContextValue::Description(
                    "one of the allowed escape characters: ['\"', '\\']",
                )))
                .parse_next(input)
        } else {
            Ok(c)
        }
    }
}

/// Represents a potentially cleared or overridden value.
///
/// This type is used for values in `package` sections where values from a `pkgbase` section might
/// be cleared or overridden.
#[derive(Clone, Debug)]
pub enum ClearableValue {
    /// A single value.
    Single(Option<String>),
    /// An array of values.
    Array(Option<Vec<String>>),
}

impl ClearableValue {
    /// Recognizes a [`ClearableValue::Single`] while handling the surroundings.
    ///
    /// Cleared values are represented by non-existing values.
    ///
    /// Consumes leading and trailing spaces while also consuming newlines.
    /// Also allows variables at the end of the file where no newline is found.
    /// Calls [`Value::parse_next_value`] for variable parsing.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`ClearableValue::Single`] is found in `input`.
    fn single_till_newline(input: &mut &str) -> ModalResult<Self> {
        let value = preceded(
            space1,
            terminated(opt(Value::parse_next_value), (space0, alt(("\n", eof)))),
        )
        .parse_next(input)?;

        let Some(value) = value else {
            return Ok(ClearableValue::Single(None));
        };

        // Returns a "Cleared" value if the value is an empty string.
        if value.is_empty() {
            return Ok(ClearableValue::Single(None));
        }

        Ok(ClearableValue::Single(Some(value)))
    }

    /// Recognizes a [`ClearableValue::Array`] while handling the surroundings.
    ///
    /// Cleared values are represented by the non-existance of a value.
    ///
    /// Consumes leading and trailing spaces while also consuming newlines.
    /// Also allows variables at the end of the file where no newline is found.
    /// Calls [`Value::parse_next_value`] for variable parsing.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`ClearableValue::Array`] is found in `input`.
    fn list_till_newline(input: &mut &str) -> ModalResult<Self> {
        let values = opt(repeat(1.., preceded(space1, Value::parse_next_value)))
            .map(ClearableValue::Array)
            .parse_next(input)?;

        // Make sure we find the end of the line after the last element in the list.
        cut_err(preceded(space0, alt(("\n", eof))))
            .context(StrContext::Label("character"))
            .context(StrContext::Expected(StrContextValue::Description(
                "end of line or end of file.",
            )))
            .parse_next(input)?;

        Ok(values)
    }
}

/// A keyword (with an optional [`Architecture`] suffix) of a variable found in the output of the
/// `alpm-pkgbuild-bridge` script.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct Keyword {
    /// The name of the keyword.
    pub keyword: String,
    /// The optional `_$suffix` that indicates a specific architecture.
    pub suffix: Option<String>,
}

impl Display for Keyword {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.keyword)?;
        if let Some(suffix) = &self.suffix {
            write!(f, "_{suffix}")?;
        }

        Ok(())
    }
}

impl Keyword {
    /// Creates a [`Keyword`] without an [`Architecture`] suffix from a string.
    pub fn simple<T: ToString>(keyword: T) -> Self {
        Self {
            keyword: keyword.to_string(),
            suffix: None,
        }
    }

    /// Recognizes a [`Keyword`] in a variable declaration line in the output of the
    /// `alpm-pkgbuild-bridge` script.
    ///
    /// This parser is aware of its surrounding and consumes preceding spaces.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`Keyword`] can be found in `input`.
    pub(crate) fn parser(input: &mut &str) -> ModalResult<Keyword> {
        let (keyword, suffix) = trace(
            "keyword",
            cut_err(preceded(
                space1,
                (
                    alphanumeric1,
                    opt(preceded('_', take_till(1.., |c| c == ' ' || c == '\n'))),
                ),
            ))
            .context(StrContext::Label(
                "keyword with potential architecture suffix, e.g. 'source_x86_64'",
            ))
            .context(StrContext::Expected(StrContextValue::Description(
                "alphabetic keyword with potential architecture suffix, e.g. 'source_x86_64'",
            ))),
        )
        .parse_next(input)?;
        Ok(Keyword {
            keyword: keyword.to_owned(),
            suffix: suffix.map(ToString::to_string),
        })
    }
}

/// A type of variable found in the output of the `alpm-pkgbuild-bridge` script.
#[derive(Debug, EnumString, VariantNames)]
#[strum(serialize_all = "UPPERCASE")]
enum VariableType {
    Array,
    String,
}

impl VariableType {
    /// Recognizes a [`VariableType`] in a variable declaration line in the output of the
    /// `alpm-pkgbuild-bridge` script.
    ///
    /// This parser is aware of its surrounding and consumes preceding spaces.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`VariableType`] is found in `input`.
    pub fn parser(input: &mut &str) -> ModalResult<VariableType> {
        trace(
            "variable_type",
            cut_err(preceded(
                space1,
                take_until(1.., ' ').try_map(VariableType::from_str),
            ))
            .context_with(iter_str_context!([VariableType::VARIANTS])),
        )
        .parse_next(input)
    }
}

/// A raw, unchecked package name.
///
/// Functions and variables may be specific for a package in a [`PKGBUILD`].
///
/// If a [`PKGBUILD`] only defines a single package, the package's `package` function will be named
/// `package`. This is represented by `RawPackageName(None)`.
///
/// If a [`PKGBUILD`] defines one or more [alpm-split-package]s, there are as many custom `package`
/// functions as there are split packages. Here, each function is named `package_<name>`, where
/// `<name>` denotes an [alpm-package-name]. This is represented by `RawPackageName(Some(name))`.
///
/// [`PKGBUILD`]: https://man.archlinux.org/man/PKGBUILD.5
/// [alpm-package-name]: https://alpm.archlinux.page/specifications/alpm-package-name.7.html
/// [alpm-split-package]: https://alpm.archlinux.page/specifications/alpm-split-package.7.html
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct RawPackageName(pub Option<String>);

impl RawPackageName {
    /// Recognizes a [`RawPackageName`] in the output of the `alpm-pkgbuild-bride` script.
    ///
    /// This parser is aware of its surrounding and consumes preceding spaces and until the eol.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`RawPackageName`] is found in `input`.
    pub(crate) fn parser(input: &mut &str) -> ModalResult<Self> {
        let package_name = trace(
            "PackageName",
            cut_err(preceded(
                (space1, "package"),
                opt(preceded('_', take_till(1.., |c| c == ' ' || c == '\n'))),
            ))
            .context(StrContext::Expected(StrContextValue::Description(
                "A 'package' function or a 'package_split-package-name' function.",
            )))
            .map(|opt| opt.map(ToString::to_string)),
        )
        .parse_next(input)?;

        Ok(Self(package_name))
    }
}

/// Represents the raw parsed, but not yet typed output of the [`alpm-pkgbuild-bridge`] script.
///
/// [`alpm-pkgbuild-bridge`]: https://gitlab.archlinux.org/archlinux/alpm/alpm-pkgbuild-bridge
#[derive(Clone, Debug)]
pub struct BridgeOutput {
    /// The map of all assigned keywords in the `pkgbase` section of the bridge output.
    pub package_base: HashMap<Keyword, Value>,
    /// The map of all assigned keywords in the `package` section of the bridge output.
    /// The keywords are grouped by package name.
    pub packages: HashMap<RawPackageName, HashMap<Keyword, ClearableValue>>,
    /// The list of all package function names that are declared in the bridge output.
    pub functions: Vec<RawPackageName>,
}

impl BridgeOutput {
    /// Creates a [`BridgeOutput`] from a [`PKGBUILD`] at a given path, by calling the
    /// [`alpm-pkgbuild-bridge`] script.
    ///
    /// [`PKGBUILD`]: https://man.archlinux.org/man/PKGBUILD.5
    /// [`alpm-pkgbuild-bridge`]: https://gitlab.archlinux.org/archlinux/alpm/alpm-pkgbuild-bridge
    pub fn from_file(pkgbuild_path: &Path) -> Result<Self, Error> {
        let input = run_bridge_script(pkgbuild_path)?;
        Self::from_script_output(&input)
    }

    /// Creates a [`BridgeOutput`] from some [`alpm-pkgbuild-bridge`] script output.
    ///
    /// This function is mostly exposed for testing, consider using [`Self::from_file`].
    ///
    /// [`alpm-pkgbuild-bridge`]: https://gitlab.archlinux.org/archlinux/alpm/alpm-pkgbuild-bridge
    pub fn from_script_output(input: &str) -> Result<Self, Error> {
        Self::parser
            .parse(input)
            .map_err(|err| Error::BridgeParseError(format!("{err}")))
    }

    /// Parse some given [alpm-pkgbuild-bridge] script output into `Self`.
    ///
    /// Use [`Self::from_script_output`] for convenient error handling.
    /// Recognizes a [`BridgeOutput`] in the output of the `alpm-pkgbuild-bridge` script.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`BridgeOutput`] can be found in `input`.
    ///
    /// [alpm-pkgbuild-bridge]: https://gitlab.archlinux.org/archlinux/alpm/alpm-pkgbuild-bridge
    fn parser(input: &mut &str) -> ModalResult<Self> {
        let package_base = Self::package_base(input)?;
        let packages = Self::packages(input)?;
        let functions = Self::functions(input)?;

        // Consume any trailing empty lines
        let _: Option<()> = opt(repeat(0.., (space0, "\n", space0))).parse_next(input)?;
        // Make sure we reached the end of the file.
        cut_err(eof)
            .context(StrContext::Expected(StrContextValue::Description(
                "end of file.",
            )))
            .parse_next(input)?;

        Ok(Self {
            package_base,
            packages,
            functions,
        })
    }

    /// Recognizes a map of [`Keyword`]-[`Value`] pairs in a string slice.
    ///
    /// Backtracks as soon as the next section is hit.
    ///
    /// # Errors
    ///
    /// Returns an error if there is not at least one [`Keyword`]-[`Value`] pair in `input`.
    fn package_base(input: &mut &str) -> ModalResult<HashMap<Keyword, Value>> {
        // We don't have to check for duplicates, and directly convert the iterator into a HashMap
        // as the bridge guarantees that variables are only declared once per keyword.
        repeat(1.., Self::package_base_line).parse_next(input)
    }

    /// Recognizes a [`Keyword`]-[`Value`] pair in a string slice.
    ///
    /// Backtracks as soon any other line prefix than `VAR GLOBAL` is encountered.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`Keyword`]-[`Value`] pair can be found in `input`.
    fn package_base_line(input: &mut &str) -> ModalResult<(Keyword, Value)> {
        // Parse the start of the line.
        //
        // The start `VAR GLOBAL` may backtrack. This indicates that the next section has been hit.
        // **Everything else** will return a hard error, as it indicates invalid or unexpected
        // bridge output.
        ("VAR GLOBAL").parse_next(input)?;

        let variable_type = VariableType::parser.parse_next(input)?;

        // Get the keyword with an optional suffix.
        let keyword = Keyword::parser.parse_next(input)?;

        let value = match variable_type {
            VariableType::Array => Value::list_till_newline(input)?,
            VariableType::String => Value::single_till_newline(input)?,
        };

        Ok((keyword, value))
    }

    /// Recognizes an entire section of package variable declarations.
    ///
    /// Delegates to [`BridgeOutput::package_line`] to recognize singular tuples of
    /// [`RawPackageName`], [`Keyword`] and [`ClearableValue`] per line and combines the data in a
    /// [`HashMap`].
    ///
    /// Backtracks as soon as the next section is hit.
    ///
    /// # Errors
    ///
    /// Returns an error if [`BridgeOutput::package_line`] does not find at least one line in
    /// `input`.
    fn packages(
        input: &mut &str,
    ) -> ModalResult<HashMap<RawPackageName, HashMap<Keyword, ClearableValue>>> {
        let lines: Vec<(RawPackageName, Keyword, ClearableValue)> =
            repeat(0.., Self::package_line).parse_next(input)?;

        let mut packages = HashMap::new();
        for (package_name, keyword, value) in lines {
            let value_map: &mut HashMap<Keyword, ClearableValue> =
                packages.entry(package_name).or_default();
            // We don't have to check for duplicates, as the bridge guarantees that variables are
            // only declared once per keyword.
            value_map.insert(keyword, value);
        }

        Ok(packages)
    }

    /// Recognizes a tuple of [`RawPackageName`], [`Keyword`] and [`ClearableValue`] in a string
    /// slice.
    ///
    /// The tuple represents a single package variable line.
    /// Backtracks as soon any other line prefix than `FUNCTION` is encountered.
    ///
    /// # Errors
    ///
    /// Returns an error if one of the components ([`RawPackageName`], [`Keyword`] and
    /// [`ClearableValue`]) is not found in `input`.
    fn package_line(input: &mut &str) -> ModalResult<(RawPackageName, Keyword, ClearableValue)> {
        // Parse the start of the line.
        //
        // The start `VAR FUNCTION` may backtrack. This indicates that the next section has been
        // hit. **Everything else** will return a hard error, as it indicates invalid or
        // unexpected bridge output.
        ("VAR FUNCTION").parse_next(input)?;

        let package_name = RawPackageName::parser.parse_next(input)?;

        let variable_type = VariableType::parser.parse_next(input)?;

        // Get the keyword with an optional suffix.
        let keyword = Keyword::parser.parse_next(input)?;

        let value = match variable_type {
            VariableType::Array => ClearableValue::list_till_newline(input)?,
            VariableType::String => ClearableValue::single_till_newline(input)?,
        };

        Ok((package_name, keyword, value))
    }

    /// Recognizes a list of [`RawPackageName`]s in a string slice.
    ///
    /// Delegates to [`BridgeOutput::function_line`] to recognize each singular [`RawPackageName`].
    /// This parser may return an empty array, which is the case for [alpm-meta-packages] without a
    /// `package` function.
    ///
    /// # Errors
    ///
    /// Returns an error if [`BridgeOutput::function_line`] fails.
    ///
    /// [alpm-meta-packages]: https://alpm.archlinux.page/specifications/alpm-meta-package.7.html
    fn functions(input: &mut &str) -> ModalResult<Vec<RawPackageName>> {
        repeat(0.., Self::function_line).parse_next(input)
    }

    /// Recognizes a [`RawPackageName`] in a string slice.
    ///
    /// # Errors
    ///
    /// Returns an error if no [`RawPackageName`] can be found in `input`.
    fn function_line(input: &mut &str) -> ModalResult<RawPackageName> {
        // Parse the start of the line.
        //
        // The start `FUNCTION` may backtrack. This indicates that the next section has been
        // hit. **Everything else** will return a hard error, as it indicates invalid or
        // unexpected bridge output.
        ("FUNCTION").parse_next(input)?;
        let package_name = RawPackageName::parser.parse_next(input)?;

        cut_err((space0, alt((eof, "\n"))))
            .context(StrContext::Label("character"))
            .context(StrContext::Expected(StrContextValue::Description(
                "end of line or end of file.",
            )))
            .parse_next(input)?;

        Ok(package_name)
    }
}
