"""Define Gaussian calculation parameters."""

import math
from typing import ClassVar
from typing import TypeVar

from autojob.coordinator import job

S = TypeVar("S", bound="EspressoJob")


class EspressoJob(job.Job):
    """A class to represent a Espresso job."""

    FILES: ClassVar[list[tuple[str, bool]]] = [
        ("pwscf.save", False),  # filename, required?
    ]

    @staticmethod
    def _create_input_params() -> list[job.CalculationParameter]:
        # Namelist: &CONTROL
        control = {
            "calculation": job.CalculationParameter(
                name="calculation",
                explicit=True,
                default="scf",
                allowed_types=[str],
                values=(
                    "scf",
                    "nscf",
                    "relax",
                    "bands",
                    "md",
                    "vc-relax",
                    "vc-md",
                ),
                description="A string describing the task to be performed.",
            ),
            "verbosity": job.CalculationParameter(
                name="verbosity",
                explicit=True,
                default="low",
                allowed_types=[str],
                values=("high", "low"),
                description="Control how much information is written to the output.",
            ),
            "restart_mode": job.CalculationParameter(
                name="restart_mode",
                explicit=True,
                default="from_scratch",
                allowed_types=[str],
                values=("from_scratch", "restart"),
                description="Control how to restart the calculation.",
            ),
            "nstep": job.CalculationParameter(
                name="nstep",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="number of molecular-dynamics or structural "
                "optimization steps performed in this run.",
            ),
            "tstress": job.CalculationParameter(
                name="tstress",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                default=False,
                description="Whether or not to calculate stress.",
            ),
            "tprnfor": job.CalculationParameter(
                name="tprnfor",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                default=False,
                description="Whether or not to calculate forces.",
            ),
            "dt": job.CalculationParameter(
                name="dt",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="time step for molecular dynamics, in Rydberg atomic "
                "units.",
            ),
            "max_seconds": job.CalculationParameter(
                name="max_seconds",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Jobs stops after max_seconds CPU time.",
            ),
            "etot_conv_thr": job.CalculationParameter(
                name="etot_conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold on total energy (a.u) for "
                "ionic minimization.",
            ),
            "forc_conv_thr": job.CalculationParameter(
                name="forc_conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold on forces (a.u) for "
                "ionic minimization.",
            ),
            "disk_io": job.CalculationParameter(
                name="disk_io",
                explicit=True,
                allowed_types=[str],
                values=("high", "medium", "low", "nowf", "minimal", "none"),
                description="Specifies the amount of disk I/O activity.",
            ),
            "tefield": job.CalculationParameter(
                name="tefield",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to add a saw-like potential "
                "simulating an electric field to the bare ionic potential.",
            ),
            "dipfield": job.CalculationParameter(
                name="dipfield",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to add a dipole correction "
                "to the bare ionic potential.",
            ),
            "lelfield": job.CalculationParameter(
                name="lelfield",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to add a homogeneous finite electric "
                "field described through the modern theory of the polarization "
                "dipole correction to the bare ionic potential.",
            ),
            "lfcp": job.CalculationParameter(
                name="lfcp",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to perform a constant bias potential "
                "(constant-mu) calculation for a system with ESM method.",
            ),
            "trism": job.CalculationParameter(
                name="trism",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to perform a 3D-RISM-SCF calculation.",
            ),
        }

        # Namelist: &SYSTEM
        system = {
            "tot_charge": job.CalculationParameter(
                name="tot_charge",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Total charge of the system.",
            ),
            "tot_magnetization": job.CalculationParameter(
                name="tot_magnetization",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Total magnetization of the system.",
            ),
            "ecutwfc": job.CalculationParameter(
                name="ecutwfc",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="kinetic energy cutoff (Ry) for wavefunctions.",
            ),
            "ecutrho": job.CalculationParameter(
                name="ecutrho",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Kinetic energy cutoff (Ry) for charge density and "
                "potential for norm-conserving pseudopotential.",
            ),
            "ecutfock": job.CalculationParameter(
                name="ecutrho",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Kinetic energy cutoff (Ry) for exact exchange "
                "operator in EXX type calculations.",
            ),
            "nosym": job.CalculationParameter(
                name="nosym",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to prevent symmetry from being used.",
            ),
            "occupations": job.CalculationParameter(
                name="occupations",
                explicit=True,
                allowed_types=[str],
                values=(
                    "smearing",
                    "tetrahedra",
                    "tetrahedra_lin",
                    "tetrahedra_opt",
                    "fixed",
                    "from_input",
                ),
                description="How to set orbital occupations.",
            ),
            "smearing": job.CalculationParameter(
                name="smearing",
                explicit=True,
                allowed_types=[str],
                values=(
                    "gaussian",
                    "methfessel-paxton",
                    "marzari-vanderbilt",
                    "fermi-dirac",
                ),
                description="Specifies the smearing method used if 'occupations' "
                "is set to 'smearing'.",
            ),
            "degauss": job.CalculationParameter(
                name="degauss",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="value of the gaussian spreading (Ry) for brillouin-"
                "zone integration in metals.",
            ),
            "nspin": job.CalculationParameter(
                name="nspin",
                explicit=True,
                allowed_types=[int],
                values=(1, 2, 4),
                description="Specify the treatment of spin-polarization.",
            ),
            "noncolin": job.CalculationParameter(
                name="noncolin",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to perform a noncollinear calculation.",
            ),
            "ecfixed": job.CalculationParameter(
                name="ecfixed",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="the value (in Rydberg) of the constant-cutoff "
                "to be used in variable-cell molecular dynamics (or in stress "
                "calculation).",
            ),
            "qcutz": job.CalculationParameter(
                name="qcutz",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="the height (in Rydberg) of the energy step for "
                "reciprocal vectors whose square modulus is greater than "
                "'ecfixed'to be used in variable-cell molecular dynamics (or "
                "in stress calculation).",
            ),
            "q2sigma": job.CalculationParameter(
                name="q2sigma",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="the width (in Rydberg) of the energy step for "
                "reciprocal vectors whose square modulus is greater than "
                "'ecfixed'to be used in variable-cell molecular dynamics (or "
                "in stress calculation).",
            ),
            "input_dft": job.CalculationParameter(
                name="input_dft",
                explicit=False,
                allowed_types=[str],
                values=(None, None, None),
                description="Exchange-correlation functional.",
            ),
            "ace": job.CalculationParameter(
                name="ace",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to use the adaptively compressed "
                "exchange operator.",
            ),
            "exx_fraction": job.CalculationParameter(
                name="exx_fraction",
                explicit=False,
                allowed_types=[float],
                values=(0, 1, "[]"),
                description="Fraction of EXX for hybrid functional "
                "calculations.",
            ),
            "screening_parameter": job.CalculationParameter(
                name="screening_parameter",
                explicit=False,
                allowed_types=[float],
                values=(0, 1, "[]"),
                description="screening_parameter for HSE like hybrid "
                "functionals.",
            ),
            "ensemble_energies": job.CalculationParameter(
                name="ensemble_energies",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not an ensemble of xc energies is "
                "calculated non-selfconsistently for perturbed exchange-"
                "enhancement factors and LDA vs. PBE correlation ratios "
                "after each converged electronic ground state calculation.",
            ),
            "edir": job.CalculationParameter(
                name="edir",
                explicit=True,
                allowed_types=[int],
                values=(1, 2, 3),
                description="The lattice vector to which the electric field or "
                "dipole correction is parallel.",
            ),
            "emaxpos": job.CalculationParameter(
                name="emaxpos",
                explicit=False,
                allowed_types=[float],
                values=(0, 1, "[]"),
                description="Position of the maximum of the saw-like potential "
                "along crystal axis edir, within the  unit cell.",
            ),
            "eopreg": job.CalculationParameter(
                name="eopreg",
                explicit=False,
                allowed_types=[float],
                values=(0, 1, "[]"),
                description="Zone in the unit cell where the saw-like potential "
                "decreases.",
            ),
            "eamp": job.CalculationParameter(
                name="eamp",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Amplitude of the electric field, in Hartree a.u..",
            ),
            "assume_isolated": job.CalculationParameter(
                name="assume_isolated",
                explicit=True,
                allowed_types=[str],
                values=(
                    "none",
                    "makov-payne",
                    "martyna-tuckerman",
                    "esm",
                    "2D",
                ),
                description="Used to perform calculation assuming the system "
                "to be isolated.",
            ),
            "esm_bc": job.CalculationParameter(
                name="esm_bc",
                explicit=True,
                allowed_types=[str],
                values=(
                    "pbc",
                    "bc1",
                    "bc2",
                    "bc3",
                ),
                description="Used to perform calculation assuming the system "
                "to be isolated.",
            ),
            "esm_w": job.CalculationParameter(
                name="esm_w",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="determines the position offset [in a.u.] of the "
                "start of the effective screening region, measured relative "
                "to the cell edge.",
            ),
            "esm_efield": job.CalculationParameter(
                name="esm_efield",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Gives the magnitude of the electric field "
                "[Ry/a.u.] to be applied between semi-infinite ESM "
                "electrodes.",
            ),
            "esm_nfit": job.CalculationParameter(
                name="esm_nfit",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="Gives the number of z-grid points for the "
                "polynomial fit along the cell edge.",
            ),
            "lgcscf": job.CalculationParameter(
                name="lgcscf",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to perform a constant bias "
                "potential (constant-mu) calculation with Grand-Canonical "
                "SCF.",
            ),
            "gcscf_mu": job.CalculationParameter(
                name="gcscf_mu",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="The target Fermi energy (eV) of GC-SCF.",
            ),
            "gcscf_conv_thr": job.CalculationParameter(
                name="gcscf_conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold of Fermi energy (eV) for "
                "GC-SCF.",
            ),
            "gcscf_beta": job.CalculationParameter(
                name="gcscf_beta",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Mixing factor for GC-SCF.",
            ),
            "vdw_corr": job.CalculationParameter(
                name="vdw_corr",
                explicit=True,
                allowed_types=[str],
                values=(
                    "none",
                    "grimme-d2",
                    "grimme-d3",
                    "tkatchenko-scheffler",
                    "many-body-dispersion",
                    "xdm",
                ),
                description="Type of Van der Waals correction.",
            ),
            "dftd3_version": job.CalculationParameter(
                name="dftd3_version",
                explicit=True,
                allowed_types=[int],
                values=(2, 3, 4, 5, 6),
                description="Version of Grimme implementation of Grimme-D3.",
            ),
            "dftd3_threebody": job.CalculationParameter(
                name="dftd3_threebody",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to turn three-body terms in "
                "Grimme-D3 on.",
            ),
        }
        # Namelist: &ELECTRONS
        electrons = {
            "electron_maxstep": job.CalculationParameter(
                name="electron_maxstep",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="maximum number of iterations in a scf step.",
            ),
            "exx_maxstep": job.CalculationParameter(
                name="exx_maxstep",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="maximum number of outer iterations in a scf "
                "calculation with exact exchange.",
            ),
            "scf_must_converge": job.CalculationParameter(
                name="scf_must_converge",
                explicit=True,
                allowed_types=[bool],
                values=(True, False),
                description="Whether or not to continue MD or ionic "
                "relaxation when electron_maxstep is reached.",
            ),
            "conv_thr": job.CalculationParameter(
                name="conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold for selfconsistency.",
            ),
            "mixing_mode": job.CalculationParameter(
                name="mixing_mode",
                explicit=True,
                allowed_types=[str],
                values=(
                    "plain",
                    "TF",
                    "local-TF",
                ),
                description="Type of density mixing used.",
            ),
            "mixing_beta": job.CalculationParameter(
                name="mixing_beta",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="mixing factor for self-consistency.",
            ),
            "mixing_ndim": job.CalculationParameter(
                name="mixing_ndim",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="Number of iterations used in mixing scheme.",
            ),
            "diagonalization": job.CalculationParameter(
                name="diagonalization",
                explicit=True,
                allowed_types=[str],
                values=(
                    "david",
                    "cg",
                    "ppcg",
                    "paro",
                    "rmm-davidson",
                ),
                description="Type of density mixing used.",
            ),
            "efield": job.CalculationParameter(
                name="efield",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Amplitude of the finite electric field.",
            ),
        }
        # Namelist: &IONS
        ions = {
            "ion_positions": job.CalculationParameter(
                name="ion_positions",
                explicit=True,
                allowed_types=[str],
                values=(
                    "from_input",
                    "default",
                ),
                description="From where to read starting atomic positions.",
            ),
            "ion_velocities": job.CalculationParameter(
                name="ion_velocities",
                explicit=True,
                allowed_types=[str],
                values=(
                    "from_input",
                    "default",
                ),
                description="From where to read starting atomic velocities.",
            ),
            "ion_dynamics": job.CalculationParameter(
                name="ion_dynamics",
                explicit=True,
                allowed_types=[str],
                values=(
                    "bfgs",
                    "fire",
                    "damp",
                    "verlet",
                    "langevin",
                    "langevin-smc",
                    "beeman",
                ),
                description="Specify the type of ionic dynamics. Note that "
                "not all values of valid for every type of calculation.",
            ),
            "ion_temperature": job.CalculationParameter(
                name="ion_temperature",
                explicit=True,
                allowed_types=[str],
                values=(
                    "rescaling",
                    "rescale-v",
                    "rescale-T",
                    "reduce-T",
                    "berendsen",
                    "andersen",
                    "svr",
                    "initial",
                    "not_controlled",
                ),
                description="The temperature controller used for molecular "
                "dynamics.",
            ),
            "tempw": job.CalculationParameter(
                name="tempw",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Starting temperature (Kelvin) in MD runs target "
                "temperature for most thermostats.",
            ),
            "delta_t": job.CalculationParameter(
                name="delta_t",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Temperature rescaling parameter.",
            ),
        }
        # Namelist: &CELL
        cell = {
            "cell_dynamics": job.CalculationParameter(
                name="cell_dynamics",
                explicit=True,
                allowed_types=[str],
                values=(
                    "none",
                    "sd",
                    "damp-pr",
                    "damp-w",
                    "bfgs",
                    "pr",
                    "w",
                ),
                description="Specify the type of dynamics for the cell.",
            ),
            "press": job.CalculationParameter(
                name="press",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Target pressure [KBar] in a variable-cell md or "
                "relaxation run.",
            ),
            "wmass": job.CalculationParameter(
                name="wmass",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Fictitious cell mass [amu] for variable-cell "
                "simulations.",
            ),
            "press_conv_thr": job.CalculationParameter(
                name="press_conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold on the pressure for "
                "variable cell relaxation.",
            ),
            "cell_dofree": job.CalculationParameter(
                name="cell_dofree",
                explicit=True,
                allowed_types=[str],
                values=(
                    "all",
                    "ibrav",
                    "a",
                    "b",
                    "c",
                    "fixa",
                    "fixb",
                    "fixc",
                    "x",
                    "y",
                    "z",
                    "xy",
                    "xz",
                    "yz",
                    "xyz",
                    "shape",
                    "volume",
                    "2Dxy",
                    "2Dshape",
                    "epitaxial_ab",
                    "epitaxial_acepitaxial_bc",
                ),
                description="Specify the type of dynamics for the cell.",
            ),
        }
        # Namelist: &FCP
        fcp = {
            "fcp_mu": job.CalculationParameter(
                name="fcp_mu",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="The target Fermi energy (eV).",
            ),
            "fcp_dynamics": job.CalculationParameter(
                name="fcp_dynamics",
                explicit=True,
                allowed_types=[str],
                values=(
                    "bfgs",
                    "newton",
                    "damp",
                    "verlet",
                    "velocity-verlet",
                    "lm",
                ),
                description="Specify the type of dynamics for the Fictitious "
                "Charge Particle (FCP).",
            ),
            "fcp_conv_thr": job.CalculationParameter(
                name="fcp_conv_thr",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Convergence threshold on force (eV) for FCP "
                "relaxation.",
            ),
            "fcp_mass": job.CalculationParameter(
                name="fcp_mass",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Mass of the FCP.",
            ),
            "fcp_velocity": job.CalculationParameter(
                name="fcp_velocity",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Initial velocity of the FCP.",
            ),
            "fcp_temperature": job.CalculationParameter(
                name="fcp_temperature",
                explicit=True,
                allowed_types=[str],
                values=(
                    "rescaling",
                    "rescale-v",
                    "rescale-T",
                    "reduce-T",
                    "berendsen",
                    "andersen",
                    "initial",
                    "not_controlled",
                ),
                description="The temperature controller used for the FCP ",
            ),
            "fcp_tempw": job.CalculationParameter(
                name="fcp_tempw",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Starting temperature (Kelvin) in FCP dynamics "
                "runs target temperature for most thermostats.",
            ),
            "fcp_tolp": job.CalculationParameter(
                name="fcp_tolp",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Tolerance for velocity rescaling.",
            ),
            "fcp_delta_t": job.CalculationParameter(
                name="fcp_delta_t",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Temperature rescaling parameter for FCP.",
            ),
        }
        # Namelist: &RISM
        rism = {
            "nsolv": job.CalculationParameter(
                name="nsolv",
                explicit=False,
                allowed_types=[int],
                values=(0, math.inf, "[)"),
                description="Number of solvents in the unit cell.",
            ),
            "closure": job.CalculationParameter(
                name="closure",
                explicit=True,
                allowed_types=[str],
                values=(
                    "kh",
                    "hnc",
                ),
                description="Specify the type of closure equation.",
            ),
            "tempv": job.CalculationParameter(
                name="tempv",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Temperature (Kelvin) of solvents.",
            ),
            "ecutsolv": job.CalculationParameter(
                name="ecutsolv",
                explicit=False,
                allowed_types=[float],
                values=(0, math.inf, "[)"),
                description="Kinetic energy cutoff (Ry) for solvent's "
                "correlation functions.",
            ),
        }

        return [
            *control.values(),
            *system.values(),
            *electrons.values(),
            *ions.values(),
            *cell.values(),
            *fcp.values(),
            *rism.values(),
        ]

    # ? Convert to abstract static property
    @staticmethod
    def input_parameters() -> list[job.CalculationParameter]:
        """Generate Gaussian job input parameters."""
        return EspressoJob._input_parameters.copy()

    _input_parameters = _create_input_params()
