"""Utilities for filtering parameter dictionaries by their values.

Example:
    >>> from autojob.coordinator.filters import Filter
    >>> from autojob.parametrizations import VariableReference
    >>> flt = Filter(
    ...     source=VariableReference(set_path=[], get_path=("inputs", "filename")),
    ...     values=("a", "b"),
    ... )
    >>> value1 = {"inputs": {"filename": "A"}}
    >>> value2 = {"inputs": {"filename": "b"}}
    >>> flt.apply(value1)
    False
    >>> flt.apply(value2)
    True
"""

from typing import TYPE_CHECKING
from typing import Any
from typing import NamedTuple

from autojob.parametrizations import VariableReference

if TYPE_CHECKING:
    from autojob.coordinator.gui.groups import SubmissionParameterGroup

STRUCTURE_FILENAME_PATH = ("task_inputs", "atoms", "info", "filename")


class Filter(NamedTuple):
    """A condition that must be satisfied.

    Attributes:
        source: A :class:`~autojob.parametrizations.VariableReference`
            indicating which input variable to reference for the comparison.
        values: The allowed values for the variable.
    """

    source: VariableReference
    values: tuple[Any]

    def apply(self, value: dict[str, Any] | object) -> bool:
        """Determine if the value passes the filter.

        Args:
            value: The value to which the filter will be applied.

        Returns:
            True if the value passes the filter. False otherwise.
        """
        return self.source.evaluate(value) in self.values


def submission_group_to_filters(
    group: "SubmissionParameterGroup",
) -> list[Filter]:
    """Convert a SubmissionParameterGroup to :class:`Filter` instances.

    Args:
        group: The group to convert.

    Returns:
        The :class:`autojob.parametrizations.VariableReference`.
    """
    # Add structures to filter
    structure_ref: VariableReference = VariableReference(
        set_path=STRUCTURE_FILENAME_PATH, get_path=STRUCTURE_FILENAME_PATH
    )
    structures = list(group.values)
    filters = [Filter(source=structure_ref, values=structures)]

    # Create prefix for calculation and task inputs
    calculation_inputs_prefix = ("calculation_inputs", "parameters")

    def merge_parameter_group(
        group: "SubmissionParameterGroup",
    ) -> dict[str, Any]:
        parameters: dict[str, list[Any]] = {}

        for structure in group.values:
            for parameter, values in group.values[structure].items():
                if parameter not in parameters:
                    parameters[str(parameter)] = []

                parameters[str(parameter)].extend(values)

        return parameters

    parameters = merge_parameter_group(group)

    # Complete get_path from calculator parameters
    for parameter, values in parameters.items():
        path = (*calculation_inputs_prefix, parameter)
        f = Filter(
            source=VariableReference(set_path=path, get_path=path),
            values=values,
        )
        filters.append(f)

    return filters
