"""Read and standardize Quantum Espresso outputs."""

import logging
from pathlib import Path
from typing import Any

from ase import Atoms
import ase.io
from ase.io.espresso import read_espresso_out
from ase.io.formats import UnknownFileTypeError

from autojob import SETTINGS
from autojob.utils.atoms import copy_atom_metadata

logger = logging.getLogger(__name__)

ALTERNATE_OUTPUT_STRUCTURES = ("relax.traj", "pwscf.save")
FILES_TO_CARRYOVER = ["pwscf.save"]
PWSCF_LOG = "pwscf.save"


# TODO: Set converged correctly
def load_calculation_results(
    src: str | Path,
) -> dict[str, Any]:
    """Load calculation outputs for a Quantum Espresso calculation.

    Note that all quantities other than the final energy are reported in atomic
    units (Hartree/Bohr).

    Args:
        src: The directory containing the Espresso (PWscf) output files.

    Returns:
        A dictionary containing Espresso calculation outputs.

    Warning:
        That a calculation has converged must be confirmed manually as the
        value of the ``"converged"`` key is always set to False.
    """
    logger.debug(
        "Loading calculation outputs for Gaussian calculation in directory: %s",
        src,
    )
    log_file = Path(src, PWSCF_LOG)
    results = {"forces": None, "energy": None}

    if log_file.exists():
        atoms: Atoms = read_espresso_out(str(log_file))
        data: dict[str, Any] = atoms.calc.results
        data["efermi"] = atoms.calc.efermi
        results["energy"] = data.pop("energy")
        results["forces"] = data.pop("forces")
        results["converged"] = False
        results["calculator_results"] = {**data, "atoms": atoms}

        logger.debug(
            "Successfully loaded calculation outputs for Espresso calculation "
            "in directory: %s",
            src,
        )
    else:
        logger.warning(
            "Espresso (PWscf) output file %s does not exist in directory: %s",
            PWSCF_LOG,
            src,
        )

    return results


# TODO: Remove since atoms read with read_espresso_out
def get_output_atoms(
    src: str | Path,
    alt_filename_index: int | None = None,
    input_atoms: Atoms | None = None,
) -> Atoms:
    """Retrieve an ``Atoms`` object representing the output structure.

    This function also copies tags and constraints from the input structure
    in the case that the output structure must be read from a non-ASE file
    (e.g., ``pwscf.save``).

    Args:
        src: The directory from which to retrieve the output structure.
        alt_filename_index: An integer pointing to which alternative structure
            file should be used. This number will be used to index
            ``ALTERNATE_OUTPUT_STRUCTURES``.
        input_atoms: An Atoms object representing the corresponding input
            structure.

    Returns:
        An Atoms object representing the output structure.
    """
    if alt_filename_index is None:
        alt_filename_index = 0
        filename = SETTINGS.OUTPUT_ATOMS_FILE
    else:
        filename = ALTERNATE_OUTPUT_STRUCTURES[alt_filename_index]
        alt_filename_index += 1

    full_filename = Path(src).joinpath(filename)
    logger.debug(f"Retrieving output atoms from {full_filename}")

    try:
        atoms = ase.io.read(full_filename)
        logger.debug(
            f"Successfully retrieved output atoms from {full_filename}"
        )
    except (FileNotFoundError, AttributeError, UnknownFileTypeError):
        msg = (
            f"Unable to retrieve atoms from: {full_filename}.\nFile not found."
        )
        logger.warning(msg)
        try:
            atoms = get_output_atoms(
                src=src,
                alt_filename_index=alt_filename_index,
                input_atoms=input_atoms,
            )
            copy_atom_metadata(
                input_atoms=input_atoms,
                output_atoms=atoms,
            )
        except IndexError as err:
            msg = (
                f"No output atoms found in {SETTINGS.OUTPUT_ATOMS_FILE} or "
                f"{ALTERNATE_OUTPUT_STRUCTURES!r}"
            )
            raise FileNotFoundError(msg) from err

    return atoms
