from pathlib import Path
import shutil

from ase.atoms import Atoms
from ase.calculators.calculator import Calculator
from ase.calculators.vasp.vasp import Vasp
import ase.io
import pytest

from autojob.harvest.harvesters.bader import load_bader_results
from autojob.harvest.harvesters.bader import parse_acf
from autojob.harvest.harvesters.ddec6 import DDEC_NET_ATOMIC_CHARGE_FILE
from autojob.harvest.harvesters.ddec6 import get_ddec6_index_map

BADER_CHARGE_JOB_ID = "jJPQg3Puwz"


class TestParseACF:
    @staticmethod
    def test_should_parse_acf(datadir: Path) -> None:
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["partial_charges"][0] == -0.04175299999999993
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0

    @staticmethod
    def test_should_parse_acf_without_poscar(datadir: Path) -> None:
        Path(datadir, "POSCAR").unlink()
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["partial_charges"] is None
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0

    @staticmethod
    def test_should_parse_acf_with_poscar_but_without_potcar(
        datadir: Path,
    ) -> None:
        Path(datadir, "POTCAR").unlink()
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["partial_charges"] is None
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0


class TestLoadBaderData:
    @staticmethod
    @pytest.mark.filterwarnings("ignore:.*Could not find CHGCAR!*:UserWarning")
    def test_should_use_internal_parser_if_pymatgen_routine_fails(
        datadir: Path,
    ) -> None:
        data = load_bader_results(dir_name=datadir)
        assert data["charge"][0] == 10.295178
        assert data["min_dist"][0] == 0.871639
        assert data["atomic_volume"][0] == 60.443714
        assert data["partial_charges"][0] == 0.7048220000000001
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0


class TestGetDDEC6IndexMap:
    @staticmethod
    @pytest.fixture(name="atoms")
    def fixture_atoms(datadir: Path) -> Atoms:
        structure = ase.io.read(Path(datadir, "POSCAR"))
        structure = structure[0] if isinstance(structure, list) else structure
        return structure

    @staticmethod
    @pytest.fixture(name="atoms_calculator")
    def fixture_atoms_calculator() -> Calculator:
        return Vasp()

    @staticmethod
    @pytest.mark.input_files("structure", "python_script")
    def test_should_return_permutation_of_atomic_indices(
        datadir: Path, atoms: Atoms, populate_inputs: Path
    ) -> None:
        shutil.copy(Path(datadir, "ACF.dat"), populate_inputs)
        shutil.copy(Path(datadir, "POTCAR"), populate_inputs)
        shutil.copy(
            Path(datadir, DDEC_NET_ATOMIC_CHARGE_FILE),
            populate_inputs,
        )
        ddec6_map = get_ddec6_index_map(src=populate_inputs)
        indices = [a.index for a in atoms]
        assert len(ddec6_map) == len(indices)
        assert all(i in ddec6_map for i in indices)

    @staticmethod
    @pytest.mark.input_files("structure", "python_script")
    def test_should_warn_when_atomic_positions_are_mismatched(
        datadir: Path, atoms: Atoms, populate_inputs: Path, structure_name: str
    ) -> None:
        shutil.copy(Path(datadir, "ACF.dat"), populate_inputs)
        shutil.copy(Path(datadir, "POTCAR"), populate_inputs)
        shutil.copy(
            Path(datadir, DDEC_NET_ATOMIC_CHARGE_FILE),
            populate_inputs,
        )
        atoms[0].position = [0.0, 0.0, 0.0]
        atoms.write(Path(populate_inputs, structure_name))
        with pytest.warns(
            UserWarning, match="Atomic displacement exceeds tolerance.*"
        ):
            _ = get_ddec6_index_map(src=populate_inputs)
