from pathlib import Path
from typing import TYPE_CHECKING

import pytest

from autojob.harvest.patch import Patch
from autojob.harvest.patch import build_metadata_patches
from autojob.study import Study
from autojob.study_group import StudyGroup
from autojob.tasks.calculation import Calculation

if TYPE_CHECKING:
    from autojob.bases.task_base import TaskBase


class TestApplyPatch:
    @staticmethod
    @pytest.fixture(name="new_files")
    def fixture_new_files() -> list[str]:
        return ["NEWFILE"]

    @staticmethod
    @pytest.fixture(name="patch")
    def fixture_patch(task: "TaskBase", new_files: list[str]) -> Patch:
        match_path = ["task_metadata", "task_id"]
        patch_path = ["task_inputs", "files_to_carry_over"]

        return Patch(
            match_path=match_path,
            match_value=task.task_metadata.task_id,
            patch_path=patch_path,
            patch_value=new_files,
        )

    @staticmethod
    def test_should_patch_task(
        task: "TaskBase", patch: Patch, new_files: list[str]
    ) -> None:
        patch.apply(task)
        assert task.task_inputs.files_to_carry_over == new_files


@pytest.mark.xfail(reason="TODO")
class TestBuildMetadataPatches:
    @staticmethod
    def test_should_build_study_group_patches(
        study_group_directory: Path, study_group: StudyGroup
    ) -> None:
        patch = build_metadata_patches(
            dir_name=study_group_directory.parent,
            metadata_type="study_group",
            legacy_mode=True,
        )[0]
        study_group_metadata = study_group.model_dump(
            mode="json", by_alias=True
        )
        assert patch.match_value == study_group.study_group_id
        assert patch.patch_value == study_group_metadata

    @staticmethod
    def test_should_build_study_patches(
        study_directory: Path, study: Study, calculation: Calculation
    ) -> None:
        patch = build_metadata_patches(
            dir_name=study_directory.parent,
            metadata_type="study",
            legacy_mode=True,
        )[0]
        study_metadata = study.model_dump(mode="json", by_alias=True)
        study_metadata["Calculations"] = [
            calculation.task_metadata.task_group_id
        ]
        del study_metadata["Tasks"]

        assert patch.match_value == study.study_id
        assert patch.patch_value == study_metadata

    @staticmethod
    def test_should_build_task_group_patches(
        task_group_directory: Path, calculation: Calculation
    ) -> None:
        patch = build_metadata_patches(
            dir_name=task_group_directory.parent,
            metadata_type="task_group",
            legacy_mode=True,
        )[0]
        calculation_metadata = calculation.task_metadata.model_dump(
            mode="json",
            exclude=(
                "workflow_step_id",
                "uri",
                "last_updated",
                "task_id",
            ),
        )
        calculation_metadata["tasks"] = [
            str(calculation.task_metadata.task_id)
        ]
        patch_value = {
            k: v for k, v in patch.patch_value.items() if k != "date_created"
        }

        assert patch.match_value == calculation.task_metadata.task_group_id
        assert patch_value == calculation_metadata
