from collections.abc import Generator
from datetime import UTC
from datetime import datetime
from pathlib import Path

import pytest

from autojob.utils.files import get_last_updated
from autojob.utils.files import get_slurm_job_id


@pytest.fixture(name="write_dummy_slurm_file")
def fixture_write_dummy_slurm_file(
    task_directory: Path,
) -> Generator[Path, None, None]:
    dummy_slurm_file = Path(task_directory, "slurm-000000000.out")
    dummy_slurm_file.touch()
    yield dummy_slurm_file
    dummy_slurm_file.unlink()


class TestGetSlurmJobID:
    @staticmethod
    @pytest.fixture(name="slurm_job_id", params=[111111111, 123456789])
    def fixture_slurm_job_id(request: pytest.FixtureRequest) -> int:
        slurm_job_id: int = request.param
        return slurm_job_id

    @staticmethod
    @pytest.mark.usefixtures("write_slurm_output_file")
    def test_should_find_valid_slurm_file(
        task_directory: Path, slurm_job_id: int
    ) -> None:
        assert get_slurm_job_id(job_dir=task_directory) == slurm_job_id

    @staticmethod
    def test_should_raise_file_not_found_error_if_no_valid_slurm_file(
        task_directory: Path,
    ) -> None:
        with pytest.raises(FileNotFoundError):
            _ = get_slurm_job_id(job_dir=task_directory)

    @staticmethod
    @pytest.mark.usefixtures("write_slurm_output_file")
    def test_should_not_return_id_of_older_job(
        task_directory: Path,
        write_dummy_slurm_file: Path,  # noqa: ARG004
        slurm_job_id: int,
    ) -> None:
        assert get_slurm_job_id(job_dir=task_directory) == slurm_job_id


class TestCreateJobStatsFile:
    pass


class TestGetLastUpdated:
    @staticmethod
    def test_should_get_time_of_last_updated_when_directory_is_not_empty(
        tmp_path: Path,
    ) -> None:
        file = Path(tmp_path, "test")
        file.touch()
        created = file.stat().st_mtime
        last_updated = get_last_updated(tmp_path)
        epoch_start = datetime(1970, 1, 1, tzinfo=UTC)
        assert (last_updated - epoch_start).total_seconds() == pytest.approx(
            created, abs=0.1
        )

    @staticmethod
    def test_should_get_time_directory_last_updated_when_directory_is_empty(
        tmp_path: Path,
    ) -> None:
        created = tmp_path.stat().st_mtime
        last_updated = get_last_updated(tmp_path)
        epoch_start = datetime(1970, 1, 1, tzinfo=UTC)
        assert (last_updated - epoch_start).total_seconds() == pytest.approx(
            created, abs=0.1
        )


class TestTemplateFile:
    @staticmethod
    @pytest.mark.xfail(reason="Not implemented")
    def test_should_template_file_with_context() -> None:
        template_file_with_context = False
        assert template_file_with_context

    @staticmethod
    @pytest.mark.xfail(reason="Not implemented")
    def test_should_write_template_to_destination() -> None:
        write_template_to_destination = False
        assert write_template_to_destination
