"""Clean up the directory of a completed task."""

import logging
import math
from pathlib import Path
import shutil

import click

logger = logging.getLogger(__name__)


@click.command(
    "clean", context_settings={"help_option_names": ["-h", "--help"]}
)
@click.option(
    "--to-delete",
    help="Specify files to delete. This option can be repeated.",
    show_default=True,
    type=click.Path(exists=True, path_type=Path),
    multiple=True,
)
@click.option(
    "--file-size-limit",
    default=math.inf,
    help="Specify a maximum file size limit in bytes. All files larger than "
    "this will be deleted.",
    show_default=True,
    type=float,
)
@click.argument(
    "paths",
    default=Path().cwd(),
    type=click.Path(
        exists=True, file_okay=False, dir_okay=True, path_type=Path
    ),
    nargs=-1,
)
def main(  # noqa: D417
    to_delete: tuple[str, ...],
    file_size_limit: float,
    paths: tuple[Path, ...],
) -> None:
    """Run a Python script under a time limit.

    Args:
        paths: The directory to clean.
    """
    for path in paths:
        logger.info("Removing files in %s", path)
        for f in path.iterdir():
            if f.name not in to_delete and f.stat().st_size < file_size_limit:
                continue

            if f.is_dir():
                logger.info("Removing directory: %s", f)
                shutil.rmtree(f)
            else:
                logger.info("Removing file: %s", f)
                f.unlink()
