"""This module defines the config-gen CLI."""

import logging
from pathlib import Path

import click

from autojob import SETTINGS

logger = logging.getLogger(__name__)


def textify_config(
    settings: dict[str, str], *, sort: bool = False
) -> list[str]:
    """Print an Options object into a list of text."""
    lines: list[str] = []
    keys = sorted(settings) if sort else list(settings)

    for key in keys:
        value = settings[key]
        if value is None:
            continue
        if isinstance(value, str):
            value = f'"{value}"'
        elif isinstance(value, bool):
            value = str(value).lower()

        lines.append(f"{key} = {value}\n")

    return lines


def create_config_file(*, sort: bool = False) -> Path:
    """Create a simple configuration file."""
    lines = [
        "[tool.autojob]\n",
        *textify_config(SETTINGS.model_dump(mode="json"), sort=sort),
    ]

    config = Path("config.toml").absolute()
    with config.open(mode="w", encoding="utf-8") as file:
        file.writelines(lines)
    return config


@click.command(
    "config-gen", context_settings={"help_option_names": ["-h", "--help"]}
)
@click.option(
    "--sort/--no-sort",
    default=False,
    help="Whether or not to sort configuration keys",
    show_default=True,
)
def main(sort: bool) -> None:
    """Generate a configuration file from autojob defaults."""
    config = create_config_file(sort=sort)
    click.echo(f"Configuration file created: {config}")
