"""CLI function for harvesting task results."""

from pathlib import Path
from typing import Literal

import click

from autojob.harvest.archive import archive
from autojob.harvest.harvest import harvest


# TODO: convert whitelist/blacklist from file paths into job directory strings
@click.group(
    "harvest",
    invoke_without_command=True,
    context_settings={"help_option_names": ["-h", "--help"]},
)
@click.option(
    "--path",
    "dir_name",
    default=Path.cwd(),
    type=click.Path(
        exists=True, file_okay=False, readable=True, path_type=Path
    ),
)
@click.option(
    "--filename",
    default="harvested",
    help="The file stem to use to name the harvested data file.",
)
@click.option(
    "--dest",
    default=Path.cwd(),
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        writable=True,
        path_type=Path,
    ),
    help="The directory in which to save the archives.",
)
@click.option(
    "--archive-mode",
    default="csv",
    help="The mode with which to archive the harvested tasks.",
    type=click.Choice(("csv", "json", "both")),
)
@click.option(
    "--strict",
    "strictness",
    flag_value="strict",
    default="relaxed",
    help="Errors will immediately halt execution.",
)
@click.option(
    "--relaxed",
    "strictness",
    flag_value="relaxed",
    default="relaxed",
    help="Every attempt will be made to harvest data from every directory. "
    "Incomplete data does not cause execution to halt.",
)
@click.option(
    "--atomic",
    "strictness",
    flag_value="atomic",
    default="relaxed",
    help="Only jobs for which no errors are thrown during harvesting will be "
    "harvested.",
)
@click.option(
    "--use-cache/--no-use-cache",
    default=False,
    show_default=False,
    help="Whether or not to use cached results. If True, then cached results "
    "will be ignored and overwritten. Otherwise, outputs will be read from an "
    "existing cache.",
)
@click.option(
    "--time-stamp/--no-time-stamp",
    default=False,
    show_default=True,
    help="Whether or not to time stamp the archived file.",
)
@click.option(
    "--whitelist",
    "whitelists",
    multiple=True,
    type=click.Path(
        exists=True,
        file_okay=True,
        dir_okay=False,
        readable=True,
        path_type=Path,
    ),
    help="Specify an exclusive list of jobs to harvest. Whitelists will be "
    "combined.",
)
@click.option(
    "--blacklist",
    "blacklists",
    multiple=True,
    type=click.Path(
        exists=True,
        file_okay=True,
        dir_okay=False,
        readable=True,
        path_type=Path,
    ),
    help="Specify a list of jobs not to harvest. Blacklists will be "
    "combined. When specified with --whitelist, --blacklist will take a "
    "higher priority.",
)
def main(
    *,
    dir_name: Path,
    use_cache: bool,
    time_stamp: bool,
    strictness: Literal["strict", "relaxed", "atomic"],
    whitelists: tuple[Path],
    blacklists: tuple[Path],
    archive_mode: Literal["csv", "json", "both"],
    filename: str,
    dest: Path,
) -> None:
    """Harvest completed tasks from a directory."""
    harvested = harvest(
        dir_name=dir_name,
        strictness=strictness,
        whitelists=whitelists or None,
        blacklists=blacklists or None,
        use_cache=use_cache,
    )
    archive(
        filename=filename,
        archive_mode=archive_mode,
        harvested=harvested,
        dest=dest,
        time_stamp=time_stamp,
    )
