"""Create new study groups, studies, task groups, and tasks.

Examples:
    .. code-block:: shell

        autojob new --type=task

        autojob new --type=task_group

        autojob new --type=study

        autojob new --type=study_group
"""

import logging
from pathlib import Path
from typing import Any
from typing import Literal

import click

from autojob.next import create_task_tree
from autojob.plugins import get_task_class
from autojob.utils.cli import configure_settings
from autojob.utils.cli import mods_to_dict

logger = logging.getLogger(__name__)


@click.command(
    "new",
    context_settings={"help_option_names": ["-h", "--help"]},
)
@click.option(
    "-v",
    "--verbose",
    "verbosity",
    default=0,
    count=True,
    help="Controls the verbosity. 0: Show messages of level warning and "
    "higher. 1: Show messages of level info and higher. 2: Show all messages"
    "-useful for debugging.",
    show_default=True,
)
@click.option(
    "--type",
    "to_create",
    type=click.Choice(choices=["task", "task_group", "study", "study_group"]),
    default="task",
    help="The type of directory to create anew.",
    show_default=True,
)
@click.option(
    "-d",
    "--dest",
    default=Path().cwd(),
    help="The directory in which to create the directory of the new task.",
    type=click.Path(path_type=Path),
)
@click.option(
    "--config",
    default={},
    multiple=True,
    callback=mods_to_dict,
    help="Configure autojob configuration settings (e.g., "
    '--config="log_file=autojob.log").  This option can be repeated.',
)
@click.option(
    "--name-template",
    help="A naming scheme to use for the restarted task directory. This can "
    "be a templated string-available fields include 'structure' and any task "
    "metadata.",
)
def main(
    *,
    verbosity: int,
    to_create: Literal["task", "task_group", "study", "study_group"],
    dest: Path,
    config: dict[str, Any],
    name_template: str | None,
) -> None:
    """Create new study group, study, task group, and task directories."""
    match verbosity:
        case 0:
            level = logging.WARNING
        case 1:
            level = logging.INFO
        case _:
            level = logging.DEBUG

    logging.basicConfig(level=level)
    configure_settings(config)

    if to_create == "task":
        new_task_type = get_task_class(to_create)
        new_task = new_task_type()
        new_task_dir = create_task_tree(
            new_task, dest, name_template=name_template
        )
        task_id = str(new_task.task_metadata.task_id)
        click.echo(f"New task created ({task_id}) {new_task_dir.name}")
    else:
        msg = f"Creating {to_create}s is not yet supported"
        raise NotImplementedError(msg)
