"""This module defines the autojob run CLI."""

from datetime import UTC
from datetime import datetime
import logging
import math
from pathlib import Path
import shlex
import subprocess
import sys
from threading import Thread
import time
from typing import NoReturn

import click

from autojob import SETTINGS
from autojob.utils.parsing import parse_time_from_slurm_script

logger = logging.getLogger(__name__)


def execute_command(
    *,
    command: tuple[str, ...],
    time_limit: float = math.inf,
    frequency: float = 300,
    buffer: float = 0.0,
) -> NoReturn:
    """Execute a command with a time limit.

    Args:
        command: The command to execute.
        time_limit: The absolute time limit in seconds.
        frequency: How frequently to check if the underlying function has
            completed.
        buffer: the time buffer for the task. A value greater than or equal
            to 0 and less than 1 will be interpreted as the fraction of the
            total time to use as a buffer. Values greater than or equal to 1
            will be interpreted as seconds.

    Note:
        This function does not return; it exits with an exit code of 0 if the
        script executes successully within the time limit (subject to the
        buffer) and an exit code of 124 if the time limit is reached.
    """
    if 0 <= buffer < 1:
        time_limit = time_limit * (1 - buffer)
    elif buffer >= 1:
        time_limit = time_limit - buffer
    else:
        msg = f"Invalid buffer value: {buffer}"
        raise ValueError(msg)

    def _target():
        logger.info("Executing command: %s", command)
        try:
            _ = subprocess.run(command, check=True)
        except subprocess.CalledProcessError as err:
            logger.error(err.args[0])
            raise

    thread = Thread(target=_target)
    thread.daemon = True
    start = datetime.now(tz=UTC)
    thread.start()
    logger.debug("Will abort after %.2f hours.", time_limit / 3600)
    logger.debug("Checking every %.2f hours.", frequency / 3600)
    elapsed = (datetime.now(tz=UTC) - start).total_seconds()

    while elapsed < time_limit:
        time.sleep(frequency)
        elapsed = (datetime.now(tz=UTC) - start).total_seconds()
        logging.debug("Checking... Elapsed time: %.2f hr", elapsed / 3600)

        if not thread.is_alive():
            logging.info(" Job appears finished. Exiting.")
            sys.exit(0)

    logger.info("Time limit reached. Elapsed time: %.2f hr", elapsed / 3600)
    sys.exit(124)


@click.command(
    "init", context_settings={"help_option_names": ["-h", "--help"]}
)
@click.option(
    "-t",
    "--time-source",
    default=SETTINGS.DEFAULT_TASK_SCRIPT_FILE,
    help="The file from which to parse the time limit",
    show_default=True,
    type=click.Path(exists=True, dir_okay=False, path_type=Path),
)
@click.option(
    "-b",
    "--buffer",
    default=0.05,
    help="Specify the time buffer for the task. A value greater than or equal "
    "to 0 and less than 1 will be interpreted as the fraction of the total "
    "time to use as a buffer. Values greater than or equal to 1 will be "
    "interpreted as seconds.",
    show_default=True,
    type=float,
)
@click.option(
    "-f",
    "--frequency",
    default=30,
    help="How frequently the process is checked for completion in seconds.",
    show_default=True,
    type=float,
)
@click.argument(
    "command",
    default=SETTINGS.DEFAULT_CALCULATION_SCRIPT_FILE,
)
def main(  # noqa: D417
    time_source: Path,
    command: str,
    buffer: float,
    frequency: float,
) -> None:
    """Run a Python script under a time limit.

    Args:
        COMMAND: The command to execute.
    """
    with time_source.open(mode="r", encoding="utf-8") as file:
        time_limit = parse_time_from_slurm_script(file)

    execute_command(
        command=shlex.split(command),
        time_limit=time_limit.total_seconds(),
        frequency=frequency,
        buffer=buffer,
    )
