"""The default calculation harvester."""

from __future__ import annotations

import logging
from pathlib import Path
from typing import Any

import ase.io
from numpy import ndarray

from autojob import SETTINGS

logger = logging.getLogger(__name__)


def load_calculation_results(src: str | Path) -> dict[str, Any]:
    """Load calculation results from a directory.

    Args:
        src: The directory from which to load the calculation results.

    Returns:
        A dictionary containing calculation results.

    Note:
        If ``SETTINGS.STRICT_MODE`` is true, this method may re-raise one of
        AttributeError, FileNotFound, or KeyError.
    """
    src = Path(src)
    atoms_file = Path(src, SETTINGS.OUTPUT_ATOMS_FILE)
    converged = atoms_file.exists()
    results = {"converged": converged}
    try:
        atoms = ase.io.read(atoms_file)
        if atoms.calc:
            results["energy"] = atoms.calc.results.pop("energy")
            results["forces"] = atoms.calc.results.pop("forces")
            results["calculator_results"] = {**atoms.calc.results}
            for key, value in results["calculator_results"].items():
                if isinstance(value, ndarray):
                    results["calculator_results"][key] = value.tolist()
    except (AttributeError, FileNotFoundError, KeyError):
        if SETTINGS.STRICT_MODE:
            raise
    return results
