"""Read and standardize Gaussian outputs."""

import logging
from pathlib import Path
from typing import Any

from ase import Atoms
import ase.io
from ase.io.formats import UnknownFileTypeError
import cclib
from cclib.parser.data import ccData

from autojob import SETTINGS
from autojob.utils.atoms import copy_atom_metadata

logger = logging.getLogger(__name__)

ALTERNATE_OUTPUT_STRUCTURES = ("relax.traj", "Gaussian.log")
FILES_TO_CARRYOVER = ["Gaussian.chk"]
GAUSSIAN_LOG = "Gaussian.log"


def load_calculation_results(
    src: str | Path,
) -> dict[str, Any]:
    """Load calculation outputs for a Gaussian calculation.

    Note that all quantities other than the final energy are reported in atomic
    units (Hartree/Bohr).

    Args:
        src: The directory containing the Gaussian output files.

    Returns:
        A dictionary containing Gaussian calculation outputs.
    """
    logger.debug(
        "Loading calculation outputs for Gaussian calculation in directory: %s",
        src,
    )
    log_file = Path(src, GAUSSIAN_LOG)
    results = {"forces": None, "energy": None}

    if not log_file.exists():
        logger.warning(
            "Gaussian output file %s does not exist in directory: %s",
            GAUSSIAN_LOG,
            src,
        )
    elif cc_data := cclib.io.ccread(str(log_file)):
        cc_data.listify()
        data = cc_data.getattributes()

        if energies := data.get("scfenergies"):
            results["energy"] = energies[-1] if energies else None

        if forces := data.get("grads"):
            results["forces"] = forces[-1] if forces else None

        results["converged"] = bool(
            # optimization statuses are defined in bit value notation
            getattr(cc_data, "optstatus", False)
            and (cc_data.optstatus[-1] & ccData.OPT_DONE)
        )
        atoms = ase.io.read(Path(src, GAUSSIAN_LOG))
        results["calculator_results"] = {**data, "atoms": atoms}

        logger.debug(
            "Successfully loaded Gaussian calculation outputs "
            "from directory: %s",
            src,
        )
    else:
        logger.warning(
            "Unable to parse Gaussian outputs from directory: %s", src
        )

    return results


# TODO: Remove since Gaussian atoms read from GAUSSIAN_LOG
def get_output_atoms(
    src: str | Path,
    alt_filename_index: int | None = None,
    input_atoms: Atoms | None = None,
) -> Atoms:
    """Retrieve an ``Atoms`` object representing the output structure.

    This function also copies tags and constraints from the input structure
    in the case that the output structure must be read from a non-ASE file
    (e.g., ``Gaussian.log``).

    Args:
        src: The directory from which to retrieve the output structure.
        alt_filename_index: An integer pointing to which alternative structure
            file should be used. This number will be used to index
            ``ALTERNATE_OUTPUT_STRUCTURES``.
        input_atoms: An Atoms object representing the corresponding input
            structure.

    Returns:
        An Atoms object representing the output structure.
    """
    if alt_filename_index is None:
        alt_filename_index = 0
        filename = SETTINGS.OUTPUT_ATOMS_FILE
    else:
        filename = ALTERNATE_OUTPUT_STRUCTURES[alt_filename_index]
        alt_filename_index += 1

    full_filename = Path(src).joinpath(filename)
    logger.debug(f"Retrieving output atoms from {full_filename}")

    try:
        atoms = ase.io.read(full_filename)
        logger.debug(
            f"Successfully retrieved output atoms from {full_filename}"
        )
    except (FileNotFoundError, AttributeError, UnknownFileTypeError):
        msg = (
            f"Unable to retrieve atoms from: {full_filename}.\nFile not found."
        )
        logger.warning(msg)
        try:
            atoms = get_output_atoms(
                src=src,
                alt_filename_index=alt_filename_index,
                input_atoms=input_atoms,
            )
            copy_atom_metadata(
                input_atoms=input_atoms,
                output_atoms=atoms,
            )
        except IndexError as err:
            msg = (
                f"No output atoms found in {SETTINGS.OUTPUT_ATOMS_FILE} or "
                f"{ALTERNATE_OUTPUT_STRUCTURES!r}"
            )
            raise FileNotFoundError(msg) from err

    return atoms
