"""Read and standardize Quantum Espresso outputs."""

import logging
from pathlib import Path
from typing import TYPE_CHECKING
from typing import Any

from ase.io.gpaw_out import read_gpaw_out

if TYPE_CHECKING:
    from ase import Atoms

logger = logging.getLogger(__name__)

ALTERNATE_OUTPUT_STRUCTURES = ("relax.traj", "pwscf.save")
FILES_TO_CARRYOVER = ["pwscf.save"]
# ! This is not a standard fileaname
GPAW_LOG = "gpaw-out.log"


# TODO: Set converged correctly
def load_calculation_results(src: str | Path) -> dict[str, Any]:
    """Load calculation outputs for a GPAW calculation.

    Args:
        src: The directory containing the GPAW output file.

    Returns:
        A dictionary containing GPAW calculation outputs.

    Warning:
        That a calculation has converged must be confirmed manually as the
        value of the ``"converged"`` key is always set to False.
    """
    logger.debug(
        "Loading calculation outputs for GPAW calculation in directory: %s",
        src,
    )
    log_file = Path(src, GPAW_LOG)
    results = {"forces": None, "energy": None}

    if log_file.exists():
        with log_file.open(mode="r", encoding="utf-8") as file:
            atoms: Atoms = read_gpaw_out(file, -1)

        data: dict[str, Any] = atoms.calc.results
        results["energy"] = data.pop("energy", None)
        results["forces"] = data.pop("forces", None)
        results["converged"] = False
        results["calculator_results"] = {**data, "atoms": atoms}

        logger.debug(
            "Successfully loaded calculation outputs for GPAW calculation "
            "in directory: %s",
            src,
        )
    else:
        logger.warning(
            "GPAW output file %s does not exist in directory: %s",
            GPAW_LOG,
            src,
        )

    return results
