"""Pytest configuration and fixture definition for unit testing suite.

To ease navigation, module-level constants (e.g., BUILDING_TASK_METADATA) have
been defined.

Fixtures have been defined for the input and output components of a job
directory to ease with testing.

Job directories can be setup using the populate_inputs and populate_outputs
fixtures which are implicitly parametrized by the various fixtures that they
request. To specify which input/output files are written, use the input_files
and output_files markers. For example:

    @pytest.mark.use_fixtures("populate_inputs")
    @pytest.mark.input_files("structure")
    class Test:...

Input and output atoms related to the job directory can be accessed by the
atoms and output_atoms fixtures, respectively. The output_atoms fixture is
the resulting Atoms object from calling the run_calc fixture, which can be
used to perform any sort of calculation.

Additional fixtures have been defined for the attributes of the various core
models of autojob (e.g., Task/Calculation/Scheduler(Inputs/Outputs)).
"""

from collections.abc import Generator
from datetime import UTC
from datetime import datetime
from datetime import timedelta
from io import TextIOWrapper
import json
import logging
from pathlib import Path
import shutil
from typing import Any
from uuid import uuid4

from ase import Atoms
from ase.calculators.calculator import Calculator
from ase.calculators.calculator import get_calculator_class
from ase.optimize.bfgs import BFGS
from ase.optimize.optimize import Optimizer
import numpy as np
from numpy.typing import NDArray
from pydantic import UUID4
from pymatgen.io.vasp import Vasprun
import pytest

from autojob import SETTINGS
from autojob import hpc
from autojob.bases.task_base import TaskOutcome
from autojob.coordinator.job import JobError
from autojob.coordinator.vasp import VaspJob
from autojob.harvest.harvesters.vasp import FILES_TO_CARRYOVER
from autojob.hpc import Partition
from autojob.hpc import SchedulerInputs
from autojob.hpc import SchedulerOutputs
from autojob.study import Study
from autojob.study_group import StudyGroup
from autojob.tasks.calculation import Calculation
from autojob.tasks.calculation import CalculationInputs
from autojob.tasks.calculation import CalculationOutputs
from autojob.tasks.task import Task
from autojob.tasks.task import TaskInputs
from autojob.tasks.task import TaskMetadata
from autojob.tasks.task import TaskOutputs
from autojob.utils.files import get_uri
from autojob.utils.parsing import TimedeltaTuple

logger = logging.getLogger(__name__)

VASP_OUTPUT_FILES = [
    "BSETFATBAND",
    "CHG",
    "CHGCAR",
    "DOSCAR",
    "EIGENVAL",
    "ELFCAR",
    "IBZKPT",
    "LOCPOT",
    "OSZICAR",
    "PARCHG",
    "PCDAT",
    "PROCAR",
    "PROOUT",
    "REPORT",
    "TMPCAR",
    "WAVECAR",
    "WAVEDER",
    "XDATCAR",
    "vasp.dipcor",
    "vasprun.xml",
]

SLURM_OUTPUT_FILES = {
    None: "no-job-error-slurm-file.txt",
    JobError.TIME_LIMIT: "time-limit-error-slurm-file.txt",
    # ! uncomment when memory limit output file comment decided on
    # JobError.MEMORY_LIMIT: "memory-limit-error-slurm-file.txt",
}

BUILDING_TASK_METADATA = """
############################
## Building Task Metadata ##
############################
"""


@pytest.fixture(name="label")
def fixture_label() -> str:
    label = "label"
    return label


@pytest.fixture(name="tags")
def fixture_tags() -> list[str]:
    tags = ["tag1"]
    return tags


@pytest.fixture(name="id_stem", params=["111111111"])
def fixture_id_stem() -> str:
    return "111111111"


@pytest.fixture(name="legacy_ids", params=[True, False])
def fixture_legacy_ids(request: pytest.FixtureRequest) -> bool:
    return bool(request.param)


@pytest.fixture(name="study_group_id")
def fixture_study_group_id(id_stem: str, legacy_ids: bool) -> UUID4 | str:
    return f"g{id_stem}" if legacy_ids else uuid4()


@pytest.fixture(name="study_id")
def fixture_study_id(id_stem: str, legacy_ids: bool) -> UUID4 | str:
    return f"s{id_stem}" if legacy_ids else uuid4()


@pytest.fixture(name="workflow_step_id")
def fixture_workflow_step_id() -> UUID4:
    return uuid4()


@pytest.fixture(name="task_group_id")
def fixture_task_group_id(id_stem: str, legacy_ids: bool) -> UUID4 | str:
    return f"c{id_stem}" if legacy_ids else uuid4()


@pytest.fixture(name="task_id")
def fixture_task_id(id_stem: str, legacy_ids: bool) -> UUID4 | str:
    return f"j{id_stem}" if legacy_ids else uuid4()


@pytest.fixture(name="date_created")
def fixture_date_created() -> datetime:
    return datetime(2022, 1, 1, tzinfo=UTC)


@pytest.fixture(name="last_updated")
def fixture_last_updated() -> datetime:
    return datetime(2024, 1, 1, tzinfo=UTC)


@pytest.fixture(name="uri")
def fixture_uri(tmp_path: Path) -> str:
    return get_uri(tmp_path)


@pytest.fixture(name="task_metadata")
def fixture_task_metadata(
    *,
    label: str,
    tags: list[str],
    uri: str,
    study_group_id: str | UUID4 | None,
    study_id: str | UUID4 | None,
    workflow_step_id: UUID4 | None,
    task_id: str | UUID4,
    task_group_id: str | UUID4,
    last_updated: datetime | None,
) -> TaskMetadata:
    data = {
        "label": label,
        "tags": tags,
        "uri": uri,
        "study_group_id": study_group_id,
        "study_id": study_id,
        "workflow_step_id": workflow_step_id,
        "task_id": task_id,
        "task_group_id": task_group_id,
        "last_updated": last_updated,
        "task_class": "task",
    }
    return TaskMetadata(**data)


@pytest.fixture(name="valid_task")
def fixture_valid_task(
    task_metadata: TaskMetadata,
) -> dict[str, Any]:
    exclude = {
        "uri",
        "last_updated",
    }
    model = task_metadata.model_dump(
        mode="json", exclude=exclude, by_alias=True
    )
    return model


@pytest.fixture(name="valid_task_group")
def fixture_valid_task_group(
    task_id: str | UUID4,
    task_group_id: str | UUID4,
    study_id: str | UUID4,
    study_group_id: str | UUID4,
    date_created: datetime,
) -> dict[str, str | UUID4 | datetime]:
    return {
        "date_created": date_created,
        "label": "Test",
        "tags": [],
        "study_group_id": study_group_id,
        "study_id": study_id,
        "task_group_id": task_group_id,
        "tasks": [task_id],
    }


@pytest.fixture(name="valid_study")
def fixture_valid_study(
    task_group_id: str | UUID4,
    study_id: str | UUID4,
    study_group_id: str | UUID4,
    date_created: datetime,
) -> dict[str, str | datetime | UUID4]:
    return {
        "date_created": date_created,
        "label": "Test",
        "tags": [],
        "study_group_id": study_group_id,
        "study_id": study_id,
        "task_groups": [task_group_id],
    }


@pytest.fixture(name="valid_study_group")
def fixture_valid_study_group(
    study_id: str | UUID4,
    study_group_id: str | UUID4,
    date_created: datetime,
) -> dict[str, str | datetime | UUID4]:
    return {
        "date_created": date_created,
        "label": "Test",
        "tags": [],
        "study_group_id": study_group_id,
        "studies": [study_id],
    }


BUILDING_TASK_INPUTS = """
##########################
## Building Task Inputs ##
##########################
"""


@pytest.fixture(name="files_to_copy", params=[VASP_OUTPUT_FILES])
def fixture_files_to_copy(request: pytest.FixtureRequest) -> list[str]:
    files_to_copy: list[str] = request.param

    return files_to_copy


@pytest.fixture(name="files_to_delete", params=[VASP_OUTPUT_FILES])
def fixture_files_to_delete(request: pytest.FixtureRequest) -> list[str]:
    files_to_delete: list[str] = request.param

    return files_to_delete


@pytest.fixture(name="files_to_carryover", params=[FILES_TO_CARRYOVER])
def fixture_files_to_carryover(request: pytest.FixtureRequest) -> list[str]:
    files_to_carryover: list[str] = request.param

    return files_to_carryover


@pytest.fixture(name="auto_restart", params=[True])
def fixture_auto_restart(request: pytest.FixtureRequest) -> bool:
    auto_restart: bool = request.param
    return auto_restart


@pytest.fixture(name="structure_name", params=("atoms",))
def fixture_structure_name(request: pytest.FixtureRequest) -> str | None:
    structure_name: str | None = request.param
    return structure_name


@pytest.fixture(name="atoms_filename")
def fixture_atoms_filename(structure_name: str) -> str:
    return f"{structure_name}.traj"


@pytest.fixture(name="input_atoms")
def fixture_input_atoms(atoms_filename: str) -> Atoms | None:
    atoms = Atoms("C")
    atoms.info["structure"] = str(atoms.symbols)
    atoms.info["filename"] = atoms_filename
    return atoms


@pytest.fixture(name="task_script")
def fixture_task_script() -> str:
    return "task.sh"


@pytest.fixture(name="task_script_template")
def fixture_task_script_template() -> str:
    return "task.sh.j2"


@pytest.fixture(name="task_inputs")
def fixture_task_inputs(
    *,
    input_atoms: Atoms,
    files_to_copy: list[str],
    files_to_delete: list[str],
    files_to_carryover: list[str],
    auto_restart: bool,
    atoms_filename: str,
    task_script_template: str,
) -> TaskInputs:
    return TaskInputs(
        atoms=input_atoms,
        files_to_copy=files_to_copy,
        files_to_delete=files_to_delete,
        files_to_carry_over=files_to_carryover,
        auto_restart=auto_restart,
        atoms_filename=atoms_filename,
        task_script_template=task_script_template,
    )


BUILDING_CALCULATION_INPUTS = """
##################################
##  Building Calculation Inputs ##
##################################
"""


@pytest.fixture(name="calculator_class")
def fixture_calculator_class() -> str:
    return "emt"


@pytest.fixture(name="calculator")
def fixture_calculator(
    calculator_class: str, task_directory: Path
) -> Calculator | None:
    if calculator_class:
        calc = get_calculator_class(calculator_class)
        return calc(directory=task_directory)
    return None


@pytest.fixture(name="calc_params")
def fixture_calc_params() -> dict[str, Any]:
    return {
        x.name.lower(): x.default
        for x in VaspJob.input_parameters()
        if x.default
    }


@pytest.fixture(name="opt_class")
def fixture_opt_class() -> type[Optimizer]:
    return BFGS


@pytest.fixture(name="opt_logfile")
def fixture_opt_logfile() -> str:
    return "opt.log"


@pytest.fixture(name="opt_traj_file")
def fixture_opt_traj_file() -> str:
    return "opt.traj"


@pytest.fixture(name="optimizer")
def fixture_optimizer(
    opt_class: type[Optimizer],
    output_atoms: Atoms,
    calculator: Calculator,
    opt_traj_file: str,
    opt_logfile: str,
    task_directory: Path,
) -> Optimizer:
    output_atoms.calc = calculator
    logfile = Path(task_directory, opt_logfile)
    traj_file = Path(task_directory, opt_traj_file)
    return opt_class(output_atoms, logfile=logfile, trajectory=traj_file)


@pytest.fixture(name="opt_params")
def fixture_opt_params() -> dict[str, Any]:
    return {"fmax": 0.1, "steps": 10}


@pytest.fixture(name="calculation_script_template")
def fixture_calculation_script_template() -> str:
    return "run.py.j2"


@pytest.fixture(name="calculation_inputs")
def fixture_calculation_inputs(
    calculator: Calculator,
    calc_params: dict[str, Any],
    calculation_script_template: str,
) -> CalculationInputs:
    name = calculator.__class__.__name__.lower()
    return CalculationInputs(
        calculator=name,
        calc_params=calc_params,
        calculation_script_template=calculation_script_template,
    )


BUILDING_SCHEDULER_INPUTS = """
##################################
###  Building Scheduler Inputs ###
##################################
"""


@pytest.fixture(name="ntasks_per_node", params=[12])
def fixture_ntasks_per_node(request: pytest.FixtureRequest) -> int:
    ntasks_per_node: int = request.param
    return ntasks_per_node


@pytest.fixture(name="nodes", params=[1])
def fixture_nodes(request: pytest.FixtureRequest) -> int:
    nodes: int = request.param
    return nodes


@pytest.fixture(name="job_name", params=["vasp.sh"])
def fixture_job_name(request: pytest.FixtureRequest) -> str:
    job_name: str = request.param
    return job_name


@pytest.fixture(name="partitions", params=[[hpc.Partition.RAZI]])
def fixture_partitions(request: pytest.FixtureRequest) -> str:
    partitions: Partition = request.param
    return partitions


@pytest.fixture(name="mem_per_cpu", params=[1024])
def fixture_mem_per_cpu(request: pytest.FixtureRequest) -> int:
    mem_per_cpu: int = request.param

    return mem_per_cpu


@pytest.fixture(name="mail_type")
def fixture_mail_type() -> list[str]:
    mail_type = ["BEGIN", "END", "FAIL", "TIME_LIMIT", "TIME_LIMIT_90"]

    return mail_type


@pytest.fixture(name="mail_user")
def fixture_mail_user() -> list[str]:
    return "john.doe@example.ca"


@pytest.fixture(name="scheduler_inputs")
def fixture_scheduler_inputs(
    elapsed: timedelta,
    ntasks_per_node: int,
    nodes: int,
    job_name: str,
    partitions: list[Partition],
    mem_per_cpu: int,
    mail_type: list[str],
    mail_user: str,
) -> SchedulerInputs:
    options = {
        "job-name": job_name,
        "partition": partitions,
        "mem-per-cpu": mem_per_cpu,
        "nodes": nodes,
        "ntasks-per-node": ntasks_per_node,
        "time": elapsed,
        "mail-user": mail_user,
        "mail-type": mail_type,
    }
    return SchedulerInputs(**options)


BUILDING_SLURM_SCRIPT = """
##############################
###  Building SLURM Script ###
##############################
"""


@pytest.fixture(name="bash_shebang", params=[True])
def fixture_bash_shebang(request: pytest.FixtureRequest) -> list[str]:
    if request.param:
        return ["#! /usr/bin/bash"]
    return []


@pytest.fixture(name="interspacer", params=("",))
def fixture_interspacer(request: pytest.FixtureRequest) -> str:
    interspacer: str = request.param
    return interspacer


@pytest.fixture(name="misplaced_slurm_option", params=[False])
def fixture_misplaced_slurm_option(
    request: pytest.FixtureRequest,
) -> list[str]:
    if request.param:
        return ["#SBATCH --output=slurm-%A.out", ""]

    return [""]


@pytest.fixture(name="scheduler_header")
def fixture_scheduler_header(
    bash_shebang: list[str],
    interspacer: str,
    scheduler_inputs: list[str],
) -> list[str]:
    """Compose a scheduler script"""
    scheduler_config = []

    for k, v in scheduler_inputs.items():
        option_name = k.replace("_", "-")
        if len(option_name) == 1:
            prefix = "-"
            suffix = " "
        else:
            prefix = "--"
            suffix = "="

        if isinstance(v, list):
            value = ",".join(str(x) for x in v)
        elif isinstance(v, timedelta):
            value = TimedeltaTuple.from_timedelta(v).to_slurm_time()
        else:
            value = v

        scheduler_config.append(
            f"#SBATCH  {prefix}{option_name}{suffix}{value}"
        )

    if interspacer:
        midpoint = int(len(scheduler_config) / 2)
        scheduler_config.insert(midpoint, interspacer.strip())

    scheduler_config.append("")

    return bash_shebang + scheduler_config


@pytest.fixture(name="write_scheduler_script")
def fixture_write_scheduler_script(
    calculation: Calculation, task_directory: Path
) -> Path:
    calculation.task_inputs.task_script_template = (
        SETTINGS.TASK_SCRIPT_TEMPLATE
    )
    return calculation.write_task_script(task_directory)


TASK_OUTPUTS = """
######################
###  Task Outputs  ###
######################
"""


@pytest.fixture(name="task_outcome")
def fixture_task_outcome() -> TaskOutcome:
    return TaskOutcome.IDLE


@pytest.fixture(name="output_atoms")
def fixture_output_atoms(input_atoms: Atoms) -> Atoms:
    return input_atoms.copy()


@pytest.fixture(name="task_outputs")
def fixture_task_outputs(
    output_atoms: Atoms, task_outcome: TaskOutcome
) -> TaskOutputs:
    return TaskOutputs(
        atoms=output_atoms,
        outcome=task_outcome,
    )


CALCULATION_OUTPUTS = """
#############################
###  Calculation Outputs  ###
#############################
"""


@pytest.fixture(name="to_calculate")
def fixture_to_calculate() -> list[str]:
    return []


@pytest.fixture(name="calculate")
def fixture_calculate(
    output_atoms: Atoms, calculator: Calculator, to_calculate: list[str]
) -> None:
    output_atoms.calc = calculator
    calculator.atoms = output_atoms
    calculator.calculate(atoms=output_atoms, properties=to_calculate)


@pytest.fixture(name="optimize")
def fixture_optimize(opt_params: dict[str, Any], optimizer: Optimizer) -> bool:
    return optimizer.run(**opt_params)


@pytest.fixture(name="forces")
def fixture_forces(
    calculator: Calculator,
    calculate: None,  # noqa: ARG001
) -> NDArray[np.floating]:
    return calculator.get_property("forces", allow_calculation=False)


@pytest.fixture(name="energy")
def fixture_energy(
    calculator: Calculator,
    calculate: None,  # noqa: ARG001
) -> float:
    return calculator.get_property("energy", allow_calculation=False)


@pytest.fixture(name="vasprun")
def fixture_vasprun(
    request: pytest.FixtureRequest, shared_datadir: Path
) -> Vasprun | None:
    marker = request.node.get_closest_marker("output_files")
    if marker and ("vasprun_xml" in marker.args or "all" in marker.args):
        return Vasprun(str(shared_datadir.joinpath("vasprun.xml")))
    return None


@pytest.fixture(name="converged")
def fixture_converged(vasprun: Vasprun | None) -> float:
    return False if vasprun is None else vasprun.converged


@pytest.fixture(name="calculation_outputs")
def fixture_calculation_outputs(
    *,
    energy: float,
    converged: bool,
    forces: list[float],
) -> CalculationOutputs:
    return CalculationOutputs(
        energy=energy,
        forces=forces,
        converged=converged,
    )


SCHEDULER_OUTPUTS = """
#######################
## Scheduler Outputs ##
#######################
"""

# SLURM Outputs: Start Time


@pytest.fixture(name="submit_time")
def fixture_submit_time() -> datetime:
    submit_time = datetime(  # noqa: DTZ001
        year=2022,
        month=7,
        day=29,
        hour=9,
        minute=48,
        second=14,
    )
    return submit_time


@pytest.fixture(name="slurm_wait_days", params=[0])
def fixture_slurm_wait_days(request: pytest.FixtureRequest) -> int:
    slurm_wait_days: int = request.param
    return slurm_wait_days


@pytest.fixture(name="slurm_wait_hours", params=[0])
def fixture_slurm_wait_hours(request: pytest.FixtureRequest) -> int:
    slurm_wait_hours: int = request.param
    return slurm_wait_hours


@pytest.fixture(name="slurm_wait_minutes", params=[0])
def fixture_slurm_wait_minutes(request: pytest.FixtureRequest) -> int:
    slurm_wait_minutes: int = request.param
    return slurm_wait_minutes


@pytest.fixture(name="slurm_wait_seconds", params=[1])
def fixture_slurm_wait_seconds(request: pytest.FixtureRequest) -> int:
    slurm_wait_seconds: int = request.param
    return slurm_wait_seconds


@pytest.fixture(name="idle")
def fixture_idle(
    slurm_wait_days: int,
    slurm_wait_hours: int,
    slurm_wait_minutes: int,
    slurm_wait_seconds: int,
) -> timedelta:
    return timedelta(
        days=slurm_wait_days,
        hours=slurm_wait_hours,
        minutes=slurm_wait_minutes,
        seconds=slurm_wait_seconds,
    )


@pytest.fixture(name="start_time")
def fixture_start_time(
    submit_time: datetime,
    idle: timedelta,
) -> datetime:
    return submit_time + idle


# SLURM Outputs: End Time


@pytest.fixture(name="slurm_days", params=[0])
def fixture_slurm_days(request: pytest.FixtureRequest) -> int:
    slurm_day: int = request.param
    return slurm_day


@pytest.fixture(name="slurm_hours", params=[0])
def fixture_slurm_hours(request: pytest.FixtureRequest) -> int:
    slurm_hours: int = request.param
    return slurm_hours


@pytest.fixture(name="slurm_minutes", params=[30])
def fixture_slurm_minutes(request: pytest.FixtureRequest) -> int:
    slurm_minutes: int = request.param
    return slurm_minutes


@pytest.fixture(name="slurm_seconds", params=[43])
def fixture_slurm_seconds(request: pytest.FixtureRequest) -> int:
    slurm_seconds: int = request.param
    return slurm_seconds


@pytest.fixture(name="elapsed")
def fixture_elapsed(
    slurm_days: int, slurm_hours: int, slurm_minutes: int, slurm_seconds: int
) -> timedelta:
    return timedelta(
        days=slurm_days,
        hours=slurm_hours,
        minutes=slurm_minutes,
        seconds=slurm_seconds,
    )


@pytest.fixture(name="end_time")
def fixture_end_time(
    start_time: datetime,
    elapsed: timedelta,
) -> datetime:
    return start_time + elapsed


@pytest.fixture(name="max_rss", params=[18049744.0])
def fixture_max_rss(request: pytest.FixtureRequest) -> int:
    max_rss: int = request.param

    return max_rss


@pytest.fixture(name="partition", params=[Partition.RAZI])
def fixture_partition(request: pytest.FixtureRequest) -> Partition:
    partition: Partition = request.param
    return partition


@pytest.fixture(name="job_stats")
def fixture_job_stats(
    max_rss: int,
    partition: Partition,
    start_time: datetime,
    end_time: datetime,
    submit_time: datetime,
    cores: int,
    nodes: int,
) -> dict[str, int | Partition | datetime]:
    return {
        "MaxRSS": max_rss,
        "Partition": partition,
        "Start": start_time,
        "End": end_time,
        "Submit": submit_time,
        "NCPUS": cores,
        "NNodes": nodes,
    }


@pytest.fixture(name="job_error", params=[None])
def fixture_job_error(request: pytest.FixtureRequest) -> bool:
    job_error: bool = request.param
    return job_error


@pytest.fixture(name="slurm_job_id", params=[123456789])
def fixture_slurm_job_id(request: pytest.FixtureRequest) -> str:
    slurm_job_id: int = request.param
    return slurm_job_id


@pytest.fixture(name="slurm_output_filename")
def fixture_slurm_output_filename(slurm_job_id: int) -> str:
    return f"slurm-{slurm_job_id}.out"


@pytest.fixture(name="write_job_stats_file")
def fixture_write_job_stats_file(
    task_directory: Path, shared_datadir: Path
) -> None:
    shutil.copy(
        Path(shared_datadir, SETTINGS.SCHEDULER_STATS_FILE),
        task_directory,
    )


@pytest.fixture(name="write_slurm_output_file")
def fixture_write_slurm_output_file(
    task_directory: Path, slurm_output_filename: str
) -> Generator[Path, None, None]:
    output_file = Path(task_directory, slurm_output_filename)
    output_file.touch()
    yield output_file
    output_file.unlink()


@pytest.fixture(name="scheduler_outputs")
def fixture_scheduler_outputs(
    elapsed: timedelta,
    idle: timedelta,
    slurm_job_id: int,
    job_error: int,
    max_rss: int,
    partition: Partition,
) -> SchedulerOutputs:
    return SchedulerOutputs(
        elapsed=elapsed,
        error=job_error,
        idle_time=idle,
        job_id=slurm_job_id,
        max_rss=max_rss,
        partition=partition,
    )


COMPLETED_OUTPUTS = """
###########################
###  Completed Outputs  ###
###########################
"""


@pytest.fixture(name="task")
def fixture_task(
    task_metadata: TaskMetadata,
    task_inputs: TaskInputs,
    task_outputs: TaskOutputs,
) -> Task:
    return Task(
        task_metadata=task_metadata,
        task_inputs=task_inputs,
        task_outputs=task_outputs,
    )


@pytest.fixture(name="calculation")
def fixture_calculation(
    task: Task,
    calculation_inputs: CalculationInputs,
    scheduler_inputs: SchedulerInputs,
) -> Calculation:
    calculation: Calculation = Calculation(
        calculation_inputs=calculation_inputs,
        scheduler_inputs=scheduler_inputs,
        **task.model_dump(),
    )
    task_class = calculation.__class__.__name__.lower()
    calculation.task_metadata.task_class = task_class
    return calculation


@pytest.fixture(name="write_task_doc")
def fixture_write_task_doc(
    task: Task, task_directory: Path
) -> Generator[TextIOWrapper, None, None]:
    task_json = Path(task_directory, SETTINGS.TASK_FILE)
    with task_json.open(mode="w", encoding="utf-8") as file:
        json.dump(task.model_dump(), file, indent=4)

    with task_json.open(mode="r", encoding="utf-8") as file:
        yield file


WORKFLOW_CONSTRUCTS = """
##############################
###  Workflow Constructs   ###
##############################
"""


@pytest.fixture(name="study")
def fixture_study(study_id: str, study_group_id: str, task: Task) -> Study:
    study: Study = Study(
        tasks=[task],
        study_id=study_id,
        study_group_id=study_group_id,
    )
    return study


@pytest.fixture(
    name="study_group",
)
def fixture_study_group(study: Study, study_group_id: str) -> StudyGroup:
    study_group = StudyGroup(studies=[study], study_group_id=study_group_id)
    return study_group


DIRECTORY_POPULATION = """
##############################
###  Directory Population  ###
##############################
"""


@pytest.fixture(name="study_group_directory")
def fixture_study_group_directory(
    tmp_path: Path, study_group_id: str | UUID4
) -> Generator[Path, None, None]:
    study_group_directory = Path(tmp_path, str(study_group_id))
    study_group_directory.mkdir(exist_ok=True)
    yield study_group_directory
    shutil.rmtree(study_group_directory)


@pytest.fixture(name="study_directory")
def fixture_study_directory(
    tmp_path: Path,
    study_group_id: str | UUID4,
    study_id: str | UUID4,
) -> Generator[Path, None, None]:
    study_directory = Path(tmp_path, str(study_group_id), str(study_id))
    study_directory.mkdir(parents=True, exist_ok=True)
    yield study_directory
    shutil.rmtree(study_directory)


@pytest.fixture(name="task_group_directory")
def fixture_task_group_directory(
    tmp_path: Path,
    study_group_id: str | UUID4,
    study_id: str | UUID4,
    task_group_id: str | UUID4,
) -> Generator[Path, None, None]:
    task_group_directory = Path(
        tmp_path, str(study_group_id), str(study_id), str(task_group_id)
    )
    task_group_directory.mkdir(parents=True, exist_ok=True)
    yield task_group_directory
    shutil.rmtree(task_group_directory)


@pytest.fixture(name="task_directory")
def fixture_task_directory(
    tmp_path: Path,
    study_group_id: str | UUID4,
    study_id: str | UUID4,
    task_group_id: str | UUID4,
    task_id: str | UUID4,
) -> Generator[Path, None, None]:
    task_directory = Path(
        tmp_path,
        str(study_group_id),
        str(study_id),
        str(task_group_id),
        str(task_id),
    )
    task_directory.mkdir(parents=True, exist_ok=True)
    yield task_directory
    shutil.rmtree(task_directory)


@pytest.fixture(name="write_task_inputs")
def fixture_write_task_inputs(task: Task, task_directory: Path) -> list[Path]:
    return task.write_inputs(task_directory)


@pytest.fixture(name="write_calculation_inputs")
def fixture_write_calculation_inputs(
    calculation: Calculation, task_directory: Path
) -> list[Path]:
    return calculation.write_inputs(task_directory)


@pytest.fixture(name="write_calculation_outputs")
def fixture_write_calculation_outputs(
    output_atoms: Atoms,
    task_directory: Path,
    calculate: None,  # noqa: ARG001
) -> Generator[None, None, None]:
    output_atoms_file = Path(task_directory, SETTINGS.OUTPUT_ATOMS_FILE)
    output_atoms.write(output_atoms_file)
    yield None
    output_atoms_file.unlink()


@pytest.fixture(name="string", params=["Test", "2", "@", "\n", " ", ""])
def fixture_string(request: pytest.FixtureRequest) -> str:
    string: str = request.param
    return string


@pytest.fixture(
    name="non_string", params=[[], {}, True, False, None, set(), 0.0, 0]
)
def fixture_non_string(request: pytest.FixtureRequest) -> Any:
    return request.param
