import json
from pathlib import Path

from pydantic import ValidationError
import pytest

from autojob.tasks.scan import BondScan
from autojob.tasks.scan import BondScanInputs


class TestBondScanInputs:
    @staticmethod
    @pytest.mark.parametrize("bond_scan_param", [(0, 1), (0, 1, [])])
    def test_should_validate_tuples_as_bond_scan_params(
        bond_scan_param: tuple,
    ) -> None:
        BondScanInputs(bond_scan_params=[bond_scan_param])

    @staticmethod
    @pytest.mark.parametrize("bond_scan_param", [(0,), (0, "a")])
    def test_should_raise_validation_error_for_invalid_bond_scan_inputs(
        bond_scan_param: tuple,
    ) -> None:
        with pytest.raises(ValidationError):
            BondScanInputs(bond_scan_params=[bond_scan_param])


class TestWriteInputs:
    @staticmethod
    def test_should_write_bond_scan_inputs_to_inputs_json(
        task_directory: Path, bond_scan: BondScan
    ) -> None:
        bond_scan.write_inputs_json(task_directory)

        inputs_json = bond_scan.write_inputs_json(task_directory)
        bond_scan_inputs = bond_scan.bond_scan_inputs.model_dump()
        with inputs_json.open(mode="r", encoding="utf-8") as file:
            data = json.load(file)

        loaded_bond_scan_inputs = BondScanInputs.model_validate(
            data.get("bond_scan_inputs")
        )
        assert loaded_bond_scan_inputs
        assert loaded_bond_scan_inputs.model_dump() == bond_scan_inputs
