from pathlib import Path
import shutil

from ase.atoms import Atoms
from ase.calculators.calculator import Calculator
from ase.calculators.vasp.vasp import Vasp
import ase.io
import pytest

from autojob.harvest.harvesters.bader import load_bader_results
from autojob.harvest.harvesters.bader import parse_acf
from autojob.harvest.harvesters.ddec6 import DDEC_NET_ATOMIC_CHARGE_FILE
from autojob.harvest.harvesters.ddec6 import get_ddec6_index_map

BADER_CHARGE_JOB_ID = "jJPQg3Puwz"


class TestParseACF:
    @staticmethod
    def test_should_parse_acf(datadir: Path) -> None:
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0

    @staticmethod
    def test_should_parse_acf_without_poscar(datadir: Path) -> None:
        Path(datadir, "POSCAR").unlink()
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0

    @staticmethod
    def test_should_parse_acf_with_poscar_but_without_potcar(
        datadir: Path,
    ) -> None:
        Path(datadir, "POTCAR").unlink()
        data = parse_acf(datadir)
        assert data["x"][0] == 7.272834
        assert data["y"][0] == 2.005745
        assert data["z"][0] == 10.000283
        assert data["charge"][0] == 6.041753
        assert data["min_dist"][0] == 0.89003
        assert data["atomic_volume"][0] == 160.280717
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0


class TestLoadBaderData:
    @staticmethod
    @pytest.mark.filterwarnings("ignore:.*Could not find CHGCAR!*:UserWarning")
    @pytest.mark.xfail(reason="No valid vasprun.xml present")
    def test_should_use_internal_parser_if_pymatgen_routine_fails(
        datadir: Path,
    ) -> None:
        data = load_bader_results(src=datadir)
        assert data["charge"][0] == 10.295178
        assert data["min_dist"][0] == 0.871639
        assert data["atomic_volume"][0] == 60.443714
        assert data["charge_transfer"][0] == 0.7048220000000001
        assert data["vacuum_charge"] == 0.0
        assert data["vacuum_volume"] == 0.0


class TestGetDDEC6IndexMap:
    @staticmethod
    @pytest.fixture(name="input_atoms")
    def fixture_input_atoms(datadir: Path) -> Atoms:
        structure = ase.io.read(Path(datadir, "POSCAR"))
        structure = structure[0] if isinstance(structure, list) else structure
        return structure

    @staticmethod
    @pytest.fixture(name="atoms_calculator")
    def fixture_atoms_calculator() -> Calculator:
        return Vasp()

    @staticmethod
    def test_should_return_permutation_of_atomic_indices(
        datadir: Path,
        input_atoms: Atoms,
        task_directory: Path,
        write_task_inputs: list[Path],  # noqa: ARG004
    ) -> None:
        shutil.copy(Path(datadir, "ACF.dat"), task_directory)
        shutil.copy(Path(datadir, "POTCAR"), task_directory)
        shutil.copy(
            Path(datadir, DDEC_NET_ATOMIC_CHARGE_FILE),
            task_directory,
        )
        ddec6_map = get_ddec6_index_map(src=task_directory)
        indices = [a.index for a in input_atoms]
        assert len(ddec6_map) == len(indices)
        assert all(i in ddec6_map for i in indices)

    @staticmethod
    def test_should_warn_when_atomic_positions_are_mismatched(
        datadir: Path,
        input_atoms: Atoms,
        task_directory: Path,
        atoms_filename: str,
        write_task_inputs: list[Path],  # noqa: ARG004
    ) -> None:
        shutil.copy(Path(datadir, "ACF.dat"), task_directory)
        shutil.copy(Path(datadir, "POTCAR"), task_directory)
        shutil.copy(
            Path(datadir, DDEC_NET_ATOMIC_CHARGE_FILE),
            task_directory,
        )
        input_atoms[0].position = [0.0, 0.0, 0.0]
        input_atoms.write(Path(task_directory, atoms_filename))
        with pytest.warns(
            UserWarning, match="Atomic displacement exceeds tolerance.*"
        ):
            _ = get_ddec6_index_map(src=task_directory)
