from collections.abc import Generator
from pathlib import Path
import tomllib

from click.testing import CliRunner
import pytest

from autojob import SETTINGS
from autojob._cli.config_gen import create_config_file
from autojob._cli.config_gen import main as config_gen
from autojob._cli.config_gen import textify_config
from autojob.settings import AutojobSettings


class TestTextifyConfig:
    @staticmethod
    def test_should_sort_settings_key_if_sort_is_true() -> None:
        lines = textify_config(SETTINGS.model_dump(), sort=True)
        first_letters_alphabetical = []
        for i, key in enumerate(lines):
            if i == 0:
                continue
            first_letters_alphabetical.append(key[0] >= lines[i - 1][0])
        assert all(first_letters_alphabetical)


class TestCreateConfigFile:
    @staticmethod
    @pytest.fixture(name="create_config_file")
    def fixture_create_config_file(
        monkeypatch: pytest.MonkeyPatch,
        tmp_path_factory: pytest.TempPathFactory,
    ) -> Generator[Path, None, None]:
        tmp_path = tmp_path_factory.mktemp("tmp_path")
        with monkeypatch.context() as m:
            m.chdir(tmp_path)
            config_file = create_config_file()
            yield config_file
            config_file.unlink()

    @staticmethod
    def test_should_create_configuration_file_in_cwd(
        create_config_file: Path,
    ) -> None:
        assert create_config_file.exists()

    @staticmethod
    def test_should_create_configuration_file_with_autojob_defaults(
        create_config_file: Path,
    ) -> None:
        with create_config_file.open(mode="rb") as file:
            data = tomllib.load(file)
        settings = SETTINGS.model_dump()
        read_settings = AutojobSettings(**data["tool"]["autojob"]).model_dump()
        assert settings == read_settings


class TestCLI:
    @staticmethod
    def test_should_create_configuration_file(
        monkeypatch: pytest.MonkeyPatch,
        tmp_path_factory: pytest.TempPathFactory,
    ) -> None:
        tmp_path = tmp_path_factory.mktemp("tmp_path")
        with monkeypatch.context() as m:
            m.chdir(tmp_path)
            runner = CliRunner()
            result = runner.invoke(config_gen)
        assert result.exit_code == 0
        assert "Configuration file created:" in result.output
