from pathlib import Path
from typing import TYPE_CHECKING
from typing import Literal

import pytest

from autojob import SETTINGS
from autojob.harvest.archive import archive
from autojob.harvest.archive import archive_json
from autojob.harvest.harvest import harvest

if TYPE_CHECKING:
    from autojob.bases.task_base import TaskBase


pytestmark = pytest.mark.usefixtures(
    "write_job_stats_file",
    "write_calculation_inputs",
    "calculate",
    "write_calculation_outputs",
)


@pytest.fixture(name="to_calculate")
def fixture_to_calculate() -> list[str]:
    return ["energy", "forces"]


@pytest.fixture(name="strictness")
def fixture_strictness() -> Literal["strict", "relaxed", "atomic"]:
    return "atomic"


@pytest.fixture(name="task_script")
def fixture_task_script() -> str:
    return "gaussian.sh"


@pytest.fixture(name="harvested")
def fixture_harvested(
    strictness: Literal["strict", "relaxed", "atomic"],
    task_directory: Path,
    calculate: None,  # noqa: ARG001
) -> "list[TaskBase]":
    return harvest(task_directory, strictness=strictness)


class TestHarvest:
    @staticmethod
    def test_should_return_harvested(harvested: "list[TaskBase]") -> None:
        assert harvested

    @staticmethod
    def test_should_harvest_from_archive_in_templated_directory(
        harvested: "list[TaskBase]", task_directory: Path
    ) -> None:
        archive_json(
            harvested,
            Path(task_directory, SETTINGS.ARCHIVE_FILE),
        )
        renamed = Path(task_directory.parent, "name")
        task_directory.rename(renamed)
        assert harvest(renamed, use_cache=True)


class TestArchive:
    @staticmethod
    def test_should_archive_harvested(
        harvested: "list[TaskBase]", datadir: Path
    ) -> None:
        filename = str(Path(datadir, "test"))
        archive(
            filename=filename,
            archive_mode="both",
            harvested=harvested,
        )
        assert filename
